#!/usr/local/bin/perl
# 
# $Header: emagent/sysman/admin/scripts/jobutil/runSQLScript.pl /main/6 2012/03/26 02:09:13 mmootha Exp $
#
# runSQLScript.pl
# 
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      runSQLScript.pl - Runs a SQL against a database
#
#    DESCRIPTION
#      Script to run arbitrary SQL script against SQL*Plus prompt
#
#    NOTES
#      Accepts the ORACLE_HOME and ORACLE_SID on the command line as mandatory arguments
#      Accepts the SQL*Plus login information on the input stream
#      Requires JobUtil.pm for Input Stream parsing
#      Requires JobDiagUtil.pm for debug logging
#
#    MODIFIED   (MM/DD/YY)
#    mmootha     03/13/12 - Pluggable DB job support
#    rdabbott    10/26/11 - fix 12653270: keep original path for sql script
#    rdabbott    06/20/11 - fix 3243419: ignore duplicate nolog
#    rdabbott    03/22/11 - rm notusenolog
#    lsatyapr    01/07/11 - Handle exits correctly
#    amathur     09/06/10 - perl script to run sql against a database
#    amathur     09/06/10 - Creation
# 
use strict;
use JobUtil;
use JobDiagUtil;

# constant for the delimiter for property name and value
use constant DELIMITER    => "=";

# constant for the end of stream for the input properties stream
use constant EOS          => "__EM_JOB_INPUT_STREAM_END__";

# constant for the sql username
use constant SQL_USER     => "__EM_JOB_SQL_USER__";

# constant for the sql password
use constant SQL_PASSWORD => "__EM_JOB_SQL_PASSWORD__";

# constant for the sql dbrole
use constant SQL_DBROLE   => "__EM_JOB_SQL_DBROLE__";

# constant for the sql machine name
use constant SQL_MACHINENAME   => "__EM_JOB_SQL_MACHINENAME__";

# constant for the sql port
use constant SQL_PORT   => "__EM_JOB_SQL_PORT__";

# constant for the sql service
use constant SQL_SERVICE   => "__EM_JOB_SQL_SERVICENAME__";

# constant for the sql prefConnString
use constant SQL_PREFCONNSTR => "__EM_JOB_SQL_PREFCONNSTR__";

# initialize global variables
my $nolog  = '/nolog';

# Bug 3243419: ignore duplicate nolog
$nolog = '' if ( grep /$nolog/i, @ARGV );
# print "nolog=$nolog\n";

my $username = "";
my $password = "";
my $dbrole = "";

#  serviceName based connection props for db job
my $machinename = "";
my $port = "";
my $service = "";

# prefConnString for db job
my $prefConnString = "";

my $connectString = "";

my $oracleHome = "$ARGV[0]";
my $oracleSid  = "$ARGV[1]";

shift;
shift;


# setup the environment required by SQL*Plus to execute
sub setupEnv
{
    $ENV{ORACLE_HOME}       = $oracleHome;
    $ENV{ORACLE_SID}        = $oracleSid;

    $ENV{LD_LIBRARY_PATH}    = "$ENV{ORACLE_HOME}/lib";
    $ENV{LD_LIBRARY_PATH_64} = "$ENV{ORACLE_HOME}/lib";
    $ENV{SHLIB_PATH}         = "$ENV{ORACLE_HOME}/lib";
    $ENV{LIBPATH}            = "$ENV{ORACLE_HOME}/lib";
    $ENV{PATH}               = "$ENV{ORACLE_HOME}/bin:$ENV{PATH}";

    $ENV{ORA_NLS} = "";
    $ENV{ORA_NLS32} = "";
    $ENV{ORA_NLS33} = "";

    JobDiagUtil::logFine("ORACLE_HOME : "        . $ENV{ORACLE_HOME});
    JobDiagUtil::logFine("ORACLE_SID  : "        . $ENV{ORACLE_SID});
    JobDiagUtil::logFine("LD_LIBRARY_PATH : "    . $ENV{LD_LIBRARY_PATH});
    JobDiagUtil::logFine("LD_LIBRARY_PATH_64 : " . $ENV{LD_LIBRARY_PATH_64});
    JobDiagUtil::logFine("SHLIB_PATH : "         . $ENV{SHLIB_PATH});
    JobDiagUtil::logFine("LIBPATH : "            . $ENV{LIBPATH});
    JobDiagUtil::logFine("PATH : "               . $ENV{PATH});
}

# get the parameters from the input stream
sub getParams
{
    # get the parameters from STDIN
    my %paramsMap  = JobUtil::getInputStreamParamsMap(DELIMITER, EOS);

    # separate out
    $username      = $paramsMap{+SQL_USER};
    $password      = $paramsMap{+SQL_PASSWORD};
    $dbrole        = $paramsMap{+SQL_DBROLE};

    $machinename   = $paramsMap{+SQL_MACHINENAME};
    $port          = $paramsMap{+SQL_PORT};
    $service       = $paramsMap{+SQL_SERVICE};
    $prefConnString= $paramsMap{+SQL_PREFCONNSTR};

    # diagnostic logging of the username, role , machinename, port and service
    JobDiagUtil::logFine("Username : $username : DBRole : $dbrole :PrefConnString :$prefConnString : MachineName : $machinename: Port : $port : DBService : $service");

    # get the connect string    
    if ($prefConnString)
    {
      $ENV{TWO_TASK} = $prefConnString;
      JobDiagUtil::logFine("TWO_TASK Env Var with PrefConnString: " . $ENV{TWO_TASK});
    }
    elsif ($machinename && $port && $service)
    {
      $ENV{TWO_TASK} = "$machinename:$port/$service";
      JobDiagUtil::logFine("TWO_TASK Env Var with ServiceName: " . $ENV{TWO_TASK});
    }
    $connectString = JobUtil::formSQLPlusConnectString($username, $password, $dbrole);
}

# execute SQL*Plus from the command line and spool the connection string and rest of the input stream to it
sub executeSQLPlus
{
    my $command = "$ENV{ORACLE_HOME}/bin/sqlplus @ARGV $nolog";
    JobDiagUtil::logInfo("Opening piped command : $command");

    # open piped command
    my $PIPED_CMD = 0;
    open PIPED_CMD, " | $command" or die("Unable to launch $command : $! \n");
    
    # write prescript
    print PIPED_CMD "WHENEVER SQLERROR EXIT SQLCODE;\n";
    print PIPED_CMD $connectString."\n";
    print PIPED_CMD "WHENEVER SQLERROR CONTINUE;\n";

    # write what ever is coming in the STDIN to this pipe
    while (<STDIN>)
    {
        print PIPED_CMD $_;
    }

    # close the pipe
    if (! close PIPED_CMD)
    {
        my $syscode = $? & 0xFFFF;
        if ( $syscode )
        {
            # The signal or core dump is in the low byte
            # program code is in the high byte
            my $sig = 0xFF & $syscode;
            my $ret = $syscode >> 8;
        
            if ( $sig )
            {
                # if there's a signal, return a negative number
                print "\nSignal from SQLPLUS: $sig\n";
                exit (- $sig);
            }
            else
            {
                # return the positive error code from sqlplus
                print "\nError from SQLPLUS: $ret\n";
                exit $ret;
            }
        }
    }
}

# logic flow for this script
sub main
{
    # setup the executing environment of the script and SQL*Plus
    setupEnv;
    
    # get the parameters from the Input Stream
    getParams;

    # execute SQL*Plus with the remaining Input Stream
    executeSQLPlus;
}

# While writing to a pipe, trap SIGPIPE. Otherwise, if a pipe is opened to a
# command that does not exist, the open() may succeed as it only reflects the
# fork()'s success, but the output would fail. Perl cannot know whether the
# command worked because that command is actually running in a separate process
# whose exec() might have failed. Therefore, while readers of bogus commands
# return just a quick end of file, writers to bogus command will trigger a
# signal.

$SIG{PIPE} = 'IGNORE';

# call the main sub routine
main;

# return success
exit 0;
