#  $Header: emagent/sysman/admin/scripts/parse-log.pl /st_emagent_10.2.0.1.0/1 2009/03/10 05:32:11 pnayak Exp $
# 
# Copyright (c) 2001, 2003, Oracle Corporation.  All rights reserved.  
#
#    NAME
#      parse-log.pl - <one-line expansion of the name>
#
#    DESCRIPTION
#      This script is used in EMD to parse log files for critical and 
#      warning patterns. The script holds the last line number searched
#      for each file in a state file for each time the script is run. The
#      next run of the script starts from the next line. The state file name 
#      is read from the environment variable $EM_STATE_FILE, which must
#      be set for the script to run.
#
#      The input of this script is :
#
#          [$logFileName|||$pattern|||$severity]* 
#
#          where $severity is 1 for critical, 2 for warning, 0 for ignore
#
#      The output of this script is :
#
#          em_result=$logFileKey|$severity|$message
#
#          where $logFileKey is 
#                <logFileName>_<startLineNum>_<endLineNum>_<timestamp>
#                $severity is as defined above
#                $message is a concatenated string of critical and warning 
#                         messages in that order. Max length = 1024.
#
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    ipoddar    09/04/03 - ipoddar_host_mgmt_enhancements 
#    ipoddar    09/04/03 - some mods 
#    ipoddar    09/03/03 - minor changes 
#    ipoddar    07/27/03 - next
#    ipoddar    07/23/03 - Creation

#-Packages--------------------------------------------------------------------
use strict;
use warnings;
use Getopt::Long;
use Time::gmtime;
require "emd_common.pl";

#-Global-Variables------------------------------------------------------------
our %Flag   = ();		# Holds global toggles and flags.
our $itemProperty = $ARGV[0];
our %logFilePatterns = ();

#-Environment-----------------------------------------------------------------
$ENV{PATH} = "/bin:/usr/bin:/usr/sbin";

#-Methods--------------------------------------------------------------

sub initLogFilePatterns {

    my ($rest, $logFilePatterns) = @_;

    while ($rest) {
        (my $logFile, my $pattern, my $severity, $rest) = 
            split /\|\|\|/, $rest, 4 ;
        my $sevKey = "critical";
        if ($severity == "1") {
            $sevKey = "critical";
        } elsif ($severity == "2") {
            $sevKey = "warning";
        } elsif ($severity == "0") {
            $sevKey = "ignore";
        }
        if (not $logFilePatterns->{$logFile}{$sevKey}) {
            $logFilePatterns->{$logFile}{$sevKey} = "$pattern";
        } else {
            $logFilePatterns->{$logFile}{$sevKey} .= "|$pattern";
        }
    }
}

sub getStateFileName {
    die "The environment variable EM_AGENT_STATE needs to be set in order to run.\n"
        unless exists $ENV{EM_AGENT_STATE};

    die "The environment variable EM_TARGET_GUID needs to be set in order to run.\n"
        unless exists $ENV{EM_TARGET_GUID};

    return "$ENV{EM_AGENT_STATE}/$ENV{EM_TARGET_GUID}-parse-log-state-file";
}

sub initFlags {

    my $logFilePatterns = shift;
 
    $ENV{EM_STATE_FILE} = getStateFileName();

    # Read the state line if we have one.
    if (open STATE, "< $ENV{EM_STATE_FILE}") {
        while (my $logFile = <STATE>) {
            chomp($logFile);
            if ($logFile and $logFilePatterns->{$logFile}) {
                $Flag{$logFile}{line} = <STATE>;
                $Flag{$logFile}{timestamp} = <STATE>; 
                $Flag{$logFile}{level} = <STATE>;
                chomp ($Flag{$logFile}{line}, 
                       $Flag{$logFile}{level}, 
                       $Flag{$logFile}{timestamp});
                EMD_PERL_DEBUG("Log File: $logFile");
                EMD_PERL_DEBUG(" State line was: $Flag{$logFile}{line}");
                EMD_PERL_DEBUG(" Time: $Flag{$logFile}{timestamp},");
                EMD_PERL_DEBUG(" Level: $Flag{$logFile}{level}\n");
            }
        }
        close STATE;
    }
}

sub saveFlags {
    # Save the state file.
    open STATE, "> $ENV{EM_STATE_FILE}" or die "Unable to open state file: $ENV{EM_STATE_FILE}. [$!]\n";

    foreach my $logFile (keys %Flag) {
        chomp($logFile);
        if ($logFile) {
            print STATE "$logFile\n";
            print STATE "$Flag{$logFile}{line}\n";
            print STATE "$Flag{$logFile}{timestamp}\n";
            print STATE "$Flag{$logFile}{level}\n";
        }
    }
    close STATE;
}

sub grepLogFile {

    # The file is the first option.
    my ($file, $patternref) = @_;

    my %pattern = %{$patternref};		

    # Debug info.
    EMD_PERL_DEBUG("Log File: $file \n");
    for (keys %pattern) {
        EMD_PERL_DEBUG("Pattern[$_]: /$pattern{$_}/\n");
    }

    my $beginline = 0;
    my $linenumber = 0;
    if ($Flag{$file}{line}) {
        $beginline = $Flag{$file}{line};
    }
    EMD_PERL_DEBUG("Begin line is: $beginline\n");

    my $timestamp = gmctime();
    $timestamp =~ s/ /\_/g;
    $timestamp =~ s/\_\_/\_/g;

    my $key = "$file\_$timestamp\_0\_0";
    $key =~ s/ /\_/g;

    my $fileOpen = open DATA, "< $file";
    # If the file could not be opened at an earlier attempt
    if (($beginline == -1) && (!$fileOpen)) {
        EMD_PERL_DEBUG("File could not be opened earlier\n");
        return;
    } elsif (($beginline == -1) && $fileOpen) {
        EMD_PERL_DEBUG("File could not be opened earlier but can now be opened\n");
        $beginline = 0;
    } elsif (!$fileOpen) { 
        print "em_result=$key|Critical|Cannot open file: $file [$!] \n"; 
        $Flag{$file}{line} = -1;
        $Flag{$file}{timestamp} = $timestamp;
        $Flag{$file}{level} = 1;
        return; 
    }

    my %lines = (
                 critical => [],
                 warning =>  []
                 );			# Hash of found lines.

    # And now find the file we need to search.

  LINE: while (my $line = <DATA>) {

      if (++$linenumber == $beginline) {
          EMD_PERL_DEBUG("Clearing. State line found.\n");
          $lines{critical} = [];
          $lines{warning}  = [];
          next LINE;
      }

      chomp($line);

      # Ignore the ignored lines.
      if ($pattern{ignore} and $line =~ /$pattern{ignore}/) {
          EMD_PERL_DEBUG("Ignoring:    $line\n");
          next LINE;
      }

      # Is there a critical, and is this it?
      if ($pattern{critical} and $line =~ /$pattern{critical}/) {
          EMD_PERL_DEBUG("Critical found: $line\n");
          push @{$lines{critical}}, $line;
          next LINE;
      }

      # Is there a warning, and is this it?
      if ($pattern{warning} and $line =~ /$pattern{warning}/) {
          EMD_PERL_DEBUG("Warning found:  $line\n");
          push @{$lines{warning}}, $line;
          next LINE;
      }   
  }

    # Debug info.
    for (keys %lines) {
        EMD_PERL_DEBUG("Lines[$_]: ", scalar @{$lines{$_}}, "\n");
    }

    $key = "$file\_$timestamp\_$beginline\_$linenumber";
    $key =~ s/ /\_/g;

    # Print the em_result.
    if (@{$lines{critical}}) {
        my $msg = "Critical messages: ";
        $msg .= join "+++ ", @{$lines{critical}};
        $msg .= ", Warning messages: " if (@{$lines{warning}});
        $msg .= join "+++ ", @{$lines{warning}} if (@{$lines{warning}});
        $msg = substr $msg, 0, 1024;
        print "em_result=$key|Critical|$msg\n";
        $Flag{$file}{level} = 1;
        $Flag{$file}{timestamp} = $timestamp;
    } elsif (@{$lines{warning}}) {
        my $msg = "Warning messages: ";
        $msg .= join "+++ ", @{$lines{warning}}; 
        $msg = substr $msg, 0, 1024;
        print "em_result=$key|Warning|$msg\n";
        $Flag{$file}{level} = 2;
        $Flag{$file}{timestamp} = $timestamp;
    }
    $Flag{$file}{line} = $linenumber;

    close DATA;
}

#-initialization--------------------------------------------------------------

if (!$itemProperty) {
    $itemProperty = "";
}
EMD_PERL_DEBUG("Input: $itemProperty\n");

initLogFilePatterns $itemProperty, \%logFilePatterns;

initFlags(\%logFilePatterns);

#-Main-()---------------------------------------------------------------------

foreach my $logFile (keys %logFilePatterns) {
    grepLogFile ($logFile, $logFilePatterns{$logFile}); 
}

saveFlags();

exit;


