#!/usr/bin/perl
#
# Copyright (c) 2003, 2004, Oracle Corporation.  All rights reserved.  
#
#    NAME
#      procResUtil.pl 
#
#    DESCRIPTION
#      Metric script that monitors CPU and resident memory utilization of processes 
#       
#
#    OUTPUT:
#
#      One or more lines of:
#      em_result=[owner]:<process>|<instance count>|<PIDs>|<total CPU%>|<max res. mem. PID>|<max resident memory in MB>
#
#    NOTES
#      Supported platforms: Solaris, HP-UX and Linux. 
#      
#      Usage: procResUtil.pl <criteria> 
#         where: 
#            <criteria> : value of USER property (ItemProperty) monRules, which  
#                         is set in the host collection file.   
#                         
#                         The value is a string in the form:
#                         <process_name>|||<owner>[|||...]
#                         
#
#    MODIFIED    (MM/DD/YY)
#      rmenon     02/23/04 - rmenon_host_mgmt_checkin 
#      lnhan      08/05/03 - Creation 


use strict;
use File::Basename;
require "emd_common.pl";

# check the number of parameters
if ( $#ARGV < 0 ) {
   print "Insufficient parameters. Note: This metric does not support real-time collection.\n";
   EMD_PERL_DEBUG("ERROR: Insufficient parameters.\nUsage: $0 <criteria>\nAbort.");
   exit -1;
}

my $monProcStr = shift @ARGV;   # string containing process selection criteria  
EMD_PERL_DEBUG("monProc = [$monProcStr]\n");
my $WARN = 2;                   # value for a warning condition


# ------------------------------------------------------------------------
# determine OS type and run ps get the list of processes and their 
# characteristics. 
# ------------------------------------------------------------------------

my $os;                         # operating system  
my $pgSize;                     # page size in bytes on HP-UX

if ( ! chomp ($os = `uname -s`) )  { 
   EMD_PERL_DEBUG("Failed to run the uname command. Abort."); 
   exit -1;
}

my @psList = ();                # list of processes

SWITCH: {
   $os eq "SunOS" && do {
          chomp (my $ver = `uname -r`);
          if ( $ver !~ /==4./ ) {
             # OS is Solaris
             @psList = `/usr/bin/ps -eo "pid user pcpu rss args"`;
             last SWITCH;
          }
       };
   $os eq "HP-UX" && do {
          # OS is HP-UX
          # Enable XPG4 environment for ps
          $ENV{UNIX95} = "XPG4";
          @psList = `/usr/bin/ps -eo "pid user pcpu sz args"`;
          $pgSize=`getconf SC_PAGE_SIZE`; 

          if ( ! defined $pgSize ) {
             EMD_PERL_DEBUG("Failed to run the getconf command. Abort.");
             exit -1;
          }

          last SWITCH;
       };
   $os eq "Linux" && do {
          # OS is Linux
          @psList = `/bin/ps -eo "pid user pcpu rss args"`;
          last SWITCH;
       };
   EMD_PERL_DEBUG("Unsupported Operating System"); 
   exit -1;
}

# remove ps header line
shift @psList;

EMD_PERL_DEBUG("monProc = [$monProcStr]\n");


# ------------------------------------------------------------------------
# validate rules for monitoring processes and combine the rules (fields) 
# for each process into an element of array monProcRules. 
# ------------------------------------------------------------------------

my @monProcs = split(/\|\|\|/,$monProcStr);
my @monProcRules;
my $numFields = 2;                         # predefined number of fields 

for (my $i=0; $i <= $#monProcs; $i += $numFields) {
   my $soughtProc    = $monProcs[$i];
   my $soughtUser    = $monProcs[$i + 1];

   if ( $soughtProc eq ""  ) {
      EMD_PERL_DEBUG("ERROR: Process field must not be empty. Abort");
      exit -1;
   }
   
   my @monProcRec;
   @monProcRec = ($soughtProc,$soughtUser); 
   @monProcRec = join '|||', @monProcRec;
   push @monProcRules, @monProcRec;

}   


# ------------------------------------------------------------------------
# screen the processes in the process list based on monitoring criteria. 
# compute and keep track of total CPU percentage and max resident memory 
# used by each user-specified process.  also keep track of the PIDs. 
# ------------------------------------------------------------------------

foreach my $rule ( @monProcRules ) {
   my ($soughtProc, $soughtUser) = split(/\|\|\|/, $rule);

   my $instanceCount = 0;                  # instance counter
   my $pidStr        = "";                 # string that holds PIDs
   my $maxMem        = 0;                  # max resident memory by a process instance 
   my $maxMemPid     = "";                 # PID of the process instance with max resident
                                           # memory usage  
   my $totPcntCpu    = 0;                  # running total CPU percentage 

   # debuging
   EMD_PERL_DEBUG("soughtproc= [$soughtProc], soughtuser= [$soughtUser]");

   my $command;
   my $soughtProcAndUser;

   foreach my $psLine ( @psList ) {

      # remove any leading space
      $psLine =~ s/^\s+//;

      my ($pid, $user, $pcntCpu, $resMem, @cmdArgs);
      my $resMemPg;                        # resident memory in pages (HP-UX)

      if ( $os eq "HP-UX" ) {
         ($pid, $user, $pcntCpu, $resMemPg, @cmdArgs) = split(/\s+/,$psLine);
	 $resMem = ($resMemPg * ($pgSize/1024));
      }
      else {
         ($pid, $user, $pcntCpu, $resMem, @cmdArgs) = split(/\s+/,$psLine);
      } 

      $command = $cmdArgs[0];
      $command = basename($command," ");

      if ( $command =~ /^$soughtProc$/ ) {
         # found a matching command

         if ( ( $soughtUser eq "" ) or ( $soughtUser eq $user ) ) {
            # username not specified OR username matches user in ps output line
    
            # compute the running total of percentage CPU 
            $totPcntCpu += $pcntCpu;  

            # keep track of max resident memory and the PID of this process  
            if ( $resMem > $maxMem ) {
               $maxMem    = $resMem;
               $maxMemPid = $pid;
            } 
            
            # append the PID of this process instance to PID list
            $pidStr .= "$pid\,";  
     
            $instanceCount++;

            next;
         }
      }
   } # end of foreach
  
   # ---------------------------------------------------------------------
   # all processes have been screened based on this rule.  
   # output results  
   # ---------------------------------------------------------------------

   if ( $pidStr ne "" ) {

          # remove trailing comma (,)
          $pidStr =~ s/,$//;
   }

   my $maxMemMB = 0;
   if ( $maxMem > 0 ) {

      # convert size in KB to size in MB and round up to 2 decimals
      $maxMemMB = sprintf("%.2f",$maxMem / 1024);
   } 

   print "em_result=[$soughtUser]:$soughtProc|$instanceCount|$pidStr|$totPcntCpu|$maxMemPid|$maxMemMB\n";
   EMD_PERL_DEBUG("em_result=[$soughtUser]:$soughtProc|$instanceCount|$pidStr|$totPcntCpu|$maxMemPid|$maxMemMB");

} # end of foreach  

# ---------------------------end of script--------------------------------
