#!/usr/local/bin/perl
#
#  $Header: emagent/sysman/admin/scripts/set_repository_url.pl /main/8 2012/03/09 17:36:31 jashukla Exp $
#
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      set_repository_url.pl - Set a new REPOSITORY_URL for the agent.
#
#    DESCRIPTION
#      This file contains logic for the setRepositoryURL request.
#
#      $ARGV[0] : the repository URL
#
#      Separate log file set_repository_url.log is being used to log only emctl
#      operation outputs that we do in this perl script. Likewise,
#      set_repository_url_secure.log is the log for emctl secure agent,
#      used to find the status of the secure step.
#
#    NOTES
#      This is organized into 5 main steps.
#      1.  Read the registration password on stdin (this signals the script
#          as to whether it is secure mode or not).
#      2.  Stop the agent.
#      3.  Modify the value of REPOSITORY_URL in emd.properties. This does not
#          use emctl setproperty agent, as the REPOSITORY_URL property is read-
#          only.
#     (4.) If secure mode, resecure the agent using the modified repository
#          URL.
#      5.  Start the agent.
#
#    MODIFIED   (MM/DD/YY)
#    jsoule      12/01/11 - handle failed emctl secure agent properly
#    jsoule      08/30/10 - moved from switch_oms.pl
#

use strict;
use IPC::Open3;
require "emd_common.pl";

EMAGENT_PERL_INFO("set_repository_url.pl: Started executing.");

my $reposUrl = shift(@ARGV);
EMAGENT_PERL_INFO("set_repository_url.pl: RepositoryURL value to be changed is $reposUrl.");

# Sleep to avoid this perl from stopping the agent before the
# agent gets the status of launched process
sleep 2;

my $emdRoot = $ENV{EMDROOT};
my $emState = $ENV{EMSTATE};

#
# 1. Read registration password (on stdin)
#
my %nameVal = get_stdinvars_with_end_line();

my $regPasswd = $nameVal{"regPasswd"};

#
# 2. Stop agent
#
my($ret) = 0xffff & system("$emdRoot/bin/emctl stop agent >> $emState/sysman/log/set_repository_url.log");

$ret >>= 8;

# Sleep some to so that let the agent get stopped
sleep 5;

EMAGENT_PERL_DEBUG("set_repository_url.pl: return value of stop agent is $ret.");

#
# 3. Modify the REPOSITORY_URL emd property
#
system `cp "$emState/sysman/config/emd.properties" "$emState/sysman/config/bk_so_emd.properties"`;

my $retStr = modifyProperty("$emState/sysman/config/emd.properties",
                            "REPOSITORY_URL", $reposUrl);

EMAGENT_PERL_DEBUG("set_repository_url.pl: return value of modifyProperty() subroutine is $retStr.");

if ($regPasswd eq "")
{
  #
  # 5. (non-secure) Start agent
  #
  EMAGENT_PERL_INFO("set_repository_url.pl: setRepositoryURL request is for non-secure OMS.");

  my($ret) = 0xffff & system("$emdRoot/bin/emctl start agent >> $emState/sysman/log/set_repository_url.log");

  $ret >>= 8;
  EMAGENT_PERL_DEBUG("set_repository_url.pl: return value of start agent is $ret.");

  # Give some time to agent to get started
  sleep 5;
}
else
{
  #
  # 4a. (secure) Secure agent
  #
  EMAGENT_PERL_INFO("set_repository_url.pl: setRepositoryURL request is for secure OMS");

  my $url = substr($reposUrl, 0, index($reposUrl, "/upload"));
  my $emdWalletSrcUrl = $url . "/wallets/emd";

  my $retStr1 = modifyProperty("$emState/sysman/config/emd.properties",
                               "emdWalletSrcUrl", $emdWalletSrcUrl);

  EMAGENT_PERL_INFO("set_repository_url.pl: return value of modifyProperty() subroutine for changing emdWalletSrcUrl is $retStr1.");

  # Gather stderr into a file.
  my $logFile = "$emState/sysman/log/set_repository_url_secure.log";
  my $errFile = "$emState/sysman/log/set_repository_url_secure.err";
  open(ERRORS, "+> $errFile")
    || die("Could not open error file $errFile: [$!]");

  # Using open3 to avoid security issue while passing regPasswd
  my $pid = open3(\*WRITE,    # stdin to emctl secure agent
                  ">&STDOUT", # stdout from emctl secure agent (unused)
                  ">&ERRORS", # stderr from emctl secure agent
                  "$emdRoot/bin/emctl secure agent > $logFile");

  print WRITE "$regPasswd\n";
  close WRITE;

  waitpid($pid, 0);

  my $secStdErr = '';
  my $secRC = 0xffff & $?;
  $secRC >>= 8;
  if ($secRC)
  {
    my $secStdErr = '';
    if (seek(ERRORS, 0, 0))
    {
      $secStdErr = join('', <ERRORS>);
    }
  }
  close ERRORS;
  unlink($errFile);

  if ($secStdErr) # secure agent failed && stderr provided
  {
    EMAGENT_PERL_INFO("set_repository_url.pl: stderr of secure agent is\n".
                      "\"$secStdErr\"".
                      "\nand return code is $secRC.");
  }
  else # success || no stderr available
  {
    EMAGENT_PERL_INFO("set_repository_url.pl: return code of secure agent is $secRC.");
  }

  #
  # 4b. Check whether the secure agent succeeded based on the return code and
  #     the log
  #
  my @all_lines;
  open(DAT, "$logFile") || die("Could not open file $logFile: [$!]");
  @all_lines = <DAT>;
  close(DAT);

  my $failed = 0;
  my $line;
  foreach ($line = @all_lines)
  {
    if (index($line, "The Agent has not been secured.") != -1)
    {
      $failed = 1;
      last;
    }
  }

  if ($failed)
  {
    EMAGENT_PERL_ERROR("set_repository_url.pl: Secure agent failed. Reverting back the emd.properties and restarting the agent");
    system `cp "$emState/sysman/config/bk_so_emd.properties" "$emState/sysman/config/emd.properties"`;

    #
    # 5. Secure failed; start agent
    #
    my($ret) = 0xffff & system("$emdRoot/bin/emctl start agent >> $emState/sysman/log/set_repository_url.log");

    $ret >>= 8;
    EMAGENT_PERL_INFO("set_repository_url.pl: return value of start agent after secure failed is $ret.");

    # Give some time to agent to get started
    sleep 5;
  }
  else
  {
    if ($secRC)
    {
      EMAGENT_PERL_INFO("set_repository_url.pl: Secure agent failed but".
                        " change REPOSITORY_URL for backward-compatibility");
    }
    else
    {
      EMAGENT_PERL_INFO("set_repository_url.pl: Secure agent succeeded");
    }

    #
    # 5. Secure succeeded; start agent
    #
    my($ret) = 0xffff & system("$emdRoot/bin/emctl start agent >> $emState/sysman/log/set_repository_url.log");

    $ret >>= 8;
    EMAGENT_PERL_DEBUG("set_repository_url.pl: return value of start agent after secure is $ret.");

    # Give some time to agent to get started
    sleep 5;
  }
}

EMAGENT_PERL_DEBUG("set_repository_url.pl: Returning from set_repository_url.pl");
