#
# Copyright (c) 2001, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#  $Id: Emc.pm /main/13 2011/02/18 02:23:24 rajverma Exp $ 
#
#
#
# NAME
#   Emc.pm
#
# DESC 
#   emc symmetrix specific subroutines
#
#
# FUNCTIONS
#
# sub generateDiskId( $ );  generate a edisk key for emc disks
# sub parsesymoutput;    parse the results of sympd to build a hash list
# sub getDiskinfo( $ );   return emc disk information from the list    
#
# NOTES
#
#
# MODIFIED  (MM/DD/YY)
# rajverma 02/16/11 - Fix bug 10051223 on 12GC
# bguhatha 09/07/20 - bug8685323
# spanchum 06/21/07 - Backport rajverma_bug-6113649 from main
# spanchum 04/03/07 - Backport spanchum_bug-5842718 from main
# spanchum 01/23/07 - use symcli if present
# ajdsouza 04/04/05 - use sq_emc_serial instead of sq_serial_no
# ajdsouza 08/18/04 - override symcli not available
# ajdsouza 08/09/04 - 
# ajdsouza 06/25/04 - storage reporting sources 
# ajdsouza 04/14/04  - 
# ajdsouza 04/08/04 -  storage reporting modules 
# ajdsouza  10/01/01 - Created
#

package storage::vendor::Emc;

require v5.6.1;

use strict;
use warnings;
use locale;
use XML::Parser;

#-------------------------------------------------
# subs defined 
#------------------------------------------------
sub generateDiskId( $ );
sub getDiskinfo( $ );
sub generateClariionDiskId($);
sub getClariionDiskinfo($);
           
#---------------------------------------------
# Variables with Global scope
#---------------------------------------------
my $ret_val = "" ;
my $pd_name = "" ;
my $dev_name = "";
my $symmetrix_id = "";
my $os_name = `uname -s`;  
 my $node ;
  my $is_frntend  = 0;
  my $is_product = 0 ;
  my $is_devinfo = 0 ;
  my $dir_port ;
  my $sym_id ;
  my $lun_name ;
  my $dir;
  my $port ;
  my $pdname ;
  my $parser ;
  my $ver65 = "V6.5";
  my $ver70 = "V7.0";
  my $tmp_pdname;


#------------------------------------------------------------------------------------
# FUNCTION : generateDiskId (Symmetrix)
#
#
# DESC
# Generate a ID for a disk device
#
#
# ARGUMENTS:
# Reference to a Hash for the disk with the name,,inode serial_no of the disk
#
#------------------------------------------------------------------------------------
sub generateDiskId( $ )
{
  my ( $diskref ) = @_;
  chomp($os_name);

  warn "Disk name passed is NULL to get EMC Disk Information \n" 
   and return 
    unless $diskref->{nameinstance};
  if($os_name eq "AIX")
    {  
      getDiskinfo( $diskref );
    } 
  else 
    {
      getDiskinfo( $diskref ) 
        unless $diskref->{storage_system_id};
  
   warn "Failed to generate a diskkey from vendor information for disk $diskref->{nameinstance}\n"
   and return 
    unless 
    (  
     $diskref->{storage_system_id}
     and $diskref->{storage_disk_device_id}
    );

   }  

   $diskref->{vendor}='EMC' 
    unless $diskref->{vendor};

  # Form a diskkey from the symid and deviceid parsed from the serial number
  if ( $diskref->{product} and $diskref->{product} =~ /SYM/i )
  {
   $diskref->{disk_key} = 
    "$diskref->{vendor}-SYMMETRIX-$diskref->{storage_system_id}-$diskref->{storage_disk_device_id}"
  }
  else
  {
   $diskref->{disk_key} = 
    "$diskref->{vendor}-$diskref->{storage_system_id}-$diskref->{storage_disk_device_id}"
  }
  
  return 1;

}

#------------------------------------------------------------------------------------
# FUNCTION : getDiskinfo (Symmetrix)
#
#
# DESC
# Add to the hash EMC symmetrix specific disk information  
#
#
# ARGUMENTS:
# Reference to a Hash for the disk with the logical name of the disk
#
#------------------------------------------------------------------------------------
sub getDiskinfo( $ )
{
  
  my $diskref = $_[0];
    $node  = "";
   $is_frntend  = 0;
   $is_product = 0 ;
   $is_devinfo = 0 ;
   $dir_port = "";
   $sym_id ="";
   $lun_name ="";
   $dir = "";
   $port = "";
   $pdname = "" ;
   $parser = "" ;
   $tmp_pdname = "";
   $ret_val = "" ;
   $pd_name = "" ;
   $dev_name = "";
   $symmetrix_id = "";
  
  warn "Disk name passed is NULL to get EMC Disk Information \n" 
   and return 
    unless $diskref->{nameinstance};
   if($os_name ne "AIX")
     { 
	  warn "Disk sequence no is NULL for EMC disk $diskref->{nameinstance}\n" 
	   and return 
	    unless $diskref->{sq_emc_serial};
      }

  # Check is the Symcli / Soulutions Enabler is installed 
  if (-e "/usr/symcli/bin/sympd" )
  {
    my $ver_cmd = "/usr/symcli/bin/symcfg -version | grep SYMCLI ";
    my $vout = `$ver_cmd` ;

    my($tmp  ,  $vtxt )  = split (/:/, $vout) ;
    my($dum, $symcliver ) = split ( / / ,$vtxt );

  my $command = "/usr/symcli/bin/sympd show  -output xml " . $diskref->{name};
  my $xml_out = `$command`;

      if($xml_out ne "")
     {
       if ( $symcliver =~ m/^$ver65/  || $symcliver =~ m/^$ver70/ )
        {
           $parser = XML::Parser->new( Handlers => {
                                      Start=>\&handle_start65,
                                      Char=>\&handle_char65,
                                      End=>\&handle_end65
                                    });
        }
      else
       {
         $parser = XML::Parser->new(Handlers =>{
                     Start=>\&handle_start,
                     });
	}

        $parser->parse($xml_out); 
       }
  }
 
  #If symcli returns data, then get device_id:port_no
  if ($ret_val ne "")
  {
     my ($symm_id, $device_id,$dir_id,$port_no) = split(/:/,$ret_val);
     $diskref->{storage_system_id}      = $symm_id;
     $diskref->{storage_disk_device_id} = $device_id.":".$dir_id.":".$port_no;
     $ret_val = "";  #reset ret_val
  }
  # Else use the default from disk serial .
  else
  {
  # Parse the symid and deviceid from the scsi serial number
  my ($symid,$deviceid,$port)  = 
   ( $diskref->{sq_emc_serial} =~ /^\s*(......)(...)(...).*/i );

  warn "Failed to get the EMC system id for disk $diskref->{nameinstance}\n"
   and return 
    unless $symid;

  $deviceid = '000' unless $deviceid;
    
  # Copy values farmed from the sq_emc_serial for EMC Symmetrix
  $diskref->{storage_system_id}      = "      ".$symid;
  $diskref->{storage_disk_device_id} = " ".$deviceid.":"."NA".":"."NA";
  } 
  return 1;

}

#-----------------------------------------------------------------------------
# FUNCTION : handle_start
# DESC     : Used to parse the XML format of SYMCLI 6.4 and 6.3
# ARGUMENTS: Nodes of XML output
#-----------------------------------------------------------------------------

sub handle_start
{
   my($expat, $element,%attrs) = @_;
   my $dir        = "";
   my $port       = "";

   if($element eq "Device")
   {
      if(%attrs)
      {
         $pd_name    = $attrs{"pd_name"};		
         $dev_name   = $attrs{"dev_name"};
      }	
   }
   if($element eq "Product" )
   {
      if(%attrs)
      {
         $symmetrix_id = $attrs{"symmetrix"};
      }
   }
   if($element eq "Port")
   {
      if(%attrs)
      {
         my $temp_pdname = $attrs{"pd_name"};
         if($pd_name eq $temp_pdname)
         {	
            $dir  = $attrs{"director"};
            $port = $attrs{"port"};
            $ret_val = $symmetrix_id.":".$dev_name.":".$dir.":".$port ;
            #return  1 ;
         }
      }	
   }	
}

#------------------------------------------------------------------------------------
# FUNCTION : handle_start65
# DESC     : Used to parse the XML format of SYMCLI 7.0 
# ARGUMENTS: Nodes of XML output
#------------------------------------------------------------------------------------


sub handle_start65 {
    my( $expat, $element, %attrs ) = @_;

        $node = $element ;

           $is_devinfo = 1
             if($element  =~ m/^Dev_Info$/);

          $is_product = 1
             if($element =~ m/^Product$/);

          $is_frntend = 1
             if($element =~ m/^Front_End$/ );  # get port for this lun.

    }


#------------------------------------------------------------------------------------
# FUNCTION : handle_char65
# DESC     : Used to parse the XML format of SYMCLI 7.0 
# ARGUMENTS: Nodes of XML output
#------------------------------------------------------------------------------------

sub handle_char65 
  {
        my( $expat, $text ) = @_ ;

        # Get Device Name
        if( $is_devinfo eq "1" )
        {
           $lun_name = trim($text)
                 if( $node =~ m/^dev_name$/  && (trim($text) ne ""));

           $pdname = trim($text)
                 if( $node =~ m/^pd_name$/  && (trim($text) ne ""));
        }

 #Get  Director : Port 
         if($is_frntend eq "1" )
          {

                 $tmp_pdname =  trim($text)
                    if($node =~ /^pd_name$/ && (trim($text) ne ""));

                if($tmp_pdname eq $pdname)
                {
                   $dir_port =  trim($text)
                       if($node =~ /^director$/ && (trim($text) ne ""));

                    $dir_port  =$dir_port.":".trim($text)
                          if($node =~ /^port$/ && (trim($text) ne ""));

                }
          }

          #Get  Symm ID
        if( $is_product eq "1" )
        {
                 $sym_id = $text
                        if( $node eq "symid"  && (trim($text) ne "") );

        }

  }

#------------------------------------------------------------------------------------
# FUNCTION : handle_end65
# DESC     : Used to parse the XML format of SYMCLI 7.0 
# ARGUMENTS: Nodes of XML output
#------------------------------------------------------------------------------------

sub handle_end65
   {

    my( $expat, $element ) = @_;
    chomp ;

             $is_frntend = 0
                if($element =~ m/^Device$/);

             $is_devinfo = 0
                if($element =~ m/^Dev_Info$/ );

             $is_product = 0
                if($element eq "Product" );

             $is_frntend = 0
               if($element =~ m/^Port$/);

        if($element =~ m/^Front_End$/ )
          {
            $ret_val = $sym_id.":".$lun_name.":".$dir_port ;
            $dir_port= "" ;
            $is_frntend = 0;
          }

  }


sub trim {
   my($string)=@_;
   for ($string) {
       s/^\s+//;
       s/\s+$//;
   }
   return $string;
}

#------------------------------------------------------------------------------------
# FUNCTION : generateClariionDiskId
#
#
# DESC
# Generate a ID for a Clariion disk device
#
#
# ARGUMENTS:
# Reference to a Hash for the disk with the name,,inode serial_no of the disk
#
#------------------------------------------------------------------------------------

sub generateClariionDiskId( $ )
{
  my ( $diskref ) = @_;

  warn "Disk name passed is NULL to get EMC Disk Information \n"
   and return
    unless $diskref->{nameinstance};

  getClariionDiskinfo( $diskref )
   unless $diskref->{storage_system_id};

  warn "Failed to generate a diskkey from vendor information for disk $diskref->{nameinstance}\n"
   and return
    unless
    (
     $diskref->{storage_system_id}
     and $diskref->{storage_disk_device_id}
    );

  $diskref->{vendor}='DGC'
   unless $diskref->{vendor};

 # Form a diskkey from the clariionid and deviceid parsed from the serial number
  if ( $diskref->{product} and $diskref->{product} =~ /DGC/i )
  {
   $diskref->{disk_key} =
    "$diskref->{vendor}-CLARIION-$diskref->{storage_system_id}-$diskref->{storage_disk_device_id}"
  }
  else
  {
   $diskref->{disk_key} =
    "$diskref->{vendor}-$diskref->{storage_system_id}-$diskref->{storage_disk_device_id}"
  }

  return 1;

}

#------------------------------------------------------------------------------------
# FUNCTION : getClariionDiskinfo
#
#
# DESC
# Add to the hash EMC Clariion specific disk information
#
#
# ARGUMENTS:
# Reference to a Hash for the disk with the logical name of the disk
#
#------------------------------------------------------------------------------------
sub getClariionDiskinfo( $ )
{

  my $diskref = $_[0];
 
  warn "Disk name passed is NULL to get EMC Disk Information \n"
   and return
    unless $diskref->{nameinstance};

  #Get the Array ID from the serial ascii
  warn "Failed to get the Clariion system id for disk $diskref->{nameinstance}\n"
   and return 
    unless  $diskref->{sq_serial_no_ascii};
  $diskref->{storage_system_id} = uc($diskref->{sq_serial_no_ascii});

  #Get the LUN UID from the page83 data.
  warn "Failed to get the Clariion device id for disk $diskref->{nameinstance}\n"
   and return 
    unless  substr($diskref->{sq_vpd_pagecode_83},8,32);
  $diskref->{storage_disk_device_id} = uc(substr($diskref->{sq_vpd_pagecode_83},8,32));

  return 1;

}

1;
