#
# Copyright (c) 2001, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      wmiquery.pl 
#
#    DESCRIPTION
#      This script is used in EMD to execute the wmi query depending on the filetype passed.
#      The state file directory is read from the environment variable $EMSTATE, which must
#      be set for the script to run.This script will set the WBEM_WQL environment and then call the 
#      nmuwmi.exe to get the metric result.
#

#
#    MODIFIED   (MM/DD/YY)
#    armanick    11/07/11 - Fix for bug#11740553. WINDOWS EVENT SEVERITY METRIC
#                           DOESN'T WORK ON JAPANESE ENVIRONMENT.Modifying the
#                           sql statement to use eventType instead of Type to
#                           query the win32_ntlogevent table.
#    cgnanasa    09/26/08 - Handling "ALL Others" condition context
#    cgnanasa    09/18/08 - Honouring condition contexts for generating results
#    sejain      05/12/08 - Backport sejain_bug-6752515 from main
#    cgnanasa    06/24/05 - EMSTATE to be used to read state directory 
#    hsharma     02/15/05 - hsharma_wmiexecutable3
#    ajayshar    02/09/05 - Creation for facilitating WMI queries (EM 10.2)


require "emd_common.pl";
require "conditionContext.pl";
use Time::gmtime;
use Time::Local;


my $maxMessageLength = 1024;

my $fileTime="0";
my $filetype=$ENV{"WBEM_FILENAME"}; # 'APPLICATION , SECURITY, SYSTEM , WINDOWS';
my $setenvstring="";
my $filename;
my $sign = '';
my $offset  = sprintf "%.1f", (timegm(localtime) - time) / 3600;

if ($offset >= 0.0)
{
$sign = '+';
}
my $UTC=sprintf "%+d", $offset*60;
my $zero=0;
my $currtime;

EMD_PERL_DEBUG("------- STARTED wmiquery.pl ( $filetype )----------");

if(($filetype ne "APPLICATION") && ($filetype ne "SYSTEM") && ($filetype ne "SECURITY") && ($filetype ne "WINDOWS"))
{

    print "em_error=Incorrect filetype (WBEM_FILENAME) in parameter.\n";
    EMD_PERL_DEBUG("$filetype : Incorrect filetype (WBEM_FILENAME) in parameter.\n\n");
    exit 1;
}

unless( exists $ENV{"EMSTATE"} or defined $ENV{"EMSTATE"} ) 
{
    print "em_error=The environment variable EMSTATE needs to be set.\n";
    EMD_PERL_DEBUG("The environment variable EMSTATE needs to be set.\n\n");
    exit 1;
}

$filename=$ENV{EMSTATE}."\\sysman\\emd\\state"."\\wmi_state_".$filetype.".dat"; 

# This block finds out whether this is a defaut collection or a real time call
#
my $isDefaultCollection = 1;
$numArgs = $#ARGV + 1;
if($numArgs == 1)
{
  my $collInterval =  $ARGV[0];
  EMD_PERL_DEBUG("_collInterval value received:".$collInterval);  
  if( $collInterval == 0)
  {
    $isDefaultCollection = 0;
    EMD_PERL_DEBUG("This is a real time call");  
  }
  else
  {
    $isDefaultCollection = 1;
    EMD_PERL_DEBUG("This is a default collection call");  
  }
}
else
{
   EMD_PERL_DEBUG( "Collection module expects one argument but $numArgs was received.\n");
}

&getTimeState();
if($isDefaultCollection == 1)
{
  &saveTimeState();
}

# This function will get the last saved time from the coresponding state file (TYPE[app/sys/sec/win]).
# If the file is not existing , it will just return indicating that the query is being run the first time.
sub getTimeState()
{  
    unless( open STATE, "< $filename" ) 
    {
        EMD_PERL_DEBUG("WARNING:Unable to open statefile ( $filename ) for reading. Should create a new file after this step.");   
        ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
        $currtime = sprintf("%4d%02d%02d%02d%02d%02d.%06d$sign%03d",$year+1900,$mon+1,$mday,$hour,$min,$sec,$zero,$UTC); 
        $fileTime=$currtime;
        return;
    }    
    my @stateData = <STATE>;
    my @timevar = split("=", $stateData[0]);
    $fileTime = $timevar[0];    
    EMD_PERL_DEBUG("Getting:$fileTime");    
    close(STATE);
    return $fileTime;  
}


# This function will save the curent time using the WQL format in the state file
# This will give error if due to some reason file cannot be created , which should never happen.
sub saveTimeState()
{

    
    unless( open STATE, "> $filename" ) 
    {
        print "em_error=Unable to open statefile ( $filename ) for Writing.\n";
        EMD_PERL_DEBUG("ERROR:Unable to open statefile ( $filename ) for Writing.\n\n");    
        exit 1;
    }    
    ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
     $currtime = sprintf("%4d%02d%02d%02d%02d%02d.%06d$sign%03d",$year+1900,$mon+1,$mday,$hour,$min,$sec,$zero,$UTC);        
    print STATE "$currtime=TIMESTATE\n";  
    EMD_PERL_DEBUG("Setting:$currtime");  
    close(STATE);  
}


if($filetype eq "WINDOWS")
{   
    $setenvstring="select logfile,sourcename,eventcode,timegenerated,recordnumber,eventtype,type,categorystring,user,message from win32_ntlogevent where timegenerated >= '$fileTime' and (eventtype=1 or eventtype=2)";
}

else
{
    $setenvstring="select sourcename,categorystring,eventcode,timegenerated,recordnumber,type,user,message from win32_ntlogevent where logfile='$filetype' and timegenerated >= '$fileTime' and (eventtype=1 or eventtype=2)";
}

EMD_PERL_DEBUG("SET Environment(WBEM_WQL):$setenvstring ");
$ENV{WBEM_WQL}=$setenvstring;
#system "nmefwmi.exe";
@nmefwmi_output= `nmefwmi.exe`;
#print "$nmefoutput[0]";

# For testing
# $data_file="C:\\chandy\\linux\\nmefwmi.out";
# open(DAT, $data_file) || die("Could not open file!");
# @nmefwmi_output=<DAT>; 
# print @nmefwmi_output;
# close(DAT);


EMD_PERL_DEBUG("------- ENDED wmiquery.pl ---------\n\n");

# Added for NLS compliancy
$errorString ="error";
$warningString = "warning";

# parse condition contexts and form DS
$conditionContextAref = getConditionContext();

# print condition contexts from DS. Solely DEBUgging purpose.
#printConditionContexts($conditionContextAref);

#Parse CC DS to form another DS suitable for Event log metric message formation.
# This DS is explained at the bottom of the file.
@allCondtionContextsAggregate = ();
createMessageDS(\@allCondtionContextsAggregate,$conditionContextAref);

# run nmefwmi and filter the rows based on allCondtionContextsAggregate 
populateMessageDS(\@allCondtionContextsAggregate, \@nmefwmi_output);

#the DS is ready with required data. Need to print it out.
printEMResult(\@allCondtionContextsAggregate);

sub printEMResult
{
  my ($lAllCondtionContextsAggregate) = @_;

  # This is going to be common across the condition contexts and hence
  # creating the timestamp once.
  $timeToReturn = scalar localtime();
  foreach my $conditionContextInfoARef (@$lAllCondtionContextsAggregate)
  {
    my @unique_cs_array = ();
    my %unique_cs_hash = ();
    my @unique_user_array = ();
    my %unique_user_hash = ();
    my $isErrorPresent = 0;
    my $isWarningPresent = 0;
    
    EMD_PERL_DEBUG("printEMResult:Inside condition context loop ---");
    # the length of this array should never be 0.
    $logfileToReturn        = ${$conditionContextInfoARef}[1];
    $srcNameToReturn        = ${$conditionContextInfoARef}[3];
    $evtCodeToReturn        = ${$conditionContextInfoARef}[5];
    EMD_PERL_DEBUG("printEMResult:logfileToReturn=".$logfileToReturn);
    EMD_PERL_DEBUG("printEMResult:srcNameToReturn=".$srcNameToReturn);
    EMD_PERL_DEBUG("printEMResult:evtCodeToReturn=".$evtCodeToReturn);


    $recordNumberToReturn   = " "; # this is no longer valid.
    $typeToReturn           = " "; # this is no longer valid.
    $categoryStringToReturn = " "; # this is no longer valid.
    $userToReturn           = " "; # this is no longer valid.
    
    # get the hashmap and iterate through the keyset to retrieve
    # the errorcount / warning count. If one of them is missing
    # the corresponding column will not be present.
    $messageToReturn = "";
    $messageToReturnLength   = 0;
    $tempMessage = "";
    $msgExceeded = 0; 
        
    $eventsAggregateHref = ${$conditionContextInfoARef}[6];
    #while ( my ($key, $eventColumnsAndTypeCountARef) = each  %{$eventsAggregateHref} ) 
    for my $key (keys %{$eventsAggregateHref} )
    {
      my $eventColumnsAndTypeCountARef = ${$eventsAggregateHref}{$key};
      EMD_PERL_DEBUG("key=$key => value=$eventColumnsAndTypeCountARef");
      
      #Actual logfilename
      $tempMessage = "[". ${$eventColumnsAndTypeCountARef}[0];
      #Actual srcname
      $tempMessage =$tempMessage.":".${$eventColumnsAndTypeCountARef}[1] ;
      #Actual evtcode
      $tempMessage =$tempMessage.":".${$eventColumnsAndTypeCountARef}[2] ;
      #Actual categorystring
      $tempMessage =$tempMessage.":".${$eventColumnsAndTypeCountARef}[3] ;
      #Actual user
      $tempMessage =$tempMessage.":".${$eventColumnsAndTypeCountARef}[4] ;
      # error count
      if( ${$eventColumnsAndTypeCountARef}[5] > 0 )
      {
        #$tempMessage =$tempMessage.":error=".${$eventColumnsAndTypeCountARef}[5];
        $tempMessage =$tempMessage.":". $errorString . "=" . ${$eventColumnsAndTypeCountARef}[5];
        $isErrorPresent = 1;
      }
      else
      {
        $tempMessage =$tempMessage.":";
      }
      if( ${$eventColumnsAndTypeCountARef}[6] > 0 )
      {
        #$tempMessage =$tempMessage.":warning=".${$eventColumnsAndTypeCountARef}[6].";";
        $tempMessage =$tempMessage.":". $warningString . "=" .${$eventColumnsAndTypeCountARef}[6];
        $isWarningPresent = 1;
      }
      else
      {
        $tempMessage =$tempMessage.":";
      }
      $tempMessage = $tempMessage ."]";
      $tempMessageLength = length ($tempMessage);
      if( ($tempMessageLength  + $messageToReturnLength) > $maxMessageLength )
      {
        EMD_PERL_DEBUG("Message field capacity exceeded. current length= ($tempMessageLength  + $messageToReturnLength)");
        if ($msgExceeded ==0) 
	{
	  $tempMessage = "...";
	  $messageToReturn = $messageToReturn . $tempMessage ;
	  $msgExceeded = 1;
	}
      }
      else
      {
        $messageToReturn = $messageToReturn . $tempMessage. " ";
        $messageToReturnLength = $messageToReturnLength + $tempMessageLength + 1 ;
      }
      #build a unique list of category strings matching the condition context
      if ( (${$eventColumnsAndTypeCountARef}[3] ne "") &&  (!exists($unique_cs_hash{${$eventColumnsAndTypeCountARef}[3]  }) ) )
      {
         push(@unique_cs_array , ${$eventColumnsAndTypeCountARef}[3]);
      }
      
      #build a unique list of users matching the condition context
      if ( (${$eventColumnsAndTypeCountARef}[4] ne "") && (!exists($unique_user_hash{${$eventColumnsAndTypeCountARef}[4]  })) )
      {
         push(@unique_user_array , ${$eventColumnsAndTypeCountARef}[4]);
      }

      #last if( ($tempMessageLength  + $messageToReturnLength) > $maxMessageLength );
    }# eventColumnsAndTypeCountARef loop ends here
    
    if( length ($messageToReturn) > 0 )   
    {
      if($logfileToReturn eq "") 
      {
      $logfileToReturn = " ";
      }
      if($srcNameToReturn eq "") 
      {
        $srcNameToReturn = " ";
      }
      if($evtCodeToReturn eq "") 
      {
        $evtCodeToReturn = " ";
      }
      if((scalar @unique_cs_array) > 0)
      {
	my %hsh;
	undef @hsh{@unique_cs_array};
	my @unique_cs_array = keys %hsh;
     
        $categoryStringToReturn = join(",", @unique_cs_array);
      }
      if((scalar @unique_user_array) > 0)
      {
	my %hsh;
	undef @hsh{@unique_user_array};
	my @unique_user_array = keys %hsh;

        $userToReturn = join(",", @unique_user_array);
      }
   
      if( $isErrorPresent == 1)
      {
        $typeToReturn  = $errorString;
      }
      if( $isWarningPresent ==1 )
      {
        $typeToReturn = $warningString;
      }
      if( ($isErrorPresent == 1)  && ( $isWarningPresent ==1 ))
      {
        $typeToReturn  =  $warningString . ",". $errorString;;
      }
      my $emresultString= "em_result=$logfileToReturn|$srcNameToReturn|$evtCodeToReturn|$timeToReturn|$recordNumberToReturn|$typeToReturn|$categoryStringToReturn|$userToReturn|$messageToReturn\n";
      print "$emresultString";
      EMD_PERL_DEBUG("$emresultString");
    }
  }#conditionContextInfoARef, per cc loop ends here
}



sub populateMessageDS()
{
  my ($lAllCondtionContextsAggregate , $lNmefwmiOutputAref) = @_;

  foreach my $nmefwmiLine (@$lNmefwmiOutputAref) 
  {
    chomp($nmefwmiLine);
    EMD_PERL_DEBUG("nmefwmi outputline:".$nmefwmiLine);
    my @wmidata        = ();
    my @tmp            = ();
    chomp(@wmidata = split ('\|',$nmefwmiLine) );

    #$wmidata[0] will have em_result=logfile , so split again on '='
    my $temp         = $wmidata[0];
    chomp(@tmp = split ('=', $temp));
    $logfile         = $tmp[1];
    $sourcename      = $wmidata[1];
    $eventcode       = $wmidata[2];
    #event type: 1 -> error and 2-> warning. Added for NLS compliancy
    $eventtype       = $wmidata[5]; 
    $type            = $wmidata[6];
    $categorystring  = $wmidata[7];
    $user            = $wmidata[8];
    
    #Now , iterate through all the condition contexts and populate hashmap
    # if the conditions match.
    
    # this key is common across cond cont, hence preparing outside.
    $key = $logfile .$sourcename .$eventcode . $categorystring . $user;
    EMD_PERL_DEBUG("formed nmefwmi key =".$key);
    foreach my $conditionContextInfoARef (@$lAllCondtionContextsAggregate)
    {
      # the length of this array should never be 0.
      $logfileMatchPattern = ${$conditionContextInfoARef}[0];
      $srcNameMatchPattern = ${$conditionContextInfoARef}[2];
      $evtCodeMatchPattern = ${$conditionContextInfoARef}[4];
      EMD_PERL_DEBUG("Trying to find a matching condition context");
      EMD_PERL_DEBUG("populateMessageDS:logfileMatchPattern=".$logfileMatchPattern);
      EMD_PERL_DEBUG("populateMessageDS:srcNameMatchPattern=".$srcNameMatchPattern);
      EMD_PERL_DEBUG("populateMessageDS:evtCodeMatchPattern=".$evtCodeMatchPattern);
      EMD_PERL_DEBUG("populateMessageDS:logfileToMatch=".$logfile);
      EMD_PERL_DEBUG("populateMessageDS:srcNameToMatch=".$sourcename);
      EMD_PERL_DEBUG("populateMessageDS:evtCodeToMatch=".$eventcode);
      
      
          # Source name matches check further
      if($sourcename =~ /$srcNameMatchPattern/i)
      {
        if( $eventcode =~ /$evtCodeMatchPattern/)
        {
          if($logfile =~ /$logfileMatchPattern/i ) 
          {
            EMD_PERL_DEBUG("populateMessageDS:record matched");
            #nmefwmi output matches the condition context. Now increase the count by type in 
            # eventsAggregateHref
            $eventsAggregateHref = ${$conditionContextInfoARef}[6];
            if ( exists($eventsAggregateHref->{$key}) )
            {
              $EventColumnsAndTypeCountARef = $eventsAggregateHref->{$key};
              if ($#{$EventColumnsAndTypeCountARef} !=  6 )
              {
                # error message
                EMD_PERL_DEBUG("Error: The array size is not  6. i.e = ". $#{$EventColumnsAndTypeCountARef});
              }
            
              if($eventtype == 1 ) # error event type
              {
                  (${$EventColumnsAndTypeCountARef}[5])= ${$EventColumnsAndTypeCountARef}[5] + 1;
                  $errorString = $type;
                  EMD_PERL_DEBUG("Key already exists: Incrementing error count to:".(${$EventColumnsAndTypeCountARef}[5]));
              }
              elsif ($eventtype == 2 ) # warning event type
              {
                  (${$EventColumnsAndTypeCountARef}[6])= ${$EventColumnsAndTypeCountARef}[6] + 1;
                  $warningString = $type;
                  EMD_PERL_DEBUG("Key already exists: Incrementing warning count to:".(${$EventColumnsAndTypeCountARef}[6]));
              }
            }
            else
            {
              EMD_PERL_DEBUG("New key:$key found. Adding event columns array");
              my @EventColumnsAndTypeCountArray = ();
              push(@EventColumnsAndTypeCountArray, $logfile);
              push(@EventColumnsAndTypeCountArray, $sourcename);
              push(@EventColumnsAndTypeCountArray, $eventcode);
              push(@EventColumnsAndTypeCountArray, $categorystring);
              push(@EventColumnsAndTypeCountArray, $user);
              #error count
              if($type =~ /error/i )
              {
                push(@EventColumnsAndTypeCountArray, 1);
              }
              else
              {
                push(@EventColumnsAndTypeCountArray, 0);
              }
              #warning count
              if($type =~ /warning/i )
              {
                push(@EventColumnsAndTypeCountArray, 1);
              }
              else
              {
                push(@EventColumnsAndTypeCountArray, 0);
              }
              $eventsAggregateHref->{$key} = \@EventColumnsAndTypeCountArray;
             
            }
          } # key matching for logfile ends here
          else
          {
            EMD_PERL_DEBUG("No Matching logfile");
          }
        }
        else
        {
          EMD_PERL_DEBUG("No Match event code");
        }
      }
      else
      {
        EMD_PERL_DEBUG("No Match src name");
      }   
    } # condition context iteration ends here
  } # nmefwmi output ends here.
  EMD_PERL_DEBUG("populateMessageDS: populating MessageDS complete");
}

sub createMessageDS
{
   my ($lAllCondtionContextsAggregate , $lConditionContextAref) = @_;
   
   # push an array into lAllCondtionContextsAggregate 
   # for every condition context an array containing the following elements as values
   # will be added. This will help in forming em_result later.
   #   [0] th elem = keyvalue to match for logfile
   #   [1] th elem = keyvalue to return for logfile
   #   [2] th elem = keyvalue to match for srcname
   #   [3] th elem = keyvalue to return for srcname
   #   [4] th elem = keyvalue to match for evtcode
   #   [5] th elem = keyvalue to return for evtcode
   #   [6] th elem = hashmap that will contain the count of errors/warning
   #                 key = (Log + Src + Evt + CatStr + User). These are actual values
   #                 of the columns returned by nmefwmi.
   #
   #                 value = A reference to an Array with the following elements.
   #                 [0]  = actual value of the logfile
   #                 [1]  = actual value of the srcname
   #                 [2]  = actual value of the evt code
   #                 [3]  = actual value of the category string
   #                 [4]  = actual value of the user
   #                 [5]  = count of error events 
   #                 [6]  = count of warning events
   #                 Logfile,srcname,eventcode, category string and user
   #                 are stored as separate elements again because delimiter
   #                 could possible exist in the value.
   #
   #
   #

   foreach my $conditionHref (@$lConditionContextAref) 
   {
     $condCount = 0;
     my @conditionContextInfoArray = ();
     my %eventsAggregate = ();
      
     my $keysAref = ${$conditionHref}{"keyColumnAref"};

      # This is done to not generate a corresponding row for those
      # condition contexts that does not have threshold values.
      my $warningThreshold = ${$conditionHref}{"warningThreshold"};
      my $criticalThreshold = ${$conditionHref}{"criticalThreshold"};
      if ($warningThreshold eq "NotDefined" &&  $criticalThreshold eq "NotDefined" )
      {
        EMD_PERL_DEBUG("Condition =".$condCount. " has no thresholds set. Moving to next condition context");
        next;
      }

     # if the condition does not have any keyColumn element defined
     # then it translates to "All Others" for all the keys in Metrics
     # and policy settings page.
     # "All Others" comes up with no key col values. This needs to be handled.
     if ($#{$keysAref} < 0 )
     {

       
       # Commenting this block for now cause of NLS issues.
       # All others as a condition context implies none of the above conditions.
       # and not %. Butcannot return "All Others" as key because it will be
       # NLS issue. Hence ignoreing all others case for now.
       # thresholds for "All Others" will be set to none.
       $logKeyToMatch=".*";
       $logKeyToReturn="%";
       $sourceKeyToMatch=".*";
       $sourceKeyToReturn="%";
       $eventKeyToMatch=".*";
       $eventKeyToReturn="%";
       EMD_PERL_DEBUG("Condition =".condCount. " has no keys. Moving to next condition context");
       next;       
     }
     else
     {
       foreach my $keyHref (@$keysAref)
       {
         if (${$keyHref}{"keyName"} eq "logfile")
         {
           $logKeyToMatch = ${$keyHref}{"keyValueToMatch"};
           $logKeyToReturn = ${$keyHref}{"keyValueToReturn"};
         }
         elsif (${$keyHref}{"keyName"} eq "sourcename")
         {
           $sourceKeyToMatch = ${$keyHref}{"keyValueToMatch"};
           $sourceKeyToReturn = ${$keyHref}{"keyValueToReturn"};
         }
         elsif (${$keyHref}{"keyName"} eq "eventcode")
         {
           $eventKeyToMatch = ${$keyHref}{"keyValueToMatch"};
           $eventKeyToReturn = ${$keyHref}{"keyValueToReturn"};
         }
       }
     }
     push(@conditionContextInfoArray, $logKeyToMatch);
     push(@conditionContextInfoArray, $logKeyToReturn);
     push(@conditionContextInfoArray, $sourceKeyToMatch);
     push(@conditionContextInfoArray, $sourceKeyToReturn);
     push(@conditionContextInfoArray, $eventKeyToMatch);
     push(@conditionContextInfoArray, $eventKeyToReturn);
     push(@conditionContextInfoArray, \%eventsAggregate);
     
     EMD_PERL_DEBUG ("Contents of one conditionContextInfoArray:");
     EMD_PERL_DEBUG ("logKeyToMatch=".$conditionContextInfoArray[0]);
     EMD_PERL_DEBUG ("logKeyToReturn=".$conditionContextInfoArray[1]);
     EMD_PERL_DEBUG ("sourceKeyToMatch=".$conditionContextInfoArray[2]);
     EMD_PERL_DEBUG ("sourceKeyToReturn=".$conditionContextInfoArray[3]);
     EMD_PERL_DEBUG ("eventKeyToMatch=".$conditionContextInfoArray[4]);
     EMD_PERL_DEBUG ("eventKeyToReturn=".$conditionContextInfoArray[5]);
     EMD_PERL_DEBUG ($conditionContextInfoArray[6]);
     push( @$lAllCondtionContextsAggregate, \@conditionContextInfoArray);
   }
   
}

#returns the size of the condition context
sub getConditionContextsSize
{
    my ($lConditionContextAref ) = @_; 
    $size = scalar @$lConditionContextAref;
    EMD_PERL_DEBUG(" The size of condition contexts=". $size );
    return $size;
}

# using DEBUG for now. Will change to info later
sub printConditionContexts
{
    my ($lConditionContextAref) = @_; 
    EMD_PERL_DEBUG("printConditionContexts:IN");
    EMD_PERL_DEBUG("printConditionContexts: Number of condition contexts passed=". scalar @$lConditionContextAref );
    EMD_PERL_DEBUG("printConditionContexts: Array  passed=".  @{$lConditionContextAref} );
    my $condCount = 0;
    foreach my $conditionHref (@$lConditionContextAref) 
    {
      EMD_PERL_DEBUG("-------------- Printing details of condition Context ". $condCount."-------------"); 
      my $keysAref = ${$conditionHref}{"keyColumnAref"};
      if ($#{$keysAref} < 0 )
      {
        EMD_PERL_DEBUG("printConditionContexts:Condition=".condCount. " has no keys");
        next;
      }
 
      EMD_PERL_DEBUG("printConditionContexts:  printing key information");
      foreach my $keyHref (@$keysAref)
      {
        EMD_PERL_DEBUG("chan2");
        EMD_PERL_DEBUG ("Keyname =". ${$keyHref}{"keyName"} );
        EMD_PERL_DEBUG ("keyValueToMatch =". ${$keyHref}{"keyValueToMatch"} );
        EMD_PERL_DEBUG ("keyValueToReturn =". ${$keyHref}{"keyValueToReturn"} );
        EMD_PERL_DEBUG ("keyOperator =". ${$keyHref}{"keyOperator"} );
      }
      EMD_PERL_DEBUG("printConditionContexts:Printing condition column information");
      EMD_PERL_DEBUG("conditionColumnName=". ${$conditionHref}{"conditionColumnName"}  );
      EMD_PERL_DEBUG("conditionOperator=". ${$conditionHref}{"conditionOperator"}  );
      EMD_PERL_DEBUG("criticalThreshold". ${$conditionHref}{"criticalThreshold"}  );
      EMD_PERL_DEBUG("warningThreshold" . ${$conditionHref}{"warningThreshold"}  );

      $condCount = $condCount+1;
    } # iterate end for condition contexts
    EMD_PERL_DEBUG("printConditionContexts:OUT");
}

