#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/db/ashviewer/ashMetric.pl /st_emgc_pt-12.1.0.4pg/1 2012/09/25 19:20:24 pbhogara Exp $
#
# ashMetric.pl
# 
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      ashMetric.pl - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    pbhogara    09/09/12 - update column numbers for 12 DB
#    pbhogara    03/23/11 - use entitry ref for special chars in XML
#    pbhogara    02/11/11 - pass event map to getReport()
#    pbhogara    10/13/10 - read the most recent ashdump from dump trace file
#    pbhogara    09/06/10 - grabbing akini_ashviewer txn
#    pbhogara    09/06/10 - Creation
# 
use strict;
use Time::Local;

require "emd_common.pl";
require "db/emergPerfUtil.pl";

package AshViewer;

##############
# Global Variables
##############
my $MAP_DIR_ROOT = $ENV{'PLUGIN_ROOT'}."/scripts/db/ashviewer/";


my %dimTable;

# stores a list of components (histograms, treemap1d, etc.) and sessions
# requested list of components, and sessions for details
# populated by parseReportLevel
my %componentTable;
my %sessionTable;

# constants used by getBucketInfo
my $MAX_MEMBERS = 5;
my $BUCKET_MAX_COUNT = 16;

# components built by this script
my $COMP_ID_HISTOGRAM = 1;
my $COMP_ID_TREEMAP1D = 2;

# session details flags 
my $SESSION_ATTRIBUTES      = 1;
my $SESSION_EVENT_HISTOGRAM = 2;
#
# Subroutine: getReport
#  $_[0] => full path to trace file
#           optional: default value = most recent trace file in T_DHOME_TRC
#  $_[1] => version of the database (e.g. '11202' or '10204')
#           optional: default value = '11202'
#  $_[2] => begin time in secs since epoch (e.g., 1254673614)
#           optional: default value = (now - 15 secs)
#  $_[3] => end time in secs since epoch (e.g., 1254673614)
#           optional: default value = (now)
#  $_[4] => report level (e.g., '<wait_class>{histogram}')
#           optional: default value = '<wait_class>{histogram}'
#  $_[5] => filter list - not supported yet
#  The next three parameters are optional. See getBucketInfo for details
#  $_[6] => bucket max count (e.g., 16)
#  $_[7] => bucket min interval (e.g., 1)
#  $_[8] => event map - to map event Ids to event name and class.
#           If this is not provided then it looks for an event map
#           file in MAP_ROOT_DIR.
# 
#  
#  Returns a string containing the XML report
#  <report>
#    <body>
#      <dim_list>
#        <dim name="wait_class" count="34710">
#          <top_mems cnt_others="0" num_others="0" >
#            <mem name="Commit">6</mem>
#            <mem name="Cluster">177</mem>
#            ..
#          </top_mems>
#        </dim>
#      </dim_list>
#      <session_details>
#        <session id="1,1659,46066" 
#                 user="1445" service="406674789" 
#                 program="httpd@amats503 (TNS V1-V3)" 
#                 module="httpd@amts503 (TNS V1-V3)">
#        <session id="1,1740,9" 
#                 user="61910" 
#                 service="406674789" 
#                 program="httpd@amts503 (TNS V1-V3)" 
#                 module="httpd@amts503 (TNS V1-V3)">
#        ..
#      </session_details>
#    </body>
#  </report>
sub getReport {

  # filterList is yet to be supported
  my ($traceFile
      ,$version
      ,$beginTime
      ,$endTime
      ,$reportLevel
      ,$filterList
      ,$bucketMaxCount
      ,$bucketMinInterval
      ,$eventIdClassMap) = @_;

  $version = _get_target_version($version); 
  initializeDimTable($version);

  my $linesRead;

  my @fields;

  my %dimStatsTable;
  my %sessionDetailsTable;

  # in general the report begin/end time as input to this function can differ 
  # from the time period covered by the xml returned. we record the latter in 
  # in these two variables and return them as attributes of the <histogram> tag
  my $histBeginTime = undef;
  my $histEndTime = undef;

  ##############
  # Set appropriate default values for all input parameters
  ##############
  if(! defined($traceFile)) {
   # if traceFile is not provided, set it to the most recent ashdump in 
   # the trace directory 

   # first get a list of trace files sorted by modification time (most recently
   # modified comes first) and separate the list by spaces
   my $trcList = `ls -t $ENV{'T_DHOME_TRC'}/*.trc`;
   $trcList =~ s/\n/ /g;

   # find the trace files that have word "ashdump" in them
   my $ashTrcList = `grep -l -i ashdump $trcList`;

   # get the most recent one out of the list
   my @ashTrcList = split("\n", $ashTrcList);
   $traceFile = $ashTrcList[0];
  }

  # default endTime = now
  if(! defined($endTime)) {
    $endTime = time;
  }

  # default beginTime = endTime - 15
  if(! defined($beginTime)) {
    $beginTime = $endTime - 15;
  }

  if(! defined($version)) {
    $version = '11202';
  }

  if(! defined($reportLevel)) {
    $reportLevel = '<wait_class>{histogram}';
  }

  ::EMAGENT_PERL_DEBUG("Input Parameters:\n");
  ::EMAGENT_PERL_DEBUG("Trace File = $traceFile\n");
  ::EMAGENT_PERL_DEBUG("Begin Time = $beginTime\n");
  ::EMAGENT_PERL_DEBUG("End Time = $endTime\n");
  ::EMAGENT_PERL_DEBUG("Version = $version\n");
  ::EMAGENT_PERL_DEBUG("Report Level = $reportLevel\n");

  # parses the report level and populates componentTable and sessionTable
  parseReportLevel($reportLevel);

  my ($bucketCount,$bucketInterval) = getBucketInfo($beginTime
                                                    ,$endTime
                                                    ,$bucketMaxCount
                                                    ,$bucketMinInterval);

  if(!(defined $eventIdClassMap))
  {
    # build the event map. This map is used to translate event ids in the 
    # dump file to their names or their wait class names
    $eventIdClassMap = 
      buildEventMap(get_eventmap_filename($version));
  }

  open(DMP_DATA, $traceFile);

  # ashdump trace file can have multiple dumps
  # read the most recent dump contents
  my $ashdump_line;
  my @tmp_ashdump_lines = ();
  my @ashdump_lines = ();
  my $check_end = 0;

  while($ashdump_line = <DMP_DATA>) 
  {

    # search for begin of dump
    while($ashdump_line)
    {
      if($ashdump_line =~ /<<<ACTIVE SESSION HISTORY - PROCESS TRACE DUMP BEGIN>>>/)
      {

        # keep old ashdump lines 
        @tmp_ashdump_lines = @ashdump_lines;

        # clear old ashdump lines to get new one
        @ashdump_lines = ();

        $check_end = 0;
 
        # skip all lines upto and including this one
        last;     
      }
      
      $ashdump_line = <DMP_DATA>;
    }
    
    NEXTLINE:while($ashdump_line) 
    {
      if($ashdump_line =~ /<<<ACTIVE SESSION HISTORY - PROCESS TRACE DUMP END>>>/)
      {
        $check_end = 1;
        last;
      }
      # this loop is for db versions where ash samples
      # are not seperated by ####
      elsif($ashdump_line =~ /^\d/)
      {
        # this is a content line; take off the trailing '\n'...
        chop $ashdump_line;

        # ...and add it to the list
        $ashdump_lines[++$#ashdump_lines] = $ashdump_line;
      }
      elsif($ashdump_line =~ /#+/)
      {
         # skip the line with # and read the next one
         $ashdump_line = <DMP_DATA>;
         chop $ashdump_line;

         my $line_count = $#ashdump_lines;
         $line_count++;
         # some times one ash sample spans multiples lines.
         # join all lines to form a single line
         while(!($ashdump_line =~ /#+/ || $ashdump_line =~ /<<<ACTIVE SESSION HISTORY - PROCESS TRACE DUMP END>>>/))
         {
            $ashdump_lines[$line_count] .= $ashdump_line;
            $ashdump_line = <DMP_DATA>;
            chop $ashdump_line;
         }
         if($ashdump_line =~ /#+/) # begin of next sample
         {
           next NEXTLINE;
         }
   
         # end of ash dump
         if($ashdump_line =~ /<<<ACTIVE SESSION HISTORY - PROCESS TRACE DUMP END>>>/)
         {
           $check_end = 1; 
           last NEXTLINE;
         }
      }
      $ashdump_line = <DMP_DATA>;
    }

    if ($check_end == 0 )
    {
      @ashdump_lines = @tmp_ashdump_lines;
    } 
  }

   READASHDUMP: foreach my $line(@ashdump_lines) {

      $line =~ s/"//g;

      # Since this script returns back an XML document, it is important to 
      # replace special characters with XML entity refs so that the parsers
      # that consume this XML document don't fail with errors.
      $line = EmergPerfUtil::use_XML_entityrefs($line);

      @fields = split(/,/,$line);

      $linesRead++;

      # computing bucket number can be expensive so we do it once here 
      my $sampleTime = "";
      if($version =~ m/^10/ || $version =~ m/^11/){
        $sampleTime = getSecsSinceEpoch($fields[3]);
      }
      else{
        $sampleTime = getSecsSinceEpoch($fields[4]); 
      }

      if($sampleTime > $endTime){
        next READASHDUMP;
      }

      if($sampleTime < $beginTime) {
        ::EMAGENT_PERL_DEBUG(" ASH sample time $sampleTime < $endTime quit ");
        last;
      }

      if(! defined($histEndTime)) {
        $histEndTime = $sampleTime;
      }
      $histBeginTime = $sampleTime;

      my $bucketNumber = int ( ($sampleTime - $beginTime) / $bucketInterval);

      ##############
      # handle session details
      ##############
      {
        my $columns = $dimTable{physical_session}->{columns};
        my $session = "$fields[$columns->[0]],$fields[$columns->[1]],$fields[$columns->[2]]";

        if(exists $sessionTable{$session}) {
          $sessionDetailsTable{$session}->{user} = 
            $fields[$dimTable{user}->{columns}->[0]];
          $sessionDetailsTable{$session}->{service} = 
            $fields[$dimTable{service}->{columns}->[0]];
          $sessionDetailsTable{$session}->{program} = 
            $fields[$dimTable{program}->{columns}->[0]];
          $sessionDetailsTable{$session}->{module} = 
            $fields[$dimTable{module}->{columns}->[0]];
          $sessionDetailsTable{$session}->{action} = 
            $fields[$dimTable{action}->{columns}->[0]];
          $sessionDetailsTable{$session}->{client_id} = 
            $fields[$dimTable{client_id}->{columns}->[0]];
          $sessionDetailsTable{$session}->{sql_id} = 
            $fields[$dimTable{sql_id}->{columns}->[0]];
          $sessionDetailsTable{$session}->{p1} = 
            $fields[$dimTable{p1}->{columns}->[0]];
          $sessionDetailsTable{$session}->{p2} = 
            $fields[$dimTable{p2}->{columns}->[0]];
          $sessionDetailsTable{$session}->{p3} = 
            $fields[$dimTable{p3}->{columns}->[0]];
          $sessionDetailsTable{$session}->{'wait_class'} = 
            $eventIdClassMap->{$fields[$dimTable{event_id}->{columns}->[0]]}->[1];
          $sessionDetailsTable{$session}->{'wait_event_text'} = 
            $eventIdClassMap->{$fields[$dimTable{event_id}->{columns}->[0]]}->[0];

          if($sessionTable{$session} == $SESSION_EVENT_HISTOGRAM) {
            my $event 
              = $eventIdClassMap->{$fields[$dimTable{event}->{columns}->[0]]}->[0];
            $sessionDetailsTable{$session}->{histogram}->[$bucketNumber]->{$event}++;
          }
        }
      }

      ##############
      # handle 1d treemaps/histograms for all dimensions
      ##############
      {

        # Phase 1 : Aggregate counts of all top members 
        for my $dim (keys %componentTable) {
          my $mem;
          my $columns = $dimTable{$dim}->{columns};
          if($dim eq 'wait_class') {
            # wait class only needs one pass over the file so we compute
            # both 1d treemaps or histograms here depends on what's requested
            if($fields[($dimTable{'wait_time'}->{columns}[0])] == 0) {
              # if the session is in a wait
              $mem = $eventIdClassMap->{$fields[$columns->[0]]}->[1];
            }
            else {
              # if the session is not in wait, its on cpu
              $mem = 'CPU';
            }
            $dimStatsTable{$dim}->{top_mems}->{$mem}++;
            $dimStatsTable{$dim}->{histogram}->[$bucketNumber]->{$mem}++;
          }
          else {
            # all other dimensions (other than 'wait class' that is) need 
            # one pass to compute 1d treemaps and two passes to compute 
            # histograms. Here we compute the 1d treemap

            # remove surrounding double quotes and weird characters (^M)
            $fields[$columns->[0]] =~ s/^"|"$|\cM//g;

            # XXX - the above doesn't remove the case where its a single "
            # this comes up when trying to parse ecid, which is the last 
            # field in 10.2.x. There was an extra ^M character which I added
            # a filter for above, but couldn't remove this single " character.
            # Once the above substitution command is fixed, we can remove this
            $fields[$columns->[0]] =~ s/^"//g;

            # construct the dimension member
            $mem = $fields[$columns->[0]];
            foreach (1..$#{$columns}) { 
              $fields[$columns->[$_]] =~ s/^"|"$//g;
              $mem .= ",$fields[$columns->[$_]]";
            }
  
            $dimStatsTable{$dim}->{top_mems}->{$mem}++;
          }
  
        }
      }
  }

  close(DMP_DATA);

  #dumpOutput(\%dimStatsTable, \%sessionDetailsTable, "allkeysbefore.out");


  # Phase 2 : Keep only top n and coalesce rest into "others" 
  for my $dim (keys %dimStatsTable) {
    my $topMemsRef = $dimStatsTable{$dim}->{top_mems};

    # For the wait_class dimension, there's no coalescing. So simply add the
    # the METADATA bucket for keeping track of various counts 
    if($dim eq 'wait_class') {
      $topMemsRef->{'METADATA'} = 
        {cnt_all => $linesRead
       , cnt_others => 0
       , num_others => 0
       , histBeginTime => $beginTime
       , histEndTime => $endTime
       , bucketCount => $bucketCount
       , bucketInterval => $bucketInterval};
    }
    else {
      # for all dims other than 'wait_class', we need to keep only the top 
      # MAX_MEMBERS members (say, 5). So for these dimensions there will be 
      # MAX_MEMBERS + 1 buckets

      # first sort the list of member names by descending order of their counts
      my @sortedMems = sort {$topMemsRef->{$b} <=> $topMemsRef->{$a}}
                         keys %{$topMemsRef};

      # a new list of top members + METADATA 
      my $newTopMemsRef = {};

      # cntAll includes nulls
      my $cntAll = 0;

      # cnt of all the members not in the top
      my $cntOthers = 0;

      my $i = 0;
      for (0 .. $#sortedMems) {
        $cntAll += $topMemsRef->{$sortedMems[$_]};

        # filter out the null or empty values
        if($sortedMems[$_] ne "") {
          # keep the top n 
          if($i++ < $MAX_MEMBERS) {
            $newTopMemsRef->{$sortedMems[$_]} = 
              $topMemsRef->{$sortedMems[$_]};
          }
          else {
            $cntOthers +=
              $topMemsRef->{$sortedMems[$_]};
          }
        }
      }
      $newTopMemsRef->{'METADATA'} = 
        {cnt_all => $cntAll
       , cnt_others => $cntOthers
       , num_others => ($#sortedMems - $MAX_MEMBERS) < 0
                         ? 0
                         : ($#sortedMems - $MAX_MEMBERS)
       , histBeginTime => $beginTime
       , histEndTime => $endTime
       , bucketCount => $bucketCount
       , bucketInterval => $bucketInterval};
      $topMemsRef = $newTopMemsRef;
    }
  }

  #dumpOutput(\%dimStatsTable, \%sessionDetailsTable, 'allkeysafter.out');

  return writeToXml(\%dimStatsTable, \%sessionDetailsTable);


  # XXX Phase 3 - return global histograms
  # If any of the components need a global histogram, then we need to cache 
  # the raw data for that dimension in say dimStatsTable{$dim}->{raw_data} as
  # an array of values and then this phase would simply run through that array
  # and construct an histogram for it. 
}

#
# Subroutine: writeToXml
#  Constructs an xml document (represented as a string) for each of the 
#  components and session details. 
#
#  $_[0] => dimStatsTable - table containing data for each of the components
#  $_[1] => sessionDetailsTable - table containing session attributes and, 
#    if requested, event histograms for each of the sessions
#
#  Returns a string containing the xml report (see docs for getReport)
sub writeToXml {
  my ($dimStatsTable,$sessionDetailsTable) = @_;
  my $outXml = "<report><body><dim_list>";

  # dim/top_mems/@count is omitted, it is simply sum(count) of all top members
  # dim/top_mems/@cnt_others - since "others" is omitted as a separate member,
  #                            this represents count of "others"
  # dim/top_mems/@num_others - this is the number of members in "others"
  foreach my $dim (keys %{$dimStatsTable}) {
    my $metadataRef = $dimStatsTable->{$dim}->{top_mems}->{METADATA};
    $outXml .= 
      "<dim name=\"$dim\" "
    . "count=\"$metadataRef->{cnt_all}\""
    . "><top_mems "
    . "cnt_others=\"$metadataRef->{cnt_others}\" "
    . "num_others=\"$metadataRef->{num_others}\" "
    . ">";

    # write out each of the top_mems
    foreach my $mem (keys % {$dimStatsTable->{$dim}->{top_mems}}) {
      if ($mem ne 'METADATA') {
        $outXml .= 
          "<mem name=\"$mem\">"
        . "$dimStatsTable->{$dim}->{top_mems}->{$mem}"
        . "</mem>";
      }
    }
    $outXml .= 
      "</top_mems>"
    . "<histogram bucketCount=\"$metadataRef->{bucketCount}\" "
    . "bucketInterval=\"$metadataRef->{bucketInterval}\" "
    . "beginTime=\"$metadataRef->{histBeginTime}\" "
    . "endTime=\"$metadataRef->{histEndTime}\">";

    # write out each bucket in the histogram
    for my $bucketNumber (0 .. $#{$dimStatsTable->{$dim}->{histogram}}) { 
      # XXX need bucket/count, top_mems/cnt_others, top_mems/num_others
      my $bucketRef = $dimStatsTable->{$dim}->{histogram}->[$bucketNumber];
      my $bucketXml = 
        "<bucket number=\"".($bucketNumber+1)."\">"
      . "<top_mems>";

      my $memCount = 0;
      # write out each mem of each bucket
      foreach my $mem (keys % {$bucketRef}) {
        if ($mem ne 'METADATA') {
          $bucketXml .= 
            "<mem name=\"$mem\">"
          . "$bucketRef->{$mem}"
          . "</mem>";
          $memCount++;
        }
      }
      $bucketXml .= "</top_mems></bucket>";
      if($memCount > 0){
        $outXml .= $bucketXml;
      }
    }
    $outXml .= "</histogram></dim>";
  }

  $outXml .= "</dim_list><session_details>";  

  # now write out session in the session_details
  foreach my $session (keys % {$sessionDetailsTable}) {
    my $sessionRef = $sessionDetailsTable->{$session};

    # first write out the attributes
    $outXml .= 
      "<session id=\"$session\" "
    . "user=\"$sessionRef->{user}\" "
    . "service=\"$sessionRef->{service}\" "
    . "program=\"$sessionRef->{program}\" "
    . "module=\"$sessionRef->{module}\" "
    . "action=\"$sessionRef->{action}\" "
    . "client_id=\"$sessionRef->{client_id}\" "
    . "sql_id=\"$sessionRef->{sql_id}\" "
    . "p1=\"$sessionRef->{p1}\" "
    . "p2=\"$sessionRef->{p2}\" " 
    . "p3=\"$sessionRef->{p3}\" "
    . "wait_event_text=\"$sessionRef->{wait_event_text}\" >";

    # then write out the event histogram
    if($sessionTable{$session} == $SESSION_EVENT_HISTOGRAM) {

      $outXml .= "<event_histogram>";

      # write out each bucket of the event histogram
      for my $bucketNumber 
                (0 .. $#{$sessionRef->{histogram}}) { 
        $outXml .= 
          "<bucket number=\"".($bucketNumber+1)."\">"
        . "<top_mems>";
        my $bucketRef = $sessionRef->{histogram}->[$bucketNumber];

        # write out each event of each bucket
        foreach my $event (keys % {$bucketRef}) {
            $outXml .= 
              "<mem name=\"$event\">"
            . "$bucketRef->{$event}"
            . "</mem>";
        }
        $outXml .= "</top_mems></bucket>";
      }
      $outXml .= "</event_histogram>";
    }
    $outXml .= "</session>";
  }

  $outXml .= "</session_details></body></report>";
  return $outXml;
}

sub get_eventmap_filename
{
  my $version = shift;
  return($MAP_DIR_ROOT . 'event_map_' . $version . '.txt');
}


#
# Subroutine: buildEventMap
#   Builds a mapping from event ids to event names and wait class. 
#   Input file contains the mapping as maintained by the fixed table x$ksled
#
#  $_[0] => file containing the mapping. This file can be created as follows
#     oradebug setmypid
#     oradebug direct_access enable trace
#     oradebug direct_access select ksledhash,kslednam,ksledclass from x$ksled
#
#  Returns a ref to map of event id to name and wait class

sub buildEventMap {
  my $eventFile = shift;
  my $line;
  my $column;
  my $eventId;
  my $eventName;
  my $class;
  my %map;

  ::EMAGENT_PERL_DEBUG("Building event map from file $eventFile");

  open(EVENT_FILE, $eventFile);

  # skip over until hitting the first interesting line 
  while($line = <EVENT_FILE>) {
    if($line =~ /KSLEDHASH/) {
      last;
    }
  }


  # iterate over runs of three lines containing the event id, name, and class
  do {
    if($line =~ /KSLEDHASH/) {

      # get event id
      ($column, $eventId) = split(/=\s*/,$line);
      chomp($eventId);
      # on windows chomp may not work
      $eventId = EmergPerfUtil::trim_new_line($eventId);

      # get event name
      $line = <EVENT_FILE>;
      ($column, $eventName) = split(/=\s*/,$line);
      chomp($eventName);
      $eventName = EmergPerfUtil::trim_new_line($eventName);

      # get wait class name
      $line = <EVENT_FILE>;
      ($column, $class) = split(/=\s*/,$line);
      chomp($class);
      $class = EmergPerfUtil::trim_new_line($class);

      # update the map
      $map{$eventId}=[$eventName,$class];
    }

  } while($line = <EVENT_FILE>);

  close(EVENT_FILE);
  ::EMAGENT_PERL_DEBUG("Found ".scalar(keys %map));
  return \%map;
}

#
# Subroutine: getSecsSinceEpoch
#   Converts time from string to number of seconds since epoch. 
#
#  $_[0] => time the following format "mo-dd-yyyy hh:mi:ss"
#
#  Returns the number of seconds since epoch 
sub getSecsSinceEpoch {
  my ($mm,$dd,$yyyy,$hh,$mi,$ss,$ms) = ($_[0] =~ /(\d+)-(\d+)-(\d+)\s(\d+):(\d+):(\d+).(\d+)/);
  my $secsSinceEpoch = Time::Local::timelocal($ss,$mi,$hh,$dd,($mm-1),$yyyy);
  return $secsSinceEpoch;
}

#
# Subroutine: parseReportLevel
#   Parses the report level string and builds two key data structures that 
#   store information about the components to be built.
#   1. componentTable - this is a mapping of dimension names to component ids
#                      e.g., componentTable{'wait_class'} = COMP_ID_HISTOGRAM
#   2. sessionTable - this is a mapping of session ids to session detail types,
#                     either SESSION_ATTRIBUTES or SESSION_EVENT_HISTOGRAM
#                      e.g., sessionTable{"1,1740,9"} = SESSION_EVENT_HISTOGRAM
#
#  $_[0] => report level string 
#             e.g., "<wait_class>{histogram} \
#                    <session_attributes>{\"1,1659,46066\",\"1,2540,7\"} \
#                    <session_event_histogram>{\"1,1740,9\"}"
#
#  Populates global variables componentTable and sessionTable  

sub parseReportLevel {
  my $level = shift;

  # break the level string into chunks 
  # for the level string given in the subroutine header, 
  #    chunk 1 = "<wait_class>{histogram}"
  #    chunk 2 = "<session_attributes>{\"1,1659,46066\",\"1,2540,7\"}"
  #    chunk 3 = "<session_event_histogram>{\"1,1740,9\"}"
  my @chunks = split(/}/, $level);

  my $dim;
  my $component;
  my @sessionList;

  foreach my $chunk (@chunks) {
    # each chunk consists of the dimension enclosed by '<', '>' and the 
    # component enclosed by '{' and '}'. We currently do not handle the case
    # where there are multiple dimensions or components in the list
    ($dim,$component) = ($chunk =~ /<(.*)>{(.*)/  ? ($1,$2) : (-1,-1)); 

    # Based on the component, we update the appropriate table
    if($dim eq 'session_attributes' || $dim eq 'session_event_histogram') {
      @sessionList = split(/","/, $component);
      foreach my $session (@sessionList) {
        $session =~ s/"//g;
        $sessionTable{$session} = ($dim eq 'session_attributes') 
                                     ? $SESSION_ATTRIBUTES 
                                     : $SESSION_EVENT_HISTOGRAM;
      }
    }
    else {
      $componentTable{$dim} = ($component eq 'histogram') 
                                  ? $COMP_ID_HISTOGRAM
                                  : $COMP_ID_TREEMAP1D;
    }
  }
}


#
# Subroutine: getBucketInfo
#   Computes two key aspects of histograms on ash data:
#     1) bucket_count
#     2) bucket_interval
#   based on the begin and end time and optionally one or both of 
#   bucket_max_count and bucket_min_interval. 
#   There can be a conflict between bucket_max_count and 
#   bucket_min_interval if both are provided. In this case, the algorithm
#   picks bucket_min_interval to compute bucket_count and returns it even
#   if it turns out to be higher than bucket_max_count
#
#  $_[0] => begin time in secs since epoch (e.g., 1254673614)
#           optional: default value = (now - 15 secs)
#  $_[1] => end time in secs since epoch (e.g., 1254673614)
#           optional: default value = (now)
#  The next two parameters are optional. See getBucketInfo for details
#  $_[2] => bucket max count (e.g., 16)
#  $_[3] => bucket min interval (e.g., 1)
#
#  Returns a two-member list (bucketCount, bucketInterval)
sub getBucketInfo {
  my ($beginTime,$endTime,$bucketMaxCount,$bucketMinInterval) = @_;

  # return values
  my $bucketCount;
  my $bucketInterval;

  # determine interval duration
  my $duration = $endTime - $beginTime;

  # if either begin or end time is not specified, we return a default 
  # interval of 8 seconds and an undefined bucket count.
  if( (! defined($beginTime)) || (! defined($endTime))) {
    $bucketInterval = 8;
    $bucketCount = undef;
    return ($bucketCount,$bucketInterval);
  }

  # set max count to the global constant if not provided by caller
  $bucketMaxCount = defined($bucketMaxCount) 
                      ? $bucketMaxCount 
                      : $BUCKET_MAX_COUNT;
  # use min interval if its provided
  if( defined($bucketMinInterval) ) {
    $bucketInterval = int($bucketMinInterval);
    $bucketInterval = ($bucketInterval < 0) ? 1 : $bucketInterval;
  }
  else {
    #
    # if we compute the bucket_interval, ensure that it rounds up to the
    # next power of 2. In this case, can only be 1, 2,4,8,16,... seconds.
    #
    $bucketInterval = 1;
    while(($duration / $bucketInterval) > $bucketMaxCount) {
      $bucketInterval *= 2;
    }
  }
  # derive bucket count from bucket interval and return both
  $bucketCount = int ($duration / $bucketInterval);
  
  # Conisder the last bucket which is not of length 
  # $bucketInterval in the bucketCount 
  $bucketCount += ($duration % $bucketInterval)!=0?1:0;

  return ($bucketCount,$bucketInterval);
}


#
# Subroutine: dumpOutput
#   Debug routine
sub dumpOutput {
  my ($dimStatsTable,$sessionDetailsTable,$out_file) = @_;

  open(OUT_DATA, ">"."$out_file");
  print OUT_DATA "\n\nAll Stats\n==========\n\n";
  foreach my $dim (keys %{$dimStatsTable}) {
    print OUT_DATA "Dimension $dim\n";
    print OUT_DATA "\tTop Members:\n";
    foreach my $mem (keys % {$dimStatsTable->{$dim}->{top_mems}}) {
      if ($mem eq 'METADATA') {
        print OUT_DATA 
          "\t\tMETADATA: cnt_all=$dimStatsTable->{$dim}->{top_mems}->{$mem}->{cnt_all}, " .
          "cnt_others=$dimStatsTable->{$dim}->{top_mems}->{$mem}->{cnt_others}, " .
          "num_others=$dimStatsTable->{$dim}->{top_mems}->{$mem}->{num_others}\n";
      }
      else {
        print OUT_DATA "\t\t$mem=$dimStatsTable->{$dim}->{top_mems}->{$mem}\n";
      }
    }
    print OUT_DATA "\n\n\tHistogram\n";
    for my $bucketNumber (0 .. $#{$dimStatsTable->{$dim}->{histogram}}) { 
      print OUT_DATA "\t\tBucket $bucketNumber\n";
      foreach my $mem 
        (keys % {$dimStatsTable->{$dim}->{histogram}->[$bucketNumber]}) {
        if ($mem eq 'METADATA') {
          print OUT_DATA 
            "METADATA: cnt_all=$dimStatsTable->{$dim}->{histogram}->[$bucketNumber]->{$mem}->{cnt_all}" .
            "METADATA: cnt_others=$dimStatsTable->{$dim}->{histogram}->[$bucketNumber]->{$mem}->{cnt_others}" .
            "METADATA: num_others=$dimStatsTable->{$dim}->{histogram}->[$bucketNumber]->{$mem}->{num_others}";
        }
        else {
          print OUT_DATA "\t\t\t$mem=$dimStatsTable->{$dim}->{histogram}->[$bucketNumber]->{$mem}\n";
        }
      }
    }
    print OUT_DATA "\n\n\n\n";
  }
  print OUT_DATA "\n\n\nSession Details\n============\n";
  foreach my $session (keys % {$sessionDetailsTable}) {
    print OUT_DATA "Session $session\n";
    print OUT_DATA "   user=$sessionDetailsTable->{$session}->{user}\n";
    print OUT_DATA "   service=$sessionDetailsTable->{$session}->{service}\n";
    print OUT_DATA "   program=$sessionDetailsTable->{$session}->{program}\n";
    print OUT_DATA "   module=$sessionDetailsTable->{$session}->{module}\n";
    print OUT_DATA "   action=$sessionDetailsTable->{$session}->{action}\n";
    print OUT_DATA "   client_id=$sessionDetailsTable->{$session}->{client_id}\n";
    print OUT_DATA "   p1=$sessionDetailsTable->{$session}->{p1}\n";
    print OUT_DATA "   p2=$sessionDetailsTable->{$session}->{p2}\n";
    print OUT_DATA "   p3=$sessionDetailsTable->{$session}->{p3}\n";
    print OUT_DATA "   sql_id=$sessionDetailsTable->{$session}->{sql_id}\n";
    print OUT_DATA "   wait_event_text=$sessionDetailsTable->{$session}->{wait_event_text}\n";
  }
  close(OUT_DATA);
}


# transform version to the form acceptable by ashMetric script 
sub _get_target_version
{
  # remove the dots in version so that the AshViewer can use it 
  my $tg_version = shift;
  $tg_version =~ s/\.//g;

  if($tg_version =~ m/^11203/)
  {
    $tg_version = "11203";
  }
  elsif($tg_version =~ m/^11202/)
  {
    $tg_version = "11202";
  }
  elsif($tg_version =~ m/^11201/)
  {
    $tg_version = "11201";
  }
  elsif($tg_version =~ m/^10205/)
  {
    $tg_version = "10205";
  }
  elsif($tg_version =~ m/^10204/)
  {
    $tg_version = "10204";
  }

  ::EMAGENT_PERL_DEBUG("Target version is $tg_version");
  return $tg_version;
}

sub initializeDimTable
{
   my $version = shift;

   if($version eq "10204" || 
      $version eq "10205")
   {
      %dimTable = (wait_class         => {columns => [25]}
                  ,event_id           => {columns => [25]}
                  ,sql_id             => {columns => [7]}
                  ,user               => {columns => [6]}
                  ,service            => {columns => [10]}
                  ,program            => {columns => [33]}
                  ,module             => {columns => [34]}
                  ,action             => {columns => [35]}
                  ,client_id          => {columns => [36]}
                  ,p1                 => {columns => [27]}
                  ,p2                 => {columns => [28]}
                  ,p3                 => {columns => [29]}
                  ,wait_time          => {columns => [30]}
                  ,physical_session   => {columns => [1,4,5]});
   }
   elsif($version =~ m/^11/) # >= 11G
   {
      %dimTable = (wait_class         => {columns => [27]}
                  ,event_id           => {columns => [27]}
                  ,p1                 => {columns => [29]}
                  ,p2                 => {columns => [30]}
                  ,p3                 => {columns => [31]}
                  ,wait_time          => {columns => [32]}
                  ,sql_id             => {columns => [8]}
                  ,user               => {columns => [7]}
                  ,service            => {columns => [46]}
                  ,program            => {columns => [47]}
                  ,module             => {columns => [48]}
                  ,action             => {columns => [49]}
                  ,client_id          => {columns => [50]}
                  ,machine            => {columns => [51]}
                  ,port               => {columns => [52]}
                  ,ecid               => {columns => [53]}
                  ,physical_session   => {columns => [1,4,5]});
   }
   else #>= 12G
   {
      %dimTable = (wait_class         => {columns => [28]}
                  ,event_id           => {columns => [28]}
                  ,p1                 => {columns => [30]}
                  ,p2                 => {columns => [31]}
                  ,p3                 => {columns => [32]}
                  ,wait_time          => {columns => [33]}
                  ,sql_id             => {columns => [9]}
                  ,user               => {columns => [10]}
                  ,service            => {columns => [47]}
                  ,program            => {columns => [48]}
                  ,module             => {columns => [49]}
                  ,action             => {columns => [50]}
                  ,client_id          => {columns => [51]}
                  ,machine            => {columns => [52]}
                  ,port               => {columns => [53]}
                  ,ecid               => {columns => [54]}
                  ,physical_session   => {columns => [1,5,6]});
   }
}
