/* This table stores the overall snapshot info */
CREATE TABLE dbsnmp.mgmt_snapshot(
       instance_number 		NUMBER NOT NULL,
       snap_id         		NUMBER NOT NULL,
       capture_time    		DATE NOT NULL,
       db_time         		NUMBER,
       cpu_time        		NUMBER,
       application_wait_time	NUMBER,
       cluster_wait_time	NUMBER,
       user_io_wait_time	NUMBER,
       concurrency_wait_time	NUMBER) NOLOGGING;
/* This table is used to store v$sql snapshot, while configuring baseline. */
CREATE TABLE dbsnmp.mgmt_snapshot_sql(
       snap_id         NUMBER NOT NULL,
       sql_id          VARCHAR2(13),
       hash_value      NUMBER NOT NULL,
       child_number    NUMBER NOT NULL,
       elapsed_time    NUMBER NOT NULL,
       executions      NUMBER NOT NULL) NOLOGGING;
/* This table stores the overall baseline info */
CREATE TABLE dbsnmp.mgmt_baseline(
       instance_number 		NUMBER NOT NULL,
       capture_time    		DATE NOT NULL,
       prev_capture_time 	DATE NOT NULL,
       db_time         		NUMBER,
       cpu_time                 NUMBER,
       application_wait_time    NUMBER,
       cluster_wait_time        NUMBER,
       user_io_wait_time        NUMBER,
       concurrency_wait_time    NUMBER);
/* The baseline table stores the identity of the SQL statements as well as
** the baseline elapsed_time_per_execution for those statements.
*/
CREATE TABLE dbsnmp.mgmt_baseline_sql(
       instance_number NUMBER NOT NULL,
       sql_text        VARCHAR2(1000),
       sql_id          VARCHAR2(13),
       hash_value      NUMBER NOT NULL,
       executions      NUMBER,
       elapsed_time    NUMBER,
       t_per_exec      NUMBER NOT NULL);
/* This table stores the overall capture info */
CREATE TABLE dbsnmp.mgmt_capture(
       instance_number 		NUMBER NOT NULL,
       capture_id      		NUMBER NOT NULL,
       capture_time    		DATE NOT NULL,
       db_time         		NUMBER,
       cpu_time                 NUMBER,
       application_wait_time    NUMBER,
       cluster_wait_time        NUMBER,
       user_io_wait_time        NUMBER,
       concurrency_wait_time    NUMBER);
/* Statistics are captured from v$sql at a regular interval and stored in
** the capture table. The columns capture_id and address together form
** the primary key. The elapsed_time and executions columns store cumulative
** values as opposed to deltas.
*/
CREATE TABLE dbsnmp.mgmt_capture_sql(
       capture_id      NUMBER NOT NULL,
       sql_id          VARCHAR2(13),
       hash_value      NUMBER NOT NULL,
       elapsed_time    NUMBER,
       executions      NUMBER);
/* Remember instance startup time in this table so that we can detect when
** the instance has been bounced.
*/
CREATE TABLE dbsnmp.mgmt_response_config(
       instance_number NUMBER NOT NULL,
       startup_time    DATE);
/* This table stores the overall latest metric info */
CREATE TABLE dbsnmp.mgmt_latest(
       instance_number 		NUMBER NOT NULL,
       capture_id      		NUMBER NOT NULL,
       capture_time    		DATE NOT NULL,
       prev_capture_time 	DATE NOT NULL,
       sql_response_time 	NUMBER NOT NULL,
       adjusted_sql_response_time NUMBER NOT NULL,
       baseline_sql_response_time NUMBER NOT NULL,
       relative_sql_response_time NUMBER NOT NULL,
       db_time         		NUMBER,
       cpu_time                 NUMBER,
       application_wait_time    NUMBER,
       cluster_wait_time        NUMBER,
       user_io_wait_time        NUMBER,
       concurrency_wait_time    NUMBER);
/* This table stores the sql info of the latest metric*/
CREATE TABLE dbsnmp.mgmt_latest_sql(
       capture_id      NUMBER NOT NULL,
       sql_id          VARCHAR2(13),
       hash_value      NUMBER NOT NULL,
       executions      NUMBER,
       elapsed_time    NUMBER,
       t_per_exec      NUMBER,
       adjusted_elapsed_time NUMBER);
/* This table stores the overall info for those metrics whose metric value
** exceeds the threshold.
*/
CREATE TABLE dbsnmp.mgmt_history(
       instance_number 		NUMBER NOT NULL,
       capture_id      		NUMBER NOT NULL,
       capture_time    		DATE NOT NULL,
       prev_capture_time 	DATE NOT NULL,
       sql_response_time 	NUMBER NOT NULL,
       adjusted_sql_response_time NUMBER NOT NULL,
       baseline_sql_response_time NUMBER NOT NULL,
       relative_sql_response_time NUMBER NOT NULL,
       db_time         		NUMBER,
       cpu_time                 NUMBER,
       application_wait_time    NUMBER,
       cluster_wait_time        NUMBER,
       user_io_wait_time        NUMBER,
       concurrency_wait_time    NUMBER);
/* this table is used to save the baseline and capture data when the sql 
** response time is exceeded four times (THRESHOLD_FOR_HISTORY) of the baseline. 
** We always keep no more than 25 collections for each instance.
*/
CREATE TABLE dbsnmp.mgmt_history_sql(
       capture_id       NUMBER NOT NULL,
       sql_id           VARCHAR2(13),
       hash_value       NUMBER NOT NULL,
       executions       NUMBER,
       elapsed_time     NUMBER,
       t_per_exec       NUMBER,
       adjusted_elapsed_time NUMBER);
CREATE GLOBAL TEMPORARY TABLE dbsnmp.mgmt_tempt_sql(
       sql_id          VARCHAR2(13),
       hash_value      NUMBER NOT NULL,
       elapsed_time    NUMBER NOT NULL,
       executions      NUMBER NOT NULL)
       ON COMMIT DELETE ROWS;
CREATE sequence dbsnmp.mgmt_response_capture_id 
       START WITH 1 INCREMENT BY 1 ORDER;
CREATE sequence dbsnmp.mgmt_response_snapshot_id 
       START WITH 1 INCREMENT BY 1 ORDER;
CREATE OR REPLACE VIEW dbsnmp.mgmt_response_baseline AS
   SELECT b.instance_number, s.sql_text, s.hash_value, v.address, s.t_per_exec
     FROM dbsnmp.mgmt_baseline b, dbsnmp.mgmt_baseline_sql s, v$sqlarea v
    WHERE b.instance_number = s.instance_number
      AND s.hash_value = v.hash_value;
CREATE OR REPLACE PACKAGE dbsnmp.mgmt_response wrapped 
0
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
3
9
9000000
1
4
0 
2a
2 :e:
1PACKAGE:
1MGMT_RESPONSE:
1TYPE:
1METRIC_CURSOR_TYPE:
1REF:
1CURSOR:
1FUNCTION:
1GETVERSION:
1RETURN:
1VARCHAR2:
1CREATE_BASELINE:
1SNAP_ID_1:
1NUMBER:
1N_STMTS:
1DELETE_SQL_FROM_BASELINE:
1HASH:
1DELETE_BASELINE:
1RESET_SNAPSHOT_STATS:
1RESET_CAPTURE_STATS:
1CAPTURE_STATS:
1TAKE_V$SQL_SNAPSHOT:
1MIN_EXECS:
1CALC_METRIC:
1METRIC:
1OUT:
1PURGE_OLD_STATS:
1PURGE_OLD_HISTORY:
1SAVE_TO_HISTORY:
1GET_METRIC_CURS:
1METRIC_CURS:
1GET_LATEST_CURS:
1GET_TIME_STATS:
1DB_TIME:
1CPU_TIME:
1APPLICATION_WAIT_TIME:
1CLUSTER_WAIT_TIME:
1USER_IO_WAIT_TIME:
1CONCURRENCY_WAIT_TIME:
1GETDURATION:
1START_DATE:
1TIMESTAMP:
1END_DATE:
0

0
0
a0
2
0 a0 97 a0 9d :2 a0 c8 77
a0 8d a0 b4 a0 2c 6a 9a
8f a0 b0 3d 8f a0 b0 3d
b4 55 6a 9a 8f a0 b0 3d
b4 55 6a 9a b4 55 6a 9a
8f a0 b0 3d b4 55 6a 9a
b4 55 6a 9a b4 55 6a 9a
8f a0 b0 3d b4 55 6a 9a
96 :2 a0 b0 54 b4 55 6a 9a
b4 55 6a 9a b4 55 6a 9a
b4 55 6a 9a 96 :2 a0 b0 54
b4 55 6a 9a 96 :2 a0 b0 54
b4 55 6a 9a 96 :2 a0 b0 54
96 :2 a0 b0 54 96 :2 a0 b0 54
96 :2 a0 b0 54 96 :2 a0 b0 54
96 :2 a0 b0 54 b4 55 6a a0
8d 8f a0 b0 3d 8f a0 b0
3d b4 :2 a0 2c 6a a0 :2 aa 59
58 1d 17 b5 
a0
2
0 3 7 11 25 19 1d 21
15 2c 30 44 48 49 4d 51
55 71 6d 6c 79 86 82 69
8e 81 93 97 9b b7 b3 7e
bf b2 c4 c8 cc af e0 e4
e8 104 100 ff 10c fc 111 115
119 12d 12e 132 136 14a 14b 14f
153 16f 16b 16a 177 167 17c 180
184 1a4 19c 1a0 19b 1ab 198 1b0
1b4 1b8 1cc 1cd 1d1 1d5 1e9 1ea
1ee 1f2 206 207 20b 20f 22f 227
22b 226 236 223 23b 23f 243 263
25b 25f 25a 26a 257 26f 273 277
297 28f 293 28e 29e 2af 2a7 2ab
28b 2b6 2c3 2bb 2bf 2a6 2ca 2db
2d3 2d7 2a3 2e2 2ef 2e7 2eb 2d2
2f6 307 2ff 303 2cf 30e 2fe 313
317 31b 31f 33b 337 2fb 343 34c
348 336 354 333 359 35d 361 365
369 36d 36f 371 374 377 378 381 
a0
2
0 1 10 4 9 1f 23 1f
:2 4 e 19 0 20 :2 4 e 1e
2b :2 1e 33 3e :2 33 1d :2 4 e
28 30 :2 28 27 :2 4 e 0 :2 4
e 23 30 :2 23 22 :2 4 e 0
:2 4 e 0 :2 4 e 22 2f :2 22
21 :2 4 e 1a 21 25 :2 1a 19
:2 4 e 0 :2 4 e 0 :2 4 e
0 :2 4 e 1e 2a 2e :2 1e 1d
:2 4 e 1e 2a 2e :2 1e 1d :2 4
e 1d 25 29 :2 1d 31 3a 3e
:2 31 46 5c 60 :2 46 8 1a 1e
:2 8 26 38 3c :2 26 8 1e 22
:2 8 1c :3 4 e 1a 28 :2 1a 33
3f :2 33 19 4a 51 :2 4 5 :7 1

a0
4
0 :2 1 :6 2 :3 3
0 :3 3 :c 4 :8 5
6 0 :2 6 :8 7
8 0 :2 8 9
0 :2 9 :8 a :9 b
c 0 :2 c d
0 :2 d e 0
:2 e :9 f :9 10 :10 11
:a 12 :5 13 :3 11 :f 14
15 :7 1 
383
4
:3 0 1 :3 0 2
:6 0 1 :2 0 3
:3 0 4 0 7
99 5 :3 0 6
:7 0 4 7 4
:4 0 7 :3 0 8
:a 0 f 2 :7 0
9 :4 0 a :3 0
c d 0 f
a e 0 99
b :a 0 1b 3
:7 0 5 7e 0
3 d :3 0 c
:7 0 13 12 :3 0
a af 0 7
d :3 0 e :7 0
17 16 :3 0 19
:2 0 1b 10 1a
0 99 f :a 0
23 4 :a 0 c
d :3 0 10 :7 0
1f 1e :3 0 21
:2 0 23 1c 22
0 99 11 :a 0
27 5 :7 0 25
:2 0 27 24 26
0 99 12 :a 0
2f 6 :7 0 10
:2 0 e d :3 0
c :7 0 2b 2a
:3 0 2d :2 0 2f
28 2e 0 99
13 :a 0 33 7
:8 0 31 :2 0 33
30 32 0 99
14 :a 0 37 8
:8 0 35 :2 0 37
34 36 0 99
15 :a 0 3f 9
:7 0 14 :2 0 12
d :3 0 16 :7 0
3b 3a :3 0 3d
:2 0 3f 38 3e
0 99 17 :a 0
48 a :7 0 18
:2 0 16 19 :3 0
d :3 0 18 :6 0
44 43 :3 0 46
:2 0 48 40 47
0 99 1a :a 0
4c b :8 0 4a
:2 0 4c 49 4b
0 99 1b :a 0
50 c :8 0 4e
:2 0 50 4d 4f
0 99 1c :a 0
54 d :8 0 52
:2 0 54 51 53
0 99 1d :a 0
5d e :7 0 1c
:2 0 1a 19 :3 0
4 :3 0 1e :6 0
59 58 :3 0 5b
:2 0 5d 55 5c
0 99 1f :a 0
66 f :7 0 20
:2 0 1e 19 :3 0
4 :3 0 1e :6 0
62 61 :3 0 64
:2 0 66 5e 65
0 99 20 :a 0
88 10 :7 0 24
2a3 0 22 19
:3 0 d :3 0 21
:6 0 6b 6a :3 0
28 2cf 0 26
19 :3 0 d :3 0
22 :6 0 70 6f
:3 0 19 :3 0 d
:3 0 23 :6 0 75
74 :3 0 2c 2fb
0 2a 19 :3 0
d :3 0 24 :6 0
7a 79 :3 0 19
:3 0 d :3 0 25
:6 0 7f 7e :3 0
35 333 0 2e
19 :3 0 d :3 0
26 :6 0 84 83
:3 0 86 :2 0 88
67 87 0 99
7 :3 0 27 :a 0
97 11 :7 0 39
:2 0 37 29 :3 0
28 :7 0 8d 8c
:3 0 29 :3 0 2a
:7 0 91 90 :3 0
9 :3 0 d :3 0
93 95 0 97
8a 96 0 99
2 :3 0 3c 9b
0 9b 99 9a
9c 2 9b 9e
0 9d 9c 9f
:8 0 
4e
4
:3 0 1 11 1
15 2 14 18
1 1d 1 20
1 29 1 2c
1 39 1 3c
1 41 1 45
1 56 1 5a
1 5f 1 63
1 68 1 6d
1 72 1 77
1 7c 1 81
6 6c 71 76
7b 80 85 1
8b 1 8f 2
8e 92 11 8
f 1b 23 27
2f 33 37 3f
48 4c 50 54
5d 66 88 97

1
4
0 
9e
0
1
14
11
22
0 1 1 1 1 1 1 1
1 1 1 1 1 1 1 1
1 0 0 0 
34 1 8
2 0 1
51 1 d
81 10 0
72 10 0
8b 11 0
5e 1 f
6d 10 0
24 1 5
5f f 0
56 e 0
28 1 6
4 1 0
8a 1 11
38 1 9
4d 1 c
30 1 7
8f 11 0
7c 10 0
29 6 0
11 3 0
68 10 0
55 1 e
49 1 b
10 1 3
1d 4 0
41 a 0
40 1 a
a 1 2
67 1 10
39 9 0
15 3 0
1c 1 4
77 10 0
0

/
CREATE OR REPLACE PACKAGE BODY dbsnmp.mgmt_response wrapped 
0
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
3
b
9000000
1
4
0 
113
2 :e:
1PACKAGE:
1BODY:
1MGMT_RESPONSE:
1PACKAGE_VERSION:
1VARCHAR2:
130:
110.2.0.1.0:
1NUMBER_OF_STMTS_TO_TRACK:
1NUMBER:
125:
1MIN_EXECS:
15:
1MAX_NUMBER_OF_STMTS_IN_HISTORY:
1THRESHOLD_FOR_HISTORY:
1500:
1MIN_DELTA_EXECS:
150:
1COLLECTION_EMPTY:
1-:
10.1:
1NEED_MORE_CAPTURE:
10.2:
1BASELINE_EMPTY:
10.3:
1BASELINE_UNAVAILABLE:
10.4:
1FUNCTION:
1GETVERSION:
1RETURN:
1TAKE_V$SQL_SNAPSHOT:
1SNAPSHOT_ID:
1INSTANCE_NUM:
1SQL_ID:
113:
1CAPTURE_TIME:
1DATE:
1DB_VERSION:
1DB_TIME:
1CPU_TIME:
1APPLICATION_WAIT_TIME:
1CLUSTER_WAIT_TIME:
1USER_IO_WAIT_TIME:
1CONCURRENCY_WAIT_TIME:
1DBMS_OUTPUT:
1PUT_LINE:
1in take_v$sql_snapshot:
1INSTANCE_NUMBER:
1TO_NUMBER:
1TRIM:
1TRAILING:
1.:
1SUBSTR:
1VERSION:
11:
12:
1 SYS$STANDARD_TRIM:
1V$INSTANCE:
1DBSNMP:
1MGMT_RESPONSE_SNAPSHOT_ID:
1NEXTVAL:
1DUAL:
1SYSDATE:
1>=:
110:
1GET_TIME_STATS:
1MGMT_SNAPSHOT:
1SNAP_ID:
1MGMT_SNAPSHOT_SQL:
1HASH_VALUE:
1CHILD_NUMBER:
1ELAPSED_TIME:
1EXECUTIONS:
1V$SQLAREA:
1>:
1PARSING_USER_ID:
1USER_ID:
1DBA_USERS:
1USERNAME:
1SYS:
1SYSMAN:
1MGMT_RESPONSE_CONFIG:
1=:
1STARTUP_TIME:
1COMMIT:
1OUT:
1TYPE:
1STATCURTYP:
1REF:
1CURSOR:
1STATS_CUR:
1SQL_STMT:
1200:
1SELECT value FROM v$sysstat WHERE stat_id IN (24469293, 1099569955, 243203433+
17, 3332107451, 3649082374, 3868577743) ORDER BY stat_id:
1VALUE:
1V$SYSSTAT:
1OPEN:
1I:
16:
1LOOP:
1EXIT:
1NOTFOUND:
13:
14:
1CASE_NOT_FOUND:
1CLOSE:
1CREATE_BASELINE:
1SNAP_ID_1:
1N_STMTS:
1HASH:
1DE:
1DT:
1T:
1PREV_CAPTURE_TIME:
1PREV_DB_TIME:
1PREV_CPU_TIME:
1PREV_APPLICATION_WAIT_TIME:
1PREV_CLUSTER_WAIT_TIME:
1PREV_USER_IO_WAIT_TIME:
1PREV_CONCURRENCY_WAIT_TIME:
1DELTA_DB_TIME:
1DELTA_CPU_TIME:
1DELTA_APPLICATION_WAIT_TIME:
1DELTA_CLUSTER_WAIT_TIME:
1DELTA_USER_IO_WAIT_TIME:
1DELTA_CONCURRENCY_WAIT_TIME:
1C1:
1INST_NUM:
1ID_1:
1MIN_EXEC:
1DELTA_E:
1DELTA_T:
1/:
1T_PER_EXEC:
1SNAP1:
1SNAP2:
1in create_baseline, snap_id = :
1||:
1RESET_SNAPSHOT_STATS:
1MGMT_BASELINE_SQL:
1MGMT_BASELINE:
1100:
1B:
1SQL_TEXT:
1V:
1DELETE_SQL_FROM_BASELINE:
1MGMT_CAPTURE_SQL:
1CAPTURE_ID:
1MGMT_CAPTURE:
1MGMT_LATEST_SQL:
1MGMT_LATEST:
1MGMT_HISTORY_SQL:
1MGMT_HISTORY:
1DELETE_BASELINE:
1OLD_STARTUP_TIME:
1CUR_STARTUP_TIME:
1!=:
10:
1RESET_CAPTURE_STATS:
1CAPTURE_STATS:
1BASE_CUR:
1HASH_IN:
1STATS_REC:
1ROWTYPE:
1in capture_stats:
1MGMT_RESPONSE_CAPTURE_ID:
1BASE_REC:
1IS NOT NULL:
1MGMT_TEMPT_SQL:
1SUM:
1PURGE_OLD_STATS:
1ID_CURS:
1CAP_ID_1:
1CAP_ID_2:
1<:
1CALC_METRIC:
1METRIC:
1DELTA_CURS:
1ID_2:
1CAPTURE1:
1CAPTURE2:
1BASELINE:
1BASELINE_EXEC:
1BASELINE_ELAP_TIME:
1H_VALUE:
1CAP_TIME_1:
1CAP_TIME_2:
1DB_TIME_1:
1CPU_TIME_1:
1APPLICATION_WAIT_TIME_1:
1CLUSTER_WAIT_TIME_1:
1USER_IO_WAIT_TIME_1:
1CONCURRENCY_WAIT_TIME_1:
1DB_TIME_2:
1CPU_TIME_2:
1APPLICATION_WAIT_TIME_2:
1CLUSTER_WAIT_TIME_2:
1USER_IO_WAIT_TIME_2:
1CONCURRENCY_WAIT_TIME_2:
1DELTA_TIME:
1DELTA_EXECS:
1ROW_COUNT:
1ADJ_ELAP_TIME:
1ADJ_ELAP_TIME_SUM:
1BASELINE_ELAP_TIME_SUM:
1ELAP_TIME_SUM:
1id1=:
1,id2=:
1h_value=:
1,delta_t=:
1,delta_e=:
1,base_exec=:
1,base_elap_time=:
1*:
1adj_elap_time=:
1ROUND:
1+:
1ADJUSTED_ELAPSED_TIME:
1SQL_RESPONSE_TIME:
1ADJUSTED_SQL_RESPONSE_TIME:
1BASELINE_SQL_RESPONSE_TIME:
1RELATIVE_SQL_RESPONSE_TIME:
11000000:
1SAVE_TO_HISTORY:
1PURGE_OLD_HISTORY:
1CAP_ID:
1in save_to_history:
1CURR_CAP_ID:
1in purge_old_history:
1row_count = :
1cap_id = :
1<=:
1GET_METRIC_CURS:
1METRIC_CURS:
1METRIC_CURSOR_TYPE:
1BASELINE_EXISTS:
1BASELINE_SQL_COUNT:
1SNAP_ID_2:
1CURR_TIME:
1PREV_TIME:
1DURATION:
1EXISTS:
1NO_DATA_FOUND:
1baseline not configured:
1no v$sql snapshot; taking v$sql snapshot:
1GETDURATION:
1300:
1have v$sql snapshot; creating baseline:
1COUNT:
1capturing stats:
1only one capture; so no metric yet:
1MAX:
1calculating metric:
1GET_LATEST_CURS:
1RELATIVE_RESPONSE_TIME:
1BASELINE_RESPONSE_TIME:
1RESPONSE_TIME:
1BASELINE_COUNT:
1CAPTURE_COUNT:
1relative_sql_response_time=:
1baseline_sql_response_time=:
1sql_response_time=:
1START_DATE:
1TIMESTAMP:
1END_DATE:
1INTERVAL:
1DAY:
19:
1SECOND:
1INTERVAL DAY TO SECOND:
1EXTRACT:
1 SYS$EXTRACT_FROM:
124:
1HOUR:
160:
1MINUTE:
0

0
0
1296
2
0 :2 a0 97 a3 a0 51 a5 1c
6e 81 b0 a3 a0 1c 51 81
b0 a3 a0 1c 51 81 b0 a3
a0 1c 51 81 b0 a3 a0 1c
51 81 b0 a3 a0 1c 51 81
b0 a3 a0 1c 7e 51 b4 2e
81 b0 a3 a0 1c 7e 51 b4
2e 81 b0 a3 a0 1c 7e 51
b4 2e 81 b0 a3 a0 1c 7e
51 b4 2e 81 b0 a0 8d a0
b4 a0 2c 6a :2 a0 65 b7 a4
a0 b1 11 68 4f 9a 8f a0
b0 3d b4 55 6a a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 51 a5 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 :2 a0 6b
6e a5 57 :4 a0 51 6e :2 a0 :2 51
a5 b b4 2e a5 b ac :3 a0
b2 ee ac e5 d0 b2 e9 :2 a0
6b a0 6b ac :2 a0 b2 ee ac
e5 d0 b2 e9 a0 ac :2 a0 b2
ee ac e5 d0 b2 e9 a0 7e
51 b4 2e :7 a0 a5 57 :2 a0 6b
:12 a0 5 d7 b2 5 e9 b7 :2 a0
6b :6 a0 5 d7 b2 5 e9 b7
:2 19 3c :2 a0 6b :9 a0 7e 51 b4
2e :2 a0 ac a0 b2 ee :2 a0 7e
b4 2e a0 4c a0 ac a0 b2
ee a0 3e :3 6e 5 48 ac d0
eb 48 a 10 ac d0 d7 b2
e9 :2 a0 6b :2 a0 7e b4 2e cd
e9 :2 a0 6b :4 a0 ac a0 b2 ee
ac d0 eb d7 b2 e9 a0 57
a0 b4 e9 b7 a4 b1 11 68
4f 9a 96 :2 a0 b0 54 96 :2 a0
b0 54 96 :2 a0 b0 54 96 :2 a0
b0 54 96 :2 a0 b0 54 96 :2 a0
b0 54 b4 55 6a a0 9d :2 a0
c8 77 a3 a0 1c 81 b0 a3
a0 51 a5 1c 6e 81 b0 a3
:2 a0 6b :2 a0 f 1c 81 b0 :3 a0
11c 11d 91 :2 51 a0 63 37 :2 a0
e9 d3 :3 a0 f 2b a0 51 :2 a0
d b7 a6 9 51 :2 a0 d b7
a6 9 51 :2 a0 d b7 a6 9
51 :2 a0 d b7 a6 9 51 :2 a0
d b7 a6 9 51 :2 a0 d b7
a6 9 a0 62 b7 9 a4 14
b7 a0 47 :2 a0 e9 c1 b7 a4
b1 11 68 4f 9a 8f a0 b0
3d 8f a0 b0 3d b4 55 6a
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 :2 a0 6b :2 a0 f 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 51 a5 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a0 f4 8f a0 b0 3d 8f
a0 b0 3d 8f a0 b0 3d b4
bf c8 :5 a0 7e :2 a0 b4 2e b9
ac :2 a0 6b :2 a0 6b :2 a0 6b a0
7e a0 6b b4 2e :2 a0 6b 7e
:2 a0 6b b4 2e a6 122 :2 a0 6b
122 124 a0 b9 :2 a0 6b a0 7e
a0 6b b4 2e :2 a0 6b 7e :2 a0
6b b4 2e a6 122 :2 a0 6b 122
124 a0 b9 ac :4 a0 ac :2 a0 6b
b2 ee :2 a0 7e b4 2e ac d0
eb a0 b9 :4 a0 ac a0 b2 ee
ac d0 eb a0 b9 b2 ee :2 a0
6b a0 7e a0 6b b4 2e ac
d0 eb b2 ee :2 a0 7e b4 2e
ac d0 a0 de ac e5 e9 bd
b7 11 a4 b1 :2 a0 6b 6e 7e
a0 b4 2e a5 57 :2 a0 a5 57
:4 a0 51 6e :2 a0 :2 51 a5 b b4
2e a5 b ac :3 a0 b2 ee ac
e5 d0 b2 e9 :2 a0 6b :2 a0 7e
b4 2e cd e9 :2 a0 6b :2 a0 7e
b4 2e cd e9 a0 ac :2 a0 b2
ee ac e5 d0 b2 e9 a0 7e
51 b4 2e :7 a0 a5 57 :7 a0 ac
:9 a0 6b b2 ee :2 a0 7e b4 2e
ac e5 d0 b2 e9 :2 a0 7e b4
2e 5a :2 a0 7e a0 b4 2e 5a
7e 51 b4 2e d b7 :2 a0 7e
51 b4 2e d b7 :2 19 3c :2 a0
7e b4 2e 5a :2 a0 7e a0 b4
2e 5a 7e 51 b4 2e d b7
:2 a0 7e 51 b4 2e d b7 :2 19
3c :2 a0 7e b4 2e 5a :2 a0 7e
a0 b4 2e 5a 7e 51 b4 2e
d b7 :2 a0 7e 51 b4 2e d
b7 :2 19 3c :2 a0 7e b4 2e 5a
:2 a0 7e a0 b4 2e 5a 7e 51
b4 2e d b7 :2 a0 7e 51 b4
2e d b7 :2 19 3c :2 a0 7e b4
2e 5a :2 a0 7e a0 b4 2e 5a
7e 51 b4 2e d b7 :2 a0 7e
51 b4 2e d b7 :2 19 3c :2 a0
7e b4 2e 5a :2 a0 7e a0 b4
2e 5a 7e 51 b4 2e d b7
:2 a0 7e 51 b4 2e d b7 :2 19
3c :2 a0 6b :12 a0 5 d7 b2 5
e9 b7 a0 ac :3 a0 6b b2 ee
:2 a0 7e b4 2e ac e5 d0 b2
e9 :2 a0 6b :6 a0 5 d7 b2 5
e9 b7 :2 19 3c :5 a0 a5 dd e9
91 51 :2 a0 63 37 :6 a0 e9 d3
5 :3 a0 f 2b :2 a0 6b :c a0 5
d7 b2 5 e9 b7 a0 47 :2 a0
e9 c1 :2 a0 6b a0 b9 :2 a0 6b
:2 a0 6b ac :2 a0 b9 b2 ee :2 a0
6b a0 7e a0 6b b4 2e ac
d0 eb e7 :2 a0 6b a0 7e b4
2e ef f9 e9 :2 a0 6b :2 a0 7e
b4 2e cd e9 :2 a0 6b :2 a0 7e
b4 2e cd e9 a0 57 a0 b4
e9 b7 a4 b1 11 68 4f 9a
8f a0 b0 3d b4 55 6a a3
a0 1c 81 b0 a0 ac :2 a0 b2
ee ac e5 d0 b2 e9 :2 a0 6b
a0 3e a0 ac :2 a0 6b b2 ee
:2 a0 7e b4 2e ac d0 eb 48
:2 a0 7e b4 2e a 10 cd e9
:2 a0 6b :2 a0 7e b4 2e :2 a0 7e
b4 2e a 10 cd e9 :2 a0 6b
a0 3e a0 ac :2 a0 6b b2 ee
:2 a0 7e b4 2e ac d0 eb 48
:2 a0 7e b4 2e a 10 cd e9
:2 a0 6b a0 3e a0 ac :2 a0 6b
b2 ee :2 a0 7e b4 2e ac d0
eb 48 :2 a0 7e b4 2e a 10
cd e9 :2 a0 6b a0 3e a0 ac
:2 a0 6b b2 ee :2 a0 7e b4 2e
ac d0 eb 48 :2 a0 7e b4 2e
a 10 cd e9 a0 57 a0 b4
e9 b7 a4 b1 11 68 4f 9a
b4 55 6a a3 a0 1c 81 b0
a0 ac :2 a0 b2 ee ac e5 d0
b2 e9 :2 a0 6b a0 3e a0 ac
:2 a0 6b b2 ee :2 a0 7e b4 2e
ac d0 eb 48 cd e9 :2 a0 6b
:2 a0 7e b4 2e cd e9 :2 a0 6b
:2 a0 7e b4 2e cd e9 :2 a0 6b
:2 a0 7e b4 2e cd e9 :2 a0 6b
a0 3e a0 ac :2 a0 6b b2 ee
:2 a0 7e b4 2e ac d0 eb 48
cd e9 :2 a0 6b :2 a0 7e b4 2e
cd e9 :2 a0 6b a0 3e a0 ac
:2 a0 6b b2 ee :2 a0 7e b4 2e
ac d0 eb 48 cd e9 :2 a0 6b
:2 a0 7e b4 2e cd e9 :2 a0 6b
a0 3e a0 ac :2 a0 6b b2 ee
:2 a0 7e b4 2e ac d0 eb 48
cd e9 :2 a0 6b :2 a0 7e b4 2e
cd e9 a0 57 a0 b4 e9 b7
a4 b1 11 68 4f 9a 8f a0
b0 3d b4 55 6a a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 :5 a0 51 6e :2 a0 :2 51 a5 b
b4 2e a5 b ac :4 a0 b2 ee
ac e5 d0 b2 e9 a0 ac :3 a0
6b b2 ee :2 a0 7e b4 2e ac
e5 d0 b2 e9 :2 a0 7e b4 2e
:2 a0 6b a0 51 e7 a0 51 e7
:2 a0 7e b4 2e ef f9 e9 a0
7e 51 b4 2e :2 a0 6b a0 51
e7 a0 51 e7 a0 51 e7 a0
51 e7 a0 51 e7 a0 51 e7
:2 a0 7e b4 2e ef f9 e9 b7
19 3c :2 a0 6b :2 a0 7e b4 2e
cd e9 :2 a0 6b :4 a0 5 d7 b2
5 e9 b7 19 3c a0 57 a0
b4 e9 b7 a4 b1 11 68 4f
9a b4 55 6a a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
:5 a0 51 6e :2 a0 :2 51 a5 b b4
2e a5 b ac :4 a0 b2 ee ac
e5 d0 b2 e9 a0 ac :3 a0 6b
b2 ee :2 a0 7e b4 2e ac e5
d0 b2 e9 :2 a0 7e b4 2e :2 a0
6b a0 51 e7 a0 51 e7 a0
3e a0 ac :2 a0 6b b2 ee :2 a0
7e b4 2e ac d0 eb 48 ef
f9 e9 a0 7e 51 b4 2e :2 a0
6b a0 51 e7 a0 51 e7 a0
51 e7 a0 51 e7 a0 51 e7
a0 51 e7 :2 a0 7e b4 2e ef
f9 e9 b7 19 3c :2 a0 6b :2 a0
7e b4 2e cd e9 :2 a0 6b :4 a0
5 d7 b2 5 e9 b7 19 3c
a0 57 a0 b4 e9 b7 a4 b1
11 68 4f 9a b4 55 6a a0
f4 8f a0 b0 3d b4 bf c8
:2 a0 6b :2 a0 6b ac :2 a0 6b a0
b9 b2 ee :2 a0 6b a0 7e b4
2e ac d0 e5 e9 bd b7 11
a4 b1 a0 f4 8f a0 b0 3d
b4 bf c8 ac :2 a0 b9 b2 ee
:2 a0 6b a0 7e b4 2e ac d0
e5 e9 bd b7 11 a4 b1 a3
:2 a0 f 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 51 a5 1c 81 b0 :2 a0 6b
6e a5 57 a0 b4 57 :4 a0 51
6e :2 a0 :2 51 a5 b b4 2e a5
b ac :3 a0 b2 ee ac e5 d0
b2 e9 :2 a0 6b a0 6b ac :2 a0
b2 ee ac e5 d0 b2 e9 a0
ac :2 a0 b2 ee ac e5 d0 b2
e9 91 :2 a0 a5 b a0 37 :2 a0
6b 7e b4 2e 91 :3 a0 6b a5
b a0 37 a0 ac :3 a0 6b b2
ee :2 a0 7e a0 6b b4 2e :2 a0
7e b4 2e a 10 ac e5 d0
b2 e9 :2 a0 6b :6 a0 6b :2 a0 6b
:2 a0 6b :2 a0 6b 5 d7 b2 5
e9 b7 a0 47 b7 19 3c b7
a0 47 a0 7e 51 b4 2e :7 a0
a5 57 :2 a0 6b :12 a0 5 d7 b2
5 e9 b7 :2 a0 6b :6 a0 5 d7
b2 5 e9 b7 :2 19 3c :2 a0 6b
:9 a0 9f a0 d2 a0 9f a0 d2
ac :2 a0 6b b2 ee :2 a0 ac d0
d7 b2 e9 a0 57 a0 b4 e9
b7 a4 b1 11 68 4f 9a b4
55 6a a0 f4 8f a0 b0 3d
b4 bf c8 a0 ac :2 a0 6b b2
ee :2 a0 7e b4 2e ac d0 a0
de ac e5 e9 bd b7 11 a4
b1 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a0 ac :2 a0 b2 ee ac e5 d0
b2 e9 :3 a0 a5 dd e9 :2 a0 e9
d3 :2 a0 f 5a :2 a0 e9 c1 a0
65 b7 19 3c :2 a0 e9 d3 :2 a0
f 5a :2 a0 e9 c1 a0 65 b7
19 3c :2 a0 e9 c1 :2 a0 6b a0
3e a0 ac :2 a0 6b b2 ee :2 a0
7e b4 2e :2 a0 7e b4 2e a
10 ac d0 eb 48 cd e9 :2 a0
6b :2 a0 7e b4 2e :2 a0 7e b4
2e a 10 cd e9 a0 57 a0
b4 e9 b7 a4 b1 11 68 4f
9a 96 :2 a0 b0 54 b4 55 6a
a0 f4 8f a0 b0 3d b4 bf
c8 :8 a0 ac :2 a0 6b b2 ee :2 a0
7e b4 2e ac d0 a0 de ac
e5 e9 bd b7 11 a4 b1 a0
f4 8f a0 b0 3d 8f a0 b0
3d 8f a0 b0 3d b4 bf c8
:2 a0 6b :2 a0 6b :2 a0 6b a0 7e
a0 6b b4 2e :2 a0 6b 7e :2 a0
6b b4 2e a6 122 :2 a0 6b 122
124 :2 a0 6b a0 7e a0 6b b4
2e :2 a0 6b 7e :2 a0 6b b4 2e
a6 122 :2 a0 6b 122 124 :2 a0 6b
:2 a0 6b :2 ac :2 a0 6b b2 ee :2 a0
7e b4 2e ac d0 eb a0 b9
ac :2 a0 6b b2 ee :2 a0 7e b4
2e ac d0 eb a0 b9 :2 a0 6b
a0 b9 b2 ee :2 a0 6b a0 7e
a0 6b b4 2e :2 a0 6b a0 7e
a0 6b b4 2e a 10 :2 a0 6b
a0 7e b4 2e a 10 :2 a0 6b
a0 7e a0 6b b4 2e :2 a0 6b
7e :2 a0 6b b4 2e a6 122 :2 a0
6b 122 124 5a a0 7e b4 2e
a 10 ac d0 e5 e9 bd b7
11 a4 b1 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 :2 a0 6b
:2 a0 f 1c 81 b0 a3 a0 51
a5 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 :2 a0 d :4 a0 51
6e :2 a0 :2 51 a5 b b4 2e a5
b ac :3 a0 b2 ee ac e5 d0
b2 e9 :3 a0 a5 dd e9 :9 a0 e9
d3 5 :2 a0 f 5a :2 a0 e9 c1
a0 65 b7 19 3c :9 a0 e9 d3
5 :2 a0 f 5a :2 a0 e9 c1 a0
65 b7 19 3c :2 a0 e9 c1 :2 a0
6b 6e 7e a0 b4 2e 7e 6e
b4 2e 7e a0 b4 2e a5 57
:2 a0 6b a0 3e a0 ac :2 a0 6b
b2 ee :2 a0 7e b4 2e ac d0
eb 48 cd e9 :2 a0 6b :2 a0 7e
b4 2e cd e9 a0 51 d a0
51 d a0 51 d a0 51 d
:5 a0 a5 dd e9 :8 a0 e9 d3 5
:3 a0 f 2b :2 a0 6b 6e 7e a0
b4 2e 7e 6e b4 2e 7e a0
b4 2e 7e 6e b4 2e 7e a0
b4 2e 7e 6e b4 2e 7e a0
b4 2e 7e 6e b4 2e 7e a0
b4 2e a5 57 :2 a0 7e a0 b4
2e d :2 a0 7e a0 b4 2e d
:2 a0 6b 6e 7e :2 a0 51 a5 b
b4 2e a5 57 :2 a0 7e a0 b4
2e d :2 a0 7e a0 b4 2e d
:2 a0 7e a0 b4 2e d :2 a0 7e
51 b4 2e d :2 a0 6b :e a0 5
d7 b2 5 e9 b7 a0 47 :2 a0
e9 c1 a0 7e 51 b4 2e :2 a0
7e a0 b4 2e 7e 51 b4 2e
d b7 19 3c a0 7e 51 b4
2e :2 a0 7e b4 2e 5a :2 a0 7e
a0 b4 2e 5a 7e 51 b4 2e
d b7 :2 a0 7e 51 b4 2e d
b7 :2 19 3c :2 a0 7e b4 2e 5a
:2 a0 7e a0 b4 2e 5a 7e 51
b4 2e d b7 :2 a0 7e 51 b4
2e d b7 :2 19 3c :2 a0 7e b4
2e 5a :2 a0 7e a0 b4 2e 5a
7e 51 b4 2e d b7 :2 a0 7e
51 b4 2e d b7 :2 19 3c :2 a0
7e b4 2e 5a :2 a0 7e a0 b4
2e 5a 7e 51 b4 2e d b7
:2 a0 7e 51 b4 2e d b7 :2 19
3c :2 a0 7e b4 2e 5a :2 a0 7e
a0 b4 2e 5a 7e 51 b4 2e
d b7 :2 a0 7e 51 b4 2e d
b7 :2 19 3c :2 a0 7e b4 2e 5a
:2 a0 7e a0 b4 2e 5a 7e 51
b4 2e d b7 :2 a0 7e 51 b4
2e d b7 :2 19 3c :2 a0 6b :13 a0
7e 51 b4 2e a0 7e 51 b4
2e a0 7e 51 b4 2e :7 a0 5
d7 b2 5 e9 b7 :2 a0 6b :d a0
7e 51 b4 2e a0 7e 51 b4
2e a0 7e 51 b4 2e a0 5
d7 b2 5 e9 b7 :2 19 3c a0
57 a0 b4 e9 :2 a0 7e b4 2e
a0 57 b3 a0 57 b3 b7 19
3c b7 a4 b1 11 68 4f 9a
b4 55 6a a3 a0 1c 81 b0
a3 a0 1c 81 b0 :2 a0 6b 6e
a5 57 a0 ac :2 a0 b2 ee ac
e5 d0 b2 e9 a0 ac :3 a0 6b
b2 ee :2 a0 7e b4 2e ac e5
d0 b2 e9 :2 a0 6b :2 a0 7e b4
2e cd e9 :2 a0 6b :2 a0 7e b4
2e cd e9 :2 a0 6b ac :2 a0 6b
b2 ee :2 a0 7e b4 2e ac d0
d7 b2 e9 :2 a0 6b ac :2 a0 6b
b2 ee :2 a0 7e b4 2e ac d0
d7 b2 e9 a0 57 a0 b4 e9
b7 a4 b1 11 68 4f 9a b4
55 6a a0 f4 8f a0 b0 3d
b4 bf c8 a0 ac :2 a0 6b b2
ee :2 a0 7e b4 2e ac d0 a0
de ac e5 e9 bd b7 11 a4
b1 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 :2 a0 6b 6e
a5 57 a0 ac :2 a0 b2 ee ac
e5 d0 b2 e9 a0 51 d a0
7e 51 b4 2e d :3 a0 a5 dd
e9 :3 a0 6b 6e 7e a0 b4 2e
a5 57 :2 a0 e9 d3 :3 a0 f 2b
:2 a0 7e b4 2e :2 a0 d a0 2b
b7 19 3c :2 a0 7e 51 b4 2e
d b7 a0 47 :2 a0 e9 c1 :2 a0
6b 6e 7e a0 b4 2e a5 57
a0 7e 51 b4 2e :2 a0 6b a0
3e a0 ac :2 a0 6b b2 ee :2 a0
7e b4 2e :2 a0 7e b4 2e a
10 ac d0 eb 48 cd e9 :2 a0
6b :2 a0 7e b4 2e :2 a0 7e b4
2e a 10 cd e9 a0 57 a0
b4 e9 b7 19 3c b7 a4 b1
11 68 4f 9a 96 :2 a0 b0 54
b4 55 6a a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a0 f4 8f a0 b0 3d b4 bf
c8 a0 ac :2 a0 6b b2 ee :2 a0
7e b4 2e ac d0 a0 de ac
e5 e9 bd b7 11 a4 b1 a0
ac :2 a0 b2 ee ac e5 d0 b2
e9 a0 51 d 51 ac :2 a0 b2
ee ac :2 a0 6b b2 ee :2 a0 7e
b4 2e ac d0 eb 7e b4 2e
ac e5 d0 b2 e9 b7 :2 a0 51
d b7 a6 9 a4 b1 11 4f
a0 7e 51 b4 2e :2 a0 6b 6e
a5 57 :3 a0 a5 dd e9 :2 a0 e9
d3 :2 a0 f 5a :2 a0 6b 6e a5
57 :2 a0 a5 57 :2 a0 e9 c1 :2 a0
d b7 :2 a0 e9 c1 a0 ac :2 a0
b2 ee ac e5 d0 b2 e9 a0
ac :3 a0 6b b2 ee :2 a0 7e b4
2e ac e5 d0 b2 e9 :4 a0 a5
b d a0 7e 51 b4 2e :2 a0
6b 6e a5 57 :3 a0 a5 57 a0
d2 9f ac :3 a0 6b b2 ee :2 a0
7e b4 2e ac e5 d0 b2 e9
a0 7e 51 b4 2e :2 a0 6b 6e
a5 57 a0 57 b3 :2 a0 6b 6e
a5 57 :2 a0 d b7 :2 a0 d b7
:2 19 3c b7 :2 a0 d b7 :2 19 3c
b7 :2 19 3c b7 a0 d2 9f ac
:3 a0 6b b2 ee :2 a0 7e b4 2e
ac e5 d0 b2 e9 a0 7e 51
b4 2e a0 ac :2 a0 b2 ee ac
e5 d0 b2 e9 a0 9f a0 d2
ac :3 a0 6b b2 ee :2 a0 7e b4
2e ac e5 d0 b2 e9 :4 a0 a5
b d a0 7e 51 b4 2e :2 a0
6b 6e a5 57 :2 a0 6b a0 6b
57 b3 :2 a0 6b a0 6b 57 b3
:2 a0 6b 6e a5 57 :2 a0 6b a0
6b a0 a5 57 b7 a0 ac :3 a0
6b b2 ee :2 a0 7e b4 2e ac
e5 d0 b2 e9 b7 :3 a0 d b7
a6 9 a4 b1 11 4f b7 :2 19
3c b7 :2 a0 d b7 :2 19 3c b7
:2 19 3c :3 a0 ac a0 b2 ee ac
e5 d0 b2 :2 e9 dd b7 a4 b1
11 68 4f 9a 96 :2 a0 b0 54
b4 55 6a a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a0 ac :2 a0 b2 ee
ac e5 d0 b2 e9 a0 51 d
a0 51 d a0 d2 9f ac :3 a0
6b b2 ee :2 a0 7e b4 2e ac
e5 d0 b2 e9 a0 7e 51 b4
2e a0 d2 9f ac :3 a0 6b b2
ee :2 a0 7e b4 2e ac e5 d0
b2 e9 a0 7e 51 b4 2e a0
d2 9f ac :3 a0 6b b2 ee :2 a0
7e b4 2e ac e5 d0 b2 e9
a0 7e 51 b4 2e :3 a0 ac :5 a0
6b b2 ee :2 a0 7e b4 2e ac
e5 d0 b2 e9 b7 :2 a0 d b7
:2 19 3c b7 :2 a0 d b7 :2 19 3c
b7 :2 a0 d b7 :2 19 3c :2 a0 6b
6e 7e a0 b4 2e a5 57 :2 a0
6b 6e 7e a0 b4 2e a5 57
:2 a0 6b 6e 7e a0 b4 2e a5
57 :5 a0 ac a0 b2 ee ac e5
d0 b2 :2 e9 dd b7 a4 b1 11
68 4f a0 8d 8f a0 b0 3d
8f a0 b0 3d b4 :2 a0 2c 6a
a3 :2 a0 51 :2 a0 63 1c 81 b0
:2 a0 7e a0 b4 2e d :3 a0 6e
:2 a0 b4 2e 7e 51 b4 2e 7e
:2 a0 6e :2 a0 b4 2e b4 2e 5a
7e 51 b4 2e 7e :2 a0 6e :2 a0
b4 2e b4 2e 5a 7e 51 b4
2e 7e :2 a0 6e :2 a0 b4 2e b4
2e 65 b7 a4 b1 11 68 4f
b1 b7 a4 11 a0 b1 56 4f
1d 17 b5 
1296
2
0 3 7 b 33 19 1d 20
21 29 2e 18 4f 3e 42 15
4a 3d 6b 5a 5e 3a 66 59
87 76 7a 56 82 75 a3 92
96 72 9e 91 bf ae b2 8e
ba ad e4 ca ce aa d6 d9
da df c9 109 ef f3 c6 fb
fe ff 104 ee 12e 114 118 eb
120 123 124 129 113 153 139 13d
110 145 148 149 14e 138 15a 15e
172 135 176 17a 17e 182 186 18a
18e 190 194 198 19a 1a6 1aa 1ac
1c8 1c4 1c3 1d0 1c0 1d5 1d9 1f2
1e1 1e5 1ed 1e0 20e 1fd 201 209
1dd 22a 215 219 21c 21d 225 1fc
246 235 239 241 1f9 25e 24d 251
259 234 27a 269 26d 275 231 292
281 285 28d 268 2ae 29d 2a1 2a9
265 2c6 2b5 2b9 2c1 29c 2e2 2d1
2d5 2dd 299 2fa 2e9 2ed 2f5 2d0
301 305 2cd 309 30e 30f 314 318
31c 320 324 327 32c 330 334 337
33a 33b 33d 33e 343 344 346 347
34b 34f 353 354 35b 35c 362 366
367 36c 370 374 377 37b 37e 37f
383 387 388 38f 390 396 39a 39b
3a0 3a4 3a5 3a9 3ad 3ae 3b5 3b6
3bc 3c0 3c1 3c6 3ca 3cd 3d0 3d1
3d6 3da 3de 3e2 3e6 3ea 3ee 3f2
3f3 3f8 3fc 400 403 407 40b 40f
413 417 41b 41f 423 427 42b 42f
433 437 43b 43f 443 447 44b 44f
457 458 45c 461 463 467 46b 46e
472 476 47a 47e 482 486 48a 492
493 497 49c 49e 4a2 4a6 4a9 4ad
4b1 4b4 4b8 4bc 4c0 4c4 4c8 4cc
4d0 4d4 4d8 4db 4de 4df 4e4 4e8
4ec 4ed 4f1 4f2 4f9 4fd 501 504
505 50a 1 50e 512 513 517 518
51f 1 523 528 52d 532 536 539
53a 53e 542 1 545 54a 54b 54f
557 558 55d 561 565 568 56c 570
573 574 579 57e 583 587 58b 58e
592 596 59a 59e 59f 5a3 5a4 5ab
5ac 5b0 5b4 5bc 5bd 5c2 5c6 5cb
5cf 5d0 5d5 5d7 5db 5dd 5e9 5ed
5ef 60f 607 60b 606 616 627 61f
623 603 62e 63b 633 637 61e 642
653 64b 64f 61b 65a 667 65f 663
64a 66e 67f 677 67b 647 686 676
68b 68f 693 6a7 69b 69f 6a3 697
6c3 6b2 6b6 6be 673 6e4 6ca 6ce
6d1 6d2 6da 6df 6b1 711 6ef 6f3
6ae 6f7 6fb 6ff 704 70c 6ee 718
71c 720 6eb 724 728 72c 72f 732
736 73a 73c 740 744 749 74e 752
756 75a 75f 765 769 76c 770 774
778 77a 77b 780 783 787 78b 78f
791 792 797 79a 79e 7a2 7a6 7a8
7a9 7ae 7b1 7b5 7b9 7bd 7bf 7c0
7c5 7c8 7cc 7d0 7d4 7d6 7d7 7dc
7df 7e3 7e7 7eb 7ed 7ee 7f3 7f7
7fa 7fc 801 805 80a 80c 810 817
81b 81f 824 826 828 82c 82e 83a
83e 840 85c 858 857 864 871 86d
854 879 86c 87e 882 89b 88a 88e
896 869 8b3 8a2 8a6 8ae 889 8e0
8be 8c2 886 8c6 8ca 8ce 8d3 8db
8bd 8fc 8eb 8ef 8f7 8ba 914 903
907 90f 8ea 930 91f 923 92b 8e7
94c 937 93b 93e 93f 947 91e 968
957 95b 963 91b 980 96f 973 97b
956 99c 98b 98f 997 953 9b4 9a3
9a7 9af 98a 9d0 9bf 9c3 9cb 987
9e8 9d7 9db 9e3 9be a04 9f3 9f7
9ff 9bb a1c a0b a0f a17 9f2 a38
a27 a2b a33 9ef a50 a3f a43 a4b
a26 a6c a5b a5f a67 a23 a84 a73
a77 a7f a5a aa0 a8f a93 a9b a57
ab8 aa7 aab ab3 a8e ad4 ac3 ac7
acf a8b aec adb adf ae7 ac2 b08
af7 afb b03 abf b20 b0f b13 b1b
af6 b3c b2b b2f b37 af3 b54 b43
b47 b4f b2a b70 b5f b63 b6b b27
b5b b77 b90 b8c b8b b98 ba5 ba1
b88 bad bb6 bb2 ba0 bbe b9d bc3
bc6 bca bce bd2 bd6 bda bde be1
be5 be9 bea bef bf1 bf2 bf6 bfa
bfd c01 c05 c08 c0c c10 c13 c17
c1a c1e c21 c22 c27 c2b c2f c32
c35 c39 c3d c40 c41 c46 c47 c49
c4d c51 c54 c56 c5a c5e c60 c64
c68 c6b c6f c72 c76 c79 c7a c7f
c83 c87 c8a c8d c91 c95 c98 c99
c9e c9f ca1 ca5 ca9 cac cae cb2
cb6 cb8 cb9 cbd cc1 cc5 cc9 cca
cce cd2 cd5 cd6 cdd ce1 ce5 ce8
ce9 cee cef cf3 cf7 cfb cfd d01
d05 d09 d0d d0e d12 d13 d1a d1b
d1f d23 d27 d29 d2a d31 d35 d39
d3c d40 d43 d47 d4a d4b d50 d51
d55 d59 d5a d61 d65 d69 d6c d6d
d72 d73 d77 d7b d7d d7e d84 d89
d8e d90 d9c da0 da2 da6 daa dad
db2 db5 db9 dba dbf dc0 dc5 dc9
dcd dce dd3 dd7 ddb ddf de3 de6
deb def df3 df6 df9 dfa dfc dfd
e02 e03 e05 e06 e0a e0e e12 e13
e1a e1b e21 e25 e26 e2b e2f e33
e36 e3a e3e e41 e42 e47 e4c e51
e55 e59 e5c e60 e64 e67 e68 e6d
e72 e77 e7b e7c e80 e84 e85 e8c
e8d e93 e97 e98 e9d ea1 ea4 ea7
ea8 ead eb1 eb5 eb9 ebd ec1 ec5
ec9 eca ecf ed3 ed7 edb edf ee3
ee7 eeb eec ef0 ef4 ef8 efc f00
f04 f08 f0c f10 f13 f14 f1b f1f
f23 f26 f27 f2c f2d f33 f37 f38
f3d f41 f45 f48 f49 f4e f51 f55
f59 f5c f60 f61 f66 f69 f6c f6f
f70 f75 f79 f7b f7f f83 f86 f89
f8a f8f f93 f95 f99 f9d fa0 fa4
fa8 fab fac fb1 fb4 fb8 fbc fbf
fc3 fc4 fc9 fcc fcf fd2 fd3 fd8
fdc fde fe2 fe6 fe9 fec fed ff2
ff6 ff8 ffc 1000 1003 1007 100b 100e
100f 1014 1017 101b 101f 1022 1026 1027
102c 102f 1032 1035 1036 103b 103f 1041
1045 1049 104c 104f 1050 1055 1059 105b
105f 1063 1066 106a 106e 1071 1072 1077
107a 107e 1082 1085 1089 108a 108f 1092
1095 1098 1099 109e 10a2 10a4 10a8 10ac
10af 10b2 10b3 10b8 10bc 10be 10c2 10c6
10c9 10cd 10d1 10d4 10d5 10da 10dd 10e1
10e5 10e8 10ec 10ed 10f2 10f5 10f8 10fb
10fc 1101 1105 1107 110b 110f 1112 1115
1116 111b 111f 1121 1125 1129 112c 1130
1134 1137 1138 113d 1140 1144 1148 114b
114f 1150 1155 1158 115b 115e 115f 1164
1168 116a 116e 1172 1175 1178 1179 117e
1182 1184 1188 118c 118f 1193 1197 119a
119e 11a2 11a6 11aa 11ae 11b2 11b6 11ba
11be 11c2 11c6 11ca 11ce 11d2 11d6 11da
11de 11e2 11e6 11ee 11ef 11f3 11f8 11fa
11fe 11ff 1203 1207 120b 120e 120f 1216
121a 121e 1221 1222 1227 1228 122e 1232
1233 1238 123c 1240 1243 1247 124b 124f
1253 1257 125b 125f 1267 1268 126c 1271
1273 1277 127b 127e 1282 1286 128a 128e
1292 129a 1295 129e 12a2 12a5 12a9 12ad
12b1 12b3 12b7 12bb 12bf 12c3 12c7 12cb
12d0 12d5 12d9 12dd 12e1 12e5 12ea 12f0
12f4 12f8 12fb 12ff 1303 1307 130b 130f
1313 1317 131b 131f 1323 1327 132b 132f
1337 1338 133c 1341 1343 1347 134e 1352
1356 135b 135d 1361 1365 1368 136c 136e
1372 1376 1379 137d 1381 1384 1385 1389
138d 138f 1390 1397 139b 139f 13a2 13a6
13a9 13ad 13b0 13b1 13b6 13b7 13bb 13bf
13c1 13c5 13c9 13cc 13d0 13d3 13d4 13d9
13df 13e0 13e5 13e9 13ed 13f0 13f4 13f8
13fb 13fc 1401 1406 140b 140f 1413 1416
141a 141e 1421 1422 1427 142c 1431 1435
143a 143e 143f 1444 1446 144a 144c 1458
145c 145e 147a 1476 1475 1482 1472 1487
148b 14a4 1493 1497 149f 1492 14ab 148f
14af 14b3 14b7 14b8 14bf 14c0 14c6 14ca
14cb 14d0 14d4 14d8 14db 1 14df 14e3
14e4 14e8 14ec 14ef 14f0 14f7 14fb 14ff
1502 1503 1508 1509 150d 1511 1514 1518
151c 151f 1520 1 1525 152a 152f 1534
1538 153c 153f 1543 1547 154a 154b 1550
1554 1558 155b 155c 1 1561 1566 156b
1570 1574 1578 157b 1 157f 1583 1584
1588 158c 158f 1590 1597 159b 159f 15a2
15a3 15a8 15a9 15ad 15b1 15b4 15b8 15bc
15bf 15c0 1 15c5 15ca 15cf 15d4 15d8
15dc 15df 1 15e3 15e7 15e8 15ec 15f0
15f3 15f4 15fb 15ff 1603 1606 1607 160c
160d 1611 1615 1618 161c 1620 1623 1624
1 1629 162e 1633 1638 163c 1640 1643
1 1647 164b 164c 1650 1654 1657 1658
165f 1663 1667 166a 166b 1670 1671 1675
1679 167c 1680 1684 1687 1688 1 168d
1692 1697 169c 16a0 16a5 16a9 16aa 16af
16b1 16b5 16b7 16c3 16c7 16c9 16dd 16de
16e2 16fb 16ea 16ee 16f6 16e9 1702 16e6
1706 170a 170e 170f 1716 1717 171d 1721
1722 1727 172b 172f 1732 1 1736 173a
173b 173f 1743 1746 1747 174e 1752 1756
1759 175a 175f 1760 1764 1768 176b 1770
1775 1779 177d 1780 1784 1788 178b 178c
1791 1796 179b 179f 17a3 17a6 17aa 17ae
17b1 17b2 17b7 17bc 17c1 17c5 17c9 17cc
17d0 17d4 17d7 17d8 17dd 17e2 17e7 17eb
17ef 17f2 1 17f6 17fa 17fb 17ff 1803
1806 1807 180e 1812 1816 1819 181a 181f
1820 1824 1828 182b 1830 1835 1839 183d
1840 1844 1848 184b 184c 1851 1856 185b
185f 1863 1866 1 186a 186e 186f 1873
1877 187a 187b 1882 1886 188a 188d 188e
1893 1894 1898 189c 189f 18a4 18a9 18ad
18b1 18b4 18b8 18bc 18bf 18c0 18c5 18ca
18cf 18d3 18d7 18da 1 18de 18e2 18e3
18e7 18eb 18ee 18ef 18f6 18fa 18fe 1901
1902 1907 1908 190c 1910 1913 1918 191d
1921 1925 1928 192c 1930 1933 1934 1939
193e 1943 1947 194c 1950 1951 1956 1958
195c 195e 196a 196e 1970 198c 1988 1987
1994 1984 1999 199d 19b6 19a5 19a9 19b1
19a4 19d2 19c1 19c5 19cd 19a1 19ea 19d9
19dd 19e5 19c0 1a06 19f5 19f9 1a01 19bd
19f1 1a0d 1a11 1a15 1a19 1a1d 1a20 1a25
1a29 1a2d 1a30 1a33 1a34 1a36 1a37 1a3c
1a3d 1a3f 1a40 1a44 1a48 1a4c 1a50 1a51
1a58 1a59 1a5f 1a63 1a64 1a69 1a6d 1a6e
1a72 1a76 1a7a 1a7d 1a7e 1a85 1a89 1a8d
1a90 1a91 1a96 1a97 1a9d 1aa1 1aa2 1aa7
1aab 1aaf 1ab2 1ab3 1ab8 1abc 1ac0 1ac3
1ac7 1aca 1acc 1ad0 1ad3 1ad5 1ad9 1add
1ae0 1ae1 1ae6 1aec 1aed 1af2 1af6 1af9
1afc 1afd 1b02 1b06 1b0a 1b0d 1b11 1b14
1b16 1b1a 1b1d 1b1f 1b23 1b26 1b28 1b2c
1b2f 1b31 1b35 1b38 1b3a 1b3e 1b41 1b43
1b47 1b4b 1b4e 1b4f 1b54 1b5a 1b5b 1b60
1b62 1b66 1b69 1b6d 1b71 1b74 1b78 1b7c
1b7f 1b80 1b85 1b8a 1b8f 1b93 1b97 1b9a
1b9e 1ba2 1ba6 1baa 1bae 1bb6 1bb7 1bbb
1bc0 1bc2 1bc6 1bc9 1bcd 1bd2 1bd6 1bd7
1bdc 1bde 1be2 1be4 1bf0 1bf4 1bf6 1c0a
1c0b 1c0f 1c28 1c17 1c1b 1c23 1c16 1c44
1c33 1c37 1c3f 1c13 1c5c 1c4b 1c4f 1c57
1c32 1c78 1c67 1c6b 1c73 1c2f 1c63 1c7f
1c83 1c87 1c8b 1c8f 1c92 1c97 1c9b 1c9f
1ca2 1ca5 1ca6 1ca8 1ca9 1cae 1caf 1cb1
1cb2 1cb6 1cba 1cbe 1cc2 1cc3 1cca 1ccb
1cd1 1cd5 1cd6 1cdb 1cdf 1ce0 1ce4 1ce8
1cec 1cef 1cf0 1cf7 1cfb 1cff 1d02 1d03
1d08 1d09 1d0f 1d13 1d14 1d19 1d1d 1d21
1d24 1d25 1d2a 1d2e 1d32 1d35 1d39 1d3c
1d3e 1d42 1d45 1d47 1 1d4b 1d4f 1d50
1d54 1d58 1d5b 1d5c 1d63 1d67 1d6b 1d6e
1d6f 1d74 1d75 1d79 1d7d 1d80 1d86 1d87
1d8c 1d90 1d93 1d96 1d97 1d9c 1da0 1da4
1da7 1dab 1dae 1db0 1db4 1db7 1db9 1dbd
1dc0 1dc2 1dc6 1dc9 1dcb 1dcf 1dd2 1dd4
1dd8 1ddb 1ddd 1de1 1de5 1de8 1de9 1dee
1df4 1df5 1dfa 1dfc 1e00 1e03 1e07 1e0b
1e0e 1e12 1e16 1e19 1e1a 1e1f 1e24 1e29
1e2d 1e31 1e34 1e38 1e3c 1e40 1e44 1e48
1e50 1e51 1e55 1e5a 1e5c 1e60 1e63 1e67
1e6c 1e70 1e71 1e76 1e78 1e7c 1e7e 1e8a
1e8e 1e90 1ea4 1ea5 1ea9 1ead 1eb1 1eca
1ec6 1ec5 1ed2 1ec2 1ed7 1eda 1ede 1ee2
1ee6 1ee9 1eed 1ef1 1ef4 1ef5 1ef9 1efd
1f00 1f04 1f06 1f07 1f0e 1f12 1f16 1f19
1f1d 1f20 1f21 1f26 1f27 1f2b 1f31 1f36
1f3b 1f3d 1f49 1f4d 1f4f 1f53 1f6c 1f68
1f67 1f74 1f64 1f79 1f7c 1f80 1f81 1f85
1f89 1f8b 1f8c 1f93 1f97 1f9b 1f9e 1fa2
1fa5 1fa6 1fab 1fac 1fb0 1fb6 1fbb 1fc0
1fc2 1fce 1fd2 1ff2 1fd8 1fdc 1fe0 1fe5
1fed 1fd7 200e 1ffd 2001 2009 1fd4 2026
2015 2019 2021 1ffc 2042 2031 2035 203d
1ff9 205a 2049 204d 2055 2030 2076 2065
2069 2071 202d 208e 207d 2081 2089 2064
20aa 2099 209d 20a5 2061 20c2 20b1 20b5
20bd 2098 20de 20cd 20d1 20d9 2095 20f6
20e5 20e9 20f1 20cc 2113 2101 20c9 2105
2106 210e 2100 211a 211e 20fd 2122 2127
2128 212d 2131 2132 2137 213b 213f 2143
2147 214a 214f 2153 2157 215a 215d 215e
2160 2161 2166 2167 2169 216a 216e 2172
2176 2177 217e 217f 2185 2189 218a 218f
2193 2197 219a 219e 21a1 21a2 21a6 21aa
21ab 21b2 21b3 21b9 21bd 21be 21c3 21c7
21c8 21cc 21d0 21d1 21d8 21d9 21df 21e3
21e4 21e9 21ed 21f1 21f5 21f6 21f8 21fc
21fe 2202 2206 2209 220c 220d 2212 2216
221a 221e 2222 2225 2226 2228 222c 222e
2232 2233 2237 223b 223f 2242 2243 224a
224e 2252 2255 2259 225c 225d 2262 2266
226a 226d 226e 1 2273 2278 2279 227f
2283 2284 2289 228d 2291 2294 2298 229c
22a0 22a4 22a8 22ac 22af 22b3 22b7 22ba
22be 22c2 22c5 22c9 22cd 22d0 22d4 22dc
22dd 22e1 22e6 22e8 22ec 22f3 22f5 22f9
22fc 22fe 2302 2309 230d 2310 2313 2314
2319 231d 2321 2325 2329 232d 2331 2335
2336 233b 233f 2343 2346 234a 234e 2352
2356 235a 235e 2362 2366 236a 236e 2372
2376 237a 237e 2382 2386 238a 238e 2392
239a 239b 239f 23a4 23a6 23aa 23ae 23b1
23b5 23b9 23bd 23c1 23c5 23c9 23cd 23d5
23d6 23da 23df 23e1 23e5 23e9 23ec 23f0
23f4 23f7 23fb 23ff 2403 2407 240b 240f
2413 2417 241b 241e 2422 2426 242a 242d
2431 2435 2436 243a 243e 2441 2442 2449
244d 2451 2452 2456 245e 245f 2464 2468
246d 2471 2472 2477 2479 247d 247f 248b
248f 2491 24a5 24a6 24aa 24ae 24b2 24cb
24c7 24c6 24d3 24c3 24d8 24db 24df 24e3
24e4 24e8 24ec 24ef 24f0 24f7 24fb 24ff
2502 2503 2508 2509 250d 2511 2513 2514
251a 251f 2524 2526 2532 2536 254d 253c
2540 2548 253b 2569 2558 255c 2564 2538
2581 2570 2574 257c 2557 2588 2554 258c
2590 2594 2595 259c 259d 25a3 25a7 25a8
25ad 25b1 25b5 25b9 25c1 25bc 25c5 25c9
25cd 25d2 25d7 25db 25df 25e4 25e7 25eb
25ef 25f4 25f6 25fa 25fe 2600 2604 2607
260b 260f 2614 2619 261d 2621 2626 2629
262d 2631 2636 2638 263c 2640 2642 2646
2649 264d 2651 2656 2658 265c 2660 2663
1 2667 266b 266c 2670 2674 2677 2678
267f 2683 2687 268a 268b 2690 2694 2698
269b 269c 1 26a1 26a6 26a7 26ab 26af
26b2 26b7 26bc 26c0 26c4 26c7 26cb 26cf
26d2 26d3 26d8 26dc 26e0 26e3 26e4 1
26e9 26ee 26f3 26f8 26fc 2701 2705 2706
270b 270d 2711 2713 271f 2723 2725 2745
273d 2741 273c 274c 2739 2751 2755 2759
275d 2776 2772 2771 277e 276e 2783 2786
278a 278e 2792 2796 279a 279e 27a2 27a6
27aa 27ab 27af 27b3 27b6 27b7 27be 27c2
27c6 27c9 27ca 27cf 27d0 27d4 27d8 27da
27db 27e1 27e6 27eb 27ed 27f9 27fd 27ff
2803 281c 2818 2817 2824 2831 282d 2814
2839 2842 283e 282c 284a 2829 284f 2852
2856 285a 285e 2861 2865 2869 286c 2870
2874 2877 287b 287e 2882 2885 2886 288b
288f 2893 2896 2899 289d 28a1 28a4 28a5
28aa 28ab 28ad 28b1 28b5 28b8 28ba 28be
28c2 28c6 28c9 28cd 28d0 28d4 28d7 28d8
28dd 28e1 28e5 28e8 28eb 28ef 28f3 28f6
28f7 28fc 28fd 28ff 2903 2907 290a 290c
2910 2914 2918 291b 291f 2923 2926 2927
2928 292c 2930 2933 2934 293b 293f 2943
2946 2947 294c 294d 2951 2955 2959 295b
295c 2960 2964 2967 2968 296f 2973 2977
297a 297b 2980 2981 2985 2989 298d 298f
2993 2997 299a 299e 29a0 29a1 29a8 29ac
29b0 29b3 29b7 29ba 29be 29c1 29c2 29c7
29cb 29cf 29d2 29d6 29d9 29dd 29e0 29e1
1 29e6 29eb 29ef 29f3 29f6 29fa 29fd
29fe 1 2a03 2a08 2a0c 2a10 2a13 2a17
2a1a 2a1e 2a21 2a22 2a27 2a2b 2a2f 2a32
2a35 2a39 2a3d 2a40 2a41 2a46 2a47 2a49
2a4d 2a51 2a54 2a56 2a5a 2a5d 2a61 2a64
2a65 1 2a6a 2a6f 2a70 2a74 2a7a 2a7f
2a84 2a86 2a92 2a96 2aad 2a9c 2aa0 2aa8
2a9b 2ac9 2ab8 2abc 2ac4 2a98 2af5 2ad0
2ad4 2ad8 2adb 2adf 2ae3 2ae8 2af0 2ab7
2b12 2b00 2ab4 2b04 2b05 2b0d 2aff 2b2e
2b1d 2b21 2b29 2afc 2b46 2b35 2b39 2b41
2b1c 2b62 2b51 2b55 2b5d 2b19 2b7a 2b69
2b6d 2b75 2b50 2b96 2b85 2b89 2b91 2b4d
2bae 2b9d 2ba1 2ba9 2b84 2bca 2bb9 2bbd
2bc5 2b81 2be2 2bd1 2bd5 2bdd 2bb8 2bfe
2bed 2bf1 2bf9 2bb5 2c16 2c05 2c09 2c11
2bec 2c32 2c21 2c25 2c2d 2be9 2c4a 2c39
2c3d 2c45 2c20 2c66 2c55 2c59 2c61 2c1d
2c7e 2c6d 2c71 2c79 2c54 2c9a 2c89 2c8d
2c95 2c51 2cb2 2ca1 2ca5 2cad 2c88 2cce
2cbd 2cc1 2cc9 2c85 2ce6 2cd5 2cd9 2ce1
2cbc 2d02 2cf1 2cf5 2cfd 2cb9 2d1a 2d09
2d0d 2d15 2cf0 2d36 2d25 2d29 2d31 2ced
2d4e 2d3d 2d41 2d49 2d24 2d6a 2d59 2d5d
2d65 2d21 2d82 2d71 2d75 2d7d 2d58 2d9e
2d8d 2d91 2d99 2d55 2db6 2da5 2da9 2db1
2d8c 2dd2 2dc1 2dc5 2dcd 2d89 2dea 2dd9
2ddd 2de5 2dc0 2e06 2df5 2df9 2e01 2dbd
2e1e 2e0d 2e11 2e19 2df4 2e3a 2e29 2e2d
2e35 2df1 2e52 2e41 2e45 2e4d 2e28 2e59
2e5d 2e61 2e65 2e69 2e6d 2e71 2e25 2e75
2e7a 2e7e 2e82 2e85 2e88 2e89 2e8b 2e8c
2e91 2e92 2e94 2e95 2e99 2e9d 2ea1 2ea2
2ea9 2eaa 2eb0 2eb4 2eb5 2eba 2ebe 2ec2
2ec6 2ece 2ec9 2ed2 2ed6 2eda 2ede 2ee2
2ee6 2eea 2eee 2ef2 2ef6 2efb 2f00 2f04
2f08 2f0c 2f11 2f14 2f18 2f1c 2f21 2f23
2f27 2f2b 2f2d 2f31 2f34 2f38 2f3c 2f40
2f44 2f48 2f4c 2f50 2f54 2f58 2f5d 2f62
2f66 2f6a 2f6e 2f73 2f76 2f7a 2f7e 2f83
2f85 2f89 2f8d 2f8f 2f93 2f96 2f9a 2f9e
2fa3 2fa5 2fa9 2fad 2fb0 2fb5 2fb8 2fbc
2fbd 2fc2 2fc5 2fca 2fcb 2fd0 2fd3 2fd7
2fd8 2fdd 2fde 2fe3 2fe7 2feb 2fee 1
2ff2 2ff6 2ff7 2ffb 2fff 3002 3003 300a
300e 3012 3015 3016 301b 301c 3020 3024
3027 302c 3031 3035 3039 303c 3040 3044
3047 3048 304d 3052 3057 305b 305e 3062
3066 3069 306d 3071 3074 3078 307c 307f
3083 3087 308b 308f 3093 3097 309f 309a
30a3 30a7 30ab 30af 30b3 30b7 30bb 30bf
30c3 30c8 30cd 30d1 30d5 30d9 30dd 30e2
30e8 30ec 30f0 30f3 30f8 30fb 30ff 3100
3105 3108 310d 310e 3113 3116 311a 311b
3120 3123 3128 3129 312e 3131 3135 3136
313b 313e 3143 3144 3149 314c 3150 3151
3156 3159 315e 315f 3164 3167 316b 316c
3171 3172 3177 317b 317f 3182 3186 3187
318c 3190 3194 3198 319b 319f 31a0 31a5
31a9 31ad 31b1 31b4 31b9 31bc 31c0 31c4
31c7 31c8 31ca 31cb 31d0 31d1 31d6 31da
31de 31e1 31e5 31e6 31eb 31ef 31f3 31f7
31fa 31fe 31ff 3204 3208 320c 3210 3213
3217 3218 321d 3221 3225 3229 322c 322f
3230 3235 3239 323d 3241 3244 3248 324c
3250 3254 3258 325c 3260 3264 3268 326c
3270 3274 3278 327c 3280 3288 3289 328d
3292 3294 3298 329f 32a3 32a7 32ac 32ae
32b2 32b5 32b8 32b9 32be 32c2 32c6 32c9
32cd 32ce 32d3 32d6 32d9 32da 32df 32e3
32e5 32e9 32ec 32f0 32f3 32f6 32f7 32fc
3300 3304 3307 3308 330d 3310 3314 3318
331b 331f 3320 3325 3328 332b 332e 332f
3334 3338 333a 333e 3342 3345 3348 3349
334e 3352 3354 3358 335c 335f 3363 3367
336a 336b 3370 3373 3377 337b 337e 3382
3383 3388 338b 338e 3391 3392 3397 339b
339d 33a1 33a5 33a8 33ab 33ac 33b1 33b5
33b7 33bb 33bf 33c2 33c6 33ca 33cd 33ce
33d3 33d6 33da 33de 33e1 33e5 33e6 33eb
33ee 33f1 33f4 33f5 33fa 33fe 3400 3404
3408 340b 340e 340f 3414 3418 341a 341e
3422 3425 3429 342d 3430 3431 3436 3439
343d 3441 3444 3448 3449 344e 3451 3454
3457 3458 345d 3461 3463 3467 346b 346e
3471 3472 3477 347b 347d 3481 3485 3488
348c 3490 3493 3494 3499 349c 34a0 34a4
34a7 34ab 34ac 34b1 34b4 34b7 34ba 34bb
34c0 34c4 34c6 34ca 34ce 34d1 34d4 34d5
34da 34de 34e0 34e4 34e8 34eb 34ef 34f3
34f6 34f7 34fc 34ff 3503 3507 350a 350e
350f 3514 3517 351a 351d 351e 3523 3527
3529 352d 3531 3534 3537 3538 353d 3541
3543 3547 354b 354e 3552 3556 3559 355d
3561 3565 3569 356d 3571 3575 3579 357d
3581 3585 3589 358d 3591 3595 3599 359d
35a1 35a5 35a8 35ab 35ac 35b1 35b5 35b8
35bb 35bc 35c1 35c5 35c8 35cb 35cc 35d1
35d5 35d9 35dd 35e1 35e5 35e9 35ed 35f1
35f9 35fa 35fe 3603 3605 3609 360d 3610
3614 3618 361c 3620 3624 3628 362c 3630
3634 3638 363c 3640 3644 3647 364a 364b
3650 3654 3657 365a 365b 3660 3664 3667
366a 366b 3670 3674 3678 3680 3681 3685
368a 368c 3690 3694 3697 369b 36a0 36a4
36a5 36aa 36ae 36b2 36b5 36b6 36bb 36bf
36c4 36c5 36c9 36ce 36cf 36d1 36d5 36d8
36da 36de 36e0 36ec 36f0 36f2 3706 3707
370b 3724 3713 3717 371f 3712 3740 372f
3733 373b 370f 372b 3747 374b 374e 3753
3754 3759 375d 375e 3762 3766 3767 376e
376f 3775 3779 377a 377f 3783 3784 3788
378c 3790 3793 3794 379b 379f 37a3 37a6
37a7 37ac 37ad 37b3 37b7 37b8 37bd 37c1
37c5 37c8 37cc 37d0 37d3 37d4 37d9 37de
37e3 37e7 37eb 37ee 37f2 37f6 37f9 37fa
37ff 3804 3809 380d 3811 3814 3815 3819
381d 3820 3821 3828 382c 3830 3833 3834
3839 383a 383e 3846 3847 384c 3850 3854
3857 3858 385c 3860 3863 3864 386b 386f
3873 3876 3877 387c 387d 3881 3889 388a
388f 3893 3898 389c 389d 38a2 38a4 38a8
38aa 38b6 38ba 38bc 38d0 38d1 38d5 38d9
38dd 38f6 38f2 38f1 38fe 38ee 3903 3906
390a 390e 390f 3913 3917 391a 391b 3922
3926 392a 392d 392e 3933 3934 3938 393c
393e 393f 3945 394a 394f 3951 395d 3961
3978 3967 396b 3973 3966 3994 3983 3987
398f 3963 39ac 399b 399f 39a7 3982 39c8
39b7 39bb 39c3 397f 39b3 39cf 39d3 39d6
39db 39dc 39e1 39e5 39e6 39ea 39ee 39ef
39f6 39f7 39fd 3a01 3a02 3a07 3a0b 3a0e
3a12 3a16 3a19 3a1c 3a1d 3a22 3a26 3a2a
3a2e 3a32 3a3a 3a35 3a3e 3a42 3a46 3a4a
3a4d 3a52 3a55 3a59 3a5a 3a5f 3a60 3a65
3a69 3a6d 3a72 3a77 3a7b 3a7f 3a83 3a88
3a8e 3a92 3a96 3a99 3a9a 3a9f 3aa3 3aa7
3aab 3aaf 3ab5 3ab7 3abb 3abe 3ac2 3ac6
3ac9 3acc 3acd 3ad2 3ad6 3ad8 3adc 3ae3
3ae7 3aeb 3af0 3af2 3af6 3afa 3afd 3b02
3b05 3b09 3b0a 3b0f 3b10 3b15 3b19 3b1c
3b1f 3b20 3b25 3b29 3b2d 3b30 1 3b34
3b38 3b39 3b3d 3b41 3b44 3b45 3b4c 3b50
3b54 3b57 3b58 3b5d 3b61 3b65 3b68 3b69
1 3b6e 3b73 3b74 3b78 3b7c 3b7f 3b84
3b89 3b8d 3b91 3b94 3b98 3b9c 3b9f 3ba0
3ba5 3ba9 3bad 3bb0 3bb1 1 3bb6 3bbb
3bc0 3bc5 3bc9 3bce 3bd2 3bd3 3bd8 3bda
3bde 3be1 3be3 3be7 3be9 3bf5 3bf9 3bfb
3c1b 3c13 3c17 3c12 3c22 3c0f 3c27 3c2b
3c44 3c33 3c37 3c3f 3c32 3c60 3c4f 3c53
3c5b 3c2f 3c78 3c67 3c6b 3c73 3c4e 3c94
3c83 3c87 3c8f 3c4b 3cac 3c9b 3c9f 3ca7
3c82 3cc8 3cb7 3cbb 3cc3 3c7f 3ce0 3ccf
3cd3 3cdb 3cb6 3cfc 3ceb 3cef 3cf7 3cb3
3d14 3d03 3d07 3d0f 3cea 3d1b 3d1f 3d38
3d34 3ce7 3d40 3d33 3d30 3d45 3d49 3d4d
3d4e 3d52 3d56 3d59 3d5a 3d61 3d65 3d69
3d6c 3d6d 3d72 3d73 3d77 3d7b 3d7d 3d7e
3d84 3d89 3d8e 3d90 3d9c 3da0 3da2 3da6
3da7 3dab 3daf 3db0 3db7 3db8 3dbe 3dc2
3dc3 3dc8 3dcc 3dcf 3dd3 3dd6 3dd7 3ddb
3ddf 3de0 3de7 3de8 3dec 3df0 3df3 3df4
3dfb 3dff 3e03 3e06 3e07 3e0c 3e0d 3e11
3e15 3e18 3e19 3e1e 3e1f 3e25 3e29 3e2a
3e2f 3e31 3e35 3e39 3e3c 3e40 3e42 3e43
3e48 3e4c 3e4e 3e5a 3e5c 3e60 3e63 3e66
3e67 3e6c 3e70 3e74 3e77 3e7c 3e7d 3e82
3e86 3e8a 3e8e 3e96 3e91 3e9a 3e9e 3ea2
3ea7 3eac 3eb0 3eb4 3eb9 3ebc 3ec0 3ec4
3ec7 3ecc 3ecd 3ed2 3ed6 3eda 3edb 3ee0
3ee4 3ee8 3eed 3eef 3ef3 3ef7 3efb 3efd
3f01 3f05 3f0a 3f0c 3f10 3f11 3f15 3f19
3f1a 3f21 3f22 3f28 3f2c 3f2d 3f32 3f36
3f37 3f3b 3f3f 3f43 3f46 3f47 3f4e 3f52
3f56 3f59 3f5a 3f5f 3f60 3f66 3f6a 3f6b
3f70 3f74 3f78 3f7c 3f80 3f81 3f83 3f87
3f8b 3f8e 3f91 3f92 3f97 3f9b 3f9f 3fa2
3fa7 3fa8 3fad 3fb1 3fb5 3fb9 3fba 3fbf
3fc3 3fc7 3fca 3fcb 3fcf 3fd3 3fd7 3fda
3fdb 3fe2 3fe6 3fea 3fed 3fee 3ff3 3ff4
3ffa 3ffe 3fff 4004 4008 400b 400e 400f
4014 4018 401c 401f 4024 4025 402a 402e
4033 4034 4038 403c 403f 4044 4045 404a
404e 4052 4056 4058 405c 4060 4064 4066
406a 406e 4071 4073 4077 407b 407f 4081
4085 4089 408c 408e 4092 4096 4099 409b
409f 40a3 40a6 40a7 40ab 40af 40b3 40b6
40b7 40be 40c2 40c6 40c9 40ca 40cf 40d0
40d6 40da 40db 40e0 40e4 40e7 40ea 40eb
40f0 40f4 40f5 40f9 40fd 40fe 4105 4106
410c 4110 4111 4116 411a 411d 4121 4125
4126 412a 412e 4132 4135 4136 413d 4141
4145 4148 4149 414e 414f 4155 4159 415a
415f 4163 4167 416b 416f 4170 4172 4176
417a 417d 4180 4181 4186 418a 418e 4191
4196 4197 419c 41a0 41a4 41a7 41ab 41ae
41b3 41b4 41b8 41bc 41bf 41c3 41c6 41cb
41cc 41d0 41d4 41d7 41dc 41dd 41e2 41e6
41ea 41ed 41f1 41f4 41f8 41f9 41fe 4200
4204 4205 4209 420d 4211 4214 4215 421c
4220 4224 4227 4228 422d 422e 4234 4238
4239 423e 4240 4244 4248 424c 4250 4252
4253 4258 425c 425e 426a 426c 426e 4272
4276 4279 427b 427f 4283 4287 4289 428d
4291 4294 4296 429a 429e 42a1 42a5 42a9
42ad 42ae 42b2 42b3 42ba 42bb 42c1 42c5
42c6 42cb 42d0 42d4 42d6 42da 42dc 42e8
42ec 42ee 430e 4306 430a 4305 4315 4302
431a 431e 4337 4326 432a 4332 4325 4353
4342 4346 434e 4322 436b 435a 435e 4366
4341 4387 4376 437a 4382 433e 439f 438e
4392 439a 4375 43bb 43aa 43ae 43b6 4372
43d3 43c2 43c6 43ce 43a9 43ef 43de 43e2
43ea 43a6 43da 43f6 43f7 43fb 43ff 4400
4407 4408 440e 4412 4413 4418 441c 441f
4423 4427 442a 442e 4432 4436 4439 443a
443e 4442 4446 4449 444a 4451 4455 4459
445c 445d 4462 4463 4469 446d 446e 4473
4477 447a 447d 447e 4483 4487 448b 448e
448f 4493 4497 449b 449e 449f 44a6 44aa
44ae 44b1 44b2 44b7 44b8 44be 44c2 44c3
44c8 44cc 44cf 44d2 44d3 44d8 44dc 44e0
44e3 44e4 44e8 44ec 44f0 44f3 44f4 44fb
44ff 4503 4506 4507 450c 450d 4513 4517
4518 451d 4521 4524 4527 4528 452d 4531
4535 4539 453a 453e 4542 4546 454a 454e
4551 4552 4559 455d 4561 4564 4565 456a
456b 4571 4575 4576 457b 457d 4581 4585
4589 458b 458f 4593 4596 4598 459c 45a0
45a4 45a6 45aa 45ae 45b1 45b3 45b7 45bb
45bf 45c1 45c5 45c9 45cc 45d0 45d4 45d7
45dc 45df 45e3 45e4 45e9 45ea 45ef 45f3
45f7 45fa 45ff 4602 4606 4607 460c 460d
4612 4616 461a 461d 4622 4625 4629 462a
462f 4630 4635 4639 463d 4641 4645 4649
464a 464e 464f 4656 4657 465d 4661 4662
4667 466c 4670 4672 4676 4678 4684 4688
468a 468e 46aa 46a6 46a5 46b2 46bf 46bb
46a2 46c7 46ba 46cc 46d0 46d4 46d8 4701
46e0 46e4 46b7 46e8 46ec 46f0 46f4 46fc
46df 4708 470c 46dc 4710 4714 4715 471a
471e 4722 4726 472a 472f 4733 4737 4738
473d 4740 4743 4744 4749 474c 4750 4754
4759 475d 4761 4762 4767 4768 476d 4770
4773 4776 4777 477c 477f 4783 4787 478c
4790 4794 4795 479a 479b 47a0 47a3 47a6
47a9 47aa 47af 47b2 47b6 47ba 47bf 47c3
47c7 47c8 47cd 47ce 47d3 47d7 47d9 47dd
47df 47eb 47ef 47f1 47f3 47f5 47f9 4805
4809 480b 480e 4810 4811 481a 
1296
2
0 1 9 15 1 11 1a 19
11 21 11 :2 1 :2 1a 24 1a :2 1
:2 b 15 b :2 1 :2 20 2a 20 :2 1
:2 17 21 17 :2 1 :2 11 1b 11 :2 1
:2 12 1c 1d :2 1c 12 :2 1 :2 13 1d
1e :2 1d 13 :2 1 :2 10 1a 1b :2 1a
10 :2 1 :2 16 20 21 :2 20 16 :2 1
a 15 0 1c :2 1 4 b 4
:2 1 5 :4 1 b 1f 2c :2 1f 1e
:2 1 4 :3 10 :2 4 :3 11 :2 4 b 14
13 :2 b :2 4 :3 11 :2 4 :3 f :2 4 :3 c
:2 4 :3 d :2 4 :3 1a :2 4 :3 16 :2 4 :3 16
:2 4 :3 1a :2 4 :2 10 19 :2 4 b 1c
26 :2 2b 34 3d 44 4d 50 :2 3d
:2 26 :2 1c b 9 17 9 4 9
:5 4 b :2 12 :2 2c b 39 4a 45
4a 45 :4 4 :2 b 18 2a 25 2a
25 :4 4 7 12 15 :2 12 7 16
1f 29 40 53 66 :2 7 20 27
20 35 46 4f 3 c 16 2d
40 53 e 1c 29 3 c 16
2d 40 53 d :4 7 18 20 27
20 35 46 4f e 1c 29 d
:4 7 :4 4 1e 25 1e b 14 1c
28 36 44 e 1b 23 2f 30
:2 2f 33 41 e c 7 c d
1a :3 18 b 1b :2 2a 37 32 37
:2 29 36 3c 45 :2 29 32 23 22
b :2 d :2 7 :3 5 10 17 10 d
1f :3 1d :2 4 10 17 10 2c 3d
f 20 f 32 2d 32 2d 8
7 :8 4 :6 1 b 1a 22 26 :2 1a
2e 37 3b :2 2e 43 59 5d :2 43
5 17 1b :2 5 23 35 39 :2 23
5 1b 1f :2 5 19 :2 1 4 9
17 1b 17 :2 4 :3 e :2 4 d 16
15 d 1e d :2 4 a 14 a
:2 1a :3 a :2 4 9 :2 17 4 8 d
10 12 d 4 d 1c :3 7 11
1b 11 7 c f 16 22 16
11 :2 a f 16 2f 16 11 :2 a
f 16 2b 16 11 :2 a f 16
2b 16 11 :2 a b 12 1d 12
d :2 6 f 16 2f 16 11 :2 a
:6 7 12 8 :2 4 a :2 4 :6 1 b
1b 28 :3 1b 26 :2 1b 1a :2 1 4
:3 11 :2 4 :3 9 :2 4 9 13 9 :2 1e
:3 9 :2 4 :3 7 :2 4 :3 7 :2 4 :3 9 :2 4
b 14 13 :2 b :2 4 :3 11 :2 4 :3 16
:2 4 :3 f :2 4 :3 c :2 4 :3 d :2 4 :3 1a
:2 4 :3 16 :2 4 :3 16 :2 4 :3 1a :2 4 :3 11
:2 4 :3 12 :2 4 :3 1f :2 4 :3 1b :2 4 :3 1b
:2 4 :3 1f :2 4 :3 12 :2 4 :3 13 :2 4 :3 20
:2 4 :3 1c :2 4 :3 1c :2 4 :3 20 :2 4 b
e 1a :3 e 16 :3 e 1a :2 e d
:2 4 e 16 22 2b 34 3b 3c
44 :3 34 e 10 :2 16 10 :2 16 1a
:2 20 2e 2b :2 34 :2 2b 44 :2 4a 55
57 :2 5d :2 44 :2 15 6d :2 73 :2 10 82
10 1a :2 20 30 2d :2 36 :2 2d 48
:2 4e 5b 5d :2 63 :2 48 :2 15 75 :2 7b
:2 10 8c :2 10 16 1e 2a 38 16
14 1b 14 f 14 15 1f :3 1d
:2 f e 25 e 16 1e 2a 38
16 14 f 14 :2 f e 1f e
9 e f :2 15 22 20 :2 28 :2 20
:2 9 :3 7 d 18 :3 15 :2 7 :2 10 :3 7
:6 4 :2 10 19 3a 3d :2 19 :3 4 19
:2 4 b 1c 26 :2 2b 34 3d 44
4d 50 :2 3d :2 26 :2 1c b 9 17
9 4 9 :5 4 10 17 10 d
1f :3 1d :2 4 10 17 10 d 1f
:3 1d :2 4 :2 b 18 2a 25 2a 25
:4 4 7 12 15 :2 12 7 16 1f
29 40 53 66 :2 7 e 1c 25
2f 46 59 6c e c 1f 2d
3c 58 70 88 c 13 c 7
c d 17 :3 15 :5 7 b 16 :3 13
:2 a 1c 24 26 :2 1c 1b 33 34
:2 1b a 24 a 1b 22 23 :2 1b
a :4 7 b 17 :3 14 :2 a 1d 26
28 :2 1d 1c 36 37 :2 1c a 26
a 1c 24 25 :2 1c a :4 7 b
24 :3 21 :2 a 2a 40 42 :2 2a 29
5d 5e :2 29 a 40 a 29 3e
3f :2 29 a :4 7 b 20 :3 1d :2 a
26 38 3a :2 26 25 51 52 :2 25
a 38 a 25 36 37 :2 25 a
:4 7 b 20 :3 1d :2 a 26 38 3a
:2 26 25 51 52 :2 25 a 38 a
25 36 37 :2 25 a :4 7 b 24
:3 21 :2 a 2a 40 42 :2 2a 29 5d
5e :2 29 a 40 a 29 3e 3f
:2 29 a :4 7 13 1a 13 28 39
47 5a 63 6d 84 97 aa e
1c 2a 3d 4c 5c 79 92 ab
d :4 7 18 :2 e :2 c 13 c 7
c d 17 :3 15 :5 7 13 1a 13
28 39 47 e 1c 2a d :4 7
:5 4 9 c 1a 25 :2 9 4 8
d 10 18 d 4 d 15 1d
23 27 2b :4 7 11 14 11 7
13 1a 13 d 1e 26 32 3e
4c 14 22 2a 30 34 38 13
:4 7 18 8 :2 4 a :2 4 b 12
b 24 b 2a 2c 2a 12 :2 14
12 22 2c 22 1d 22 12 :2 14
21 1f :2 23 :2 1f 1d b a 2a
d :2 f 21 :3 1f :3 4 10 17 10
d 17 :3 15 :2 4 10 17 10 d
1f :3 1d :7 4 :6 1 b 25 2d :2 25
24 :2 1 4 :3 11 4 :2 b 20 32
2d 32 2d :4 4 10 17 10 :2 d
:2 20 c 13 c 7 c d 1f
:3 1d 7 19 18 :2 d 1a :3 18 :2 d
:2 4 10 17 10 d 1f :3 1d d
1a :3 18 :2 d :2 4 10 17 10 :2 d
:2 23 33 3a 33 2e 33 d 1f
:3 1d 2e 1c 1b :2 d 1a :3 18 :2 d
:2 4 10 17 10 :2 d :2 23 33 3a
33 2e 33 22 34 :3 32 2e 1c
1b :2 d 1a :3 18 :2 d :2 4 10 17
10 :2 d :2 23 33 3a 33 2e 33
22 34 :3 32 2e 1c 1b :2 d 1a
:3 18 :2 d :7 4 :6 1 b 0 :2 1 4
:3 11 4 :2 b 20 32 2d 32 2d
:4 4 10 17 10 :2 d :2 20 c 13
c 7 c d 1f :3 1d 7 19
18 d :2 4 10 17 10 d 1f
:3 1d :2 4 10 17 10 d 1f :3 1d
:2 4 10 17 10 d 1f :3 1d :2 4
10 17 10 :2 d :2 23 33 3a 33
2e 33 22 34 :3 32 2e 1c 1b
d :2 4 10 17 10 d 1f :3 1d
:2 4 10 17 10 :2 d :2 23 33 3a
33 2e 33 22 34 :3 32 2e 1c
1b d :2 4 10 17 10 d 1f
:3 1d :2 4 10 17 10 :2 d :2 23 33
3a 33 2e 33 22 34 :3 32 2e
1c 1b d :2 4 10 17 10 d
1f :3 1d :7 4 :6 1 b 20 2d :2 20
1f :2 1 3 :3 14 :2 3 :3 14 :2 3 :3 14
:2 3 :3 14 3 a 1b 29 33 :2 38
41 4a 51 5a 5d :2 4a :2 33 :2 29
:2 a 18 2a a 5 a 5 :4 3
:4 a 11 a 5 :2 a 1c :3 1a 5
:4 3 6 1a :3 17 c 13 :2 c 19
c 1c 29 1c c 16 :3 14 :3 5
7 12 15 :2 12 e 15 :2 e 16
e 19 22 19 25 3b 25 3e
50 3e 53 65 53 68 7e 68
e 18 :3 16 :3 7 18 :2 4 c 13
:2 c 1e :3 1c :2 5 c 13 c 28
39 d 1b c :4 5 2b :7 3 :6 1
b 0 :2 1 3 :3 14 :2 3 :3 14 :2 3
:3 14 :2 3 :3 14 3 a 1b 29 33
:2 38 41 4a 51 5a 5d :2 4a :2 33
:2 29 :2 a 18 2a a 5 a 5
:4 3 :4 a 11 a 5 :2 a 1c :3 1a
5 :4 3 6 1a :3 17 c 13 :2 c
19 c 1c 29 1c :2 c :2 22 32
39 32 2d 32 c 1e :3 1c 2d
1b 1a c :3 5 7 12 15 :2 12
e 15 :2 e 16 e 19 22 19
25 3b 25 3e 50 3e 53 65
53 68 7e 68 e 20 :3 1e :3 7
18 :2 4 c 13 :2 c 1e :3 1c :2 5
c 13 c 28 39 d 1b c
:4 5 2b :7 3 :6 1 b 0 :2 1 4
b 14 21 :2 14 13 :2 4 e :2 10
18 :2 1a e 2a 31 2a 43 2a
25 2a 16 :2 18 2a :3 28 25 :3 7
:6 4 b 15 1d :2 15 14 :2 4 e
15 1f 15 10 15 27 :2 29 36
:3 34 10 :3 7 :6 4 e 18 :3 e :2 4
:3 11 :2 4 :3 f :2 4 :3 11 :2 4 :3 f :2 4
:3 c :2 4 :3 d :2 4 :3 1a :2 4 :3 16 :2 4
:3 16 :2 4 :3 1a :2 4 b 14 13 :2 b
:2 4 :2 10 19 :5 4 b 1c 26 :2 2b
34 3d 44 4d 50 :2 3d :2 26 :2 1c
b 9 17 9 4 9 :5 4 b
:2 12 :2 2b b 38 48 43 48 43
:4 4 :2 b 18 2a 25 2a 25 :4 4
8 14 1d :2 14 :2 4 a :2 13 :3 a
e 1b 25 :2 2e :2 1b :2 a :2 14 20
2c 33 2c 27 2c 14 1f 1e
:2 29 :2 1e 14 24 :3 23 :2 14 27 :4 d
19 20 19 2f 37 43 51 18
:2 21 29 :2 33 1d :2 27 35 :2 3f 17
:4 d a e a 2a :2 7 4 8
4 7 12 15 :2 12 7 16 1f
29 40 53 66 :2 7 13 1a 13
27 38 44 52 5b 65 7c 8f
a2 e 1c 28 36 3f 49 60
73 86 d :4 7 18 13 1a 13
27 38 44 e 1c 28 d :4 7
:4 4 10 17 10 b 17 1f 2b
39 f 1b 23 :2 2f 33 2f :2 42
46 42 f 13 1a 13 e 13
17 23 e 8 :8 4 :6 1 b 0
:2 1 4 b 13 1f :2 13 12 :2 4
:2 e c 13 c 7 c d 1f
:3 1d :2 7 :2 10 :3 7 :6 4 :3 11 :2 4 :3 11
:2 4 :3 11 4 :2 b 20 32 2d 32
2d :5 4 9 11 :2 9 4 a 17
:2 4 8 10 8 :2 7 d :4 7 1a
:2 4 a 17 :2 4 8 10 8 :2 7
d :4 7 1a :3 4 a :2 4 10 17
10 :2 a :2 e c 13 c 7 c
d 1f :3 1d b 18 :3 16 :2 d :2 7
18 a :2 4 10 17 10 d 1f
:3 1d d 1a :3 18 :2 d :7 4 :6 1 b
17 1e 22 :2 17 16 :2 1 4 b
13 1f :2 13 12 :2 4 e 1a 28
31 3b 52 65 78 e c 13
c 7 c d 1f :3 1d :2 7 :2 10
:3 7 :6 4 b 16 1e :2 16 26 2e
:2 26 36 46 :2 36 15 :2 4 e :2 17
23 :2 2c 18 :2 21 31 2e :2 3a :2 2e
4c :2 55 62 64 :2 6d :2 4c :2 13 7f
:2 88 :2 e 18 :2 21 2f 2c :2 38 :2 2c
48 :2 51 5c 5e :2 67 :2 48 :2 13 77
:2 80 :3 e :2 17 23 :2 2c e 14 1b
22 1b 16 1b 39 46 :3 44 16
d c 4c c 14 1b 22 1b
16 1b 39 46 :3 44 16 d c
4c :2 c 13 c 25 c 7 c
d :2 16 23 21 :2 2c :2 21 d :2 16
23 21 :2 2c :2 21 :3 d :2 16 28 :3 26
:2 d 18 :2 21 2f 2c :2 38 :2 2c 48
:2 51 5c 5e :2 67 :2 48 :2 13 77 :2 80
:2 e d 93 :3 90 :2 d :4 7 :6 4 :3 12
:2 4 :3 17 :2 4 d 17 d :2 22 :3 d
:2 4 11 1a 19 :2 11 :2 4 :3 11 :2 4
:3 11 :2 4 :3 11 :2 4 :3 11 :2 4 :3 11 :2 4
:3 11 :2 4 :3 11 :2 4 :3 1c :2 4 :3 18 :2 4
:3 18 :2 4 :3 1c :2 4 :3 11 :2 4 :3 f :2 4
:3 1c :2 4 :3 18 :2 4 :3 18 :2 4 :3 1c :2 4
:3 12 :2 4 :3 13 :2 4 :3 20 :2 4 :3 1c :2 4
:3 1c :2 4 :3 20 :2 4 :3 11 :2 4 :3 11 :2 4
:3 11 :2 4 :3 11 :2 4 :3 14 :2 4 :3 1a :2 4
:3 1b :2 4 :3 13 :2 4 :3 11 :2 4 e 4
b 1c 26 :2 2b 34 3d 44 4d
50 :2 3d :2 26 :2 1c b 9 17 9
4 9 :6 4 9 11 :2 9 4 a
17 21 2d 38 44 5d 72 87
:3 4 8 10 8 :2 7 d :4 7 1a
:2 4 a 17 21 2d 38 44 5d
72 87 :3 4 8 10 8 :2 7 d
:4 7 1a :3 4 a :3 4 :2 10 19 20
23 :2 19 2c 2f :2 19 37 3a :2 19
:2 4 10 17 10 :2 a :2 20 30 37
30 2b 30 a 1c :3 1a 2b 19
18 a :2 4 10 17 10 a 1c
:3 1a :3 4 11 :2 4 19 :2 4 1e :2 4
15 :2 4 9 14 1e 27 :2 9 :2 4
d 1d 26 2e 1d 2a 39 :4 7
11 1c 11 :2 7 :2 13 a 15 18
:2 a 20 :3 a 16 19 :2 a 24 27
:2 a 33 36 :2 a 42 :3 a 18 1b
:2 a 29 2c :2 a 3f 42 :2 a :2 7
a 18 22 23 :2 18 :2 a 1b 25
26 :2 1b :2 a :2 16 1f 30 33 39
47 :2 33 :2 1f :3 a 1f 31 33 :2 1f
a 3 1d 34 36 :2 1d :2 3 14
22 24 :2 14 3 a 17 21 23
:2 17 a 16 1d 16 6 12 1a
26 32 40 4c 11 1b 23 2c
39 45 51 10 :4 a 4 8 1
4 a :2 4 7 11 13 :2 11 7
11 23 25 :2 11 3c 3e :2 11 7
15 :2 4 7 12 15 :2 12 b 18
:3 15 :2 a 1c 26 28 :2 1c 1b 32
33 :2 1b a 23 a 1b 24 25
:2 1b a :4 7 b 19 :3 16 :2 a 1d
28 2a :2 1d 1c 35 36 :2 1c a
25 a 1c 26 27 :2 1c a :4 7
b 26 :3 23 :2 a 2a 42 44 :2 2a
29 5c 5d :2 29 a 3f a 29
40 41 :2 29 a :4 7 b 22 :3 1f
:2 a 26 3a 3c :2 26 25 50 51
:2 25 a 37 a 25 38 39 :2 25
a :4 7 b 22 :3 1f :2 a 26 3a
3c :2 26 25 50 51 :2 25 a 37
a 25 38 39 :2 25 a :4 7 b
26 :3 23 :2 a 2a 42 44 :2 2a 29
5c 5d :2 29 a 3f a 29 40
41 :2 29 a :4 7 13 1a 13 26
37 43 51 64 3 1f 3b 3
c 16 2d 40 53 e 1c 26
32 3e 4b 4c :2 3e 55 66 67
:2 55 3 19 1a :2 3 23 3 12
22 3f 58 71 d :4 7 18 13
1a 13 26 37 43 51 64 11
2d 49 e 1c 26 32 3e 4b
4c :2 3e 55 66 67 :2 55 11 27
28 :2 11 31 d :4 7 :9 4 6 f
:3 d :6 7 25 :2 3 :6 1 b 0 :2 1
4 :3 11 :2 4 :3 b :2 4 :2 10 19 :2 4
:2 b 20 32 2d 32 2d :4 4 :2 b
1b 9 10 9 4 9 a 1c
:3 1a :5 4 10 17 10 2a 37 :3 35
:2 4 10 17 10 2e 3b :3 39 :2 4
10 17 10 d 14 1b 14 f
14 c 19 :3 17 f 6 :3 4 10
17 10 d 14 1b 14 f 14
c 19 :3 17 f 6 :8 4 :6 1 b
0 :2 1 4 b 13 1f :2 13 12
:2 4 :2 e c 13 c 7 c d
1f :3 1d :2 7 :2 10 :3 7 :6 4 :3 f :2 4
:3 11 :2 4 :3 11 :2 4 :3 e :2 4 :2 10 19
:2 4 :2 b 20 32 2d 32 2d :5 4
11 :2 4 e f :2 e :2 4 9 11
:2 9 :2 4 5 :2 11 1a 29 2c :2 1a
:2 5 b 18 :2 5 7 11 19 11
7 8 14 :3 12 7 11 :3 7 33
:3 5 12 1c 1e :2 12 5 4 8
1 4 a :3 4 :2 10 19 25 28
:2 19 :2 4 7 e 11 :2 e 13 1a
13 :2 d :2 11 f 16 f a f
10 22 :3 20 e 1c :3 19 :2 10 :2 a
1b d :2 7 13 1a 13 f 21
:3 1f d 1b :3 18 :2 f :7 7 13 :2 4
:6 1 b 1b 27 2b :2 1b 1a :2 1
4 :3 11 :2 4 :3 b :2 4 :3 14 :2 4 :3 17
:2 4 :3 e :2 4 :3 e :2 4 :3 e :2 4 :3 e
:2 4 :3 d :2 4 b 13 1f :2 13 12
:2 4 :2 e c 13 c 7 c d
1f :3 1d :2 7 :2 10 :3 7 :5 4 :2 b 20
32 2d 32 2d :5 4 17 4 :2 e
15 2a 25 2a 1f 26 2d 26
21 26 1e 30 :3 2e 21 18 17
:3 10 25 :4 7 4 c a 1d a
1a :2 7 4 :3 1 7 17 19 :2 17
7 :2 13 1c :3 7 c 14 :2 c 7
d 1a :2 7 b 13 b :2 a :2 16
1f :3 a 1e :3 a 10 :3 a 14 a
1d a 10 :2 a :2 11 1e 2d 28
2d 28 :4 a :2 11 23 32 39 32
2d 32 4d 5f :3 5d 2d :5 a 16
22 2d :2 16 a d 16 19 :2 16
d :2 19 22 :3 d 1d 28 :2 d :6 14
1b 14 f :2 14 26 :3 24 f :4 d
10 23 25 :2 23 10 :2 1c 25 :6 10
:2 1c 25 :3 10 1a 10 27 10 1a
10 :4 d 1d d 17 d :4 a :4 7
1b :6 e 15 e 9 :2 e 20 :3 1e
9 :4 7 a 1d 1f :2 1d :2 11 1e
2d 28 2d 28 :4 a :2 11 15 :2 11
28 37 3e 37 32 37 51 63
:3 61 32 :5 a 16 22 2d :2 16 a
d 16 19 :2 16 d :2 19 22 :3 d
:2 14 :2 22 :3 d :2 14 :2 22 :3 d :2 19 22
:3 d :2 14 :2 22 2e :2 d 1d :4 17 1e
17 12 :2 17 29 :3 27 12 :4 10 d
15 13 1d 13 23 :2 10 d :7 a
21 a 14 a :4 7 :5 4 9 :2 20
2c 27 2c 27 :4 19 4 19 :6 1
b 1b 27 2b :2 1b 1a :2 1 4
:3 11 :2 4 :3 1b :2 4 :3 1b :2 4 :3 12 :2 4
:3 11 :2 4 :3 13 :2 4 :3 17 :2 4 :3 12 4
:2 b 20 32 2d 32 2d :5 4 1e
:2 4 15 4 :6 b 12 b 6 :2 b
1d :3 1b 6 :4 4 7 16 18 :2 16
:6 e 15 e 9 :2 e 20 :3 1e 9
:4 7 a 1d 1f :2 1d :6 11 18 11
c :2 11 23 :3 21 c :4 a d 1b
1e :2 1b 14 30 4c :2 14 2c 44
14 1b 14 f :2 14 26 :3 24 f
:4 d 20 d 27 d :4 a 21 a
24 a :4 7 1a 7 21 7 :5 4
:2 10 19 37 3a :2 19 :3 4 :2 10 19
37 3a :2 19 :3 4 :2 10 19 2e 31
:2 19 :3 4 9 20 38 50 20 63
5e 63 5e :4 19 4 19 :7 1 a
16 24 :2 16 2f 3b :2 2f 15 46
4d :2 1 4 d 16 1a 20 :3 16
d :2 4 10 19 1b :2 10 :2 4 d
:2 15 1e :3 d 27 28 :2 d 2b d
:2 15 1f :5 d c 29 2a :2 c 2d
d :2 15 21 :3 d :2 c b 2b 2c
:2 b 2f d :2 15 21 :3 d :2 b 4
:a 1 5 :6 1 
1296
4
0 :3 1 :8 7 :6 8
:6 9 :6 a :6 b :6 c
:9 d :9 e :9 f :9 10
:3 13 0 :3 13 :3 15
:2 14 16 :4 13 :8 1c
:5 1d :5 1e :7 1f :5 22
:5 23 :5 24 :5 25 :5 26
:5 27 :5 28 :5 29 :6 2b
:11 2d :2 2e :4 2f :4 2d
:f 31 :b 33 :5 36 :9 37
:6 39 :6 3a :3 3b :6 3c
3b :4 39 36 :6 3e
:4 3f :4 3e 3d :3 36
:3 47 :6 48 :a 49 :3 4a
:5 4b :7 4c :7 4d :4 4c
:2 4b 4a 49 :3 47
:3 50 :5 51 :2 50 :5 53
:9 54 :3 53 :5 56 :2 2a
:4 1c :10 59 :a 5a :5 5b
:3 59 :6 5c :5 5d :8 5e
:a 5f :5 61 :6 62 :4 63
:5 64 65 :7 66 :7 67
:7 68 :7 69 :7 6a :7 6b
:6 65 62 6d 62
:4 6e :2 60 :4 59 :5 7a
:4 7b :3 7a :5 7c :5 7d
:a 7e :5 7f :5 80 :5 81
:7 82 :5 83 :5 84 :5 85
:5 86 :5 87 :5 88 :5 89
:5 8a :5 8b :5 8c :5 8d
:5 8e :5 8f :5 90 :5 91
:5 92 :5 93 :5 94 :5 95
:5 96 :5 97 :6 98 :4 99
:4 9a :3 98 :c 9b :3 9e
:3 9f :1b a0 :1b a1 9e
:5 a3 :5 a4 :5 a5 a4
:2 a3 a5 a3 :5 a7
:4 a8 :2 a7 a8 a7
a2 a3 :9 aa a2
9e 9d 9c 9d
:5 ac 9c 9b :3 ad
:2 9b :5 98 :a af :4 b2
:11 b4 :2 b5 :4 b6 :4 b4
:3 b8 :5 b9 :2 b8 :3 bb
:5 bc :2 bb :b be :5 c0
:9 c1 :8 c3 :7 c4 :5 c5
:5 c6 c5 :4 c3 :6 c8
:c c9 c8 :7 cb ca
:3 c8 :6 cd :c ce cd
:7 d0 cf :3 cd :6 d2
:c d3 d2 :7 d5 d4
:3 d2 :6 d7 :c d8 d7
:7 da d9 :3 d7 :6 dc
:c dd dc :7 df de
:3 dc :6 e1 :c e2 e1
:7 e4 e3 :3 e1 :c e7
:a e8 :4 e7 c0 :2 ea
eb :5 ec :5 ed ec
:4 ea :6 ef :4 f0 :4 ef
e9 :3 c0 :8 f4 :6 f5
:9 f6 :5 f7 :3 f8 :6 f9
:7 fa :4 f8 f5 fb
f5 :4 fc :8 ff :9 100
:9 101 :3 100 ff :7 102
:3 ff :3 105 :5 106 :2 105
:3 108 :5 109 :2 108 :5 10a
:2 ae :4 7a :8 111 :5 112
:b 114 :3 116 :4 117 :5 118
:5 119 118 :3 117 :5 11a
:2 117 :2 116 :3 11c :5 11d
:5 11e :2 11d :2 11c :3 120
:9 121 :5 122 :4 121 :5 123
:2 121 :2 120 :3 125 :9 126
:5 127 :4 126 :5 128 :2 126
:2 125 :3 12a :9 12b :5 12c
:4 12b :5 12d :2 12b :2 12a
:5 12f :2 113 :4 111 132
0 :2 132 :5 133 :b 135
:3 137 :4 138 :5 139 :5 13a
139 :3 138 :2 137 :3 13c
:5 13d :2 13c :3 13f :5 140
:2 13f :3 142 :5 143 :2 142
:3 145 :9 146 :5 147 :4 146
:2 145 :3 149 :5 14a :2 149
:3 14c :9 14d :5 14e :4 14d
:2 14c :3 150 :5 151 :2 150
:3 153 :9 154 :5 155 :4 154
:2 153 :3 157 :5 158 :2 157
:5 15a :2 134 :4 132 :8 160
:5 162 :5 163 :5 164 :5 165
:12 16a :3 16b :4 16c :4 16a
:2 16f 170 :5 171 :5 172
171 :4 16f :5 174 :3 177
:6 178 :5 179 :3 177 :5 17b
:3 17d :12 17e :5 17f :3 17d
:3 17b :3 185 :5 186 :2 184
:5 189 :3 18a :4 188 :3 174
:5 18e :2 167 :4 160 194
0 :2 194 :5 196 :5 197
:5 198 :5 199 :12 19e :3 19f
:4 1a0 :4 19e :2 1a3 1a4
:5 1a5 :5 1a6 1a5 :4 1a3
:5 1a8 :3 1ab :6 1ac :9 1ad
:5 1ae :4 1ad :3 1ab :5 1b0
:3 1b2 :12 1b3 :5 1b4 :3 1b2
:3 1b0 :3 1ba :5 1bb :2 1b9
:5 1be :3 1bf :4 1bd :3 1a8
:5 1c3 :2 19b :4 194 1c9
0 :2 1c9 :9 1ca :e 1cb
:7 1cc :4 1cb :5 1ca :9 1cd
:11 1ce :5 1cd :7 1cf :5 1d0
:5 1d1 :5 1d2 :5 1d3 :5 1d4
:5 1d5 :5 1d6 :5 1d7 :5 1d8
:5 1d9 :7 1da :6 1dc :3 1df
:11 1e1 :2 1e2 :4 1e3 :4 1e1
:f 1e5 :b 1e7 :5 1ea 1eb
1ea :6 1ec :7 1ed 1ee
1ed :8 1ef :7 1f0 :5 1f1
:2 1f0 :5 1ef :7 1f2 :6 1f3
:6 1f4 1f3 :4 1f2 1ee
1f5 1ed :3 1ec 1eb
1f7 1ea :5 1f9 :9 1fa
:c 1fc :a 1fd :4 1fc 1f9
:6 1ff :4 200 :4 1ff 1fe
:3 1f9 :3 203 :5 204 :c 205
:5 206 :3 207 205 :3 203
:5 208 :2 1db :4 1c9 20e
0 :2 20e :9 20f :2 210
:5 211 :5 212 211 210
:3 213 :2 210 :5 20f :5 214
:5 215 :5 216 :b 218 :6 21a
:4 21b :4 21c :4 21d :2 21e
:3 21c :4 220 :4 221 :4 222
:2 223 :3 221 :4 225 :3 227
:2 228 :2 229 :5 22a :5 22b
:5 22c :2 22b 22a 229
:2 228 :2 227 :3 22e :5 22f
:5 230 :2 22f :2 22e :5 232
:2 217 :4 20e :9 23c :9 23d
:9 23e :5 23f :5 240 23f
23e :3 241 :2 23e :5 23d
:11 242 :6 243 :19 244 :19 245
:6 246 243 :10 247 :10 248
:5 249 :2 247 :9 24a :9 24b
:2 24a :7 24c :2 24a :1e 24d
:2 24a 247 :3 243 :5 242
:5 24e :5 24f :a 250 :7 251
:5 252 :5 253 :5 254 :5 255
:5 256 :5 257 :5 258 :5 259
:5 25a :5 25b :5 25c :5 25d
:5 25e :5 25f :5 260 :5 261
:5 262 :5 263 :5 264 :5 265
:5 266 :5 267 :5 268 :5 269
:5 26a :5 26b :5 26c :5 26d
:5 26e :5 26f :5 270 :5 271
:3 273 :11 275 :2 276 :4 277
:4 275 :6 27a :c 27b :4 27c
:4 27d :2 27e :3 27c :c 280
:4 281 :4 282 :2 283 :3 281
:4 285 :12 286 :3 288 :9 289
:5 28a :4 289 :2 288 :3 28c
:5 28d :2 28c :3 28f :3 290
:3 291 :3 292 :8 293 294
:4 296 :3 297 :3 296 :5 298
:3 29a :6 29b 29c :2 29b
:2 29c :2 29b :2 29c :2 29b
:2 29c :2 29b 29c 29d
:2 29b :2 29d :2 29b :2 29d
:2 29b :2 29d :2 29b :2 29a
:7 2a2 :7 2a3 :e 2a4 :7 2a5
:7 2a6 :7 2a7 :7 2a8 :3 2aa
:7 2ab :8 2ac :4 2aa 294
2af 272 :4 2b0 :5 2b2
:b 2b3 :3 2b2 :5 2b6 :6 2b7
:c 2b8 2b7 :7 2ba 2b9
:3 2b7 :6 2bc :c 2bd 2bc
:7 2bf 2be :3 2bc :6 2c1
:c 2c2 2c1 :7 2c4 2c3
:3 2c1 :6 2c6 :c 2c7 2c6
:7 2c9 2c8 :3 2c6 :6 2cb
:c 2cc 2cb :7 2ce 2cd
:3 2cb :6 2d0 :c 2d1 2d0
:7 2d3 2d2 :3 2d0 :8 2d6
:3 2d7 :6 2d8 :e 2d9 :6 2da
:6 2db 2d9 :4 2d6 2b6
:8 2dd :3 2de :e 2df :6 2e0
2df :4 2dd 2dc :3 2b6
:5 2e2 :5 2e8 :3 2e9 :3 2ea
:3 2e8 :2 272 :4 23c 2f3
0 :2 2f3 :5 2f4 :5 2f5
:6 2f7 :b 2f9 :3 2fb :5 2fc
:5 2fd 2fc :4 2fb :a 2ff
:a 301 :3 303 :6 304 :5 305
:2 304 :3 303 :3 307 :6 308
:5 309 :2 308 :3 307 :5 30b
:2 2f6 :4 2f3 312 0
:2 312 :9 313 :2 314 :5 315
:5 316 315 314 :3 317
:2 314 :5 313 :5 318 :5 319
:5 31a :5 31b :6 31d :b 31e
:3 320 :6 321 :6 322 323
:a 324 :4 325 :5 326 :5 327
:3 328 :2 329 :3 327 :7 32b
323 32d 31c :4 32e
:a 330 :5 331 :3 332 :2 333
:2 334 :5 335 :5 336 :5 337
:2 336 335 334 :2 333
:2 332 :3 339 :5 33a :5 33b
:2 33a :2 339 :5 33c :3 331
:2 31c :4 312 :9 343 :5 344
:5 345 :5 346 :5 347 :5 348
:5 349 :5 34a :5 34b :5 34c
:9 34d :2 34e :5 34f :5 350
34f 34e :3 351 :2 34e
:5 34d :b 353 :3 356 :6 358
:6 359 :5 35a :6 359 :5 358
357 35c :3 35d :3 35c
35b :3 352 :5 361 :6 363
:6 365 :4 367 :4 368 :6 36a
:4 36b :4 36c :3 36d 368
:4 370 :b 374 :12 375 :7 376
:5 377 :6 379 :5 37a :4 37c
37d :5 37e :5 37f 37e
:4 37c :5 381 :6 383 :3 384
:6 387 :3 388 381 :3 38a
389 :3 381 377 :3 38d
38c :3 377 36e :3 368
361 :4 391 392 :5 393
:5 394 393 :4 391 :5 396
:b 399 :15 39a :7 39b :5 39c
:6 39d :7 39e :7 39f :6 3a1
:8 3a2 39c :2 3a5 3a6
:5 3a7 :5 3a8 3a7 :4 3a5
3a4 3aa :3 3ab :3 3aa
3a9 :4 3a3 :3 39c 396
:3 3af 3ae :3 396 390
:3 361 :e 3b3 :2 352 :4 343
:9 3bb :5 3bc :5 3bd :5 3be
:5 3bf :5 3c0 :5 3c1 :5 3c2
:5 3c3 :b 3c5 :3 3c7 :3 3c8
:4 3cb 3cc :5 3cd :5 3ce
3cd :4 3cb :5 3d0 :4 3d2
3d3 :5 3d4 :5 3d5 3d4
:4 3d2 :5 3d7 :4 3d9 3da
:5 3db :5 3dc 3db :4 3d9
:5 3de :4 3df :3 3e0 :5 3e1
:5 3e2 3e1 :4 3df 3de
:3 3e4 3e3 :3 3de 3d7
:3 3e7 3e6 :3 3d7 3d0
:3 3ea 3e9 :3 3d0 :a 3ed
:a 3ee :a 3ef :10 3f1 :2 3c4
:4 3bb :f 3f5 :a 3f6 :7 3f8
:d 3f9 :7 3fa :3 3f9 :2 3fa
:2 3f9 3fa :7 3fb :3 3f9
:2 3fb :2 3f9 3fb :7 3fc
:3 3f9 :2 3f7 :4 3f5 :4 13
401 :6 1 
481c
4
:3 0 1 :3 0 2
:3 0 3 :6 0 1
:2 0 a :2 0 :2 5
:3 0 6 :2 0 3
5 7 :6 0 7
:4 0 b 8 9
1290 0 4 :6 0
c :2 0 7 9
:3 0 d :7 0 11
e f 1290 0
8 :6 0 a :2 0
:2 9 :3 0 13 :7 0
17 14 15 1290
0 b :6 0 f
:2 0 b 9 :3 0
19 :7 0 1d 1a
1b 1290 0 d
:6 0 11 :2 0 d
9 :3 0 1f :7 0
23 20 21 1290
0 e :6 0 13
:2 0 f 9 :3 0
25 :7 0 29 26
27 1290 0 10
:6 0 13 :2 0 13
9 :3 0 2b :7 0
14 :2 0 11 2d
2f :3 0 32 2c
30 1290 0 12
:6 0 13 :2 0 17
9 :3 0 34 :7 0
16 :2 0 15 36
38 :3 0 3b 35
39 1290 0 15
:6 0 13 :2 0 1b
9 :3 0 3d :7 0
18 :2 0 19 3f
41 :3 0 44 3e
42 1290 0 17
:9 0 1f 9 :3 0
46 :7 0 1a :2 0
1d 48 4a :3 0
4d 47 4b 1290
0 19 :6 0 1b
:3 0 1c :a 0 5d
2 :7 0 1d :3 0
5 :3 0 51 52
0 5d 4f 53
:2 0 1d :3 0 4
:3 0 56 :2 0 58
21 5c :3 0 5c
1c :4 0 5c 5b
58 59 :6 0 5d
1 0 4f 53
5c 1290 :2 0 1e
:a 0 16e 3 :7 0
25 :2 0 23 9
:3 0 b :7 0 62
61 :3 0 64 :2 0
16e 5f 65 :2 0
29 1f9 0 27
9 :3 0 68 :7 0
6b 69 0 16c
0 1f :6 0 2f
231 0 2d 9
:3 0 6d :7 0 70
6e 0 16c 0
20 :6 0 5 :3 0
22 :2 0 2b 72
74 :6 0 77 75
0 16c 0 21
:6 0 33 265 0
31 24 :3 0 79
:7 0 7c 7a 0
16c 0 23 :6 0
9 :3 0 7e :7 0
81 7f 0 16c
0 25 :6 0 37
299 0 35 9
:3 0 83 :7 0 86
84 0 16c 0
26 :6 0 9 :3 0
88 :7 0 8b 89
0 16c 0 27
:6 0 3b 2cd 0
39 9 :3 0 8d
:7 0 90 8e 0
16c 0 28 :6 0
9 :3 0 92 :7 0
95 93 0 16c
0 29 :6 0 a0
a1 0 3d 9
:3 0 97 :7 0 9a
98 0 16c 0
2a :6 0 9 :3 0
9c :7 0 9f 9d
0 16c 0 2b
:6 0 2c :3 0 2d
:3 0 2e :4 0 3f
a2 a4 :2 0 16a
2f :3 0 30 :3 0
38 :3 0 32 :3 0
c :2 0 33 :4 0
34 :3 0 35 :3 0
36 :2 0 37 :2 0
41 ac b0 45
a8 b2 :3 0 49
a7 b4 4b 20
:3 0 25 :3 0 39
:3 0 4e ba :2 0
bc :4 0 be bf
:5 0 b6 bb 0
50 0 bd :2 0
16a 3a :3 0 3b
:3 0 c1 c2 0
3c :3 0 c3 c4
0 53 1f :3 0
3d :3 0 55 c9
:2 0 cb :4 0 cd
ce :5 0 c6 ca
0 57 0 cc
:2 0 16a 3e :3 0
59 23 :3 0 3d
:3 0 5b d4 :2 0
d6 :4 0 d8 d9
:5 0 d1 d5 0
5d 0 d7 :2 0
16a 25 :3 0 3f
:2 0 40 :2 0 61
dc de :3 0 41
:3 0 26 :3 0 27
:3 0 28 :3 0 29
:3 0 2a :3 0 2b
:3 0 64 e0 e7
:2 0 103 3a :3 0
42 :2 0 4 e9
ea 0 2f :3 0
43 :3 0 23 :3 0
26 :3 0 27 :3 0
28 :3 0 29 :3 0
2a :3 0 2b :3 0
20 :3 0 1f :3 0
23 :3 0 26 :3 0
27 :3 0 28 :3 0
29 :3 0 2a :3 0
2b :3 0 6b :3 0
eb 100 101 102
:4 0 75 7f :4 0
ff :2 0 103 81
113 3a :3 0 42
:2 0 4 104 105
0 2f :3 0 43
:3 0 23 :3 0 20
:3 0 1f :3 0 23
:3 0 84 :3 0 106
10f 110 111 :4 0
88 8c :4 0 10e
:2 0 112 8e 114
df 103 0 115
0 112 0 115
90 0 16a 3a
:3 0 44 :2 0 4
116 117 0 43
:3 0 21 :3 0 45
:3 0 46 :3 0 47
:3 0 48 :3 0 1f
:3 0 21 :3 0 45
:3 0 13 :2 0 36
:2 0 93 122 124
:3 0 47 :3 0 48
:3 0 95 49 :3 0
9c 12a 144 0
145 :3 0 48 :3 0
b :3 0 4a :2 0
a0 12e 12f :3 0
4b :3 0 4c :3 0
a3 4d :3 0 a5
136 13e 0 13f
:3 0 4e :3 0 4f
:4 0 50 :4 0 3a
:4 0 a7 :3 0 138
139 13d :2 0 134
137 0 140 :3 0
131 132 141 130
143 142 :4 0 128
12b 0 118 148
146 149 :4 0 ab
0 147 :2 0 16a
3a :3 0 51 :2 0
4 14a 14b 0
2f :3 0 20 :3 0
52 :2 0 b4 14f
150 :3 0 14c 151
0 153 :2 0 152
:2 0 16a 3a :3 0
51 :2 0 4 154
155 0 2f :3 0
53 :3 0 2f :3 0
53 :3 0 b7 39
:3 0 ba 15d :2 0
15f :5 0 15b 15e
0 160 :3 0 156
163 161 164 :4 0
bc 0 162 :2 0
16a 54 :3 0 167
168 :2 0 169 54
:5 0 166 :2 0 16a
bf 16d :3 0 16d
c9 16d 16c 16a
16b :6 0 16e 1
0 5f 65 16d
1290 :2 0 41 :a 0
1ff 4 :7 0 d7
61b 0 d5 55
:3 0 9 :3 0 26
:6 0 174 173 :3 0
db 647 0 d9
55 :3 0 9 :3 0
27 :6 0 179 178
:3 0 55 :3 0 9
:3 0 28 :6 0 17e
17d :3 0 df 673
0 dd 55 :3 0
9 :3 0 29 :6 0
183 182 :3 0 55
:3 0 9 :3 0 2a
:6 0 188 187 :3 0
e8 6ae 0 e1
55 :3 0 9 :3 0
2b :6 0 18d 18c
:3 0 18f :2 0 1ff
170 190 :2 0 56
:3 0 193 0 196
1fd 58 :3 0 59
:7 0 57 196 193
:4 0 1a6 1a7 0
ec 57 :3 0 199
:7 0 19c 19a 0
1fd 0 5a :6 0
5 :3 0 5c :2 0
ea 19e 1a0 :6 0
5d :4 0 1a4 1a1
1a2 1fd 0 5b
:6 0 1b0 1b1 0
ee 5f :3 0 5e
:2 0 4 56 :3 0
56 :2 0 1 1a8
1aa :3 0 1ab :7 0
1ae 1ac 0 1fd
0 5e :6 0 60
:3 0 5a :3 0 5b
:3 0 1b2 :2 0 1fb
61 :3 0 36 :2 0
62 :2 0 63 :3 0
1b5 1b6 :2 0 1b4
1b8 5a :3 0 5e
:4 0 1bd :2 0 1f4
1ba 1bb :3 0 64
:3 0 5a :3 0 65
:3 0 1bf 1c0 :4 0
1c1 :3 0 1f4 61
:3 0 36 :2 0 27
:3 0 5e :3 0 1c5
1c6 0 1c8 f0
1ca f2 1c9 1c8
:2 0 1f2 37 :2 0
28 :3 0 5e :3 0
1cc 1cd 0 1cf
f4 1d1 f6 1d0
1cf :2 0 1f2 66
:2 0 29 :3 0 5e
:3 0 1d3 1d4 0
1d6 f8 1d8 fa
1d7 1d6 :2 0 1f2
67 :2 0 2a :3 0
5e :3 0 1da 1db
0 1dd fc 1df
fe 1de 1dd :2 0
1f2 c :2 0 26
:3 0 5e :3 0 1e1
1e2 0 1e4 100
1e6 102 1e5 1e4
:2 0 1f2 62 :2 0
2b :3 0 5e :3 0
1e8 1e9 0 1eb
104 1ed 106 1ec
1eb :2 0 1f2 68
:2 0 40 1ee 0
1f0 108 1f1 0
1f0 :2 0 1f2 10a
:2 0 1f3 1c3 1f2
0 1f4 0 112
1f6 63 :3 0 1b9
1f4 :4 0 1fb 69
:3 0 5a :4 0 1fa
:2 0 1fb 1f8 0
116 1fe :3 0 1fe
11a 1fe 1fd 1fb
1fc :6 0 1ff 1
0 170 190 1fe
1290 :2 0 6a :a 0
51f 6 :7 0 121
869 0 11f 9
:3 0 6b :7 0 204
203 :3 0 126 886
0 123 9 :3 0
6c :7 0 208 207
:3 0 20a :2 0 51f
201 20b :2 0 218
219 0 128 9
:3 0 20e :7 0 211
20f 0 51d 0
20 :6 0 9 :3 0
213 :7 0 216 214
0 51d 0 61
:6 0 12c 8e7 0
12a 49 :3 0 45
:2 0 4 56 :3 0
56 :2 0 1 21a
21c :3 0 21d :7 0
220 21e 0 51d
0 6d :6 0 130
91b 0 12e 9
:3 0 222 :7 0 225
223 0 51d 0
6e :6 0 9 :3 0
227 :7 0 22a 228
0 51d 0 6f
:6 0 136 953 0
134 9 :3 0 22c
:7 0 22f 22d 0
51d 0 70 :6 0
5 :3 0 22 :2 0
132 231 233 :6 0
236 234 0 51d
0 21 :6 0 13a
987 0 138 24
:3 0 238 :7 0 23b
239 0 51d 0
23 :6 0 24 :3 0
23d :7 0 240 23e
0 51d 0 71
:6 0 13e 9bb 0
13c 9 :3 0 242
:7 0 245 243 0
51d 0 25 :6 0
9 :3 0 247 :7 0
24a 248 0 51d
0 26 :6 0 142
9ef 0 140 9
:3 0 24c :7 0 24f
24d 0 51d 0
27 :6 0 9 :3 0
251 :7 0 254 252
0 51d 0 28
:6 0 146 a23 0
144 9 :3 0 256
:7 0 259 257 0
51d 0 29 :6 0
9 :3 0 25b :7 0
25e 25c 0 51d
0 2a :6 0 14a
a57 0 148 9
:3 0 260 :7 0 263
261 0 51d 0
2b :6 0 9 :3 0
265 :7 0 268 266
0 51d 0 72
:6 0 14e a8b 0
14c 9 :3 0 26a
:7 0 26d 26b 0
51d 0 73 :6 0
9 :3 0 26f :7 0
272 270 0 51d
0 74 :6 0 152
abf 0 150 9
:3 0 274 :7 0 277
275 0 51d 0
75 :6 0 9 :3 0
279 :7 0 27c 27a
0 51d 0 76
:6 0 156 af3 0
154 9 :3 0 27e
:7 0 281 27f 0
51d 0 77 :6 0
9 :3 0 283 :7 0
286 284 0 51d
0 78 :6 0 15a
b27 0 158 9
:3 0 288 :7 0 28b
289 0 51d 0
79 :6 0 9 :3 0
28d :7 0 290 28e
0 51d 0 7a
:6 0 15e b5b 0
15c 9 :3 0 292
:7 0 295 293 0
51d 0 7b :6 0
9 :3 0 297 :7 0
29a 298 0 51d
0 7c :6 0 59
:3 0 9 :3 0 29c
:7 0 29f 29d 0
51d 0 7d :6 0
7e :a 0 7 337
:4 0 162 b9d 0
160 9 :3 0 7f
:7 0 2a4 2a3 :3 0
166 :2 0 164 9
:3 0 80 :7 0 2a8
2a7 :3 0 9 :3 0
81 :7 0 2ac 2ab
:3 0 2a1 2b0 0
2ae :3 0 21 :3 0
45 :3 0 82 :3 0
83 :3 0 83 :3 0
84 :2 0 82 :3 0
85 :3 0 16a 2b6
2b9 :3 0 2ba 2b8
16d 86 :3 0 21
:3 0 2bd 2be 0
86 :3 0 45 :3 0
2c0 2c1 0 87
:3 0 48 :3 0 2c3
2c4 0 86 :3 0
3f :2 0 48 :3 0
2c6 2c8 0 175
2c7 2ca :3 0 87
:3 0 48 :3 0 2cc
2cd 0 13 :2 0
86 :3 0 48 :3 0
2d0 2d1 0 178
2cf 2d3 :3 0 17b
2d5 2d4 87 :3 0
48 :3 0 2d7 2d8
:2 0 2d9 0 17d
:2 0 82 :3 0 2db
2dc 87 :3 0 47
:3 0 2de 2df 0
86 :3 0 3f :2 0
47 :3 0 2e1 2e3
0 182 2e2 2e5
:3 0 87 :3 0 47
:3 0 2e7 2e8 0
13 :2 0 86 :3 0
47 :3 0 2eb 2ec
0 185 2ea 2ee
:3 0 188 2f0 2ef
87 :3 0 47 :3 0
2f2 2f3 :2 0 2f4
0 18a :2 0 83
:3 0 2f6 2f7 18d
21 :3 0 45 :3 0
47 :3 0 48 :3 0
192 3a :3 0 44
:2 0 4 2ff 300
0 197 302 308
0 309 :3 0 43
:3 0 80 :3 0 52
:2 0 19b 306 307
:5 0 2fe 303 0
30a :3 0 86 :3 0
30b 30c 21 :3 0
45 :3 0 47 :3 0
48 :3 0 19e 49
:3 0 1a3 314 :2 0
316 :5 0 312 315
0 317 :3 0 87
:3 0 318 319 1a5
31b 325 0 326
:3 0 86 :3 0 45
:3 0 31d 31e 0
87 :3 0 52 :2 0
45 :3 0 320 322
0 1aa 321 324
:5 0 2f9 31c 0
327 :3 0 1ad 329
32f 0 330 :3 0
82 :3 0 81 :3 0
3f :2 0 1b1 32d
32e :5 0 2bc 32a
0 82 :3 0 2
332 1b4 331 0
334 :4 0 335 :2 0
338 2a1 2b0 339
0 51d 1b6 339
33b 338 33a :6 0
337 1 :6 0 339
2c :3 0 2d :3 0
33c 33d 0 88
:4 0 89 :2 0 6b
:3 0 1b8 340 342
:3 0 1bb 33e 344
:2 0 51b 8a :3 0
6b :3 0 1bd 346
348 :2 0 51b 2f
:3 0 30 :3 0 38
:3 0 32 :3 0 c
:2 0 33 :4 0 34
:3 0 35 :3 0 36
:2 0 37 :2 0 1bf
350 354 1c3 34c
356 :3 0 1c7 34b
358 1c9 20 :3 0
25 :3 0 39 :3 0
1cc 35e :2 0 360
:4 0 362 363 :5 0
35a 35f 0 1ce
0 361 :2 0 51b
3a :3 0 8b :2 0
4 365 366 0
2f :3 0 20 :3 0
52 :2 0 1d3 36a
36b :3 0 367 36c
0 36e :2 0 36d
:2 0 51b 3a :3 0
8c :2 0 4 36f
370 0 2f :3 0
20 :3 0 52 :2 0
1d8 374 375 :3 0
371 376 0 378
:2 0 377 :2 0 51b
3e :3 0 1db 23
:3 0 3d :3 0 1dd
37d :2 0 37f :4 0
381 382 :5 0 37a
37e 0 1df 0
380 :2 0 51b 25
:3 0 3f :2 0 40
:2 0 1e3 385 387
:3 0 41 :3 0 26
:3 0 27 :3 0 28
:3 0 29 :3 0 2a
:3 0 2b :3 0 1e6
389 390 :2 0 47e
23 :3 0 26 :3 0
27 :3 0 28 :3 0
29 :3 0 2a :3 0
2b :3 0 1ed 71
:3 0 72 :3 0 73
:3 0 74 :3 0 75
:3 0 76 :3 0 77
:3 0 3a :3 0 42
:2 0 4 3a1 3a2
0 1f5 3a4 3aa
0 3ab :3 0 43
:3 0 6b :3 0 52
:2 0 1f9 3a8 3a9
:4 0 3ad 3ae :5 0
399 3a5 0 1fc
0 3ac :2 0 47e
26 :3 0 72 :3 0
3f :2 0 206 3b2
3b3 :3 0 3b4 :2 0
78 :3 0 26 :3 0
13 :2 0 72 :3 0
209 3b8 3ba :3 0
3bb :2 0 84 :2 0
8d :2 0 20c 3bd
3bf :3 0 3b6 3c0
0 3c2 20f 3cb
78 :3 0 26 :3 0
84 :2 0 8d :2 0
211 3c5 3c7 :3 0
3c3 3c8 0 3ca
214 3cc 3b5 3c2
0 3cd 0 3ca
0 3cd 216 0
47e 27 :3 0 73
:3 0 3f :2 0 21b
3d0 3d1 :3 0 3d2
:2 0 79 :3 0 27
:3 0 13 :2 0 73
:3 0 21e 3d6 3d8
:3 0 3d9 :2 0 84
:2 0 8d :2 0 221
3db 3dd :3 0 3d4
3de 0 3e0 224
3e9 79 :3 0 27
:3 0 84 :2 0 8d
:2 0 226 3e3 3e5
:3 0 3e1 3e6 0
3e8 229 3ea 3d3
3e0 0 3eb 0
3e8 0 3eb 22b
0 47e 28 :3 0
74 :3 0 3f :2 0
230 3ee 3ef :3 0
3f0 :2 0 7a :3 0
28 :3 0 13 :2 0
74 :3 0 233 3f4
3f6 :3 0 3f7 :2 0
84 :2 0 8d :2 0
236 3f9 3fb :3 0
3f2 3fc 0 3fe
239 407 7a :3 0
28 :3 0 84 :2 0
8d :2 0 23b 401
403 :3 0 3ff 404
0 406 23e 408
3f1 3fe 0 409
0 406 0 409
240 0 47e 29
:3 0 75 :3 0 3f
:2 0 245 40c 40d
:3 0 40e :2 0 7b
:3 0 29 :3 0 13
:2 0 75 :3 0 248
412 414 :3 0 415
:2 0 84 :2 0 8d
:2 0 24b 417 419
:3 0 410 41a 0
41c 24e 425 7b
:3 0 29 :3 0 84
:2 0 8d :2 0 250
41f 421 :3 0 41d
422 0 424 253
426 40f 41c 0
427 0 424 0
427 255 0 47e
2a :3 0 76 :3 0
3f :2 0 25a 42a
42b :3 0 42c :2 0
7c :3 0 2a :3 0
13 :2 0 76 :3 0
25d 430 432 :3 0
433 :2 0 84 :2 0
8d :2 0 260 435
437 :3 0 42e 438
0 43a 263 443
7c :3 0 2a :3 0
84 :2 0 8d :2 0
265 43d 43f :3 0
43b 440 0 442
268 444 42d 43a
0 445 0 442
0 445 26a 0
47e 2b :3 0 77
:3 0 3f :2 0 26f
448 449 :3 0 44a
:2 0 7d :3 0 2b
:3 0 13 :2 0 77
:3 0 272 44e 450
:3 0 451 :2 0 84
:2 0 8d :2 0 275
453 455 :3 0 44c
456 0 458 278
461 7d :3 0 2b
:3 0 84 :2 0 8d
:2 0 27a 45b 45d
:3 0 459 45e 0
460 27d 462 44b
458 0 463 0
460 0 463 27f
0 47e 3a :3 0
8c :2 0 4 464
465 0 2f :3 0
23 :3 0 71 :3 0
26 :3 0 27 :3 0
28 :3 0 29 :3 0
2a :3 0 2b :3 0
20 :3 0 23 :3 0
71 :3 0 78 :3 0
79 :3 0 7a :3 0
7b :3 0 7c :3 0
7d :3 0 282 :3 0
466 47b 47c 47d
:4 0 28c 296 :4 0
47a :2 0 47e 298
4a0 23 :3 0 2a2
71 :3 0 3a :3 0
42 :2 0 4 482
483 0 2a4 485
48b 0 48c :3 0
43 :3 0 6b :3 0
52 :2 0 2a8 489
48a :4 0 48e 48f
:5 0 480 486 0
2ab 0 48d :2 0
49f 3a :3 0 8c
:2 0 4 491 492
0 2f :3 0 23
:3 0 71 :3 0 20
:3 0 23 :3 0 71
:3 0 2ad :3 0 493
49c 49d 49e :4 0
2b1 2b5 :4 0 49b
:2 0 49f 2b7 4a1
388 47e 0 4a2
0 49f 0 4a2
2ba 0 51b 60
:3 0 7e :3 0 20
:3 0 6b :3 0 10
:3 0 2bd 4a4 4a8
0 4a9 :2 0 51b
4a4 4a8 :2 0 61
:3 0 36 :2 0 6c
:3 0 63 :3 0 4ac
4ad :2 0 4ab 4af
7e :3 0 21 :3 0
6d :3 0 6e :3 0
6f :3 0 70 :4 0
4b8 :2 0 4d3 4b1
4b9 :3 0 2c1 :3 0
64 :3 0 7e :3 0
65 :3 0 4bb 4bc
:4 0 4bd :3 0 4d3
3a :3 0 8b :2 0
4 4bf 4c0 0
2f :3 0 21 :3 0
45 :3 0 48 :3 0
47 :3 0 85 :3 0
20 :3 0 21 :3 0
6d :3 0 6e :3 0
6f :3 0 70 :3 0
2c7 :3 0 4c1 4d0
4d1 4d2 :4 0 2ce
2d5 :4 0 4cf :2 0
4d3 2d7 4d5 63
:3 0 4b0 4d3 :4 0
51b 69 :3 0 7e
:4 0 4d9 :2 0 51b
4d7 0 3a :3 0
8b :2 0 4 4da
4db 0 8e :3 0
4dc 4dd 8e :3 0
8f :2 0 4 4df
4e0 0 90 :3 0
8f :3 0 4e2 4e3
0 2db 49 :3 0
90 :3 0 4e6 4e7
2dd 4e9 4f3 0
4f4 :3 0 90 :3 0
45 :3 0 4eb 4ec
0 8e :3 0 52
:2 0 45 :3 0 4ee
4f0 0 2e1 4ef
4f2 :5 0 4e5 4ea
0 4f5 :3 0 4e1
4f6 8e :3 0 2f
:3 0 4f8 4f9 0
20 :3 0 52 :2 0
2e6 4fc 4fd :3 0
4de 500 4fe 0
501 0 2e9 0
4ff :2 0 51b 3a
:3 0 44 :2 0 4
502 503 0 43
:3 0 6b :3 0 52
:2 0 2ed 507 508
:3 0 504 509 0
50b :2 0 50a :2 0
51b 3a :3 0 42
:2 0 4 50c 50d
0 2f :3 0 20
:3 0 52 :2 0 2f2
511 512 :3 0 50e
513 0 515 :2 0
514 :2 0 51b 54
:3 0 518 519 :2 0
51a 54 :5 0 517
:2 0 51b 2f5 51e
:3 0 51e 304 51e
51d 51b 51c :6 0
51f 1 0 201
20b 51e 1290 :2 0
91 :a 0 5cb 9
:7 0 324 :2 0 322
9 :3 0 6d :7 0
524 523 :3 0 526
:2 0 5cb 521 527
:2 0 328 :2 0 326
9 :3 0 52a :7 0
52d 52b 0 5c9
0 20 :6 0 2f
:3 0 20 :3 0 39
:3 0 32a 532 :2 0
534 :4 0 536 537
:5 0 52f 533 0
32c 0 535 :2 0
5c7 3a :3 0 44
:2 0 4 539 53a
0 43 :3 0 43
:3 0 32e 3a :3 0
42 :2 0 4 540
541 0 330 543
549 0 54a :3 0
2f :3 0 20 :3 0
52 :2 0 334 547
548 :5 0 53f 544
0 54b :3 0 53c
53d 54c 45 :3 0
6d :3 0 52 :2 0
339 550 551 :3 0
54d 553 552 :2 0
53b 554 0 556
:2 0 555 :2 0 5c7
3a :3 0 8b :2 0
4 557 558 0
2f :3 0 20 :3 0
52 :2 0 33e 55c
55d :3 0 45 :3 0
6d :3 0 52 :2 0
343 561 562 :3 0
55e 564 563 :2 0
559 565 0 567
:2 0 566 :2 0 5c7
3a :3 0 92 :2 0
4 568 569 0
93 :3 0 93 :3 0
346 3a :3 0 94
:2 0 4 56f 570
0 348 572 578
0 579 :3 0 2f
:3 0 20 :3 0 52
:2 0 34c 576 577
:5 0 56e 573 0
57a :3 0 56b 56c
57b 45 :3 0 6d
:3 0 52 :2 0 351
57f 580 :3 0 57c
582 581 :2 0 56a
583 0 585 :2 0
584 :2 0 5c7 3a
:3 0 95 :2 0 4
586 587 0 93
:3 0 93 :3 0 354
3a :3 0 96 :2 0
4 58d 58e 0
356 590 596 0
597 :3 0 2f :3 0
20 :3 0 52 :2 0
35a 594 595 :5 0
58c 591 0 598
:3 0 589 58a 599
45 :3 0 6d :3 0
52 :2 0 35f 59d
59e :3 0 59a 5a0
59f :2 0 588 5a1
0 5a3 :2 0 5a2
:2 0 5c7 3a :3 0
97 :2 0 4 5a4
5a5 0 93 :3 0
93 :3 0 362 3a
:3 0 98 :2 0 4
5ab 5ac 0 364
5ae 5b4 0 5b5
:3 0 2f :3 0 20
:3 0 52 :2 0 368
5b2 5b3 :5 0 5aa
5af 0 5b6 :3 0
5a7 5a8 5b7 45
:3 0 6d :3 0 52
:2 0 36d 5bb 5bc
:3 0 5b8 5be 5bd
:2 0 5a6 5bf 0
5c1 :2 0 5c0 :2 0
5c7 54 :3 0 5c4
5c5 :2 0 5c6 54
:5 0 5c3 :2 0 5c7
370 5ca :3 0 5ca
378 5ca 5c9 5c7
5c8 :6 0 5cb 1
0 521 527 5ca
1290 :2 0 99 :a 0
682 a :8 0 5ce
:2 0 682 5cd 5cf
:2 0 37c :2 0 37a
9 :3 0 5d2 :7 0
5d5 5d3 0 680
0 20 :6 0 2f
:3 0 20 :3 0 39
:3 0 37e 5da :2 0
5dc :4 0 5de 5df
:5 0 5d7 5db 0
380 0 5dd :2 0
67e 3a :3 0 44
:2 0 4 5e1 5e2
0 43 :3 0 43
:3 0 382 3a :3 0
42 :2 0 4 5e8
5e9 0 384 5eb
5f1 0 5f2 :3 0
2f :3 0 20 :3 0
52 :2 0 388 5ef
5f0 :5 0 5e7 5ec
0 5f3 :3 0 5e4
5e5 5f4 5e3 5f5
0 5f7 :2 0 5f6
:2 0 67e 3a :3 0
42 :2 0 4 5f8
5f9 0 2f :3 0
20 :3 0 52 :2 0
38d 5fd 5fe :3 0
5fa 5ff 0 601
:2 0 600 :2 0 67e
3a :3 0 8c :2 0
4 602 603 0
2f :3 0 20 :3 0
52 :2 0 392 607
608 :3 0 604 609
0 60b :2 0 60a
:2 0 67e 3a :3 0
8b :2 0 4 60c
60d 0 2f :3 0
20 :3 0 52 :2 0
397 611 612 :3 0
60e 613 0 615
:2 0 614 :2 0 67e
3a :3 0 92 :2 0
4 616 617 0
93 :3 0 93 :3 0
39a 3a :3 0 94
:2 0 4 61d 61e
0 39c 620 626
0 627 :3 0 2f
:3 0 20 :3 0 52
:2 0 3a0 624 625
:5 0 61c 621 0
628 :3 0 619 61a
629 618 62a 0
62c :2 0 62b :2 0
67e 3a :3 0 94
:2 0 4 62d 62e
0 2f :3 0 20
:3 0 52 :2 0 3a5
632 633 :3 0 62f
634 0 636 :2 0
635 :2 0 67e 3a
:3 0 95 :2 0 4
637 638 0 93
:3 0 93 :3 0 3a8
3a :3 0 96 :2 0
4 63e 63f 0
3aa 641 647 0
648 :3 0 2f :3 0
20 :3 0 52 :2 0
3ae 645 646 :5 0
63d 642 0 649
:3 0 63a 63b 64a
639 64b 0 64d
:2 0 64c :2 0 67e
3a :3 0 96 :2 0
4 64e 64f 0
2f :3 0 20 :3 0
52 :2 0 3b3 653
654 :3 0 650 655
0 657 :2 0 656
:2 0 67e 3a :3 0
97 :2 0 4 658
659 0 93 :3 0
93 :3 0 3b6 3a
:3 0 98 :2 0 4
65f 660 0 3b8
662 668 0 669
:3 0 2f :3 0 20
:3 0 52 :2 0 3bc
666 667 :5 0 65e
663 0 66a :3 0
65b 65c 66b 65a
66c 0 66e :2 0
66d :2 0 67e 3a
:3 0 98 :2 0 4
66f 670 0 2f
:3 0 20 :3 0 52
:2 0 3c1 674 675
:3 0 671 676 0
678 :2 0 677 :2 0
67e 54 :3 0 67b
67c :2 0 67d 54
:5 0 67a :2 0 67e
3c4 681 :3 0 681
3d1 681 680 67e
67f :6 0 682 1
0 5cd 5cf 681
1290 :2 0 8a :a 0
72c b :7 0 3d5
:2 0 3d3 9 :3 0
6b :7 0 687 686
:3 0 689 :2 0 72c
684 68a :2 0 3d9
19bd 0 3d7 24
:3 0 68d :7 0 690
68e 0 72a 0
9a :6 0 3dd 19f1
0 3db 24 :3 0
692 :7 0 695 693
0 72a 0 9b
:6 0 9 :3 0 697
:7 0 69a 698 0
72a 0 20 :6 0
2f :3 0 9 :3 0
69c :7 0 69f 69d
0 72a 0 25
:6 0 53 :3 0 30
:3 0 38 :3 0 32
:3 0 c :2 0 33
:4 0 34 :3 0 35
:3 0 36 :2 0 37
:2 0 3df 6a7 6ab
3e3 6a3 6ad :3 0
3e7 6a2 6af 3e9
20 :3 0 9b :3 0
25 :3 0 39 :3 0
3ed 6b6 :2 0 6b8
:4 0 6ba 6bb :5 0
6b1 6b7 0 3ef
0 6b9 :2 0 728
53 :3 0 3f3 9a
:3 0 3a :3 0 51
:2 0 4 6c0 6c1
0 3f5 6c3 6c9
0 6ca :3 0 2f
:3 0 20 :3 0 52
:2 0 3f9 6c7 6c8
:4 0 6cc 6cd :5 0
6be 6c4 0 3fc
0 6cb :2 0 728
9b :3 0 9a :3 0
9c :2 0 400 6d1
6d2 :3 0 3a :3 0
44 :2 0 4 6d4
6d5 0 47 :3 0
9d :2 0 6d7 6d8
48 :3 0 9d :2 0
6da 6db 43 :3 0
6b :3 0 52 :2 0
405 6df 6e0 :3 0
6d6 6e3 6e1 0
6e4 0 408 0
6e2 :2 0 720 25
:3 0 3f :2 0 40
:2 0 40d 6e6 6e8
:3 0 3a :3 0 42
:2 0 4 6ea 6eb
0 26 :3 0 9d
:2 0 6ed 6ee 27
:3 0 9d :2 0 6f0
6f1 28 :3 0 9d
:2 0 6f3 6f4 29
:3 0 9d :2 0 6f6
6f7 2a :3 0 9d
:2 0 6f9 6fa 2b
:3 0 9d :2 0 6fc
6fd 43 :3 0 6b
:3 0 52 :2 0 412
701 702 :3 0 6ec
705 703 0 706
0 415 0 704
:2 0 707 41c 708
6e9 707 0 709
41e 0 720 3a
:3 0 51 :2 0 4
70a 70b 0 2f
:3 0 20 :3 0 52
:2 0 422 70f 710
:3 0 70c 711 0
713 :2 0 712 :2 0
720 3a :3 0 51
:2 0 4 714 715
0 2f :3 0 53
:3 0 20 :3 0 9b
:3 0 425 :3 0 716
71d 71e 71f :4 0
428 42b :4 0 71c
:2 0 720 42d 721
6d3 720 0 722
432 0 728 54
:3 0 725 726 :2 0
727 54 :5 0 724
:2 0 728 434 72b
:3 0 72b 439 72b
72a 728 729 :6 0
72c 1 0 684
68a 72b 1290 :2 0
9e :a 0 7df c
:8 0 72f :2 0 7df
72e 730 :2 0 440
1c2f 0 43e 24
:3 0 733 :7 0 736
734 0 7dd 0
9a :6 0 444 1c63
0 442 24 :3 0
738 :7 0 73b 739
0 7dd 0 9b
:6 0 9 :3 0 73d
:7 0 740 73e 0
7dd 0 20 :6 0
2f :3 0 9 :3 0
742 :7 0 745 743
0 7dd 0 25
:6 0 53 :3 0 30
:3 0 38 :3 0 32
:3 0 c :2 0 33
:4 0 34 :3 0 35
:3 0 36 :2 0 37
:2 0 446 74d 751
44a 749 753 :3 0
44e 748 755 450
20 :3 0 9b :3 0
25 :3 0 39 :3 0
454 75c :2 0 75e
:4 0 760 761 :5 0
757 75d 0 456
0 75f :2 0 7db
53 :3 0 45a 9a
:3 0 3a :3 0 51
:2 0 4 766 767
0 45c 769 76f
0 770 :3 0 2f
:3 0 20 :3 0 52
:2 0 460 76d 76e
:4 0 772 773 :5 0
764 76a 0 463
0 771 :2 0 7db
9b :3 0 9a :3 0
9c :2 0 467 777
778 :3 0 3a :3 0
92 :2 0 4 77a
77b 0 47 :3 0
9d :2 0 77d 77e
48 :3 0 9d :2 0
780 781 93 :3 0
93 :3 0 46a 3a
:3 0 94 :2 0 4
787 788 0 46c
78a 790 0 791
:3 0 2f :3 0 20
:3 0 52 :2 0 470
78e 78f :5 0 786
78b 0 792 :3 0
783 784 793 77c
796 794 0 797
0 473 0 795
:2 0 7d3 25 :3 0
3f :2 0 40 :2 0
478 799 79b :3 0
3a :3 0 94 :2 0
4 79d 79e 0
26 :3 0 9d :2 0
7a0 7a1 27 :3 0
9d :2 0 7a3 7a4
28 :3 0 9d :2 0
7a6 7a7 29 :3 0
9d :2 0 7a9 7aa
2a :3 0 9d :2 0
7ac 7ad 2b :3 0
9d :2 0 7af 7b0
2f :3 0 20 :3 0
52 :2 0 47d 7b4
7b5 :3 0 79f 7b8
7b6 0 7b9 0
480 0 7b7 :2 0
7ba 487 7bb 79c
7ba 0 7bc 489
0 7d3 3a :3 0
51 :2 0 4 7bd
7be 0 2f :3 0
20 :3 0 52 :2 0
48d 7c2 7c3 :3 0
7bf 7c4 0 7c6
:2 0 7c5 :2 0 7d3
3a :3 0 51 :2 0
4 7c7 7c8 0
2f :3 0 53 :3 0
20 :3 0 9b :3 0
490 :3 0 7c9 7d0
7d1 7d2 :4 0 493
496 :4 0 7cf :2 0
7d3 498 7d4 779
7d3 0 7d5 49d
0 7db 54 :3 0
7d8 7d9 :2 0 7da
54 :5 0 7d7 :2 0
7db 49f 7de :3 0
7de 4a4 7de 7dd
7db 7dc :6 0 7df
1 0 72e 730
7de 1290 :2 0 9f
:a 0 95f d :8 0
7e2 :2 0 95f 7e1
7e3 :2 0 59 :3 0
a0 :a 0 e 807
:4 0 4ab :2 0 4a9
9 :3 0 20 :7 0
7e9 7e8 :3 0 7e6
7ed 0 7eb :3 0
8e :3 0 21 :3 0
7ee 7ef 0 8e
:3 0 45 :3 0 7f1
7f2 0 4ad 3a
:3 0 8b :2 0 4
7f5 7f6 0 8e
:3 0 7f7 7f8 4b0
7fa 802 0 803
:3 0 8e :3 0 2f
:3 0 7fc 7fd 0
20 :3 0 52 :2 0
4b4 800 801 :5 0
7f4 7fb 0 804
:6 0 805 :2 0 808
7e6 7ed 809 0
95d 4b7 809 80b
808 80a :6 0 807
1 :6 0 809 59
:3 0 5a :a 0 f
826 :4 0 4bb :2 0
4b9 9 :3 0 a1
:7 0 810 80f :3 0
80d 814 0 812
:4 0 49 :3 0 90
:3 0 816 817 4bd
819 821 0 822
:3 0 90 :3 0 45
:3 0 81b 81c 0
a1 :3 0 52 :2 0
4c1 81f 820 :5 0
815 81a 0 823
:6 0 824 :2 0 827
80d 814 828 0
95d 4c4 828 82a
827 829 :6 0 826
1 :6 0 828 4c8
1ff9 0 4c6 49
:3 0 a3 :3 0 82c
82d :3 0 82e :7 0
831 82f 0 95d
0 a2 :6 0 4cc
202d 0 4ca 9
:3 0 833 :7 0 836
834 0 95d 0
20 :6 0 9 :3 0
838 :7 0 83b 839
0 95d 0 93
:6 0 4d0 2061 0
4ce 24 :3 0 83d
:7 0 840 83e 0
95d 0 23 :6 0
9 :3 0 842 :7 0
845 843 0 95d
0 25 :6 0 4d4
2095 0 4d2 9
:3 0 847 :7 0 84a
848 0 95d 0
26 :6 0 9 :3 0
84c :7 0 84f 84d
0 95d 0 27
:6 0 4d8 20c9 0
4d6 9 :3 0 851
:7 0 854 852 0
95d 0 28 :6 0
9 :3 0 856 :7 0
859 857 0 95d
0 29 :6 0 22
:2 0 4da 9 :3 0
85b :7 0 85e 85c
0 95d 0 2a
:6 0 9 :3 0 860
:7 0 863 861 0
95d 0 2b :6 0
86b 86c 0 4de
5 :3 0 4dc 865
867 :6 0 86a 868
0 95d 0 21
:6 0 2c :3 0 2d
:3 0 a4 :4 0 4e0
86d 86f :2 0 95b
9e :4 0 871 872
:2 0 95b 2f :3 0
30 :3 0 38 :3 0
32 :3 0 c :2 0
33 :4 0 34 :3 0
35 :3 0 36 :2 0
37 :2 0 4e2 87a
87e 4e6 876 880
:3 0 4ea 875 882
4ec 20 :3 0 25
:3 0 39 :3 0 4ef
888 :2 0 88a :4 0
88c 88d :5 0 884
889 0 4f1 0
88b :2 0 95b 3a
:3 0 a5 :3 0 88f
890 0 3c :3 0
891 892 0 4f4
93 :3 0 3d :3 0
4f6 897 :2 0 899
:4 0 89b 89c :5 0
894 898 0 4f8
0 89a :2 0 95b
3e :3 0 4fa 23
:3 0 3d :3 0 4fc
8a2 :2 0 8a4 :4 0
8a6 8a7 :5 0 89f
8a3 0 4fe 0
8a5 :2 0 95b a6
:3 0 a0 :3 0 20
:3 0 500 8aa 8ac
63 :3 0 8a9 8ad
a6 :3 0 45 :3 0
8b0 8b1 0 a7
:2 0 502 8b3 8b4
:3 0 a2 :3 0 5a
:3 0 a6 :3 0 45
:3 0 8b8 8b9 0
504 8b7 8bb 63
:3 0 8b6 8bc 21
:3 0 506 21 :3 0
3a :3 0 8b :2 0
4 8c2 8c3 0
508 8c5 8d4 0
8d5 :3 0 45 :3 0
a2 :3 0 52 :2 0
45 :3 0 8c8 8ca
0 50c 8c9 8cc
:3 0 2f :3 0 20
:3 0 52 :2 0 511
8d0 8d1 :3 0 8cd
8d3 8d2 :3 0 8d7
8d8 :5 0 8c0 8c6
0 514 0 8d6
:2 0 8f2 3a :3 0
a8 :2 0 4 8da
8db 0 21 :3 0
45 :3 0 47 :3 0
48 :3 0 a6 :3 0
21 :3 0 8e1 8e2
0 a2 :3 0 45
:3 0 8e4 8e5 0
a2 :3 0 47 :3 0
8e7 8e8 0 a2
:3 0 48 :3 0 8ea
8eb 0 516 :3 0
8dc 8ef 8f0 8f1
:4 0 51b 520 :4 0
8ee :2 0 8f2 522
8f4 63 :3 0 8be
8f2 :4 0 8f5 525
8f6 8b5 8f5 0
8f7 527 0 8f8
529 8fa 63 :3 0
8af 8f8 :4 0 95b
25 :3 0 3f :2 0
40 :2 0 52d 8fc
8fe :3 0 41 :3 0
26 :3 0 27 :3 0
28 :3 0 29 :3 0
2a :3 0 2b :3 0
530 900 907 :2 0
923 3a :3 0 94
:2 0 4 909 90a
0 2f :3 0 93
:3 0 23 :3 0 26
:3 0 27 :3 0 28
:3 0 29 :3 0 2a
:3 0 2b :3 0 20
:3 0 93 :3 0 23
:3 0 26 :3 0 27
:3 0 28 :3 0 29
:3 0 2a :3 0 2b
:3 0 537 :3 0 90b
920 921 922 :4 0
541 54b :4 0 91f
:2 0 923 54d 933
3a :3 0 94 :2 0
4 924 925 0
2f :3 0 93 :3 0
23 :3 0 20 :3 0
93 :3 0 23 :3 0
550 :3 0 926 92f
930 931 :4 0 554
558 :4 0 92e :2 0
932 55a 934 8ff
923 0 935 0
932 0 935 55c
0 95b 3a :3 0
92 :2 0 4 936
937 0 93 :3 0
21 :3 0 45 :3 0
47 :3 0 48 :3 0
93 :3 0 21 :3 0
45 :3 0 a9 :3 0
a9 :2 0 47 :3 0
942 0 943 0
a9 :3 0 a9 :2 0
48 :3 0 946 0
947 0 55f 3a
:3 0 a8 :2 0 4
94a 94b 0 565
94d :2 0 951 :3 0
45 :3 0 21 :3 0
567 0 949 94e
0 938 954 952
955 :4 0 56a 0
953 :2 0 95b 54
:3 0 958 959 :2 0
95a 54 :5 0 957
:2 0 95b 570 95e
:3 0 95e 57a 95e
95d 95b 95c :6 0
95f 1 0 7e1
7e3 95e 1290 :2 0
aa :a 0 a04 12
:8 0 962 :2 0 a04
961 963 :2 0 59
:3 0 ab :a 0 13
981 :4 0 58b :2 0
589 9 :3 0 7f
:7 0 969 968 :3 0
966 96d 0 96b
:3 0 93 :3 0 58d
3a :3 0 94 :2 0
4 970 971 0
58f 973 979 0
97a :3 0 2f :3 0
7f :3 0 52 :2 0
593 977 978 :5 0
96f 974 0 93
:3 0 2 97c 596
97b 0 97e :4 0
97f :2 0 982 966
96d 983 0 a02
598 983 985 982
984 :6 0 981 1
:6 0 983 59c 2554
0 59a 9 :3 0
987 :7 0 98a 988
0 a02 0 ac
:6 0 5a0 :2 0 59e
9 :3 0 98c :7 0
98f 98d 0 a02
0 ad :6 0 9
:3 0 991 :7 0 994
992 0 a02 0
20 :6 0 2f :3 0
20 :3 0 39 :3 0
5a2 999 :2 0 99b
:4 0 99d 99e :5 0
996 99a 0 5a4
0 99c :2 0 a00
60 :3 0 ab :3 0
20 :3 0 5a6 9a1
9a3 0 9a4 :2 0
a00 9a1 9a3 :2 0
ab :3 0 ac :4 0
9a9 :2 0 a00 9a6
9a7 :3 0 ab :3 0
65 :3 0 9aa 9ab
:3 0 9ac :2 0 69
:3 0 ab :4 0 9b1
:2 0 9b4 9af 0
1d :6 0 9b4 5a8
9b5 9ad 9b4 0
9b6 5ab 0 a00
ab :3 0 ad :4 0
9ba :2 0 a00 9b7
9b8 :3 0 ab :3 0
65 :3 0 9bb 9bc
:3 0 9bd :2 0 69
:3 0 ab :4 0 9c2
:2 0 9c5 9c0 0
1d :6 0 9c5 5ad
9c6 9be 9c5 0
9c7 5b0 0 a00
69 :3 0 ab :4 0
9cb :2 0 a00 9c9
0 3a :3 0 92
:2 0 4 9cc 9cd
0 93 :3 0 93
:3 0 5b2 3a :3 0
94 :2 0 4 9d3
9d4 0 5b4 9d6
9e3 0 9e4 :3 0
2f :3 0 20 :3 0
52 :2 0 5b8 9da
9db :3 0 93 :3 0
ad :3 0 ae :2 0
5bd 9df 9e0 :3 0
9dc 9e2 9e1 :4 0
9d2 9d7 0 9e5
:3 0 9cf 9d0 9e6
9ce 9e7 0 9e9
:2 0 9e8 :2 0 a00
3a :3 0 94 :2 0
4 9ea 9eb 0
2f :3 0 20 :3 0
52 :2 0 5c2 9ef
9f0 :3 0 93 :3 0
ad :3 0 ae :2 0
5c7 9f4 9f5 :3 0
9f1 9f7 9f6 :2 0
9ec 9f8 0 9fa
:2 0 9f9 :2 0 a00
54 :3 0 9fd 9fe
:2 0 9ff 54 :5 0
9fc :2 0 a00 5ca
a03 :3 0 a03 5d5
a03 a02 a00 a01
:6 0 a04 1 0
961 963 a03 1290
:2 0 af :a 0 e1b
14 :7 0 5dc :2 0
5da 55 :3 0 9
:3 0 b0 :6 0 a0a
a09 :3 0 a0c :2 0
e1b a06 a0d :2 0
59 :3 0 ab :a 0
15 a32 :4 0 5e0
:2 0 5de 9 :3 0
7f :7 0 a13 a12
:3 0 a10 a17 0
a15 :3 0 93 :3 0
23 :3 0 26 :3 0
27 :3 0 28 :3 0
29 :3 0 2a :3 0
2b :3 0 5e2 3a
:3 0 94 :2 0 4
a21 a22 0 5eb
a24 a2a 0 a2b
:3 0 2f :3 0 7f
:3 0 52 :2 0 5ef
a28 a29 :5 0 a20
a25 0 93 :3 0
2 a2d 5f2 a2c
0 a2f :4 0 a30
:2 0 a33 a10 a17
a34 0 e19 5f4
a34 a36 a33 a35
:6 0 a32 1 :6 0
a34 59 :3 0 b1
:a 0 16 aef :4 0
5f8 2829 0 5f6
9 :3 0 80 :7 0
a3b a3a :3 0 5fc
:2 0 5fa 9 :3 0
b2 :7 0 a3f a3e
:3 0 9 :3 0 20
:7 0 a43 a42 :3 0
a38 a47 0 a45
:3 0 b3 :3 0 45
:3 0 a48 a49 0
b3 :3 0 21 :3 0
a4b a4c 0 b3
:3 0 47 :3 0 a4e
a4f 0 b4 :3 0
3f :2 0 47 :3 0
a51 a53 0 602
a52 a55 :3 0 b3
:3 0 47 :3 0 a57
a58 0 13 :2 0
b4 :3 0 47 :3 0
a5b a5c 0 605
a5a a5e :3 0 608
a60 a5f b3 :3 0
47 :3 0 a62 a63
:2 0 a64 0 60a
:2 0 b3 :3 0 48
:3 0 a67 a68 0
b4 :3 0 3f :2 0
48 :3 0 a6a a6c
0 60f a6b a6e
:3 0 b3 :3 0 48
:3 0 a70 a71 0
13 :2 0 b4 :3 0
48 :3 0 a74 a75
0 612 a73 a77
:3 0 615 a79 a78
b3 :3 0 48 :3 0
a7b a7c :2 0 a7d
0 617 :2 0 b5
:3 0 48 :3 0 a80
a81 0 b5 :3 0
47 :3 0 a83 a84
0 61a 0 3a
:3 0 92 :2 0 4
a88 a89 0 621
a8b a91 0 a92
:3 0 93 :3 0 80
:3 0 52 :2 0 625
a8f a90 :5 0 a87
a8c 0 a93 :3 0
b3 :3 0 a94 a95
0 3a :3 0 92
:2 0 4 a98 a99
0 628 a9b aa1
0 aa2 :3 0 93
:3 0 b2 :3 0 52
:2 0 62c a9f aa0
:5 0 a97 a9c 0
aa3 :3 0 b4 :3 0
aa4 aa5 3a :3 0
8b :2 0 4 aa7
aa8 0 b5 :3 0
aa9 aaa 62f aac
aea 0 aeb :3 0
b3 :3 0 45 :3 0
aae aaf 0 b4
:3 0 52 :2 0 45
:3 0 ab1 ab3 0
635 ab2 ab5 :3 0
b3 :3 0 45 :3 0
ab7 ab8 0 b5
:3 0 52 :2 0 45
:3 0 aba abc 0
63a abb abe :3 0
ab6 ac0 abf :2 0
b5 :3 0 2f :3 0
ac2 ac3 0 20
:3 0 52 :2 0 63f
ac6 ac7 :3 0 ac1
ac9 ac8 :2 0 b3
:3 0 48 :3 0 acb
acc 0 b4 :3 0
3f :2 0 48 :3 0
ace ad0 0 644
acf ad2 :3 0 b3
:3 0 48 :3 0 ad4
ad5 0 13 :2 0
b4 :3 0 48 :3 0
ad8 ad9 0 647
ad7 adb :3 0 64a
add adc b3 :3 0
48 :3 0 adf ae0
:2 0 ae1 0 64c
:2 0 ae3 :2 0 10
:3 0 3f :2 0 651
ae6 ae7 :3 0 aca
ae9 ae8 :4 0 a86
aad 0 aec :6 0
aed :2 0 af0 a38
a47 af1 0 e19
654 af1 af3 af0
af2 :6 0 aef 1
:6 0 af1 658 2ab4
0 656 9 :3 0
af5 :7 0 af8 af6
0 e19 0 b6
:6 0 22 :2 0 65a
9 :3 0 afa :7 0
afd afb 0 e19
0 b7 :6 0 49
:3 0 45 :2 0 4
aff b00 0 56
:3 0 56 :2 0 1
b01 b03 :3 0 b04
:7 0 b07 b05 0
e19 0 b8 :6 0
660 2b19 0 65e
5 :3 0 65c b09
b0b :6 0 b0e b0c
0 e19 0 21
:6 0 664 2b4d 0
662 9 :3 0 b10
:7 0 b13 b11 0
e19 0 ac :6 0
9 :3 0 b15 :7 0
b18 b16 0 e19
0 ad :6 0 668
2b81 0 666 24
:3 0 b1a :7 0 b1d
b1b 0 e19 0
b9 :6 0 24 :3 0
b1f :7 0 b22 b20
0 e19 0 ba
:6 0 66c 2bb5 0
66a 9 :3 0 b24
:7 0 b27 b25 0
e19 0 25 :6 0
9 :3 0 b29 :7 0
b2c b2a 0 e19
0 bb :6 0 670
2be9 0 66e 9
:3 0 b2e :7 0 b31
b2f 0 e19 0
bc :6 0 9 :3 0
b33 :7 0 b36 b34
0 e19 0 bd
:6 0 674 2c1d 0
672 9 :3 0 b38
:7 0 b3b b39 0
e19 0 be :6 0
9 :3 0 b3d :7 0
b40 b3e 0 e19
0 bf :6 0 678
2c51 0 676 9
:3 0 b42 :7 0 b45
b43 0 e19 0
c0 :6 0 9 :3 0
b47 :7 0 b4a b48
0 e19 0 c1
:6 0 67c 2c85 0
67a 9 :3 0 b4c
:7 0 b4f b4d 0
e19 0 c2 :6 0
9 :3 0 b51 :7 0
b54 b52 0 e19
0 c3 :6 0 680
2cb9 0 67e 9
:3 0 b56 :7 0 b59
b57 0 e19 0
c4 :6 0 9 :3 0
b5b :7 0 b5e b5c
0 e19 0 c5
:6 0 684 2ced 0
682 9 :3 0 b60
:7 0 b63 b61 0
e19 0 c6 :6 0
9 :3 0 b65 :7 0
b68 b66 0 e19
0 78 :6 0 688
2d21 0 686 9
:3 0 b6a :7 0 b6d
b6b 0 e19 0
79 :6 0 9 :3 0
b6f :7 0 b72 b70
0 e19 0 7a
:6 0 68c 2d55 0
68a 9 :3 0 b74
:7 0 b77 b75 0
e19 0 7b :6 0
9 :3 0 b79 :7 0
b7c b7a 0 e19
0 7c :6 0 690
2d89 0 68e 9
:3 0 b7e :7 0 b81
b7f 0 e19 0
7d :6 0 9 :3 0
b83 :7 0 b86 b84
0 e19 0 c7
:6 0 694 2dbd 0
692 9 :3 0 b88
:7 0 b8b b89 0
e19 0 c8 :6 0
9 :3 0 b8d :7 0
b90 b8e 0 e19
0 85 :6 0 698
2df1 0 696 9
:3 0 b92 :7 0 b95
b93 0 e19 0
c9 :6 0 9 :3 0
b97 :7 0 b9a b98
0 e19 0 ca
:6 0 69c 2e25 0
69a 9 :3 0 b9c
:7 0 b9f b9d 0
e19 0 cb :6 0
9 :3 0 ba1 :7 0
ba4 ba2 0 e19
0 cc :6 0 c
:2 0 69e 9 :3 0
ba6 :7 0 ba9 ba7
0 e19 0 cd
:6 0 9 :3 0 bab
:7 0 bae bac 0
e19 0 20 :6 0
b0 :3 0 12 :3 0
baf bb0 0 e17
2f :3 0 30 :3 0
38 :3 0 32 :3 0
33 :4 0 34 :3 0
35 :3 0 36 :2 0
37 :2 0 6a0 bb8
bbc 6a4 bb4 bbe
:3 0 6a8 bb3 bc0
6aa 20 :3 0 25
:3 0 39 :3 0 6ad
bc6 :2 0 bc8 :4 0
bca bcb :5 0 bc2
bc7 0 6af 0
bc9 :2 0 e17 60
:3 0 ab :3 0 20
:3 0 6b2 bce bd0
0 bd1 :2 0 e17
bce bd0 :2 0 ab
:3 0 ac :3 0 b9
:3 0 bb :3 0 bc
:3 0 bd :3 0 be
:3 0 bf :3 0 c0
:4 0 bdd :2 0 e17
bd3 bde :3 0 6b4
:3 0 ab :3 0 65
:3 0 bdf be0 :3 0
be1 :2 0 69 :3 0
ab :4 0 be6 :2 0
be9 be4 0 1d
:6 0 be9 6bd bea
be2 be9 0 beb
6c0 0 e17 ab
:3 0 ad :3 0 ba
:3 0 c1 :3 0 c2
:3 0 c3 :3 0 c4
:3 0 c5 :3 0 c6
:4 0 bf6 :2 0 e17
bec bf7 :3 0 6c2
:3 0 ab :3 0 65
:3 0 bf8 bf9 :3 0
bfa :2 0 69 :3 0
ab :4 0 bff :2 0
c02 bfd 0 1d
:6 0 c02 6cb c03
bfb c02 0 c04
6ce 0 e17 69
:3 0 ab :4 0 c08
:2 0 e17 c06 0
2c :3 0 2d :3 0
c09 c0a 0 ce
:4 0 89 :2 0 ac
:3 0 6d0 c0d c0f
:3 0 89 :2 0 cf
:4 0 6d3 c11 c13
:3 0 89 :2 0 ad
:3 0 6d6 c15 c17
:3 0 6d9 c0b c19
:2 0 e17 3a :3 0
95 :2 0 4 c1b
c1c 0 93 :3 0
93 :3 0 6db 3a
:3 0 96 :2 0 4
c22 c23 0 6dd
c25 c2b 0 c2c
:3 0 2f :3 0 20
:3 0 52 :2 0 6e1
c29 c2a :5 0 c21
c26 0 c2d :3 0
c1e c1f c2e c1d
c2f 0 c31 :2 0
c30 :2 0 e17 3a
:3 0 96 :2 0 4
c32 c33 0 2f
:3 0 20 :3 0 52
:2 0 6e6 c37 c38
:3 0 c34 c39 0
c3b :2 0 c3a :2 0
e17 c9 :3 0 9d
:2 0 c3c c3d 0
e17 cb :3 0 9d
:2 0 c3f c40 0
e17 cc :3 0 9d
:2 0 c42 c43 0
e17 cd :3 0 9d
:2 0 c45 c46 0
e17 60 :3 0 b1
:3 0 ac :3 0 ad
:3 0 20 :3 0 6e9
c49 c4d 0 c4e
:2 0 e17 c49 c4d
:2 0 63 :3 0 b1
:3 0 b8 :3 0 21
:3 0 c7 :3 0 c8
:3 0 b6 :3 0 b7
:4 0 c59 :2 0 cd8
c51 c5a :3 0 6ed
:3 0 64 :3 0 b1
:3 0 65 :3 0 c5c
c5d :4 0 c5e :3 0
cd8 2c :3 0 2d
:3 0 c60 c61 0
d0 :4 0 89 :2 0
b8 :3 0 6f4 c64
c66 :3 0 89 :2 0
d1 :4 0 6f7 c68
c6a :3 0 89 :2 0
c7 :3 0 6fa c6c
c6e :3 0 89 :2 0
d2 :4 0 6fd c70
c72 :3 0 89 :2 0
c8 :3 0 700 c74
c76 :3 0 89 :2 0
d3 :4 0 703 c78
c7a :3 0 89 :2 0
b6 :3 0 706 c7c
c7e :3 0 89 :2 0
d4 :4 0 709 c80
c82 :3 0 89 :2 0
b7 :3 0 70c c84
c86 :3 0 70f c62
c88 :2 0 cd8 85
:3 0 c7 :3 0 84
:2 0 c8 :3 0 711
c8c c8e :3 0 c8a
c8f 0 cd8 ca
:3 0 85 :3 0 d5
:2 0 b6 :3 0 714
c93 c95 :3 0 c91
c96 0 cd8 2c
:3 0 2d :3 0 c98
c99 0 d6 :4 0
89 :2 0 d7 :3 0
ca :3 0 36 :2 0
717 c9d ca0 71a
c9c ca2 :3 0 71d
c9a ca4 :2 0 cd8
cb :3 0 cb :3 0
d8 :2 0 ca :3 0
71f ca8 caa :3 0
ca6 cab 0 cd8
cc :3 0 cc :3 0
d8 :2 0 b7 :3 0
722 caf cb1 :3 0
cad cb2 0 cd8
cd :3 0 cd :3 0
d8 :2 0 c7 :3 0
725 cb6 cb8 :3 0
cb4 cb9 0 cd8
c9 :3 0 c9 :3 0
d8 :2 0 36 :2 0
728 cbd cbf :3 0
cbb cc0 0 cd8
3a :3 0 95 :2 0
4 cc2 cc3 0
93 :3 0 21 :3 0
45 :3 0 48 :3 0
47 :3 0 85 :3 0
d9 :3 0 ac :3 0
21 :3 0 b8 :3 0
c8 :3 0 c7 :3 0
85 :3 0 ca :3 0
72b :3 0 cc4 cd5
cd6 cd7 :4 0 733
73b :4 0 cd4 :2 0
cd8 73d cda 63
:4 0 cd8 :4 0 e17
69 :3 0 b1 :4 0
cde :2 0 e17 cdc
0 c9 :3 0 4a
:2 0 9d :2 0 74b
ce0 ce2 :3 0 b0
:3 0 cb :3 0 84
:2 0 cc :3 0 74e
ce6 ce8 :3 0 d5
:2 0 8d :2 0 751
cea cec :3 0 ce4
ced 0 cef 754
cf0 ce3 cef 0
cf1 756 0 e17
25 :3 0 3f :2 0
40 :2 0 75a cf3
cf5 :3 0 bb :3 0
c1 :3 0 3f :2 0
75f cf9 cfa :3 0
cfb :2 0 78 :3 0
bb :3 0 13 :2 0
c1 :3 0 762 cff
d01 :3 0 d02 :2 0
84 :2 0 8d :2 0
765 d04 d06 :3 0
cfd d07 0 d09
768 d12 78 :3 0
bb :3 0 84 :2 0
8d :2 0 76a d0c
d0e :3 0 d0a d0f
0 d11 76d d13
cfc d09 0 d14
0 d11 0 d14
76f 0 ddb bc
:3 0 c2 :3 0 3f
:2 0 774 d17 d18
:3 0 d19 :2 0 79
:3 0 bc :3 0 13
:2 0 c2 :3 0 777
d1d d1f :3 0 d20
:2 0 84 :2 0 8d
:2 0 77a d22 d24
:3 0 d1b d25 0
d27 77d d30 79
:3 0 bc :3 0 84
:2 0 8d :2 0 77f
d2a d2c :3 0 d28
d2d 0 d2f 782
d31 d1a d27 0
d32 0 d2f 0
d32 784 0 ddb
bd :3 0 c3 :3 0
3f :2 0 789 d35
d36 :3 0 d37 :2 0
7a :3 0 bd :3 0
13 :2 0 c3 :3 0
78c d3b d3d :3 0
d3e :2 0 84 :2 0
8d :2 0 78f d40
d42 :3 0 d39 d43
0 d45 792 d4e
7a :3 0 bd :3 0
84 :2 0 8d :2 0
794 d48 d4a :3 0
d46 d4b 0 d4d
797 d4f d38 d45
0 d50 0 d4d
0 d50 799 0
ddb be :3 0 c4
:3 0 3f :2 0 79e
d53 d54 :3 0 d55
:2 0 7b :3 0 be
:3 0 13 :2 0 c4
:3 0 7a1 d59 d5b
:3 0 d5c :2 0 84
:2 0 8d :2 0 7a4
d5e d60 :3 0 d57
d61 0 d63 7a7
d6c 7b :3 0 be
:3 0 84 :2 0 8d
:2 0 7a9 d66 d68
:3 0 d64 d69 0
d6b 7ac d6d d56
d63 0 d6e 0
d6b 0 d6e 7ae
0 ddb bf :3 0
c5 :3 0 3f :2 0
7b3 d71 d72 :3 0
d73 :2 0 7c :3 0
bf :3 0 13 :2 0
c5 :3 0 7b6 d77
d79 :3 0 d7a :2 0
84 :2 0 8d :2 0
7b9 d7c d7e :3 0
d75 d7f 0 d81
7bc d8a 7c :3 0
bf :3 0 84 :2 0
8d :2 0 7be d84
d86 :3 0 d82 d87
0 d89 7c1 d8b
d74 d81 0 d8c
0 d89 0 d8c
7c3 0 ddb c0
:3 0 c6 :3 0 3f
:2 0 7c8 d8f d90
:3 0 d91 :2 0 7d
:3 0 c0 :3 0 13
:2 0 c6 :3 0 7cb
d95 d97 :3 0 d98
:2 0 84 :2 0 8d
:2 0 7ce d9a d9c
:3 0 d93 d9d 0
d9f 7d1 da8 7d
:3 0 c0 :3 0 84
:2 0 8d :2 0 7d3
da2 da4 :3 0 da0
da5 0 da7 7d6
da9 d92 d9f 0
daa 0 da7 0
daa 7d8 0 ddb
3a :3 0 96 :2 0
4 dab dac 0
2f :3 0 93 :3 0
23 :3 0 71 :3 0
da :3 0 db :3 0
dc :3 0 dd :3 0
26 :3 0 27 :3 0
28 :3 0 29 :3 0
2a :3 0 2b :3 0
20 :3 0 ac :3 0
b9 :3 0 ba :3 0
cd :3 0 84 :2 0
de :2 0 7db dc1
dc3 :3 0 cb :3 0
84 :2 0 de :2 0
7de dc6 dc8 :3 0
cc :3 0 84 :2 0
de :2 0 7e1 dcb
dcd :3 0 b0 :3 0
78 :3 0 79 :3 0
7a :3 0 7b :3 0
7c :3 0 7d :3 0
7e4 :3 0 dad dd8
dd9 dda :4 0 7f3
802 :4 0 dd7 :2 0
ddb 804 e01 3a
:3 0 96 :2 0 4
ddc ddd 0 2f
:3 0 93 :3 0 23
:3 0 71 :3 0 da
:3 0 db :3 0 dc
:3 0 dd :3 0 20
:3 0 ac :3 0 b9
:3 0 ba :3 0 cd
:3 0 84 :2 0 de
:2 0 80c dec dee
:3 0 cb :3 0 84
:2 0 de :2 0 80f
df1 df3 :3 0 cc
:3 0 84 :2 0 de
:2 0 812 df6 df8
:3 0 b0 :3 0 815
:3 0 dde dfd dfe
dff :4 0 81e 827
:4 0 dfc :2 0 e00
829 e02 cf6 ddb
0 e03 0 e00
0 e03 82b 0
e17 54 :3 0 e06
e07 :2 0 e08 54
:5 0 e05 :2 0 e17
b0 :3 0 e :3 0
4a :2 0 830 e0b
e0c :3 0 df :3 0
e0e e10 :2 0 e14
0 e0 :3 0 e11
e13 :2 0 e14 0
833 e15 e0d e14
0 e16 836 0
e17 838 e1a :3 0
e1a 84f e1a e19
e17 e18 :6 0 e1b
1 0 a06 a0d
e1a 1290 :2 0 df
:a 0 e91 18 :8 0
e1e :2 0 e91 e1d
e1f :2 0 878 372b
0 876 9 :3 0
e22 :7 0 e25 e23
0 e8f 0 20
:6 0 2c :3 0 9
:3 0 e27 :7 0 e2a
e28 0 e8f 0
e1 :6 0 2d :3 0
e2b e2c 0 e2
:4 0 87a e2d e2f
:2 0 e8d 2f :3 0
87c 20 :3 0 39
:3 0 87e e35 :2 0
e37 :4 0 e39 e3a
:5 0 e32 e36 0
880 0 e38 :2 0
e8d 93 :3 0 882
e1 :3 0 3a :3 0
96 :2 0 4 e3f
e40 0 884 e42
e48 0 e49 :3 0
2f :3 0 20 :3 0
52 :2 0 888 e46
e47 :4 0 e4b e4c
:5 0 e3d e43 0
88b 0 e4a :2 0
e8d 3a :3 0 98
:2 0 4 e4e e4f
0 93 :3 0 e1
:3 0 52 :2 0 88f
e53 e54 :3 0 e50
e55 0 e57 :2 0
e56 :2 0 e8d 3a
:3 0 97 :2 0 4
e58 e59 0 93
:3 0 e1 :3 0 52
:2 0 894 e5d e5e
:3 0 e5a e5f 0
e61 :2 0 e60 :2 0
e8d 3a :3 0 98
:2 0 4 e62 e63
:2 0 3a :3 0 96
:2 0 4 e66 e67
0 897 e69 e6f
0 e70 :3 0 93
:3 0 e1 :3 0 52
:2 0 89b e6d e6e
:5 0 e65 e6a 0
e64 e73 e71 e74
:6 0 e72 :2 0 e8d
3a :3 0 97 :2 0
4 e75 e76 :2 0
3a :3 0 95 :2 0
4 e79 e7a 0
89e e7c e82 0
e83 :3 0 93 :3 0
e1 :3 0 52 :2 0
8a2 e80 e81 :5 0
e78 e7d 0 e77
e86 e84 e87 :6 0
e85 :2 0 e8d 54
:3 0 e8a e8b :2 0
e8c 54 :5 0 e89
:2 0 e8d 8a5 e90
:3 0 e90 8ae e90
e8f e8d e8e :6 0
e91 1 0 e1d
e1f e90 1290 :2 0
e0 :a 0 f65 19
:8 0 e94 :2 0 f65
e93 e95 :2 0 59
:3 0 ab :a 0 1a
eb3 :4 0 8b3 :2 0
8b1 9 :3 0 7f
:7 0 e9b e9a :3 0
e98 e9f 0 e9d
:3 0 93 :3 0 8b5
3a :3 0 98 :2 0
4 ea2 ea3 0
8b7 ea5 eab 0
eac :3 0 2f :3 0
7f :3 0 52 :2 0
8bb ea9 eaa :5 0
ea1 ea6 0 93
:3 0 2 eae 8be
ead 0 eb0 :4 0
eb1 :2 0 eb4 e98
e9f eb5 0 f63
8c0 eb5 eb7 eb4
eb6 :6 0 eb3 1
:6 0 eb5 8c4 397f
0 8c2 9 :3 0
eb9 :7 0 ebc eba
0 f63 0 e1
:6 0 8c8 39b3 0
8c6 9 :3 0 ebe
:7 0 ec1 ebf 0
f63 0 e3 :6 0
9 :3 0 ec3 :7 0
ec6 ec4 0 f63
0 20 :6 0 2c
:3 0 9 :3 0 ec8
:7 0 ecb ec9 0
f63 0 c9 :6 0
2d :3 0 ecc ecd
0 e4 :4 0 8ca
ece ed0 :2 0 f61
2f :3 0 8cc 20
:3 0 39 :3 0 8ce
ed6 :2 0 ed8 :4 0
eda edb :5 0 ed3
ed7 0 8d0 0
ed9 :2 0 f61 c9
:3 0 9d :2 0 edd
ede 0 f61 e1
:3 0 13 :2 0 36
:2 0 8d2 ee1 ee3
:3 0 ee0 ee4 0
f61 60 :3 0 ab
:3 0 20 :3 0 8d4
ee7 ee9 0 eea
:2 0 f61 ee7 ee9
:2 0 63 :3 0 2c
:3 0 2d :3 0 eed
eee 0 e5 :4 0
89 :2 0 c9 :3 0
8d6 ef1 ef3 :3 0
8d9 eef ef5 :2 0
f14 ab :3 0 e3
:4 0 efa :2 0 f14
ef7 ef8 :3 0 64
:3 0 ab :3 0 65
:3 0 efc efd :4 0
efe :3 0 f14 c9
:3 0 d :3 0 52
:2 0 8dd f02 f03
:3 0 e1 :3 0 e3
:3 0 f05 f06 0
f0a 64 :8 0 f0a
8e0 f0b f04 f0a
0 f0c 8e3 0
f14 c9 :3 0 c9
:3 0 d8 :2 0 36
:2 0 8e5 f0f f11
:3 0 f0d f12 0
f14 8e8 f16 63
:4 0 f14 :4 0 f61
69 :3 0 ab :4 0
f1a :2 0 f61 f18
0 2c :3 0 2d
:3 0 f1b f1c 0
e6 :4 0 89 :2 0
e1 :3 0 8ee f1f
f21 :3 0 8f1 f1d
f23 :2 0 f61 e1
:3 0 3f :2 0 9d
:2 0 8f5 f26 f28
:3 0 3a :3 0 97
:2 0 4 f2a f2b
0 93 :3 0 93
:3 0 8f8 3a :3 0
98 :2 0 4 f31
f32 0 8fa f34
f41 0 f42 :3 0
2f :3 0 20 :3 0
52 :2 0 8fe f38
f39 :3 0 93 :3 0
e1 :3 0 e7 :2 0
903 f3d f3e :3 0
f3a f40 f3f :4 0
f30 f35 0 f43
:3 0 f2d f2e f44
f2c f45 0 f47
:2 0 f46 :2 0 f5e
3a :3 0 98 :2 0
4 f48 f49 0
2f :3 0 20 :3 0
52 :2 0 908 f4d
f4e :3 0 93 :3 0
e1 :3 0 e7 :2 0
90d f52 f53 :3 0
f4f f55 f54 :2 0
f4a f56 0 f58
:2 0 f57 :2 0 f5e
54 :3 0 f5b f5c
:2 0 f5d 54 :5 0
f5a :2 0 f5e 910
f5f f29 f5e 0
f60 914 0 f61
916 f64 :3 0 f64
920 f64 f63 f61
f62 :6 0 f65 1
0 e93 e95 f64
1290 :2 0 e8 :a 0
113f 1c :7 0 928
:2 0 926 55 :3 0
ea :3 0 e9 :6 0
f6b f6a :3 0 f6d
:2 0 113f f67 f6e
:2 0 92c 3c4b 0
92a 9 :3 0 f71
:7 0 f74 f72 0
113d 0 20 :6 0
930 3c7f 0 92e
9 :3 0 f76 :7 0
f79 f77 0 113d
0 b0 :6 0 9
:3 0 f7b :7 0 f7e
f7c 0 113d 0
eb :6 0 934 3cb3
0 932 9 :3 0
f80 :7 0 f83 f81
0 113d 0 ec
:6 0 9 :3 0 f85
:7 0 f88 f86 0
113d 0 6b :6 0
938 3ce7 0 936
9 :3 0 f8a :7 0
f8d f8b 0 113d
0 ed :6 0 24
:3 0 f8f :7 0 f92
f90 0 113d 0
ee :6 0 93c 3d30
0 93a 24 :3 0
f94 :7 0 f97 f95
0 113d 0 ef
:6 0 9 :3 0 f99
:7 0 f9c f9a 0
113d 0 f0 :6 0
59 :3 0 ab :a 0
1d fb9 :4 0 f9e
fa5 0 93e 9
:3 0 7f :7 0 fa1
fa0 :3 0 fa3 :3 0
43 :3 0 940 3a
:3 0 42 :2 0 4
fa8 fa9 0 942
fab fb1 0 fb2
:3 0 2f :3 0 7f
:3 0 52 :2 0 946
faf fb0 :5 0 fa7
fac 0 43 :3 0
2 fb4 949 fb3
0 fb6 :4 0 fb7
:2 0 fba f9e fa5
fbb 0 113d 94b
fbb fbd fba fbc
:6 0 fb9 1 :6 0
fbb 2f :3 0 94d
20 :3 0 39 :3 0
94f fc2 :2 0 fc4
:4 0 fc6 fc7 :5 0
fbf fc3 0 951
0 fc5 :2 0 113b
eb :3 0 9d :2 0
fc9 fca 0 113b
36 :2 0 953 eb
:3 0 3d :3 0 955
fd0 fe2 0 fe3
:4 0 3a :3 0 8c
:2 0 4 fd3 fd4
0 957 fd6 fdc
0 fdd :3 0 2f
:3 0 20 :3 0 52
:2 0 95b fda fdb
:5 0 fd2 fd7 0
fde :3 0 f1 :2 0
95e fe0 fe1 :4 0
fe5 fe6 :5 0 fcd
fd1 0 960 0
fe4 :2 0 fe8 962
ff2 f2 :3 0 eb
:3 0 9d :2 0 fea
feb 0 fed 964
fef 966 fee fed
:2 0 ff0 968 :2 0
ff2 0 ff2 ff1
fe8 ff0 :6 0 113b
1c :3 0 eb :3 0
52 :2 0 9d :2 0
96c ff5 ff7 :3 0
2c :3 0 2d :3 0
ff9 ffa 0 f3
:4 0 96f ffb ffd
:2 0 1096 60 :3 0
ab :3 0 20 :3 0
971 1000 1002 0
1003 :2 0 1096 1000
1002 :2 0 ab :3 0
6b :4 0 1008 :2 0
1096 1005 1006 :3 0
ab :3 0 65 :3 0
1009 100a :3 0 100b
:2 0 2c :3 0 2d
:3 0 100d 100e 0
f4 :4 0 973 100f
1011 :2 0 101e 1e
:3 0 b :3 0 975
1013 1015 :2 0 101e
69 :3 0 ab :4 0
101a :2 0 101e 1018
0 b0 :3 0 19
:3 0 101b 101c 0
101e 977 1093 69
:3 0 ab :4 0 1022
:2 0 1092 1020 0
3e :3 0 97c ee
:3 0 3d :3 0 97e
1027 :2 0 1029 :4 0
102b 102c :5 0 1024
1028 0 980 0
102a :2 0 1092 23
:3 0 982 ef :3 0
3a :3 0 42 :2 0
4 1031 1032 0
984 1034 103a 0
103b :3 0 2f :3 0
20 :3 0 52 :2 0
988 1038 1039 :4 0
103d 103e :5 0 102f
1035 0 98b 0
103c :2 0 1092 f0
:3 0 f5 :3 0 ef
:3 0 ee :3 0 98d
1041 1044 1040 1045
0 1092 f0 :3 0
3f :2 0 f6 :2 0
992 1048 104a :3 0
2c :3 0 2d :3 0
104c 104d 0 f7
:4 0 995 104e 1050
:2 0 108a 6a :3 0
6b :3 0 8 :3 0
997 1052 1055 :2 0
108a f8 :3 0 1059
:3 0 f8 :2 0 99a
ec :3 0 3a :3 0
8b :2 0 4 105c
105d 0 99c 105f
1065 0 1066 :3 0
2f :3 0 20 :3 0
52 :2 0 9a0 1063
1064 :4 0 1068 1069
:5 0 105a 1060 0
9a3 0 1067 :2 0
108a ec :3 0 4a
:2 0 9d :2 0 9a7
106c 106e :3 0 2c
:3 0 2d :3 0 1070
1071 0 f9 :4 0
9aa 1072 1074 :2 0
1082 9f :3 0 1076
1078 :2 0 1082 0
2c :3 0 2d :3 0
1079 107a 0 fa
:4 0 9ac 107b 107d
:2 0 1082 b0 :3 0
15 :3 0 107f 1080
0 1082 9ae 1087
b0 :3 0 17 :3 0
1083 1084 0 1086
9b3 1088 106f 1082
0 1089 0 1086
0 1089 9b5 0
108a 9b8 108f b0
:3 0 19 :3 0 108b
108c 0 108e 9bd
1090 104b 108a 0
1091 0 108e 0
1091 9bf 0 1092
9c2 1094 100c 101e
0 1095 0 1092
0 1095 9c8 0
1096 9cb 112a f8
:3 0 1099 :3 0 f8
:2 0 9d0 ec :3 0
3a :3 0 8b :2 0
4 109c 109d 0
9d2 109f 10a5 0
10a6 :3 0 2f :3 0
20 :3 0 52 :2 0
9d6 10a3 10a4 :4 0
10a8 10a9 :5 0 109a
10a0 0 9d9 0
10a7 :2 0 1129 ec
:3 0 4a :2 0 9d
:2 0 9dd 10ac 10ae
:3 0 3e :3 0 9e0
ee :3 0 3d :3 0
9e2 10b4 :2 0 10b6
:4 0 10b8 10b9 :5 0
10b1 10b5 0 9e4
0 10b7 :2 0 1121
fb :3 0 fb :2 0
23 :3 0 10bc 0
10bd 0 9e6 ef
:3 0 3a :3 0 94
:2 0 4 10c1 10c2
0 9e8 10c4 10ca
0 10cb :3 0 2f
:3 0 20 :3 0 52
:2 0 9ec 10c8 10c9
:4 0 10cd 10ce :5 0
10bf 10c5 0 9ef
0 10cc :2 0 1121
f0 :3 0 f5 :3 0
ef :3 0 ee :3 0
9f1 10d1 10d4 10d0
10d5 0 1121 f0
:3 0 3f :2 0 f6
:2 0 9f6 10d8 10da
:3 0 2c :3 0 2d
:3 0 10dc 10dd 0
f9 :4 0 9f9 10de
10e0 :2 0 10fe 3a
:3 0 3 :3 0 10e2
10e3 0 9f :3 0
10e4 10e5 0 10e6
10e8 :2 0 10fe 0
3a :3 0 3 :3 0
10e9 10ea 0 aa
:3 0 10eb 10ec 0
10ed 10ef :2 0 10fe
0 2c :3 0 2d
:3 0 10f0 10f1 0
fc :4 0 9fb 10f2
10f4 :2 0 10fe 3a
:3 0 3 :3 0 10f6
10f7 0 af :3 0
10f8 10f9 0 b0
:3 0 9fd 10fa 10fc
:2 0 10fe 9ff 111e
dd :3 0 a05 b0
:3 0 3a :3 0 96
:2 0 4 1102 1103
0 a07 1105 110b
0 110c :3 0 2f
:3 0 20 :3 0 52
:2 0 a0b 1109 110a
:4 0 110e 110f :5 0
1100 1106 0 a0e
0 110d :2 0 1111
a10 111b f2 :3 0
b0 :3 0 15 :3 0
1113 1114 0 1116
a12 1118 a14 1117
1116 :2 0 1119 a16
:2 0 111b 0 111b
111a 1111 1119 :6 0
111d 1c :3 0 a18
111f 10db 10fe 0
1120 0 111d 0
1120 a1a 0 1121
a1d 1126 b0 :3 0
17 :3 0 1122 1123
0 1125 a22 1127
10af 1121 0 1128
0 1125 0 1128
a24 0 1129 a27
112b ff8 1096 0
112c 0 1129 0
112c a2a 0 113b
60 :3 0 e9 :3 0
b0 :3 0 a2d 3d
:3 0 a2f 1132 :2 0
1134 :4 0 1136 1137
:5 0 1130 1133 :3 0
1135 :4 0 113a :2 0
113b 112e 1138 :2 0
a31 113e :3 0 113e
a37 113e 113d 113b
113c :6 0 113f 1
0 f67 f6e 113e
1290 :2 0 fd :a 0
122e 20 :7 0 a44
:2 0 a42 55 :3 0
ea :3 0 e9 :6 0
1145 1144 :3 0 1147
:2 0 122e 1141 1148
:2 0 a48 433e 0
a46 9 :3 0 114b
:7 0 114e 114c 0
122c 0 20 :6 0
a4c 4372 0 a4a
9 :3 0 1150 :7 0
1153 1151 0 122c
0 fe :6 0 9
:3 0 1155 :7 0 1158
1156 0 122c 0
ff :6 0 a50 43a6
0 a4e 9 :3 0
115a :7 0 115d 115b
0 122c 0 100
:6 0 24 :3 0 115f
:7 0 1162 1160 0
122c 0 23 :6 0
a54 43da 0 a52
9 :3 0 1164 :7 0
1167 1165 0 122c
0 101 :6 0 9
:3 0 1169 :7 0 116c
116a 0 122c 0
ec :6 0 2f :3 0
9 :3 0 116e :7 0
1171 116f 0 122c
0 102 :6 0 a56
20 :3 0 39 :3 0
a58 1176 :2 0 1178
:4 0 117a 117b :5 0
1173 1177 0 a5a
0 1179 :2 0 122a
ff :3 0 9d :2 0
117d 117e 0 122a
100 :3 0 9d :2 0
1180 1181 0 122a
f8 :3 0 1185 :3 0
f8 :2 0 a5c 101
:3 0 3a :3 0 8c
:2 0 4 1188 1189
0 a5e 118b 1191
0 1192 :3 0 2f
:3 0 20 :3 0 52
:2 0 a62 118f 1190
:4 0 1194 1195 :5 0
1186 118c 0 a65
0 1193 :2 0 122a
101 :3 0 4a :2 0
9d :2 0 a69 1198
119a :3 0 f8 :3 0
119e :3 0 f8 :2 0
a6c ec :3 0 3a
:3 0 8b :2 0 4
11a1 11a2 0 a6e
11a4 11aa 0 11ab
:3 0 2f :3 0 20
:3 0 52 :2 0 a72
11a8 11a9 :4 0 11ad
11ae :5 0 119f 11a5
0 a75 0 11ac
:2 0 11f4 ec :3 0
4a :2 0 9d :2 0
a79 11b1 11b3 :3 0
f8 :3 0 11b7 :3 0
f8 :2 0 a7c 102
:3 0 3a :3 0 94
:2 0 4 11ba 11bb
0 a7e 11bd 11c3
0 11c4 :3 0 2f
:3 0 20 :3 0 52
:2 0 a82 11c1 11c2
:4 0 11c6 11c7 :5 0
11b8 11be 0 a85
0 11c5 :2 0 11ec
102 :3 0 3f :2 0
37 :2 0 a89 11ca
11cc :3 0 dd :3 0
dc :3 0 db :3 0
a8c fe :3 0 ff
:3 0 100 :3 0 3a
:3 0 96 :2 0 4
11d5 11d6 0 a90
11d8 11de 0 11df
:3 0 2f :3 0 20
:3 0 52 :2 0 a94
11dc 11dd :4 0 11e1
11e2 :5 0 11d1 11d9
0 a97 0 11e0
:2 0 11e4 a9b 11e9
fe :3 0 15 :3 0
11e5 11e6 0 11e8
a9d 11ea 11cd 11e4
0 11eb 0 11e8
0 11eb a9f 0
11ec aa2 11f1 fe
:3 0 17 :3 0 11ed
11ee 0 11f0 aa5
11f2 11b4 11ec 0
11f3 0 11f0 0
11f3 aa7 0 11f4
aaa 11f9 fe :3 0
19 :3 0 11f5 11f6
0 11f8 aad 11fa
119b 11f4 0 11fb
0 11f8 0 11fb
aaf 0 122a 2c
:3 0 2d :3 0 11fc
11fd 0 103 :4 0
89 :2 0 fe :3 0
ab2 1200 1202 :3 0
ab5 11fe 1204 :2 0
122a 2c :3 0 2d
:3 0 1206 1207 0
104 :4 0 89 :2 0
ff :3 0 ab7 120a
120c :3 0 aba 1208
120e :2 0 122a 2c
:3 0 2d :3 0 1210
1211 0 105 :4 0
89 :2 0 100 :3 0
abc 1214 1216 :3 0
abf 1212 1218 :2 0
122a 60 :3 0 e9
:3 0 fe :3 0 ff
:3 0 100 :3 0 ac1
3d :3 0 ac5 1221
:2 0 1223 :4 0 1225
1226 :5 0 121f 1222
:3 0 1224 :4 0 1229
:2 0 122a 121b 1227
:2 0 ac7 122d :3 0
122d ad1 122d 122c
122a 122b :6 0 122e
1 0 1141 1148
122d 1290 :2 0 1b
:3 0 f5 :a 0 1289
21 :7 0 adc 46b7
0 ada 107 :3 0
106 :7 0 1234 1233
:3 0 10b :2 0 ade
107 :3 0 108 :7 0
1238 1237 :3 0 1d
:3 0 9 :3 0 123a
123c 0 1289 1231
123d :2 0 13 :2 0
ae1 109 :3 0 10a
:3 0 10c :3 0 10d
:3 0 1242 :3 0 1244
1245 :6 0 1248 1246
0 1287 0 109
:6 0 109 :3 0 108
:3 0 106 :3 0 ae3
124b 124d :3 0 1249
124e 0 1285 1d
:3 0 10e :3 0 10a
:3 0 10a :4 0 109
:3 0 10f :3 0 ae6
1255 1256 :3 0 d5
:2 0 110 :2 0 ae9
1258 125a :3 0 d8
:2 0 10e :3 0 111
:3 0 111 :4 0 109
:3 0 10f :3 0 aec
1261 1262 :3 0 aef
125c 1264 :3 0 1265
:2 0 d5 :2 0 112
:2 0 af2 1267 1269
:3 0 d8 :2 0 10e
:3 0 113 :3 0 113
:4 0 109 :3 0 10f
:3 0 af5 1270 1271
:3 0 af8 126b 1273
:3 0 1274 :2 0 d5
:2 0 112 :2 0 afb
1276 1278 :3 0 d8
:2 0 10e :3 0 10c
:3 0 10c :4 0 109
:3 0 10f :3 0 afe
127f 1280 :3 0 b01
127a 1282 :3 0 1283
:2 0 1285 b04 1288
:3 0 1288 b07 1288
1287 1285 1286 :6 0
1289 1 0 1231
123d 1288 1290 :3 0
128e 0 128e :3 0
128e 1290 128c 128d
:6 0 1291 :2 0 3
:3 0 b09 0 3
128e 1294 :3 0 1293
1291 1295 :8 0 
b24
4
:3 0 1 6 1
4 1 c 1
12 1 18 1
1e 1 24 1
2e 1 2a 1
37 1 33 1
40 1 3c 1
49 1 45 1
57 1 60 1
63 1 67 1
6c 1 73 1
71 1 78 1
7d 1 82 1
87 1 8c 1
91 1 96 1
9b 1 a3 3
ad ae af 3
b1 ab aa 1
b3 2 a6 b5
1 b9 2 b7
b8 1 c5 1
c8 1 c7 1
d0 1 d3 1
d2 1 dd 2
db dd 6 e1
e2 e3 e4 e5
e6 9 f5 f6
f7 f8 f9 fa
fb fc fd 9
ec ed ee ef
f0 f1 f2 f3
f4 1 fe 2
e8 102 3 10a
10b 10c 3 107
108 109 1 10d
1 111 2 113
114 1 123 6
11f 120 121 125
126 127 1 129
1 12d 2 12c
12d 1 133 1
135 3 13a 13b
13c 6 119 11a
11b 11c 11d 11e
1 14e 2 14d
14e 2 159 15a
1 15c 2 157
158 9 a5 c0
cf da 115 149
153 164 169 b
6a 6f 76 7b
80 85 8a 8f
94 99 9e 1
171 1 176 1
17b 1 180 1
185 1 18a 6
175 17a 17f 184
189 18e 1 198
1 19f 1 19d
1 1a5 1 1c7
1 1c4 1 1ce
1 1cb 1 1d5
1 1d2 1 1dc
1 1d9 1 1e3
1 1e0 1 1ea
1 1e7 1 1ef
7 1ca 1d1 1d8
1df 1e6 1ed 1f1
3 1bc 1c2 1f3
3 1b3 1f6 1f9
4 197 19b 1a3
1ad 1 202 1
206 2 205 209
1 20d 1 212
1 217 1 221
1 226 1 22b
1 232 1 230
1 237 1 23c
1 241 1 246
1 24b 1 250
1 255 1 25a
1 25f 1 264
1 269 1 26e
1 273 1 278
1 27d 1 282
1 287 1 28c
1 291 1 296
1 29b 1 2a2
1 2a6 1 2aa
3 2a5 2a9 2ad
2 2b5 2b7 5
2b1 2b2 2b3 2b4
2bb 1 2c9 2
2c5 2c9 2 2ce
2d2 1 2cb 2
2d6 2da 1 2e4
2 2e0 2e4 2
2e9 2ed 1 2e6
2 2f1 2f5 4
2bf 2c2 2dd 2f8
4 2fa 2fb 2fc
2fd 1 301 1
305 2 304 305
4 30e 30f 310
311 1 313 2
30d 31a 1 323
2 31f 323 1
328 1 32c 2
32b 32c 1 333
1 336 2 33f
341 1 343 1
347 3 351 352
353 3 355 34f
34e 1 357 2
34a 359 1 35d
2 35b 35c 1
369 2 368 369
1 373 2 372
373 1 379 1
37c 1 37b 1
386 2 384 386
6 38a 38b 38c
38d 38e 38f 7
392 393 394 395
396 397 398 1
3a3 1 3a7 2
3a6 3a7 7 39a
39b 39c 39d 39e
39f 3a0 1 3b1
2 3b0 3b1 2
3b7 3b9 2 3bc
3be 1 3c1 2
3c4 3c6 1 3c9
2 3cb 3cc 1
3cf 2 3ce 3cf
2 3d5 3d7 2
3da 3dc 1 3df
2 3e2 3e4 1
3e7 2 3e9 3ea
1 3ed 2 3ec
3ed 2 3f3 3f5
2 3f8 3fa 1
3fd 2 400 402
1 405 2 407
408 1 40b 2
40a 40b 2 411
413 2 416 418
1 41b 2 41e
420 1 423 2
425 426 1 429
2 428 429 2
42f 431 2 434
436 1 439 2
43c 43e 1 441
2 443 444 1
447 2 446 447
2 44d 44f 2
452 454 1 457
2 45a 45c 1
45f 2 461 462
9 470 471 472
473 474 475 476
477 478 9 467
468 469 46a 46b
46c 46d 46e 46f
1 479 9 391
3af 3cd 3eb 409
427 445 463 47d
1 47f 1 484
1 488 2 487
488 1 481 3
497 498 499 3
494 495 496 1
49a 2 490 49e
2 4a0 4a1 3
4a5 4a6 4a7 5
4b2 4b3 4b4 4b5
4b6 6 4c8 4c9
4ca 4cb 4cc 4cd
6 4c2 4c3 4c4
4c5 4c6 4c7 1
4ce 3 4b7 4be
4d2 1 4e4 1
4e8 1 4f1 2
4ed 4f1 1 4fb
2 4fa 4fb 1
4f7 1 506 2
505 506 1 510
2 50f 510 e
345 349 364 36e
378 383 4a2 4aa
4d5 4d8 501 50b
515 51a 1d 210
215 21f 224 229
22e 235 23a 23f
244 249 24e 253
258 25d 262 267
26c 271 276 27b
280 285 28a 28f
294 299 29e 337
1 522 1 525
1 529 1 52e
1 531 1 530
1 53e 1 542
1 546 2 545
546 1 54f 2
54e 54f 1 55b
2 55a 55b 1
560 2 55f 560
1 56d 1 571
1 575 2 574
575 1 57e 2
57d 57e 1 58b
1 58f 1 593
2 592 593 1
59c 2 59b 59c
1 5a9 1 5ad
1 5b1 2 5b0
5b1 1 5ba 2
5b9 5ba 7 538
556 567 585 5a3
5c1 5c6 1 52c
1 5d1 1 5d6
1 5d9 1 5d8
1 5e6 1 5ea
1 5ee 2 5ed
5ee 1 5fc 2
5fb 5fc 1 606
2 605 606 1
610 2 60f 610
1 61b 1 61f
1 623 2 622
623 1 631 2
630 631 1 63c
1 640 1 644
2 643 644 1
652 2 651 652
1 65d 1 661
1 665 2 664
665 1 673 2
672 673 c 5e0
5f7 601 60b 615
62c 636 64d 657
66e 678 67d 1
5d4 1 685 1
688 1 68c 1
691 1 696 1
69b 3 6a8 6a9
6aa 3 6ac 6a6
6a5 1 6ae 3
6a0 6a1 6b0 1
6b5 3 6b2 6b3
6b4 1 6bd 1
6c2 1 6c6 2
6c5 6c6 1 6bf
1 6d0 2 6cf
6d0 1 6de 2
6dd 6de 2 6d9
6dc 1 6e7 2
6e5 6e7 1 700
2 6ff 700 6
6ef 6f2 6f5 6f8
6fb 6fe 1 706
1 708 1 70e
2 70d 70e 2
719 71a 2 717
718 1 71b 4
6e4 709 713 71f
1 721 4 6bc
6ce 722 727 4
68f 694 699 69e
1 732 1 737
1 73c 1 741
3 74e 74f 750
3 752 74c 74b
1 754 3 746
747 756 1 75b
3 758 759 75a
1 763 1 768
1 76c 2 76b
76c 1 765 1
776 2 775 776
1 785 1 789
1 78d 2 78c
78d 2 77f 782
1 79a 2 798
79a 1 7b3 2
7b2 7b3 6 7a2
7a5 7a8 7ab 7ae
7b1 1 7b9 1
7bb 1 7c1 2
7c0 7c1 2 7cc
7cd 2 7ca 7cb
1 7ce 4 797
7bc 7c6 7d2 1
7d4 4 762 774
7d5 7da 4 735
73a 73f 744 1
7e7 1 7ea 2
7f0 7f3 1 7f9
1 7ff 2 7fe
7ff 1 806 1
80e 1 811 1
818 1 81e 2
81d 81e 1 825
1 82b 1 832
1 837 1 83c
1 841 1 846
1 84b 1 850
1 855 1 85a
1 85f 1 866
1 864 1 86e
3 87b 87c 87d
3 87f 879 878
1 881 2 874
883 1 887 2
885 886 1 893
1 896 1 895
1 89e 1 8a1
1 8a0 1 8ab
1 8b2 1 8ba
1 8bf 1 8c4
1 8cb 2 8c7
8cb 1 8cf 2
8ce 8cf 1 8c1
4 8e3 8e6 8e9
8ec 4 8dd 8de
8df 8e0 1 8ed
2 8d9 8f1 1
8f4 1 8f6 1
8f7 1 8fd 2
8fb 8fd 6 901
902 903 904 905
906 9 915 916
917 918 919 91a
91b 91c 91d 9
90c 90d 90e 90f
910 911 912 913
914 1 91e 2
908 922 3 92a
92b 92c 3 927
928 929 1 92d
1 931 2 933
934 5 93e 93f
940 944 948 1
94c 2 94f 950
5 939 93a 93b
93c 93d 9 870
873 88e 89d 8a8
8fa 935 955 95a
e 807 826 830
835 83a 83f 844
849 84e 853 858
85d 862 869 1
967 1 96a 1
96e 1 972 1
976 2 975 976
1 97d 1 980
1 986 1 98b
1 990 1 995
1 998 1 997
1 9a2 2 9b0
9b3 1 9b5 2
9c1 9c4 1 9c6
1 9d1 1 9d5
1 9d9 2 9d8
9d9 1 9de 2
9dd 9de 1 9ee
2 9ed 9ee 1
9f3 2 9f2 9f3
a 99f 9a5 9a8
9b6 9b9 9c7 9ca
9e9 9fa 9ff 4
981 989 98e 993
1 a07 1 a0b
1 a11 1 a14
8 a18 a19 a1a
a1b a1c a1d a1e
a1f 1 a23 1
a27 2 a26 a27
1 a2e 1 a31
1 a39 1 a3d
1 a41 3 a3c
a40 a44 1 a54
2 a50 a54 2
a59 a5d 1 a56
2 a61 a65 1
a6d 2 a69 a6d
2 a72 a76 1
a6f 2 a7a a7e
6 a4a a4d a66
a7f a82 a85 1
a8a 1 a8e 2
a8d a8e 1 a9a
1 a9e 2 a9d
a9e 3 a96 aa6
aab 1 ab4 2
ab0 ab4 1 abd
2 ab9 abd 1
ac5 2 ac4 ac5
1 ad1 2 acd
ad1 2 ad6 ada
1 ad3 2 ade
ae2 1 ae5 2
ae4 ae5 1 aee
1 af4 1 af9
1 afe 1 b0a
1 b08 1 b0f
1 b14 1 b19
1 b1e 1 b23
1 b28 1 b2d
1 b32 1 b37
1 b3c 1 b41
1 b46 1 b4b
1 b50 1 b55
1 b5a 1 b5f
1 b64 1 b69
1 b6e 1 b73
1 b78 1 b7d
1 b82 1 b87
1 b8c 1 b91
1 b96 1 b9b
1 ba0 1 ba5
1 baa 3 bb9
bba bbb 3 bbd
bb7 bb6 1 bbf
2 bb2 bc1 1
bc5 2 bc3 bc4
1 bcf 8 bd4
bd5 bd6 bd7 bd8
bd9 bda bdb 2
be5 be8 1 bea
8 bed bee bef
bf0 bf1 bf2 bf3
bf4 2 bfe c01
1 c03 2 c0c
c0e 2 c10 c12
2 c14 c16 1
c18 1 c20 1
c24 1 c28 2
c27 c28 1 c36
2 c35 c36 3
c4a c4b c4c 6
c52 c53 c54 c55
c56 c57 2 c63
c65 2 c67 c69
2 c6b c6d 2
c6f c71 2 c73
c75 2 c77 c79
2 c7b c7d 2
c7f c81 2 c83
c85 1 c87 2
c8b c8d 2 c92
c94 2 c9e c9f
2 c9b ca1 1
ca3 2 ca7 ca9
2 cae cb0 2
cb5 cb7 2 cbc
cbe 7 ccc ccd
cce ccf cd0 cd1
cd2 7 cc5 cc6
cc7 cc8 cc9 cca
ccb 1 cd3 b
c58 c5f c89 c90
c97 ca5 cac cb3
cba cc1 cd7 1
ce1 2 cdf ce1
2 ce5 ce7 2
ce9 ceb 1 cee
1 cf0 1 cf4
2 cf2 cf4 1
cf8 2 cf7 cf8
2 cfe d00 2
d03 d05 1 d08
2 d0b d0d 1
d10 2 d12 d13
1 d16 2 d15
d16 2 d1c d1e
2 d21 d23 1
d26 2 d29 d2b
1 d2e 2 d30
d31 1 d34 2
d33 d34 2 d3a
d3c 2 d3f d41
1 d44 2 d47
d49 1 d4c 2
d4e d4f 1 d52
2 d51 d52 2
d58 d5a 2 d5d
d5f 1 d62 2
d65 d67 1 d6a
2 d6c d6d 1
d70 2 d6f d70
2 d76 d78 2
d7b d7d 1 d80
2 d83 d85 1
d88 2 d8a d8b
1 d8e 2 d8d
d8e 2 d94 d96
2 d99 d9b 1
d9e 2 da1 da3
1 da6 2 da8
da9 2 dc0 dc2
2 dc5 dc7 2
dca dcc e dbc
dbd dbe dbf dc4
dc9 dce dcf dd0
dd1 dd2 dd3 dd4
dd5 e dae daf
db0 db1 db2 db3
db4 db5 db6 db7
db8 db9 dba dbb
1 dd6 7 d14
d32 d50 d6e d8c
daa dda 2 deb
ded 2 df0 df2
2 df5 df7 8
de7 de8 de9 dea
def df4 df9 dfa
8 ddf de0 de1
de2 de3 de4 de5
de6 1 dfb 1
dff 2 e01 e02
1 e0a 2 e09
e0a 2 e0f e12
1 e15 16 bb1
bcc bd2 bdc beb
bf5 c04 c07 c1a
c31 c3b c3e c41
c44 c47 c4f cda
cdd cf1 e03 e08
e16 26 a32 aef
af7 afc b06 b0d
b12 b17 b1c b21
b26 b2b b30 b35
b3a b3f b44 b49
b4e b53 b58 b5d
b62 b67 b6c b71
b76 b7b b80 b85
b8a b8f b94 b99
b9e ba3 ba8 bad
1 e21 1 e26
1 e2e 1 e31
1 e34 1 e33
1 e3c 1 e41
1 e45 2 e44
e45 1 e3e 1
e52 2 e51 e52
1 e5c 2 e5b
e5c 1 e68 1
e6c 2 e6b e6c
1 e7b 1 e7f
2 e7e e7f 8
e30 e3b e4d e57
e61 e74 e87 e8c
2 e24 e29 1
e99 1 e9c 1
ea0 1 ea4 1
ea8 2 ea7 ea8
1 eaf 1 eb2
1 eb8 1 ebd
1 ec2 1 ec7
1 ecf 1 ed2
1 ed5 1 ed4
1 ee2 1 ee8
2 ef0 ef2 1
ef4 1 f01 2
f00 f01 2 f07
f09 1 f0b 2
f0e f10 5 ef6
ef9 eff f0c f13
2 f1e f20 1
f22 1 f27 2
f25 f27 1 f2f
1 f33 1 f37
2 f36 f37 1
f3c 2 f3b f3c
1 f4c 2 f4b
f4c 1 f51 2
f50 f51 3 f47
f58 f5d 1 f5f
9 ed1 edc edf
ee5 eeb f16 f19
f24 f60 5 eb3
ebb ec0 ec5 eca
1 f68 1 f6c
1 f70 1 f75
1 f7a 1 f7f
1 f84 1 f89
1 f8e 1 f93
1 f98 1 f9f
1 fa2 1 fa6
1 faa 1 fae
2 fad fae 1
fb5 1 fb8 1
fbe 1 fc1 1
fc0 1 fcc 1
fcf 1 fd5 1
fd9 2 fd8 fd9
1 fdf 1 fce
1 fe7 1 fec
1 fe9 1 fef
1 ff6 2 ff4
ff6 1 ffc 1
1001 1 1010 1
1014 4 1012 1016
1019 101d 1 1023
1 1026 1 1025
1 102e 1 1033
1 1037 2 1036
1037 1 1030 2
1042 1043 1 1049
2 1047 1049 1
104f 2 1053 1054
1 1058 1 105e
1 1062 2 1061
1062 1 105b 1
106d 2 106b 106d
1 1073 1 107c
4 1075 1077 107e
1081 1 1085 2
1087 1088 4 1051
1056 106a 1089 1
108d 2 108f 1090
5 1021 102d 103f
1046 1091 2 1093
1094 4 ffe 1004
1007 1095 1 1098
1 109e 1 10a2
2 10a1 10a2 1
109b 1 10ad 2
10ab 10ad 1 10b0
1 10b3 1 10b2
1 10be 1 10c3
1 10c7 2 10c6
10c7 1 10c0 2
10d2 10d3 1 10d9
2 10d7 10d9 1
10df 1 10f3 1
10fb 5 10e1 10e7
10ee 10f5 10fd 1
10ff 1 1104 1
1108 2 1107 1108
1 1101 1 1110
1 1115 1 1112
1 1118 1 111b
2 111e 111f 4
10ba 10cf 10d6 1120
1 1124 2 1126
1127 2 10aa 1128
2 112a 112b 1
112f 1 1131 5
fc8 fcb ff2 112c
1139 a f73 f78
f7d f82 f87 f8c
f91 f96 f9b fb9
1 1142 1 1146
1 114a 1 114f
1 1154 1 1159
1 115e 1 1163
1 1168 1 116d
1 1172 1 1175
1 1174 1 1184
1 118a 1 118e
2 118d 118e 1
1187 1 1199 2
1197 1199 1 119d
1 11a3 1 11a7
2 11a6 11a7 1
11a0 1 11b2 2
11b0 11b2 1 11b6
1 11bc 1 11c0
2 11bf 11c0 1
11b9 1 11cb 2
11c9 11cb 3 11ce
11cf 11d0 1 11d7
1 11db 2 11da
11db 3 11d2 11d3
11d4 1 11e3 1
11e7 2 11e9 11ea
2 11c8 11eb 1
11ef 2 11f1 11f2
2 11af 11f3 1
11f7 2 11f9 11fa
2 11ff 1201 1
1203 2 1209 120b
1 120d 2 1213
1215 1 1217 3
121c 121d 121e 1
1220 9 117c 117f
1182 1196 11fb 1205
120f 1219 1228 8
114d 1152 1157 115c
1161 1166 116b 1170
1 1232 1 1236
2 1235 1239 1
123f 2 124a 124c
2 1254 1253 2
1257 1259 2 1260
125f 2 125b 1263
2 1266 1268 2
126f 126e 2 126a
1272 2 1275 1277
2 127e 127d 2
1279 1281 2 124f
1284 1 1247 1a
a 10 16 1c
22 28 31 3a
43 4c 5d 16e
1ff 51f 5cb 682
72c 7df 95f a04
e1b e91 f65 113f
122e 1289 
1
4
0 
1294
0
1
28
21
c3
0 1 1 1 4 1 6 6
1 1 1 1 1 d d d
10 1 12 1 14 14 14 1
1 19 19 1 1c 1c 1c 1
1 0 0 0 0 0 0 0

b50 14 0
24 1 0
7e1 1 d
3 0 1
af4 14 0
221 6 0
e1d 1 18
a38 14 16
f98 1c 0
f8e 1c 0
b0f 14 0
986 12 0
85f d 0
25f 6 0
18a 4 0
9b 3 0
b14 14 0
98b 12 0
ebd 19 0
850 d 0
250 6 0
17b 4 0
8c 3 0
1232 21 0
eb8 19 0
e26 18 0
27d 6 0
1141 1 20
226 6 0
1154 20 0
b28 14 0
84b d 0
26e 6 0
24b 6 0
176 4 0
87 3 0
b46 14 0
5cd 1 a
b73 14 0
291 6 0
45 1 0
b2d 14 0
b4b 14 0
c 1 0
123f 21 0
1142 20 0
f68 1c 0
b37 14 0
684 1 b
b55 14 0
b8c 14 0
2a 1 0
837 d 0
b96 14 0
1a5 4 0
ba5 14 0
b64 14 0
282 6 0
1231 1 21
5f 1 3
269 6 0
7e6 d e
193 4 0
f93 1c 0
b19 14 0
278 6 0
b1e 14 0
2aa 7 0
e93 1 19
b08 14 0
864 d 0
230 6 0
71 3 0
67 3 0
114a 20 0
f9f 1d 0
f70 1c 0
e99 1a 0
ec2 19 0
e21 18 0
a41 16 0
a11 15 0
baa 14 0
967 13 0
990 12 0
7e7 e 0
832 d 0
73c c 0
696 b 0
5d1 a 0
529 9 0
2a2 7 0
20d 6 0
6c 3 0
72e 1 c
80d d f
198 4 0
b7d 14 0
29b 6 0
1236 21 0
8a9 10 0
b9b 14 0
b6e 14 0
85a d 0
732 c 0
68c b 0
28c 6 0
25a 6 0
185 4 0
96 3 0
33 1 0
f84 1c 0
f7a 1c 0
b69 14 0
685 b 0
287 6 0
23c 6 0
202 6 0
f89 1c 0
b87 14 0
8b6 11 0
82b d 0
1159 20 0
b82 14 0
846 d 0
246 6 0
171 4 0
82 3 0
f67 1 1c
961 1 12
4ab 8 0
212 6 0
1b4 5 0
201 1 6
522 9 0
217 6 0
116d 20 0
1168 20 0
1163 20 0
f7f 1c 0
f75 1c 0
a07 14 0
a06 1 14
115e 20 0
b41 14 0
83c d 0
2a1 6 7
264 6 0
237 6 0
78 3 0
b5f 14 0
22b 6 0
4f 1 2
4 1 0
114f 20 0
afe 14 0
273 6 0
170 1 4
b23 14 0
841 d 0
741 c 0
69b b 0
241 6 0
7d 3 0
60 3 0
3c 1 0
1e 1 0
18 1 0
12 1 0
f9e 1c 1d
e98 19 1a
b78 14 0
b3c 14 0
af9 14 0
a10 14 15
966 12 13
737 c 0
691 b 0
296 6 0
206 6 0
b5a 14 0
521 1 9
a39 16 0
ba0 14 0
2a6 7 0
ec7 19 0
a3d 16 0
b91 14 0
80e f 0
19d 4 0
b32 14 0
855 d 0
255 6 0
180 4 0
91 3 0
0

/
/*
set timing on
variable rc refcursor;
begin
end;
/
print rc
prompt @@  content of mgmt_snapshot
SELECT snap_id FROM dbsnmp.mgmt_snapshot;
prompt @@  content of mgmt_baseline_sql
SELECT sql_id, hash_value, executions, elapsed_time FROM dbsnmp.mgmt_baseline_sql
order by hash_value;
prompt @@  content of mgmt_capture_sql
select capture_id, sql_id, hash_value, executions, elapsed_time from dbsnmp.mgmt_capture_sql
order by hash_value, capture_id;
prompt @@  content of mgmt_history_sql
select capture_id, sql_id, hash_value, executions, elapsed_time, t_per_exec, adjusted_elapsed_time
from dbsnmp.mgmt_history_sql
order by capture_id, sql_id, hash_value DESC;
prompt @@ unique capture_id in mgmt_history_sql
select unique capture_id 
from dbsnmp.mgmt_history_sql;
*/
show errors
