#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/db/criticalSessions.pl /main/3 2010/08/30 07:28:09 pbhogara Exp $
#
# criticalSessions.pl
# 
# Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      criticalSessions.pl
#
#    DESCRIPTION
#      Gets the list of critical sessions.
#
#    NOTES
#
#
#    MODIFIED   (MM/DD/YY)
#    pbhogara    06/09/10 - Creation
# 


use strict;

require "emd_common.pl";
require "db/direct_access.pl";

package CriticalSessions;

my $INDX       = "INDX";
my $KSUPRSER   = "KSUPRSER";
my $KSUPRPNAME = "KSUPRPNAME";
my $KSUPRFLG   = "KSUPRFLG";
my $ADDR       = "ADDR";
my $KSUPRPID   = "KSUPRPID";

my $KSUSESER   = "KSUSESER";
my $INST_ID    = "INST_ID";
my $KSUSEPRO   = "KSUSEPRO";
my $SESSION_ID = "SESSION_ID";

# Projection list used for querying x$ksupr
my @QUERY_KSUPR = ($INDX,       # oracle pid
                   $KSUPRSER,   # process serial number 
                   $KSUPRPNAME, # process name 
                   $KSUPRFLG,
                   $ADDR,       # addr of process state object
                   $KSUPRPID,); # os pid of the process


# Projection list used for querying x$ksuse
my @QUERY_KSUSE = ($INDX,       # sid
                   $KSUSESER,   # session serial no
                   $INST_ID,    # inst id
                   $KSUSEPRO,); # address of parent state object


my $target_version;
my @inst_critical_sessions;

sub setContext
{
  my $oracle_home    = shift;
  my $oracle_sid     = shift;
  my $connect_string = shift;
  $target_version    = shift;
  my $use_prelim     = shift;

  DirectAccess::setConnectionContext($oracle_home,
                                     $oracle_sid,
                                     $connect_string,
                                     $target_version,
                                     $use_prelim,
                                    );
}

#
# Subroutine: get_critical_sessions
#
# queries x$ksupr & x$ksuse in prelim mode to get the 
# set of instance critical sessions.
# 
sub get_critical_sessions
{

  @inst_critical_sessions = ();

  my @fatal_processes = get_fatal_processes();

  my @ksuse_sessions  = get_ksuse_sessions();

  
  my $count = 0;
  foreach my $row1(@ksuse_sessions)
  {
    foreach my $row2(@fatal_processes) 
    {
      if($row1->{$KSUSEPRO} eq $row2->{$ADDR})
      {
        my %row;
        $row{$SESSION_ID} = $row1->{$INDX}."_".
                            $row1->{$KSUSESER}."_".
                            $row1->{$INST_ID};
        $row{$KSUPRPID}    = $row2->{$KSUPRPID};
        $row{$KSUPRPNAME}  = $row2->{$KSUPRPNAME};
        push(@inst_critical_sessions,\%row);
        $count++;
      }
    }
  } 

  ::EMAGENT_PERL_DEBUG(" Number of inst critical sessions found ".$count);

}


sub get_ksuse_sessions
{
  my @session_list = ();

  DirectAccess::setQueryContext($DirectAccess::UNSAFE_MODE,
                                $DirectAccess::DISABLE_TRACE,
                                $DirectAccess::OUTPUT_CSV);

  my $query_begin = time;
  my @preds = ("$KSUSEPRO!=0");
  my @ksuse_rows  = DirectAccess::getFixedTable('X$KSUSE',
                                                \@QUERY_KSUSE,);

  my $query_end   = time;
  ::EMAGENT_PERL_DEBUG(" direct_access query on x\$ksuse took - ".
                         ($query_end-$query_begin)." secs");


  return @ksuse_rows; 
}

sub get_fatal_processes
{
 
  my @fatal_processes = ();

  DirectAccess::setQueryContext($DirectAccess::UNSAFE_MODE,
                                $DirectAccess::DISABLE_TRACE,
                                $DirectAccess::OUTPUT_CSV);

  my $query_begin = time;
  my @ksupr_rows  = DirectAccess::getFixedTable('X$KSUPR',
                                                \@QUERY_KSUPR,);

  my $query_end   = time;
  ::EMAGENT_PERL_DEBUG(" direct_access query on x\$ksupr took - ".
                         ($query_end-$query_begin)." secs");

  my $count=0;
  foreach my $row(@ksupr_rows)
  {
    my $fatal_flag = $row->{$KSUPRFLG};

    # If bitand(ksuprflg,4) !=0 then process is fatal
    if($fatal_flag & 4)
    {
      push(@fatal_processes,$row); 
      $count++;
    }
    else
    {
      ::EMAGENT_PERL_DEBUG(" Process ".$row->{$KSUPRPNAME}.
                           $row->{$KSUPRSER}." is not fatal");
    }
  }
  ::EMAGENT_PERL_DEBUG(" $count fatal processes found");

  return @fatal_processes;
}


#
# Subroutine: print_metric_data
# 
# Returns: metric columns seperated by delimiter "|" and
#          rows separated by new line.
#          For example:
#          35FCAA04|19534|PSP0
#          35FCB4E0|19541|VKTM
sub print_metric_data
{
  my $res = "";
  foreach my $row(@inst_critical_sessions)
  {
    $res .= $row->{$SESSION_ID}."|". #col1
            $row->{$KSUPRPID}."|".   #col2
            $row->{$KSUPRPNAME}.     #col3
            "\n"; #new row
  }
  ::EMAGENT_PERL_DEBUG(" Critical sessions -- $res");

  return $res;
}

sub _trim
{
  my $str = shift;
  $str =~ s/^\s+//;
  $str =~ s/\s+$//;
  return $str;
}

1;
