#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/db/dbAshWaitClass.pl /main/6 2011/06/29 00:45:50 pbhogara Exp $
#
# dbAshWaitClass.pl
# 
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      dbAshWaitClass.pl - get ASH wait class summary from ASH dump.
#
#    DESCRIPTION
#      Uses script ashMetric.pl to get ASH wait class summary. 
#      Along with this data it returns the following information
#       a) Various attributes(eg: service,module,action..) of sessions 
#          whose ids are passed to the script.
#       b) Event histogram of these sessions for the ASH dump duration
#      This kind of piggybacking avoids another ash dump(which could be expensive) 
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    pbhogara    02/11/11 - Pass event map to ashscript
#    pbhogara    07/07/10 - Creation
# 

use strict;

require "emd_common.pl";

require "db/oradebug_dump.pl";
require "db/direct_access.pl";

package AshWaitClass;

our $ERROR_CODE;

my $target_version;
my $oracle_home;
my $cleanup_flag = $OradebugDump::modes{'SPACE_SAVING'}; 

my $HASH       = "KSLEDHASH";
my $EVENT_NAME = "KSLEDNAM";
my $CLASS_NAME = "KSLEDCLASS";

my @QUERY_KSLED = ($HASH,
                   $EVENT_NAME,
                   $CLASS_NAME);

#
# Subroutine: setContext
#  $_[0] => oracle_home
#  $_[1] => oracle_sid
#  $_[2] => connect string (should be as SYSDBA)
#  $_[3] => prelim/non-prelim
#
sub setContext
{ 
  $oracle_home    = shift;
  my $oracle_sid     = shift;
  my $connect_string = shift;
  $target_version      = shift;
  my $use_prelim     = shift;
  
  OradebugDump::setContextAttributes($oracle_home,
                                     $oracle_sid,
                                     $connect_string,
                                     $use_prelim,
                                    );

  DirectAccess::setConnectionContext($oracle_home,
                                     $oracle_sid,
                                     $connect_string,
                                     $target_version,
                                     $use_prelim,
                                    );
}


#
# Subroutine: getAshMetric
#  $_[0] => duration of ash dump in millisecs
#  $_[1] => histogram bucket interval in millisecs.
#  $_[2] => This UTC epoch time stamp represents the time when ths metric was
#           requested from the client( mid-tier client).
#  $_[3] => List of sessions for which we need the attribute values
#           which will be used to populate hanganalysis screens.
#  $_[4] => List of sessions for which we need the event histogram
#
sub getAshMetric
{
 
  require "db/ashviewer/ashMetric.pl";

  my $duration = shift;
  $duration = 60000 if !defined $duration;

  my $bucket_interval = shift;
  $bucket_interval = 15000 if !defined $bucket_interval;

  my $dump_request_time = shift;

  ::EMAGENT_PERL_DEBUG(" duration $duration ");
  ::EMAGENT_PERL_DEBUG(" dump request time $dump_request_time");

  # convert all timestamps to secs
  $duration = int($duration/1000);
  $bucket_interval = int($bucket_interval/1000); 

  my $session_list = shift;

  my $end_time = time;
  my $begin_time = $end_time - $duration;
  my $delay = 0;
  if(defined $dump_request_time)
  {
     $dump_request_time = int($dump_request_time/1000);
     $begin_time = $dump_request_time - $duration;

     # The delay in receiving the request from client
     $delay = $end_time - $dump_request_time; 
  }

  # Add delay to duration because we have to dump this extra bit for
  # retrieving what we need.
  $duration += $delay;

  my $maxBuckets = int($duration/$bucket_interval);
  $maxBuckets += ($duration % $bucket_interval!=0)?1:0; 

  # Duration in minutes. 
  my $duration_min = int($duration/60);
  $duration_min += ($duration % 60!=0)?1:0;

  # Generate ASH dump
  my $trace_file = generate_ash_dump($duration_min);
  if($ERROR_CODE)
  {
    goto EXIT;
  }

  ::EMAGENT_PERL_DEBUG(" Max buckets: ".$maxBuckets.",  Bucket interval: ".$bucket_interval.", Duration: ".$duration_min."mins");

  ::EMAGENT_PERL_DEBUG(" trace file name - ".$trace_file." begin:$begin_time, end:$end_time");

  my $report_level = "<wait_class>{histogram}";
  my $str = "";
  if(defined $session_list)
  {
    $report_level .= "<session_attributes>{";
    my @arr = split(/#/,$session_list);

    # Convert sid_serialno_instid to
    # instid,sid,serialno format
    my @sid_arr = split(/_/,$arr[0]);
    $report_level .= "\"$sid_arr[2],$sid_arr[0],$sid_arr[1]\"";

    for(my $i=1;$i<=$#arr;$i++)
    {
      @sid_arr = split(/_/,$arr[$i]);
      $report_level .= ",\"$sid_arr[2],$sid_arr[0],$sid_arr[1]\"";
    } 
    $report_level .= "}"; 
  }

  my $res = "";
  if(is11gR1_or_higher())
  {
    my $begin = time;
    my $event_map_ref = get_event_map();
    if($ERROR_CODE)
    {
      goto EXIT;
    }
    $res = AshViewer::getReport($trace_file,$target_version, $begin_time, $dump_request_time,$report_level,undef,$maxBuckets,$bucket_interval, $event_map_ref); 
    my $end = time;
    ::EMAGENT_PERL_DEBUG("Time taken for querying ksled ".($end-$begin));
  }
  else
  {
    $res = AshViewer::getReport($trace_file,$target_version, $begin_time, $dump_request_time,$report_level,undef,$maxBuckets,$bucket_interval); 
  }

  EXIT:
  if($cleanup_flag && (defined $trace_file) && (-e $trace_file)){
    unlink $trace_file;
  }

  ::EMAGENT_PERL_DEBUG(" ASH wait class output ".$res); 

  return $res;
}


#
# Subroutine: generate_ash_dump 
#  Use the ASH dump module to generate ash dump
#
#  $_[0] => duration of ash dump in minutes, default is 1min
#
sub generate_ash_dump
{
  my $duration = shift;
  my $ashdump_command   = "oradebug dump ashdump $duration";

  ################################
  # Produce an ORAERROR ASH Dump trace file and get its name.
  # Also record ORA- errors.
  ################################
 
  my $tracefile_name = OradebugDump::generateDump(# oradebug command
                                                  $ashdump_command,
                                                  # set tracefile cleanup mode
                                                  $cleanup_flag,
                                                 );
  
  if ($OradebugDump::ERROR_CODE)
  { 
    ::EMAGENT_PERL_ERROR("received error code $OradebugDump::ERROR_CODE while dumping ash");
    $ERROR_CODE = $OradebugDump::ERROR_CODE;
    return;
  }

  return $tracefile_name; 
}

sub get_event_map
{
  DirectAccess::setQueryContext($DirectAccess::UNSAFE_MODE,
                                $DirectAccess::DISABLE_TRACE,
                                $DirectAccess::OUTPUT_XML);
  
  my @event_map_arr = DirectAccess::getFixedTable('X$ksled',
                                                  \@QUERY_KSLED);
  if ($DirectAccess::ERROR_CODE)
  { 
    ::EMAGENT_PERL_ERROR("received error code $DirectAccess::ERROR_CODE while querying x\$ksled");
    $ERROR_CODE = $DirectAccess::ERROR_CODE;
    return;
  }

  my %event_map;
  my $count = 0;
  foreach my $map(@event_map_arr)
  {
    $event_map{$map->{$HASH}} = [$map->{$EVENT_NAME},$map->{$CLASS_NAME}];
  }
  my $count = scalar(keys %event_map);
  ::EMAGENT_PERL_DEBUG("Number of events in x\$ksled - $count");
  return \%event_map;
}

sub trim
{ 
  my $str = shift;
  $str =~ s/^\s+//;
  $str =~ s/\s+$//;
  return $str;
}

sub is11gR1_or_higher
{
  $target_version = trim($target_version);
  if($target_version eq "")
  {
    return 1;
  }
  my @arr = split(/\./,$target_version);
  if($arr[0] >= 11)
  {
    return 1;
  }
  return 0;
}
1;
