#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/db/dbHost.pl /main/4 2011/03/31 04:40:42 pbhogara Exp $
#
# dbHost.pl
# 
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      dbHost.pl
#
#    DESCRIPTION
#      This script gets host metrics from X$KEWMDRMV through oradebug direct_access.
#
#    NOTES
#
#
#    MODIFIED   (MM/DD/YY)
#    pbhogara    06/06/10 - Creation
# 


use strict;

require "emd_common.pl";
require "db/direct_access.pl";
require "db/emergPerfUtil.pl";

package DbHost;

our $ERROR_CODE = 0;

# Projection list used for querying x$kewmdrmv
my @QUERY_KEWMDRMV = ("METRICID",
                      "VALUE",
                      "GROUPID",
                      "INTSIZE_CSEC",
                      "NAME",
                      "BEGTIME",
                      "ENDTIME",);

my $target_version;
sub setContext
{
  my $oracle_home    = shift;
  my $oracle_sid     = shift;
  my $connect_string = shift;
  $target_version    = shift;
  my $use_prelim     = shift;

  DirectAccess::setConnectionContext($oracle_home,
                                     $oracle_sid,
                                     $connect_string,
                                     $target_version,
                                     $use_prelim,
                                    );
}

my $RUNQUEUE_METRICID       = 2135;
my $HOST_CPU                = 2155;
my $BACKGROUND_CPU_METRICID = 2153;
my $FOREGROUND_CPU_METRICID = 2075;
my $GROUP_ID                = 2;

my @metric_ids =($FOREGROUND_CPU_METRICID,
                 $BACKGROUND_CPU_METRICID,
                 $HOST_CPU,
                 $RUNQUEUE_METRICID,);

# array of hashmaps containing timestamp and metric values 
# For example, each entry in the array is as follows:
# (ts => "05-06-2010 12:14:30" , 2135 => .41 ,2155 => 18.12, 2153 => 0.016 ,2075 => 0.57)
my @metric_result;


#
# Subroutine: get_Host_metrics
#  $_[0] => Duration(in millisecs) for which we need data. 
#           If it is -1/undef then we get the latest sample.
#  $_[1] => The time(UTC epoch) at which metric request was sent from 
#           client. Duration is considered with reference to this time
#           stamp.
#
sub get_Host_metrics
{
  my $duration = shift;
  my $metric_request_time = shift;

  my $history_flag = 1;

  if(!defined $duration || $duration == -1){
    $history_flag = 0; 
  }

  ::EMAGENT_PERL_DEBUG(" host metric duration,metric request time stamp - ".$duration.",".$metric_request_time);

  my $curr_time_stamp = time;

  my $end_time   = $curr_time_stamp*1000;
  my $begin_time = $curr_time_stamp*1000 - $duration;

  if(defined $metric_request_time)
  {
     # Get samples in the period $begin_time to $end_time
     $begin_time = ($metric_request_time - $duration); # millisecs
     $end_time   = $metric_request_time; 
  }

  DirectAccess::setQueryContext($DirectAccess::UNSAFE_MODE,
                                $DirectAccess::DISABLE_TRACE,
                                $DirectAccess::OUTPUT_CSV);

  my $fixed_indx_pred = 'GROUPID==2';

  # Get only long duration metrics
  my @preds = ("INTSIZE_CSEC>5900");

  my $query_begin = time;
  my @kewmdrmv_metrics = DirectAccess::getFixedTable('X$KEWMDRMV',
                                                     \@QUERY_KEWMDRMV,
                                                     \@preds,
                                                     $DirectAccess::LOGICAL_OP{AND},
                                                     $fixed_indx_pred);
  if ($DirectAccess::ERROR_CODE)
  {
    ::EMAGENT_PERL_ERROR("received error code $DirectAccess::ERROR_CODE while querying for x\$kewdrmv - dbhost");
    $ERROR_CODE = $DirectAccess::ERROR_CODE;
    return;
  }
  my $query_end   = time;
  ::EMAGENT_PERL_DEBUG(" query on x\$kewmdrmv took - ".($query_end-$query_begin)." secs");

  my $i             = 0;
  my $first_metric  = undef;
  my $ts_and_values_ref = {}; 
  my $num_samples   = 0;

  LOOP:foreach my $metric_row(@kewmdrmv_metrics)
  {
    my $metric_id  = EmergPerfUtil::trim($metric_row->{'METRICID'});
    my $metric_val = EmergPerfUtil::trim($metric_row->{'VALUE'});

    if(grep {$_ == $metric_id} @metric_ids)
    {
      # Timestamp obtained from direct_access output is of the form 
      # DD-MM-YYYY HH24:MI:SS. Convert it to epoch time.
      my $sample_beg_time = EmergPerfUtil::get_epoch_time($metric_row->{'BEGTIME'});
      my $sample_end_time = EmergPerfUtil::get_epoch_time($metric_row->{'ENDTIME'});

      if($sample_end_time > $end_time && $history_flag){
        next LOOP;
      }

      if($sample_end_time < $begin_time && $history_flag){
        last;
      }

      if(!defined $first_metric ||
         $metric_id eq $first_metric)
      {
        if($num_samples == 1 && !$history_flag)
        {
          last; # break after first sample if history flag is unset 
        }
        $ts_and_values_ref = {};
        push(@metric_result,$ts_and_values_ref);
        $first_metric = $metric_id;
        $num_samples++;
      }
      $ts_and_values_ref->{'TIMESTAMP'} = $sample_end_time;
      $metric_val = 0 if $metric_val eq "";
      $ts_and_values_ref->{$metric_id} = $metric_val;
    }
  }
}


# Subroutine: print_metric_data
# 
# Returns: metric columns seperated by delimiter "|" and
#          rows separated by new line.
#          For example:
#           1277458886000|3.40|48.81|4.76|197.02|1.42
#           1277458946000|2.42|38.51|0.85|91.50|0.00
#
sub print_metric_data
{
  my $res    = "";
  my $i=0;

  # Read the rows in ascending order of time stamps
  for($i=$#metric_result;$i>=0;$i--)
  {
    my $hash_ref = $metric_result[$i];

    my $time_stamp       = EmergPerfUtil::trim($hash_ref->{'TIMESTAMP'});
    my $run_queue_length = EmergPerfUtil::trim($hash_ref->{$metric_ids[3]});
    my $db_host_cpu      = EmergPerfUtil::trim($hash_ref->{$metric_ids[2]})/100;
    my $background_cpu   = EmergPerfUtil::trim($hash_ref->{$metric_ids[1]})/100;
    my $foreground_cpu   = EmergPerfUtil::trim($hash_ref->{$metric_ids[0]})/100;

    my $non_db_host_cpu  = ($db_host_cpu - $background_cpu - $foreground_cpu);
    $non_db_host_cpu = $db_host_cpu<0?0:$db_host_cpu;

    $res .= $time_stamp."|". #col1
            sprintf("%.2f",$run_queue_length)."|".#col2
            sprintf("%.2f",$non_db_host_cpu)."|". #col3
            sprintf("%.2f",$background_cpu)."|".  #col4
            sprintf("%.2f",$foreground_cpu)."|".  #col5
            "\n"; #new row
  }

  ::EMAGENT_PERL_DEBUG("Host metrics $res");
  return $res;
}

1;
