#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/db/direct_access.pl /st_emgc_pt-12.1.0.4pg/2 2012/09/25 19:20:25 pbhogara Exp $
#
# direct_access.pl
#
# Copyright (c) 2006, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      direct_access.pl - Provide oradebug direct_access support.
#
#    DESCRIPTION
#      This file encapsulates the support for oradebug direct_access.
#
#    NOTES
#      Not every potential replacement character is listed here.  Only will the
#      special characters:
#       '$', '#'
#      be properly translated in query columns.
#
#    MODIFIED   (MM/DD/YY)
#    pbhogara    08/22/12 - use set_env_var() for setting the env variables
#    pbhogara    03/04/12 - on windows chomp does not remove ^M
#    pbhogara    03/07/11 - support 'select *' when o/p is XML
#    pbhogara    02/15/11 - print TZ and local time before spawning sqlplus
#    pbhogara    04/14/10 - Support simple predicate based filtering
#    pbhogara    05/20/09 - enable prelim mode
#    jsoule      05/01/07 - XbranchMerge jsoule_bug-5975841 from st_emdbsa_11.1
#    jsoule      11/30/06 - avoid open("-|") on NT
#    jsoule      11/16/06 - use new direct_access syntax/output format
#    jsoule      10/04/06 - better tracing
#    jsoule      09/15/06 - Creation
#


use vars qw($NT);
use strict;
use XML::Parser;

require "emd_common.pl";
require "db/db_common.pl";
require "db/emergPerfUtil.pl";

package DirectAccess;

################################
## context attributes
################################

our $DISABLE_TRACE       = "disable";
our $ENABLE_TRACE        = "enable";
our $SAFE_MODE           = "safe";
our $UNSAFE_MODE         = "unsafe";
our $OUTPUT_XML          = "text/xml";
our $OUTPUT_CSV          = "text/csv";
our $OUTPUT_TEXT         = "text";

my $oracle_home;
my $oracle_sid;
my $connect_string;
my $enable_prelim = 1; # default is true
my $target_version;    # database version

my $trace_option      = $ENABLE_TRACE; # default
my $mode              = $SAFE_MODE;
my $content_type      = undef;

my $fixed_indx_supp   = 0;
my $csv_support       = 0;

#
# Subroutine: setContextAttributes
#  $_[0] => oracle_home
#  $_[1] => oracle_sid
#  $_[2] => connect string (should be as SYSDBA)
#  $_[3] => prelim/non-prelim
#
sub setConnectionContext
{
  $oracle_home    = shift;
  $oracle_sid     = shift;
  $connect_string = shift;
  $target_version = shift;
  $enable_prelim  = shift;

  # check if the target version >= 11.2.0.2
  if(::compareVer($target_version,"11.2.0.2.0")>=0)
  {
    $fixed_indx_supp = 1;
    $csv_support     = 1;
  }

  # Default to prelim
  $enable_prelim = 1 if !(defined $enable_prelim);

  ::EMAGENT_PERL_DEBUG("oracle_home: $oracle_home");
  ::EMAGENT_PERL_DEBUG("oracle_sid: $oracle_sid");
}


#
# Subroutine: setQueryContext
#  $_[0] => mode(safe/unsafe) 
#  $_[1] => disable/enable uts trace 
#  $_[2] => query output format(text,XML,CSV) 
#
sub setQueryContext
{
  $mode              = shift;
  $trace_option      = shift;
  $content_type      = shift;

  ::EMAGENT_PERL_DEBUG(" setting $mode mode ");
  ::EMAGENT_PERL_DEBUG(" setting $trace_option trace option ");
  ::EMAGENT_PERL_DEBUG(" setting $content_type content type ");
}



################################
## parsing state
################################

my $fixed_table;
my @fixed_table_rows;
my @fixed_table_colnames;
my @fixed_table_predicates;
my $fixed_table_colname;
my $fixed_table_colval;
my $filterRow = 1;

our %RELATIONAL_OP =( EQUAL_TO                  => "==",
                      LESS_THAN                 => "<",
                      GREATER_THAN              => ">",
                      LESS_THAN_OR_EQUAL_TO     => "<=",
                      GREATER_THAN_OR_EQUAL_TO  => ">=",
                      NOT_EQUAL_TO              => "!=");

our %LOGICAL_OP = (AND => 0,
                   OR  => 1);

my $logical_op = $LOGICAL_OP{AND};

################################
## NT XML output file
################################

my $outfile_name;

################################
## public variable for error code
################################

our $ERROR_CODE;  # error codes are as follows:
                  #     0  - no error
                  #   750  - unable to spawn SQL*Plus
                  #   942  - unsupported fixed table
                  #  1017  - invalid SYSDBA username/password
                  #  15657 - column does not have a fixed index defined

#
# Subroutine: parseXML
#  $_[0] => fixed table name
#  $_[1] => filehandle for reading
#  $_[2] => reference to list of fixed table column names to track.
#           If this array contains just one element -  ('*') then 
#           this method returns all columns in the table.
#  $_[3] => reference to list of predicates to be applied on columns
#           predicates are of the form (<operand1><operator><operand2>)
#  $_[4] => logical 0/1(AND/OR) to applied on predicates
#
# Returns: An array of hashes, one hash per row of the fixed table.
#          Each hash is an associative array of [column_name]->[column value]
#
sub parseXML
{
  $fixed_table                   = shift;
  my $output_stream              = shift;
  my $fixed_table_colnames_ref   = shift;
  my $fixed_table_predicates_ref = shift;
  $logical_op = shift;

  @fixed_table_colnames   = @$fixed_table_colnames_ref;
  @fixed_table_predicates = @$fixed_table_predicates_ref if defined $fixed_table_predicates_ref;

  my $parser = new XML::Parser(ErrorContext => 2);

  $parser->setHandlers(Start => \&start_handler,
                       End   => \&end_handler,
                       Char  => \&char_handler);

  $parser->parse($output_stream);

  return @fixed_table_rows;
}

# 
# Subroutine: parseCSV
#
# Same as parseXML. The only difference is that this
# function parses CSV and the columns names array CANNOT
# have "*".
#
sub parseCSV
{
  $fixed_table                   = shift;
  my $output_stream              = shift;
  my $fixed_table_colnames_ref   = shift;
  my $fixed_table_predicates_ref = shift;
  $logical_op = shift;

  @fixed_table_colnames   = @$fixed_table_colnames_ref;
  @fixed_table_predicates = @$fixed_table_predicates_ref if defined $fixed_table_predicates_ref;

  @fixed_table_rows = ();
  while(my $line = <$output_stream>)
  {
    if($line =~ /DIRECT_ACCESS/ )
    {
      next;
    }
    my %row = ();
    my @vals = split(/,/,$line);
    for(my $i=0;$i<=$#fixed_table_colnames;$i++)
    {
      my $val = (EmergPerfUtil::trim($vals[$i]) eq '""') ? "":(EmergPerfUtil::trim($vals[$i]));
      chomp $val;
      # on windows chomp may not work
      $val = EmergPerfUtil::trim_new_line($val); 

      $row{$fixed_table_colnames[$i]} = $val;
    }

    my $filter_row =  !defined $fixed_table_predicates_ref
                      ?1
                      :apply_predicates(\%row,
                                       $fixed_table_predicates_ref,
                                       $logical_op);
    if($filter_row > 0)
    {
      push(@fixed_table_rows,\%row);
    }
  } 
}


#
# Subroutine: parseXMLFailure
#
# The last thing this process does after parse failure.
#
sub parseXMLFailure
{
  if (defined($outfile_name) && open(DIRECTACCESS_XML, "$outfile_name"))
  {
    while (<DIRECTACCESS_XML>) { ::EMD_PERL_ERROR($_); }
    close(DIRECTACCESS_XML);
  }
}

#
# Subroutine: getFixedTable
#  $_[0] => fixed table name
#  $_[1] => ref to list of fixed table column names to supply (projection list)
#           When the output format is XML then the column names array can have
#           single element "*" which will return all columns in the table.
#           Please not that this is not supported for CSV output.
#  $_[3] => ref to list of predicates to be applied on columns.
#           predicates are of the form (<operand1><operator><operand2>).
#           These predicates are applied after direct_access result is obtained.
#  $_[4] => logical operator 0/1(AND/OR) to be applied on all
#           predicates. Defaults to AND.
#  $_[5] => Fixed index predicate - 11.2.0.2 onwards direct_access supports
#           single predicate on fixed index which is evaluated in the callback.
#
# Returns: see parseXML
#
# Note: a) Set the context attributes first.
#       b) Set the query context before calling this if default 
#          behaviour needs to be overridden
#       c) Make sure that the column names included in predicate list are
#          part of projection list also(TODO: Need to improve this later).
#       d) All predicates are logically combined based on the logical
#          operator provided. Supports AND & OR.
#          Only one operator(AND/OR) can be applied on all predicates at a time.
#          Mixed expressions are not supported.
#
sub getFixedTable
{
  $fixed_table                   = shift;
  my $fixed_table_colnames_ref   = shift;
  my $fixed_table_predicates_ref = shift;
  $logical_op                    = shift;
  my $fixed_index_pred           = shift;

  @fixed_table_colnames = @$fixed_table_colnames_ref;

  if(defined $fixed_table_predicates_ref)
  {
    @fixed_table_predicates = @$fixed_table_predicates_ref;
  }
  $logical_op   = $LOGICAL_OP{AND} if (!(defined $logical_op) || $logical_op eq "");

  my $prelim_mode = $enable_prelim?"-prelim":"";

  $ERROR_CODE   = 0;

  ################################
  ## establish the "start parsing" prompt
  ################################
  my $prompt = "direct_access XML";

  ################################
  # create a temporary file with the script
  ################################
  my $direct_access_stmt =
   "select ".join(', ', @fixed_table_colnames)." from $fixed_table";

  # From 11.2.0.2 onwards we have support for single predicate
  # on fixed index. 
  if(defined $fixed_index_pred && $fixed_indx_supp)
  {
    my @parr = split(/(==)/,$fixed_index_pred);
    my $opr1  = EmergPerfUtil::trim($parr[0]);
    my $opr2  = EmergPerfUtil::trim($parr[2]);
    $direct_access_stmt .= " where $opr1"."=".$opr2;
  }

  ::EMAGENT_PERL_DEBUG("direct_access statement: $direct_access_stmt");
  my ($sqlfile_handle, $sqlfile_name) = ::create_temp_file(".sql");

  if(!(defined $content_type))
  {
    if($csv_support)
    {
      $content_type = $OUTPUT_CSV; # force CSV. This improves response time
    }
    else
    {
      $content_type = $OUTPUT_XML;
    } 
  } 

  print $sqlfile_handle <<"EOS";
oradebug setmypid;
oradebug direct_access set content_type='$content_type';
oradebug direct_access set mode=$mode;
oradebug direct_access $trace_option trace;
EOS
  if(::compareVer($target_version,"11.2.0.2.0")>=0)
  {
     # Fix the date format for the date fields. 
     # This is used by dbIO and dbHost metrics to query 
     # endtime in x$kewmdrmv.
     print $sqlfile_handle <<"EOS";
oradebug direct_access SET NLS_DATE_FORMAT='DD-MM-YYYY HH24:MI:SS';
oradebug direct_access SET NLS_TIMESTAMP_FORMAT = 'DD-MM-YYYY HH24:MI:SSxFF';
oradebug direct_access SET NLS_TIMESTAMP_TZ_FORMAT = 'DD-MM-YYYY HH24:MI:SSxFF TZH:TZM';
EOS
  }
  print $sqlfile_handle <<"EOS";
prompt $prompt
prompt <DIRECT_ACCESS>
oradebug direct_access $direct_access_stmt;
prompt </DIRECT_ACCESS>
exit;
EOS
  close $sqlfile_handle;

  ################################
  ## spawn sqlplus, piping back the output
  ################################
  if (!$::NT)
  {
    if (!open(SQLPLUS, "-|"))
    {
      # Locally instantiate the relevant pieces of the ENV array so that
      #  SQL*Plus is spawned in the correct context.
      # Override the ORACLE_HOME, ORACLE_SID, LD_LIBRARY_PATH env variables.
      ::set_env_var($oracle_home,$oracle_sid,"TRUE");

      # log the value of TZ env variable and local time
      ::EMAGENT_PERL_DEBUG(" Value of ENV{TZ} is ".$ENV{'TZ'});
      my $now_string = localtime;
      ::EMAGENT_PERL_DEBUG(" Local time is $now_string ");

      my $exit_status =
       system("$oracle_home/bin/sqlplus $prelim_mode -S \"$connect_string\" < $sqlfile_name");

      if ($exit_status)
      {
        ################################
        ## when this fails, it may be because SQL*Plus could not be found
        ## assume it is because $oracle_home is improperly set and append
        ##   an artificial error to the output
        ## note: any true error will override this
        ################################
        ::EMAGENT_PERL_ERROR("$oracle_home/bin/sqlplus command returned status $exit_status");
        print "SP2-00750\n";
      }
      exit;
    }
  }
  else
  {
    # Locally instantiate the relevant pieces of the ENV array so that
    #  SQL*Plus is spawned in the correct context.
    # Override the ORACLE_HOME, ORACLE_SID, LD_LIBRARY_PATH env variables.
    $ENV{'ORACLE_HOME'}     = $oracle_home;
    $ENV{'ORACLE_SID'}      = $oracle_sid;
    $ENV{'LD_LIBRARY_PATH'} = $oracle_home."\\lib:".$ENV{'LD_LIBRARY_PATH'};

    # open("-|") is not permitted on NT, but it provides better diagnostics
    # so we keep it for non-NT platforms.
    $outfile_name = ::create_temp_file(".xml");
    my $exit_status =
      system("$oracle_home\\bin\\sqlplus $prelim_mode -S \"$connect_string\" < $sqlfile_name > $outfile_name");
    if ($exit_status)
    {
      ################################
      ## when this fails, it may be because SQL*Plus could not be found
      ## assume it is because $oracle_home is improperly set and append
      ##   an artificial error to the output
      ################################
      ::EMAGENT_PERL_ERROR("$oracle_home\\bin\\sqlplus command returned status $exit_status");
      print "SP2-00750\n";
      exit;
    }
    else
    {
      open(SQLPLUS, "$outfile_name");
    }
  }

  ################################
  ## peel off everything before the prompt...
  ################################
  while (<SQLPLUS>)
  {
    if (/^$prompt/)
    {
      last;
    }
    elsif (/[A-Z][0-9a-zA-Z]{2}-([0-9]+)/)
    {
      ::EMAGENT_PERL_DEBUG("Direct Access Error: $_");
      if ($1 != 1012)
      {
        ################################
        # ...but abort on the first error (except the expected 01012)
        ################################
        $ERROR_CODE = $1;
        last;
      }
    }
  }

  if (!$ERROR_CODE)
  {

    if($content_type eq $OUTPUT_XML)
    {
      ################################
      ## parse the rest as an XML document
      ################################
      $SIG{__DIE__} = \&parseXMLFailure;
      parseXML($fixed_table,
               *SQLPLUS,
               $fixed_table_colnames_ref,
               $fixed_table_predicates_ref,
               $logical_op);

      $SIG{__DIE__} = 'DEFAULT';
    }
    elsif($content_type eq $OUTPUT_CSV && $csv_support)
    {
      parseCSV($fixed_table,
               *SQLPLUS,
               $fixed_table_colnames_ref,
               $fixed_table_predicates_ref,
               $logical_op); 
    }
  }

  close(SQLPLUS);

  ################################
  ## return the parse results
  ################################
  ::EMAGENT_PERL_INFO("found ".(@fixed_table_rows + 0)." rows in $fixed_table");
  return @fixed_table_rows;
}

################################
# replacement characters
################################

my @replacements = ('x0023', 'x0024');
my @originals    = ('#',     '$');

#
# Subroutine: start_handler
#  $_[0] => expat (unused)
#  $_[1] => XML element
#
# This is a callback for parsing an XML start-element tag.
#
sub start_handler
{
  my $self = shift;
  my $el   = shift;

  my $replace;
  for ($replace = 0; $replace < @replacements; $replace++)
  {
    $el =~ s/_$replacements[$replace]_/$originals[$replace]/g;
  }

  if ($el eq "RESULT")
  {
    # initialize the array of rows
    @fixed_table_rows = qw();
  }
  elsif ($el eq "ROW")
  {
    # add a new hash for the current row
    push(@fixed_table_rows, {});
  }
  elsif ($el ne "DIRECT_ACCESS")
  {
    # set the column being parsed
    $fixed_table_colname = $el;
  }
}

#
# Subroutine: end_handler
#  $_[0] => expat (unused)
#  $_[1] => XML element
#
# This is a callback for parsing an XML end-element tag.
#
sub end_handler
{
  my $self = shift;
  my $el   = shift;

  my $replace;
  for ($replace = 0; $replace < @replacements; $replace++)
  {
    $el =~ s/_$replacements[$replace]_/$originals[$replace]/g;
  }

  if ($el eq "RESULT")
  {
    # nothing to finalize
  }
  elsif ($el eq "ROW")
  {
    my $filter_row =  defined \@fixed_table_predicates?
                      apply_predicates($fixed_table_rows[$#fixed_table_rows],
                                       \@fixed_table_predicates,
                                       $logical_op)
                      :1;
    if($filter_row <= 0)
    {
      pop(@fixed_table_rows);
    }

  }
  elsif ($el ne "DIRECT_ACCESS")
  {
    if ($el eq $fixed_table_colname)
    {
      if ((grep {/$fixed_table_colname/} @fixed_table_colnames) ||
         ($fixed_table_colnames[0] eq "*"))
      {
        # add the column name/value to this row
        $fixed_table_rows[$#fixed_table_rows]->{$fixed_table_colname} =
          $fixed_table_colval;
      }
    }
    else
    {
      # this should never be
      ::EMAGENT_PERL_ERROR("column $fixed_table_colname is inconsistent");
    }

    # reset column name/value
    $fixed_table_colname = '';
    $fixed_table_colval  = '';
  }
}

#
# Subroutine: char_handler
#  $_[0] => expat (unused)
#  $_[1] => character string
#
# This is a callback for parsing character data in an XML element.
#
sub char_handler
{
  my $self = shift;
  my $ch   = shift;

  if ($fixed_table_colname)
  {
    # append these characters
    $fixed_table_colval .= $ch;
  }
  elsif (!@fixed_table_rows)
  {
    if ($ch eq "Non existent or unsupported table")
    {
      ::EMAGENT_PERL_WARN("table or view $fixed_table does not exist");

      $ERROR_CODE = 942;
    }
    elsif ($ch =~ /ORA-(\d+)/)
    {
      ::EMAGENT_PERL_WARN("ORA error encountered: $ch");

      $ERROR_CODE = $1;
    }
  }
  else
  {
    # ::EMAGENT_PERL_WARN("processing character $ch outside of table cell");
  }
}


#
# Subroutine: apply_predicates
#  $_[0] => represents a row in the the result. It is a reference
#           to hashmap which associates "column name" => "column value"
#  $_[1] => reference to predicates to be applied on the row
#  $_[2] => logical AND/OR(0/1) operation on predicates
#
#  This subroutine helps in matching the set of predicates on a row
#  obtained from direct_access query result.
#
#  Returns: returns 1 if the column values match the predicates.
#           returns 0 if predicate match fails
#
sub apply_predicates
{
  my $row_ref = shift;
  my $predicates_ref = shift;
  $logical_op = shift;

  my %row = %$row_ref;
  my @predicates = @$predicates_ref;
  my $match_count = 0;

  if(@predicates <= 0)
  {
    # no predicates.
    return 1;
  }

  if(!(defined $logical_op))
  {
    # Default is AND
    $logical_op = $LOGICAL_OP{AND};
  }
 
  my $regexp = $RELATIONAL_OP{LESS_THAN_OR_EQUAL_TO}    . "|" .
               $RELATIONAL_OP{GREATER_THAN_OR_EQUAL_TO} . "|" .
               $RELATIONAL_OP{NOT_EQUAL_TO}             . "|" .
               $RELATIONAL_OP{LESS_THAN}                . "|" .
               $RELATIONAL_OP{GREATER_THAN}             . "|" .
               $RELATIONAL_OP{EQUAL_TO};

  for my $pred (@predicates)
  {
    # divides the predicate expression into three parts
    # <operand1>, <comparison op>, <operand2>
    my @arr = split(/($regexp)/,$pred);

    $arr[0] = uc(EmergPerfUtil::trim($arr[0])); # Operand 1
    $arr[1] = uc(EmergPerfUtil::trim($arr[1])); # Operator
    $arr[2] = uc(EmergPerfUtil::trim($arr[2])); # Operand 2

    if(!(exists ($row{$arr[0]})))
    {
      ::EMAGENT_PERL_WARN(" Column $arr[0] does not exist in the output ");
      return -1;
    }

    if(exists $row{$arr[2]})
    { $arr[2] = uc($row{$arr[2]}); }

    # Check if value is a number
    if( $arr[2] =~ /^[+-]?\d+.?\d*$/ )
    {
      $match_count++ if(_numeric_compare($row{$arr[0]},$arr[1],$arr[2]));
    }
    else
    {
      $match_count++ if(_string_compare(uc($row{$arr[0]}),$arr[1],$arr[2]));
    }

    if($match_count > 0 and
       $logical_op eq $LOGICAL_OP{OR})
    {
      return 1; # Atleast one has matched (OR)
    }
  }
  if($match_count == @predicates and
     $logical_op eq $LOGICAL_OP{AND})
  {
    return 1; # All predicates have matched (AND)
  }

  return 0;
}


#
# Subroutine: _numeric_compare
#  $_[0] => operand1
#  $_[1] => operator ( One of the operators defined in %RELATIONAL_OP )
#  $_[2] => operand2
#
#  This subroutine helps in comparing two numeric values.
#
#  Returns: returns true if comparison succeeds,
#           returns 0 otherwise.
#
sub _numeric_compare
{
  my @arr = @_;

  if( $arr[1] eq $RELATIONAL_OP{EQUAL_TO})
  { return($arr[0] == $arr[2]); }

  elsif($arr[1] eq $RELATIONAL_OP{LESS_THAN_OR_EQUAL_TO})
  { return($arr[0] <= $arr[2]); }

  elsif($arr[1] eq $RELATIONAL_OP{GREATER_THAN_OR_EQUAL_TO})
  { return($arr[0] >= $arr[2]); }

  elsif($arr[1] eq $RELATIONAL_OP{NOT_EQUAL_TO})
  { return($arr[0] != $arr[2]); }

  elsif($arr[1] eq $RELATIONAL_OP{LESS_THAN})
  { return($arr[0] < $arr[2]); }

  elsif( $arr[1] eq $RELATIONAL_OP{GREATER_THAN})
  { return($arr[0] > $arr[2]); }

  return 0;
}


#
# Subroutine: _string_compare
#  $_[0] => operand1
#  $_[1] => operator ( One of the operators defined in %RELATIONAL_OP )
#  $_[2] => operand2
#
#  This subroutine helps in comparing two string values.
#
#  Returns: returns true if comparison succeeds,
#           returns 0 otherwise.
#
sub _string_compare
{
  my @arr = @_;

  if( $arr[1] eq $RELATIONAL_OP{EQUAL_TO})
  { return($arr[0] eq $arr[2]); }

  elsif($arr[1] eq $RELATIONAL_OP{LESS_THAN_OR_EQUAL_TO})
  { return($arr[0] le $arr[2]); }

  elsif($arr[1] eq $RELATIONAL_OP{GREATER_THAN_OR_EQUAL_TO})
  { return($arr[0] ge $arr[2]); }

  elsif($arr[1] eq $RELATIONAL_OP{NOT_EQUAL_TO})
  { return($arr[0] ne $arr[2]); }

  elsif($arr[1] eq $RELATIONAL_OP{LESS_THAN})
  { return($arr[0] gt $arr[2]); }

  elsif( $arr[1] eq $RELATIONAL_OP{GREATER_THAN})
  { return($arr[0] lt $arr[2]); }

  return 0;
}

1;
