#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/db/emergPerfData.pl /main/13 2012/03/30 17:25:03 pbhogara Exp $
#
# emergPerfData.pl
# 
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      emergPerfData.pl - This script gets the emergency performance data
#                         based on the metric name. 
#
#    DESCRIPTION
#      This is a driver script which calls other perl modules to get
#      emergency performance data based on the metric name. The metric
#      data is retrieved using "oradebug direct_access".
#
#    NOTES
#      Data can be obtained in prelim mode OR non-prelim mode. But, 
#      emergency performance data is collected in prelim mode. 
#      The metric result is in XML format.
#
#    MODIFIED   (MM/DD/YY)
#    pbhogara    06/23/11 - add session details metric
#    pbhogara    02/22/11 - Set TZ env variable before metric collection
#    pbhogara    06/05/10 - Creation
# 

use strict;

require "emd_common.pl";
require "db/direct_access.pl";
require "db/sessionDetails.pl";
require "db/emergPerfUtil.pl";
require "db/oradebug_dump.pl";

use Time::Local;

my $oracle_home        = EmergPerfUtil::trim($ARGV[0]);
my $oracle_sid         = EmergPerfUtil::trim($ARGV[1]);
my $metric_name        = EmergPerfUtil::trim($ARGV[2]); 

my $use_prelim = 1; #default true
if ($#ARGV > 2)
{
  if($ARGV[3] eq "false")
  {
    $use_prelim = 0;
  }
  else
  {
    $use_prelim = 1;
  }
}

my $HANGANALYSIS      = "HANGANALYSIS";
my $ASH_WAIT_CLASS    = "ASH_WAIT_CLASS";
my $SESSION_DETAILS   = "SESSION_DETAILS";
my $CRITICAL_SESSIONS = "CRITICAL_SESSIONS";
my $DBIO              = "DBIO";
my $DBHOST            = "DBHOST";

my $metric_result     = "";

my $target_version_cat = EmergPerfUtil::trim($ENV{'EM_TARGET_VERSION_CAT'});  # version category
my $target_version     = EmergPerfUtil::trim($ENV{'EM_TARGET_VERSION'}); # version

# stdin
my %stdinArgs = get_stdinvars();
my $username  = $stdinArgs{"EM_SYSDBA_USERNAME"};
my $password  = $stdinArgs{"EM_SYSDBA_PASSWORD"};
my $tg_tzoffset = $stdinArgs{"TARGET_TIMEZONE_OFFSET"};

# environment variables
my $connect_string = $ENV{"EM_TARGET_ADDRESS"};
if ($password)
{
  $connect_string = "$username/$password as sysdba";
}
else
{
  EMAGENT_PERL_INFO("trying BEQ protocol and OS authentication");

  $connect_string = "/ as sysdba";
}

#### verfication
OradebugDump::setContextAttributes($oracle_home,
                                   $oracle_sid,
                                   $connect_string,
                                   $use_prelim,
                                   );
# Use oradebug command that dumps very little data and then verify the
# credentials
my $tracefile_name = 
         OradebugDump::generateDump(# oradebug command
                                    "oradebug dump record_callstack 1000001",$OradebugDump::modes{'SPACE_SAVING'});

if ($OradebugDump::ERROR_CODE)
{
  print("em_error= received error code $OradebugDump::ERROR_CODE");
  exit -1;
}

# 
if(compareVer($target_version,"11.2.0.2.0")>=0)
{
  ::EMAGENT_PERL_DEBUG(" Current time zone ".$ENV{'TZ'});

  # First get timezone offset of current timezone
  my @t = localtime(time);
  my $curr_gmt_offset = (timegm(@t) - timelocal(@t))*1000;
  ::EMAGENT_PERL_DEBUG(" Current time zone utc offset ".$curr_gmt_offset);
  my $currts = time * 1000;

  # get the latest ASH sample time stamp from x$kewam
  DirectAccess::setConnectionContext($oracle_home,
                                     $oracle_sid,
                                     $connect_string,
                                     $target_version,
                                     $use_prelim,
                                    );
  DirectAccess::setQueryContext($DirectAccess::UNSAFE_MODE,
                                $DirectAccess::DISABLE_TRACE);
  my @cols = ('LATEST_SAMPLE_TIME');
  my @kewam_res =  DirectAccess::getFixedTable('X$KEWAM',\@cols);
  my $latest_sample_ts = $kewam_res[0]->{'LATEST_SAMPLE_TIME'};
  ::EMAGENT_PERL_DEBUG(" latest sample time from x\$kewam".$latest_sample_ts);

  # get_epoch_time interprets the given latest time stamp to be in current time   # zone and returns the UTC epoch for it. So, we get the time difference 
  # between agent and target in the following step
  my $ag_tg_tzdiff = EmergPerfUtil::get_epoch_time($latest_sample_ts) - $currts;

  #
  # The time difference between agent and target timezones can
  # be obtained by 
  # ag_tg_tzdiff = (UTC - (ag_tzoffset)) - ( UTC - (tg_tzoffset))
  #              =     -ag_tzoffset      +      tg_tzoffset
  # tg_tzoffset = ag_tg_tzdiff + ag_tzoffset             
  #
  $tg_tzoffset = $ag_tg_tzdiff + $curr_gmt_offset;
}

if(defined($tg_tzoffset))
{
   ::EMAGENT_PERL_DEBUG("offset in millisecs $tg_tzoffset");
   my $sign = 1;
   if($tg_tzoffset < 0)
   {
     $sign = -1;
     $tg_tzoffset = -($tg_tzoffset);
   }
   # offset is in millisecs 
   my $secs  = int($tg_tzoffset / 1000);
   my $msecs = $tg_tzoffset % 1000;

   my $mins  = int($secs / 60);
   $secs  = $secs % 60;

   my $hours = int($mins / 60);
   $mins  = $mins % 60;

   if(!($hours =~ /\d\d/))
   {
      $hours = "0".$hours;
   }
   if(!($mins =~ /\d\d/)){ $mins = "0".$mins;}
   if(!($secs =~ /\d\d/)){ $secs = "0".$secs;}


   if($sign == -1)
   {
      ::EMAGENT_PERL_DEBUG("Offset is -$hours:$mins:$secs");
      $ENV{'TZ'} = "UTC+$hours:$mins:$secs";
   }
   else
   {
      ::EMAGENT_PERL_DEBUG("Offset is +$hours:$mins:$secs");
      $ENV{'TZ'} = "UTC-$hours:$mins:$secs";
   }

   ::EMAGENT_PERL_DEBUG("Setting TZ to ".$ENV{'TZ'});
}

my $begin = time;

if($metric_name eq $HANGANALYSIS)
{
  ################################
  ## HANGANALYSIS  metric
  ################################

  my $use_hanganalyze_dump = 0;
  if($target_version_cat eq "10gR203" ||
     $target_version_cat eq "10gR204" ||
     $target_version_cat eq "10gR205" )
  {
    $use_hanganalyze_dump = 1;
  }
  else
  {
    # For 11g databases use oradebug direct_access instead of hanganalyze dump
    $use_hanganalyze_dump = 0;
  }

  if($use_hanganalyze_dump)
  {
    require "db/hanganalyze.pl";
  }
  else
  {
    require "db/waitChains.pl";
  }

  my $analysisAreaOption = (defined $stdinArgs{"ANALYSIS_AREA_OPTION"})
                           ?$stdinArgs{"ANALYSIS_AREA_OPTION"}
                           :'local';

  if($use_hanganalyze_dump)
  {
    Hanganalyze::setContext($oracle_home,
                            $oracle_sid,
                            $connect_string,
                            $target_version,
                            $use_prelim,
                           );

    my $waiting_sessions_ref = Hanganalyze::identify_waitchains($analysisAreaOption);
    if ($Hanganalyze::ERROR_CODE)
    {
      EMAGENT_PERL_ERROR("received error code while querying for hanganalyze metric $Hanganalyze::ERROR_CODE");
      $metric_result = print_error($Hanganalyze::ERROR_CODE);
    }
    else
    {
      $metric_result = "em_result=".Hanganalyze::print_metric_data($waiting_sessions_ref)."|";
      $metric_result.= "\n";
    }
  }
  else  # >= 11g
  {
    WaitChains::setContext($oracle_home, 
                         $oracle_sid, 
                         $connect_string,
                         $target_version,
                         $use_prelim);

    my $waiting_sessions_ref = WaitChains::identify_waitchains($analysisAreaOption);

    if ($WaitChains::ERROR_CODE)
    {
      EMAGENT_PERL_ERROR("received error code while querying for waitchains metric $WaitChains::ERROR_CODE");
      $metric_result = print_error($WaitChains::ERROR_CODE);
    }
    else
    {
      # print metric result
      $metric_result ="em_result=". WaitChains::print_metric_data($waiting_sessions_ref)."|";
      $metric_result.= "\n";
    }
  }
}
elsif($metric_name eq $SESSION_DETAILS)
{
#  ################################
#  ## SESSION_DETAILS  metric
#  ################################

   require "db/sessionDetails.pl";

   # list of session for which details are required
   my $session_list   = $stdinArgs{"SESSIONID_LIST"};

   ::EMAGENT_PERL_DEBUG(" session list $session_list");

   my @sessions_arr = split(/#/,$session_list);

   SessionDetails::setContext($oracle_home,
                              $oracle_sid,
                              $connect_string,
                              $target_version,
                              $use_prelim);  
   SessionDetails::get_session_details(@sessions_arr); 

   $metric_result ="em_result=".SessionDetails::print_XML()."|";
   $metric_result.= "\n";
}
if($metric_name eq $ASH_WAIT_CLASS)
{

  require "db/dbAshWaitClass.pl";

  ################################
  ## ASH_WAIT_CLASS  metric
  ################################
  AshWaitClass::setContext($oracle_home,
                           $oracle_sid,
                           $connect_string,
                           $target_version,
                           $use_prelim);

  my $duration = (defined $stdinArgs{"DURATION"})
                          ?$stdinArgs{"DURATION"}
                          :60000;#millsecs - 1 min

  my $bucket_interval = (defined $stdinArgs{"BUCKET_INTERVAL"})
                                 ?$stdinArgs{"BUCKET_INTERVAL"}
                                :15000; # 15 sec

  my $dump_request_time = $stdinArgs{"METRIC_REQUEST_TIME_EPOCH"};

  # list of sessions for which we need attribute values like user,service,program,
  # module,action,client_id. 
  my $sess_list_for_att  = $stdinArgs{"SESSIONLIST_ATT"};

  # list of sessions for which we need event histogram for the ASH dump period 
  my $sess_list_for_hist = $stdinArgs{"SESSIONLIST_EVENT_HISTOGRAM"};

  my $res = AshWaitClass::getAshMetric($duration,
                                       $bucket_interval,
                                       $dump_request_time,
                                       $sess_list_for_att,
                                       $sess_list_for_hist)."|";
  if ($AshWaitClass::ERROR_CODE)
  {
    EMAGENT_PERL_ERROR("received error code while querying for ashwaitclass metric $AshWaitClass::ERROR_CODE");
    $metric_result = print_error($AshWaitClass::ERROR_CODE);
  }
  else
  {
    # print metric result
    $metric_result ="em_result=$res|\n";
  }
}

if($metric_name eq $DBHOST)
{

  ################################
  ## DBHOST metric
  ################################

  require "db/dbHost.pl";

  # Latest sample if duration is -1
  my $duration = (defined $stdinArgs{"DURATION"})
                          ?$stdinArgs{"DURATION"}
                          :-1; # Latest sample

  my $metric_request_time = $stdinArgs{"METRIC_REQUEST_TIME_EPOCH"};


  DbHost::setContext($oracle_home,
                     $oracle_sid,
                     $connect_string,
                     $target_version,
                     $use_prelim);

  DbHost::get_Host_metrics($duration,$metric_request_time);
  if ($DbHost::ERROR_CODE)
  {
    EMAGENT_PERL_ERROR("received error code while querying for dbhost metric $DbHost::ERROR_CODE");
    $metric_result = "";
  }
  else
  {
    $metric_result =DbHost::print_metric_data();
  }
}
if($metric_name eq $DBIO)
{

  ################################
  ## DBIO metric
  ################################

  require "db/dbIO.pl";

  # Latest sample if duration is -1
  my $duration = (defined $stdinArgs{"DURATION"})
                          ?$stdinArgs{"DURATION"}
                          :-1; # Latest sample

  my $metric_request_time = $stdinArgs{"METRIC_REQUEST_TIME_EPOCH"};


  DbIO::setContext($oracle_home,
                   $oracle_sid,
                   $connect_string,
                   $target_version,
                   $use_prelim);
    
  DbIO::get_IO_metrics($duration,$metric_request_time);
  if ($DbIO::ERROR_CODE)
  {
    EMAGENT_PERL_ERROR("received error code while querying for dbio metric $DbIO::ERROR_CODE");
    $metric_result = "";
  }
  else
  {
    $metric_result = DbIO::print_metric_data();
  }
} 
if($metric_name eq $CRITICAL_SESSIONS)
{
  ################################
  ## CRITICAL_SESSIONS metric
  ################################

  require "db/criticalSessions.pl";

  CriticalSessions::setContext($oracle_home,
                               $oracle_sid,
                               $connect_string,
                               $target_version,
                               $use_prelim);

  CriticalSessions::get_critical_sessions();

  $metric_result = CriticalSessions::print_metric_data();
}

my $end = time;

::EMAGENT_PERL_DEBUG(" Time taken to get metric $metric_name is ".($end-$begin)." secs");

::EMAGENT_PERL_DEBUG(" Metric result is $metric_result");
print $metric_result;

sub print_error
{
  my $error_code = shift;
  return "em_result=<emergperf_error code='".$error_code."'/>|\n";
}
