# $Header: emdb/sysman/admin/scripts/db/hanganalyze.pl /main/22 2012/03/30 17:25:04 pbhogara Exp $
#
# hanganalyze.pl
# 
# Copyright (c) 2004, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      hanganalyze.pl - Script for hang analysis
#
#    DESCRIPTION
#      This script invokes ORADEBUG HANGANALYZE and returns
#      the <blocking-session, blocked-session> pairs
#      It also identfies the chains and assigns the chain
#      id to each session in the output.
#
#    NOTES
#
#      The following conventions must be followed.
#
#      1) SYSDBA user/password are to be passed on stdin.
#         If the password is missing, that signals the script to attempt
#         OS authentication using BEQ protocol.
#      2) The TNS descriptor is passed as an environment variable.
#         This is unused if password is missing (use "/ as sysdba").
#      3) Command-line parameters are $ORACLE_HOME $ORACLE_SID [global]
#         If global is present, this gets the data for all instances.
#      
#
#    MODIFIED   (MM/DD/YY)
#    pbhogara    03/04/12 - on windows chomp does not remove ^M
#    pbhogara    11/08/10 - Remove wait_event info from output XML
#    pbhogara    04/14/10 - Updating the script to support identifying chains
#
#    rtakeish    03/28/07 - bug5940949, hanganalysis
#                           doesn't work for rac
#    jsoule      05/31/05 - get trace file properly 
#    zsyed       04/19/05 - Fixing GC new issue for GRID AGENT 
#    zsyed       02/02/05 - Fixing new break 
#    zsyed       11/22/04 - Removing references to emdw 
#    zsyed       10/27/04 - Retrieving OS pid 
#    zsyed       06/22/04 - Creation
#

use strict;

require "emd_common.pl";

require "db/oradebug_dump.pl";
require "db/emergPerfUtil.pl";

package Hanganalyze;


# A session is stored as an hash of its attribute values.
# Following is the list of attributes.
my $SID                    = "SID"; 
my $INST_ID                = "INST_ID";
my $SESS_SERIAL_NO         = "SESS_SERIAL#"; 
my $NODE_ID                = "NODE_ID";
my $IN_WAIT                = "IN_WAIT"; 
#my $WAIT_EVENT             = "WAIT_EVENT"; 
my $OSPID                  = "OSID"; 
my $CHAIN_ID               = "CHAIN_ID";
my $CHAIN_IS_CYCLE         = "CHAIN_IS_CYCLE";
my $NUM_WAITERS            = "NUM_WAITERS"; 
my $BLOCKER_SID            = "BLOCKER_SID";
my $BLOCKER_SESS_SERIAL_NO = "BLOCKER_SESS_SERIAL#";
my $BLOCKER_INST_ID        = "BLOCKER_INST_ID";
my $BLOCKER_NODE_ID        = "BLOCKER_NODE_ID";
my $BLOCKER_CHAIN_ID       = "BLOCKER_CHAIN_ID";
my $PRED_NODE_ID           = "PRED_NODE_ID";

my $ID = "ID"; # SID_SERIAL#_INSTID
my $BLOCKER_ID = "BLOCKER_ID"; # SID_SERIAL#_INSTID of blocker
#my $WAIT_EVENT_TEXT = "WAIT_EVENT_TEXT";

# Session attributes obtained from hanganalyze dump
my @SESSION_ATTS    = ($SID,
                       $INST_ID,
                       $SESS_SERIAL_NO,
                       $NODE_ID,
                       $IN_WAIT,
                       #$WAIT_EVENT_TEXT,
                       $OSPID,
                       $CHAIN_ID,
                       $CHAIN_IS_CYCLE,
                       $NUM_WAITERS,
                       $BLOCKER_SID,
                       $BLOCKER_SESS_SERIAL_NO,
                       $BLOCKER_INST_ID,
                       $BLOCKER_NODE_ID,
                       $BLOCKER_CHAIN_ID,
                       $PRED_NODE_ID,);

# Output column names
our @OUTPUT_COLUMNS = (#$WAIT_EVENT,
                       $OSPID,
                       $CHAIN_ID,
                       $CHAIN_IS_CYCLE,
                       $NUM_WAITERS,
                       $BLOCKER_ID,
                       $BLOCKER_CHAIN_ID);


our $ERROR_CODE = 0;

# Keep track of waiters behind final blockers
# (final blocker id => hash of waiters below fb) 
my %wtrs_below_fb = ();

# cycle_id => final blocker
my %cycle_fb = ();

my $target_version;

#
# Subroutine: setContext
#  $_[0] => oracle_home
#  $_[1] => oracle_sid
#  $_[2] => connect string (should be as SYSDBA)
#  $_[3] => prelim/non-prelim
#
sub setContext
{ 
  my $oracle_home    = shift;
  my $oracle_sid     = shift;
  my $connect_string = shift;
  my $target_version = shift;
  my $use_prelim     = shift;
  
  OradebugDump::setContextAttributes($oracle_home,
                                     $oracle_sid,
                                     $connect_string,
                                     $use_prelim,
                                    );
}


my $analysisAreaLocal  = 'local';
my $analysisAreaOption = $analysisAreaLocal; 

#
# Subroutine: identify_waitchains
#  $_[0] => Local/global hang analysis 
#
# Returns: Reference to a hashmap containing all sessions.
#          The key used to identify a session is "SID_SERIAL#_INSTID".
#          Each session info is represented by an hash of attribute values.
#
# Note: Set the context attributes first.
#
sub identify_waitchains
{

  $analysisAreaOption = shift;

  my $hanganalyze_local = "oradebug hanganalyze";
  my $hanganalyze_rac   = "oradebug -g def hanganalyze";
  my $oradebug_command  = $analysisAreaOption eq "local"?$hanganalyze_local:$hanganalyze_rac;
  my $cleanup_flag      = $analysisAreaOption eq "local"?$OradebugDump::modes{'SPACE_SAVING'}:
                                                         $OradebugDump::modes{'NORMAL'};


  ################################
  # Produce an ORADEBUG HANGANALYZE trace file and get its name.
  # Also record ORA- errors.
  ################################

  my $tracefile_name = OradebugDump::generateDump(# oradebug command
                                                  $oradebug_command,

                                                  # set tracefile cleanup mode
                                                  $cleanup_flag,
                                                 );
  ::EMAGENT_PERL_DEBUG("Hang Analyze trace file : ".$tracefile_name);

  if ($OradebugDump::ERROR_CODE)
  {
    ::EMAGENT_PERL_ERROR("received error code $OradebugDump::ERROR_CODE");
    $ERROR_CODE = $OradebugDump::ERROR_CODE;
    return;
  }

  # Contains all sessions participating in waitchains. Each entry in this
  # hashmap is as follows:
  # (SID_SERIAL#_INST_ID => (SID => 1,SERIAL# => 123, INST_ID => 1,OSPID => 123))
  my %sessions_in_waitchains = ();

  # number of cycles
  my $num_cycles = 0;

  # number of open chains 
  my $num_open_chains = 0;

  # Total number of waiting sessions(blocked/blockers)
  my $tot_sessions = 0;

  # position in trace file where "HANG ANALYSIS" section starts. In case of a
  # trace file with multiple entries( Ex: diag trace file in cluster wide dump)
  # this stores the position of the last one.
  my $start_pos = 0;

  my $num_open_chains = 0;

  # Sessions obtained from "Cycle" entries in Hang Analysis trace file.
  my $sessions_in_cycles_ref;

  # Sessions obtained from "State of nodes" section excluding sessions in
  # cycles.
  my $sessions_in_open_chains_ref;

  my %all_sessions = ();


  ################################
  # Get raw data from trace file
  ################################

  open(TRACE_RDR, $tracefile_name) or die "Cannot open file $tracefile_name";
  my @waitfor_lines = ();

  # Read state of nodes section and get all waiting sessions
  # Get the offset of "Hang Analysis" section in trace file through $start_pos.
  @waitfor_lines =  read_state_of_nodes_sec(*TRACE_RDR,
                                            \$start_pos);

  $tot_sessions = @waitfor_lines;

  # Set the read pointer to the line where "HANG ANALYSIS" section begins.
  seek TRACE_RDR,$start_pos,0;

  # Read the "Cycle" entries
  my @cycle_lines = read_cycle_lines(*TRACE_RDR);
 
  $num_cycles = @cycle_lines;

  if($num_cycles == 0)
  {
    ::EMAGENT_PERL_DEBUG("No cycles found");
  }
  else
  {
    ::EMAGENT_PERL_DEBUG("Number of cycles found - $num_cycles");
  }

  # We are done with reading the required data from trace file. So, close it.
  close(TRACE_RDR);

  if($cleanup_flag)
  {
    unlink $tracefile_name;
  }


  ################################
  # Process the raw data to get cycles and chains
  ################################

  # Get the sessions participating in cycles
  $sessions_in_cycles_ref = get_sessions_in_cycles(@cycle_lines);

  if(::EMAGENT_isPerlDebugEnabled())
  {
    ::EMAGENT_PERL_DEBUG("******* Printing sessions in cycles -- START *******");
    my $tempstr = tostring_hashmap_of_hashmaps($sessions_in_cycles_ref);
    ::EMAGENT_PERL_DEBUG($tempstr);
    ::EMAGENT_PERL_DEBUG("******* Printing sessions in cycles -- END   *******");
  }


  # Identify the open chains using the information read from state of nodes
  # section.
  $sessions_in_open_chains_ref = find_open_chains($sessions_in_cycles_ref,
                                                  $num_cycles,
                                                  @waitfor_lines);

  if(::EMAGENT_isPerlDebugEnabled())
  {
    ::EMAGENT_PERL_DEBUG("******* Printing sessions in open-chains -- START *******");
    my $tempstr = tostring_hashmap_of_hashmaps($sessions_in_open_chains_ref);
    ::EMAGENT_PERL_DEBUG($tempstr);
    ::EMAGENT_PERL_DEBUG("******* Printing sessions in open-chains -- END   *******");
  }


  %all_sessions = (%$sessions_in_cycles_ref,%$sessions_in_open_chains_ref);

  if(::EMAGENT_isPerlDebugEnabled())
  {
    ::EMAGENT_PERL_DEBUG("******* Printing all sessions -- START *******");
    my $tempstr = tostring_hashmap_of_hashmaps(\%all_sessions);
    ::EMAGENT_PERL_DEBUG($tempstr);
    ::EMAGENT_PERL_DEBUG("******* Printing all Sessions -- END   *******");
  }

  return \%all_sessions;
}


#
# Subroutine: read_cycle_lines
#  $_[0] => Trace file handle
#
# Returns: lines in trace file under "Cycle" section.
#
sub read_cycle_lines
{
  my $trc_rdr_ref = shift;

  my @cycle_lines = ();
  my $line = "";

  OUTER: while ($line = <$trc_rdr_ref>)
  {
    # Cycle entries are followed by "Open chains found" section.
    # But, to be safe check for "State of nodes" section also.
    if($line =~ m/Open chains found/ || $line =~ m/State of nodes/)
    {
      last;
    }
    if($line =~ m{Cycle\s*\d+\s*:\s*
                  <cnode/sid/sess_srno/ospid/wait_event>\s*:}x)
    {

      # Cycle entries are as follows:
      # Cycle 1 : <cnode/sid/sess_srno/ospid/wait_event> :
      #       <0/140/23984/26911/enq: TM - contention>
      #    -- <0/124/7974/26863/enq: TM - contention>

      # Read each session in the cycle and combine all of them into one
      # line like this:
      # <0/140/23984/26911/enq: TM - contention> -- <0/124/7974/26863/enq: TM - contention>

      my $pos = 0;
      my $cycle_line = "";
      while($line = <$trc_rdr_ref>)
      {
        chomp $line; # Remove the new line

        # on windows chomp may not work
        $line = EmergPerfUtil::trim_new_line($line);

        # remove spaces at the begining and end of line
        $line = EmergPerfUtil::trim($line);

        # Check for end of "Cycle" entry
        if($line eq ""                   ||
           $line =~ m/Open chains found/ || 
           $line =~ m/State of nodes/)
        {
          if($cycle_line ne "")
          {
             push @cycle_lines, $cycle_line;
          }

          next OUTER;
        }

        # Check for start of new "Cycle" entry.
        if($line =~ m{Cycle\s*\d+\s*:\s*
                      <cnode/sid/sess_srno/ospid/wait_event>\s*:}x)
        {
          push @cycle_lines, $cycle_line; # save the existing cycle info

          # Go back to the "Cycle" line just read and continue the outer loop
          # for processing the new cycle.
          seek $trc_rdr_ref, $pos, 0;

          next OUTER;
        }
        else
        {
          $cycle_line = $cycle_line . $line;
        }
        $pos = tell $trc_rdr_ref; 
      }
    }
  }
  return @cycle_lines;
}


#
# Subroutine: read_state_of_nodes_sec
#  $_[0] => Trace file handle
#
# Returns: lines in state of nodes section
#
sub read_state_of_nodes_sec
{
  my $trace_rdr_ref = shift;
  my $start_pos_ref = shift;

  my $check_end = 0;
  my @waitfor_lines = ();

  # Useful, when there are multiple hang analysis dump sections in the trace
  # file. For example: global hanganalyze dump.
  my @tmp_waitfor_lines = ();

  my $temp_start_pos = 0;  # to save the start of Hang Analysis section.

  my $line = "";

  # Read "state of nodes" section
  while (<$trace_rdr_ref>)
  {
    while ($line = <$trace_rdr_ref>)
    {
      if ($line =~ /State of nodes/)
      {
        # keep old waitfor_line
        @tmp_waitfor_lines = @waitfor_lines;

        # clear old waitfor_line to get new one
        @waitfor_lines = ();

        # skip all lines upto and including this one
        last;
      }
      if($line =~ /HANG ANALYSIS:/)
      {
        # Start of Hang Analysis section
        $temp_start_pos = tell $trace_rdr_ref;
        $check_end = 0;
      }
    }
    while ($line = <$trace_rdr_ref>)
    {
      if ($line =~ /END OF HANG ANALYSIS/)
      {
        $check_end = 1;

        ${$start_pos_ref} = $temp_start_pos;

        # back out the one before this match...
        if ( $analysisAreaOption eq $analysisAreaLocal )
        {
          $#waitfor_lines--;
        }
        else
        {
          $#waitfor_lines = $#waitfor_lines - 3;
        }

        # ...and break
        last;
      }
      else
      {
        # this is a content line; take off the trailing '\n'...
        chop $line;

        # ...and add it to the list
        $waitfor_lines[++$#waitfor_lines] = $line;
      }
    }
  }

  # If "END OF HANG ANALYSIS" is not found before end of file,
  # back to the last waitfor_lines
  if ( $check_end == 0 )
  {
    @waitfor_lines = @tmp_waitfor_lines;
  }

  return @waitfor_lines;
}


#
# Subroutine: get_sessions_in_cycles
#  $_[*] => Cycle lines from trace file
#
# Returns: Hashmap of sessions involved in cycles. Each session entry in the
#          hashmap is an hashmap containing session details. (SID_SERIAL#_INSTID)
#          is used as the hashmap key.
# 
# This function numbers each cycle and stores this value in CHAIN_ID field of
# each session in the cycle. NUM_WAITERS on a session is (chain_length - 1). 
# CHAIN_IS_CYCLE field is set to 1 for all the sessions,
#
sub get_sessions_in_cycles
{
  my @cycle_lines = @_;

  my $num_cycles = 0;
  my %sess_in_cycles = ();


  for(my $it1 = 0;$it1 <= $#cycle_lines; $it1++)
  {
    my $line = $cycle_lines[$it1];
    my @arr = split(/--/,$line);

    my $cycle_length = @arr;
    my @cycle_trace = ();
    my $fb_id = "";

    my $prev_sess_ref;
    my $first_sess_ref;
    my $last_sess_ref;

    for(my $it2 = 0; $it2 <= $#arr; $it2++)
    {
      $arr[$it2] = EmergPerfUtil::trim($arr[$it2]);
      my %sess_info = ();

      # Each session is of the form <cnode/sid/sess_srno/ospid/wait_event>
      # extract the information and build a hash array.

      $arr[$it2] =~ s/^<//;  # Remove the first "<"
      $arr[$it2] =~ s/>$//;  # Remove the last ">"

      my @atts = split(/\//,$arr[$it2]);

      $sess_info{$INST_ID} = $atts[0]+1;  # Add 1 to cnode to get inst id
      $sess_info{$SID} = $atts[1];
      $sess_info{$SESS_SERIAL_NO} = $atts[2];

      my $unique_sid = $sess_info{$SID}."_".
                       $sess_info{$SESS_SERIAL_NO}."_".
                       $sess_info{$INST_ID};

      if((scalar @arr) == 4)
      {
        # In RAC env, remote node session entries don't have
        # ospid information.
        $sess_info{$OSPID}      = "Unavailable";
        #$sess_info{$WAIT_EVENT} = $atts[3];
      }
      else
      {
        $sess_info{$OSPID} = $atts[3];
        #$sess_info{$WAIT_EVENT} = $atts[4];
      }

      $sess_info{$CHAIN_IS_CYCLE} = 1;
      $sess_info{$CHAIN_ID} = $it1+1;
      $sess_info{$NUM_WAITERS} = $cycle_length - 1;

      if(defined $prev_sess_ref->{$SID})
      {
        $sess_info{$BLOCKER_SID} = $prev_sess_ref->{$SID};
        $sess_info{$BLOCKER_SESS_SERIAL_NO} = $prev_sess_ref->{$SESS_SERIAL_NO};
        $sess_info{$BLOCKER_INST_ID} = $prev_sess_ref->{$INST_ID};

        push(@cycle_trace,$unique_sid);
      }
      else
      {
        # This is the designated final blocker for
        # this cycle, used for displaying in UI.
        $fb_id = $unique_sid;

        ::EMAGENT_PERL_DEBUG(" $fb_id is designted fb for cycle ".
                               $sess_info{$CHAIN_ID});

        $cycle_fb{$sess_info{$CHAIN_ID}} = $fb_id;

        $sess_info{$BLOCKER_SID} = "";
        $sess_info{$BLOCKER_SESS_SERIAL_NO} = "";
        $sess_info{$BLOCKER_INST_ID} = "";
      }


      $sess_in_cycles{$unique_sid} = \%sess_info;

      $prev_sess_ref = \%sess_info;

      if($it2 == 0)
      {
        # Save the reference to the first node session
        $first_sess_ref = \%sess_info;
      }
      if($it2 == $#arr)
      {
        # Save the reference to the last session entry
        $last_sess_ref = \%sess_info;
      }
    }

    # Last session is the blocker of first session because
    # this is a cycle.
    $first_sess_ref->{$BLOCKER_SID} = $last_sess_ref->{$SID};
    $first_sess_ref->{$BLOCKER_SESS_SERIAL_NO} = $last_sess_ref->{$SESS_SERIAL_NO};
    $first_sess_ref->{$BLOCKER_INST_ID} = $last_sess_ref->{$BLOCKER_INST_ID};

    my %wtrs = ();
    #my $non_critical_sess_found = 0;
    for(my $it=0;$it<=$#cycle_trace;$it++)
    {
      if($it<=1)  
      {
        # keep the first two waiters
        $wtrs{$cycle_trace[$it]} = 1;

         ::EMAGENT_PERL_DEBUG(" Adding $cycle_trace[$it] below $fb_id");

    #    if(!is_critical($cycle_trace[$it]))
    #    {
    #      $non_critical_sess_found = 1;
    #    }
      }
    }
    #  else
    #  {
        # If any of the first two waiters are critical then 
        # find the next non-critical session
    #    if($non_critical_sess_found)
    #    {
    #      last;
    #    }
    #    elsif(!is_critical($cycle_trace[$it]))
    #    {
    #      push(@wtrs,$cycle_trace[$it]);
    #    }
    #  }
    $wtrs_below_fb{$fb_id} = \%wtrs;
  }
  return \%sess_in_cycles;
}


#
# Subroutine: find_open_chains
#  $_[0] => Sessions in cycles
#  $_[1] => Number of chains found until now
#  $_[*] => Waitfor lines from "State of nodes" section
#
# Returns: Hashmap of sessions participating in open chains.
#          Each session entry in the hashmap is a hashmap containing session
#          details. (SID_SERIAL#_INSTID) is used as the hashmap key.
#
# This routine numbers each chain and stores this value in CHAIN_ID field of
# each session hashmap. It also calculates the NUM_WAITERS(total number
# of direct/indirect waiters) on a session.
#
# Suppose, we have a chain C1 with sessions (S1,S2,S3,S4) and chain C2 with
# sessions (S7,S8,S2,S3,S4). C2 intersects C1 at S2. Therefore, CHAIN_ID attribute 
# of S1,S2,S3,S4 is set to 1 and S7,S8 is set to 2. The common part of chain is 
# captured by setting BLOCKER_CHAIN_ID of S8 to 2.
#
sub find_open_chains
{


  my $sess_in_cycles_ref = shift;
  my $num_chain_index = shift;
  my @waitfor_lines = @_;

  my %sess_in_open_chains = ();
  my %nodeid_sessid_map = ();

  ################################
  # Parse the waitfor lines and get the session info.
  ################################
  my $i = 0;
  my $nodenumcol = 0;
  my $cnodecol = -1;
  my $sidcol = 0;
  my $sesssrncol = 0;
  my $sesscol = 0;
  my $adjlistcol = 0;
  my $predlistcol = 0;
  my $pidcol = 0;

  my $colname;
  my @colnames = split("/", $waitfor_lines[0]);

  foreach $colname (@colnames)
  {
    if($colname =~ /nodenum/)
    {
        $nodenumcol = $i;
    }
    elsif($colname =~ /cnode/)
    {
        $cnodecol = $i;
    }
    elsif($colname =~ /ospid/)
    {
        $pidcol = $i;
    }
    elsif($colname =~ /sid/)
    {
        $sidcol = $i;
    }
    elsif($colname =~ /adjlist/)
    {
        $adjlistcol = $i;
    }
    elsif($colname =~ /sess_srno/)
    {
        $sesssrncol = $i;
    }
    elsif($colname =~ /session/)
    {
        $sesscol = $i;
    }
    elsif($colname =~ /predecessor/)
    {
        $predlistcol = $i;
    }
    $i++;
  }

  my $count;
  my $adj_entry;
  my $ospid;
  my $instNum;
  my $nodenum;
  my @colvalues;


  for($count = 1; $count < @waitfor_lines; $count++) # Ignore header (0th entry)
  {
    my $adj_sess_node_id;
    @colvalues = split("/", $waitfor_lines[$count]);

    if (($colvalues[$adjlistcol] ne "") || ($colvalues[$predlistcol] ne "none"))
    {
      # If session is a root blocker, return "none" as adjacency list element
      if (!$colvalues[$adjlistcol])
      {
        $adj_entry = "none";
      }
      else
      {
        # If adjacency list is [12][43][34] then chose the last entry.
        # This is the edge that x$ksuse stores. TODO: Need to verify this.

        $adj_entry = $colvalues[$adjlistcol];
        while ($adj_entry =~ m/\[(\d+)\]/g)
        {
          $adj_sess_node_id = $1;
        }
      }

      # If ospid is on other node, return "Unavailable" instead of ""
      if (!$colvalues[$pidcol])
      {
        $ospid = "Unavailable";
      }
      else
      {
        $ospid = $colvalues[$pidcol];
      }

      # If cnode is null, return -1
      # If not, return instance_number = cnode + 1
      if ( $cnodecol == -1 )
      {
        $instNum = -1;
      }
      else
      {
        $instNum = $colvalues[$cnodecol] + 1;
      }

      $colvalues[$nodenumcol] =~ m/\[(\d+)\]/g;
      $nodenum = $1;

      my %sess_info = ();
      $sess_info{$SID} = $colvalues[$sidcol];
      $sess_info{$SESS_SERIAL_NO} = $colvalues[$sesssrncol];
      $sess_info{$INST_ID} = $instNum;
      $sess_info{$NODE_ID} = $nodenum;
      $sess_info{$OSPID} = $ospid;
      #$sess_info{$WAIT_EVENT} = "";
      $sess_info{$CHAIN_IS_CYCLE} = 0;
      $sess_info{$CHAIN_ID} = "";
      $sess_info{$BLOCKER_SID} = "";
      $sess_info{$BLOCKER_SESS_SERIAL_NO} = "";
      $sess_info{$BLOCKER_INST_ID} = "";
      $sess_info{$BLOCKER_NODE_ID} = $adj_sess_node_id;
      $sess_info{$BLOCKER_CHAIN_ID} = "";
      $sess_info{$PRED_NODE_ID}=$colvalues[$predlistcol];

      my $unique_sid = $sess_info{$SID}."_".
                       $sess_info{$SESS_SERIAL_NO}."_".
                       $sess_info{$INST_ID};

      # Add to open chains only if the session was not found in cycles section.
      if(!(exists $sess_in_cycles_ref->{$unique_sid}))
      {
        $sess_in_open_chains{$unique_sid} = \%sess_info;
      }

      # Blockers are referred to by node id in adjacency list.
      # Create a map of (nodeid => session id) to be used later.
      $nodeid_sessid_map{$nodenum} = $unique_sid;
   }
 }

  ::EMAGENT_PERL_DEBUG("Identifying open chains");

  ################################
  # Identify open chains by traversing through adj entry.
  ################################
  my $num_open_chains = 0;


  OUTER:for my $nodeid(keys %sess_in_open_chains)
  {

    my $ch_id        = $sess_in_open_chains{$nodeid}->{$CHAIN_ID};
    my $blkr_id      = $sess_in_open_chains{$nodeid}->{$BLOCKER_NODE_ID}; 
    my $pred_node_id = $sess_in_open_chains{$nodeid}->{$PRED_NODE_ID};
    my $pred_blkr_id = $sess_in_open_chains{$nodeid_sessid_map{$pred_node_id}}->{$BLOCKER_NODE_ID};

    # validate if this node's predecessor's blocker is this node itself. If not then
    # set it to none.
    if($pred_blkr_id ne $sess_in_open_chains{$nodeid}->{$NODE_ID})
    {
      $sess_in_open_chains{$nodeid}->{$PRED_NODE_ID} = $pred_node_id = 'none';
    }

    # Start from waiting sessions which don't have any waiters
    if(($ch_id eq "") &&
       ($blkr_id ne "") &&
       ($pred_node_id eq 'none'))
    {
      $num_chain_index++; # Increment the overall chain index count
      $num_open_chains++; # Increment the number of open chains

      ::EMAGENT_PERL_DEBUG("Starting traversal from $nodeid - $num_chain_index");

      my $temp_sess_ref = $sess_in_open_chains{$nodeid};

      my $num_waiters = 0;
      my $final_blkr_id = "";
      my @chain_trace = ();

      while($temp_sess_ref->{$BLOCKER_NODE_ID} ne "")
      {
        my $unique_sid = $temp_sess_ref->{$SID}."_".
                         $temp_sess_ref->{$SESS_SERIAL_NO}."_".
                         $temp_sess_ref->{$INST_ID};

        push(@chain_trace,$unique_sid);

        my $blocker_nodeid = $temp_sess_ref->{$BLOCKER_NODE_ID};
        my $blocker_unique_id = $nodeid_sessid_map{$blocker_nodeid};

        ::EMAGENT_PERL_DEBUG("edge $unique_sid -- $blocker_unique_id");

        $final_blkr_id = $blocker_unique_id; # This will have final blocker id at the end

        my $blocker_sess_ref;

        if(exists $sess_in_cycles_ref->{$blocker_unique_id})
        {
          $blocker_sess_ref = $sess_in_cycles_ref->{$blocker_unique_id};
        }
        else
        {
          $blocker_sess_ref = $sess_in_open_chains{$blocker_unique_id};
        }

        # Update the current session with blocker information
        $temp_sess_ref->{$BLOCKER_SID} = $blocker_sess_ref->{$SID};
        $temp_sess_ref->{$BLOCKER_SESS_SERIAL_NO} =
                                        $blocker_sess_ref->{$SESS_SERIAL_NO};
        $temp_sess_ref->{$BLOCKER_INST_ID} = $blocker_sess_ref->{$INST_ID};
        $temp_sess_ref->{$CHAIN_ID} = $num_chain_index;
        $temp_sess_ref->{$NUM_WAITERS} = $num_waiters;

        ::EMAGENT_PERL_DEBUG(" num_waiters on $unique_sid -- $num_waiters");

        if($blocker_sess_ref->{$CHAIN_ID} ne "" ||
           $blocker_sess_ref->{$CHAIN_IS_CYCLE})
        {

          ::EMAGENT_PERL_DEBUG(" Intersecting node found ");

          # If blocker is already part of a chain/cycle then update the blocker
          # chain id. This is an intersecting chain and this blocker node
          # is the intersecting node.
          $temp_sess_ref->{$BLOCKER_CHAIN_ID} = $blocker_sess_ref->{$CHAIN_ID};

          # Add (NUM_WAITERS of temp_sess_ref + 1) to all the sessions  
          # starting from the this intersecting node.
          update_numwaiters_in_intersecting_chains(\%sess_in_open_chains,
                                                   $sess_in_cycles_ref,
                                                   $blocker_sess_ref,
                                                   ($num_waiters+1),
                                                   \@chain_trace);    
          next OUTER;
        }
        $temp_sess_ref = $blocker_sess_ref;
        $num_waiters++;
      }

      my $wtrs_ref = exists $wtrs_below_fb{$final_blkr_id}?$wtrs_below_fb{$final_blkr_id}:{};
      for(my $it=0;$it<=1;$it++)
      {
        $wtrs_ref->{$chain_trace[$#chain_trace - $it]} = 1;
        ::EMAGENT_PERL_DEBUG(" Adding $chain_trace[$#chain_trace - $it] below". 
                             " final blocker $final_blkr_id"); 
      }
      $wtrs_below_fb{$final_blkr_id} = $wtrs_ref;

      $temp_sess_ref->{$CHAIN_ID}    = $num_chain_index;
      $temp_sess_ref->{$NUM_WAITERS} = $num_waiters;
    }
   }



   if(::EMAGENT_isPerlDebugEnabled())
   {
     if($num_open_chains > 0)
     {
       ::EMAGENT_PERL_DEBUG(" Open chains found - $num_open_chains ");
     }
     else
     {
       ::EMAGENT_PERL_DEBUG(" No open chains found ");
     }
   }


   # According to the algorithm we use to identify chains there is a 
   # chance that some of the waiters/blockers are not identified as
   # as part of any chains. Filter such sessions.
   foreach my $unique_sid(keys %sess_in_open_chains)
   {
     my $sess_ref = $sess_in_open_chains{$unique_sid};

     # Filter the sessions that are not part of any chain.
     if($sess_ref->{$CHAIN_ID} eq "")
     {
       delete $sess_in_open_chains{$unique_sid};
     }
   }

   return \%sess_in_open_chains;
}



#
# Subroutine: update_numwaiters_in_intersecting_chains
#  $_[0] => reference to hashmap of sessions in open chains 
#  $_[1] => reference to hashmap of sessions in cycle
#  $_[2] => intersecting node
#  $_[3] => number of waiters to be added
#  $_[4] => trace of session found while traversal
#
#  In case of intersecting chains we add the num_waiters of final node
#  on one intersecting chain to the nodes in the other intersecting chain.
#  For example: Chain C1 - S1,S2,S3,S4,S5
#               Chain C2 - S6,S7,S2
#  Add (num waiters on S7 + 1) to S2,S3,S4,S5
#  In this example S2 is the intersecting node.
#
sub update_numwaiters_in_intersecting_chains
{
  my $sess_in_open_chains_ref = shift;
  my $sess_in_cycles_ref      = shift;
  my $inter_node_ref          = shift;
  my $num_waiters             = shift;

  my $chain_trace_ref         = shift;
  my @chain_trace = @$chain_trace_ref;

  my %visited = (); # to keep track of visiting cycle nodes

  my $unique_id = $inter_node_ref->{$SID}."_".
                  $inter_node_ref->{$SESS_SERIAL_NO}."_".
                  $inter_node_ref->{$INST_ID};

  my $final_blkr_id = "";
  my $touched_cycle = 0;
  my $cycle_length  = 0;

  ::EMAGENT_PERL_DEBUG("updating intersecting chain");

  my $temp_node_ref = $inter_node_ref;
  while(!$visited{$unique_id} && $unique_id ne "")
  {
    ::EMAGENT_PERL_DEBUG(" node - $unique_id");
    if($temp_node_ref->{$CHAIN_IS_CYCLE} == 1)
    {
      $cycle_length++;
      $touched_cycle = 1;
      $final_blkr_id = $cycle_fb{$temp_node_ref->{$CHAIN_ID}};

      #if(!$touched_cycle)
      #{
      #  push(@chain_trace,$unique_id);  # push the first node in cycle
      #  $touched_cycle = 1;
      #  $final_blkr_id = $cycle_fb{$temp_node_ref{'CHAIN_ID'}};
      #}
    }
    else
    {
      $final_blkr_id = $unique_id; # This will have the fb value at the end
      push(@chain_trace,$unique_id);
    }

    $visited{$unique_id} = 1;
    $temp_node_ref->{$NUM_WAITERS} += $num_waiters;


    if($temp_node_ref->{$BLOCKER_SID} eq "")
    {
      $unique_id = "";
    } 
    else
    {
      # Get next session in the chain
      $unique_id = $temp_node_ref->{$BLOCKER_SID}."_".
                   $temp_node_ref->{$BLOCKER_SESS_SERIAL_NO}."_".
                   $temp_node_ref->{$BLOCKER_INST_ID};
    }

    if(exists($sess_in_open_chains_ref->{$unique_id}))
    {
      $temp_node_ref = $sess_in_open_chains_ref->{$unique_id};
    }
    else
    {
      $temp_node_ref = $sess_in_cycles_ref->{$unique_id};
    }
  }

  my $wtrs_ref = exists $wtrs_below_fb{$final_blkr_id}
                 ? $wtrs_below_fb{$final_blkr_id}
                 : {};

  if($touched_cycle && $cycle_length == 2)
  {
    # Add the node that is adjacen to cycle node to
    # waiters list
    my $id = pop(@chain_trace);
    $wtrs_ref->{$id} = 1;

    $final_blkr_id = $cycle_fb{$temp_node_ref->{$CHAIN_ID}};

    ::EMAGENT_PERL_DEBUG(" Adding $id below fb $final_blkr_id");
  }
  else
  {
    pop(@chain_trace); # pop the final blocker
    for(my $it=0;$it<=1;$it++)
    {
      $wtrs_ref->{$chain_trace[$#chain_trace - $it]} = 1;

      ::EMAGENT_PERL_DEBUG(" Adding $chain_trace[$#chain_trace - $it] below". 
                           " final blocker $final_blkr_id"); 
    }
  }

  $wtrs_below_fb{$final_blkr_id} = $wtrs_ref;


  #my $non_critical_sess_found = 0;
  #for(my $it=0;$it<=$#cycle_trace;$it++)
  #{
  #  if($it<=1 && !$end_with_cycle)
  #  {
  #    # keep the first two waiters
  #    push(@wtrs,$cycle_trace[$it]);
  #   if(!is_critical($cycle_trace[$it]))
  #    {
  #      $non_critical_sess_found = 1;
  #    }
  #  }
  #  else
  #  {
      # If any of the first two waiters are critical then 
      # find the next non-critical session
  #    if($non_critical_sess_found)
  #    {
  #      last;
  #    }
  #    elsif(!is_critical($cycle_trace[$it]))
  #    {
  #      push(@wtrs,$cycle_trace[$it]);
  #    }
  #  }
  #}
}


#
# Subroutine: print_metric_data 
#  $_[0] => reference to hashmap of all sessions 
#
#  returns hanganalysis data in XML format.
#
sub print_metric_data
{
  my $all_sessions_ref = shift;

  my @fblkrs = sort {$all_sessions_ref->{$b}{$NUM_WAITERS}
                     <=> $all_sessions_ref->{$a}{$NUM_WAITERS}
                    } keys %wtrs_below_fb;

  ::EMAGENT_PERL_DEBUG("Final blockers @fblkrs ordered by num_waiters"); 

  my $xml_str = "";

  $xml_str .= "<hanganalysis_data dbversion='".$target_version."' >";

  $xml_str .= print_waiters_XML($all_sessions_ref,\@fblkrs);

  $xml_str .= print_fb_XML($all_sessions_ref,\@fblkrs);

  $xml_str .= print_all_sessions_XML($all_sessions_ref,\@fblkrs);

  $xml_str .= "</hanganalysis_data>";

  ::EMAGENT_PERL_DEBUG("hanganalyze metric -- $xml_str");

  return $xml_str;
}


#
# Subroutine: print_all_sessions_XML 
#  $_[0] => reference to hashmap of all sessions 
#
#  returns all sessions with all attributes in XML format.
#
sub print_all_sessions_XML
{

  my $all_sessions_ref = shift;
  my $fblkrs_arr_ref   = shift;

  my $xml_str = "";

  $xml_str .= "<session_details>";

#  $xml_str .= "<att_info>";
#  for(my $it=0;$it<=$#OUTPUT_COLUMNS;$it++)
#  {
    # Reserve 1,2,3 ids for sid,serialno,instid
#    $xml_str .= "<att id='".($it + 4)."' name='".lc($OUTPUT_COLUMNS[$it])."' />"; 
#  } 
#  $xml_str .= "</att_info>";

  foreach my $fblkr_id(@$fblkrs_arr_ref)
  {

    #$xml_str .= "<session id='$fblkr_id' >";
    $xml_str .= "<session id='$fblkr_id'";

    my $sess_ref = $all_sessions_ref->{$fblkr_id};

    for(my $it=0;$it<=$#OUTPUT_COLUMNS;$it++)
    {
      my $val = $sess_ref->{$OUTPUT_COLUMNS[$it]};
      if($OUTPUT_COLUMNS[$it] eq $BLOCKER_ID)
      {
        $val = $sess_ref->{$BLOCKER_SID}."_".
               $sess_ref->{$BLOCKER_SESS_SERIAL_NO}."_".
               $sess_ref->{$BLOCKER_INST_ID};
      }
      #if($OUTPUT_COLUMNS[$it] eq $WAIT_EVENT_TEXT)
      #{
      #  $val = $sess_ref->{$WAIT_EVENT};
      #}
      #$xml_str .= "<att_ref ref='."($it + 4)."' val='$val'/>";
      $xml_str .= " ".lc($OUTPUT_COLUMNS[$it])."='".EmergPerfUtil::use_XML_entityrefs($val)."'";
    }
    #$xml_str .= "</session>";
    $xml_str .= "/>";

    my $waiters_ref = $wtrs_below_fb{$fblkr_id};

    foreach my $wtrid(keys %$waiters_ref)
    {
      $sess_ref = $all_sessions_ref->{$wtrid};

      my $unique_id = $sess_ref->{$SID}."_".
                      $sess_ref->{$SESS_SERIAL_NO}."_".
                      $sess_ref->{$INST_ID}; 

      #$xml_str .= "<session id='$unique_id' >";
      $xml_str .= "<session id='$unique_id'";

      for(my $it=0;$it<=$#OUTPUT_COLUMNS;$it++)
      {
        my $val = $sess_ref->{$OUTPUT_COLUMNS[$it]};
        if($OUTPUT_COLUMNS[$it] eq $BLOCKER_ID)
        {
          $val = $sess_ref->{$BLOCKER_SID}."_".
                 $sess_ref->{$BLOCKER_SESS_SERIAL_NO}."_".
                 $sess_ref->{$BLOCKER_INST_ID};
        }
        #$xml_str .= "<att_ref ref='".($it + 4)."' val='$val'/>";
        $xml_str .= " ".lc($OUTPUT_COLUMNS[$it])."='".EmergPerfUtil::use_XML_entityrefs($val)."'";
      }
      #$xml_str .= "</session>";
      $xml_str .= "/>";
    }
  }
  $xml_str .= "</session_details>";
}


#
# Subroutine: print_fb_XML 
#  $_[0] => reference to hashmap of all sessions 
#
#  returns final blockers in XML format.
#
sub print_fb_XML
{
  my $all_sessions_ref = shift;
  my $fblkrs_arr_ref   = shift;

  my $xml_str = "";

  $xml_str .= "<top_final_blockers>";

  foreach my $fblkr_id(@$fblkrs_arr_ref) 
  {

    ::EMAGENT_PERL_DEBUG("Printing final blocker $fblkr_id");

    $xml_str .= "<fblkr ref='$fblkr_id'>";

    $xml_str .= "<waiters>";

    my $waiters_ref = $wtrs_below_fb{$fblkr_id};

    foreach my $wtrid(keys %$waiters_ref)
    {
      $xml_str .= "<sess_ref ref='$wtrid'/>";
    }    

    $xml_str .= "</waiters>";

    $xml_str .= "</fblkr>";
  }
  $xml_str .= "</top_final_blockers>";
  return $xml_str;
}


#
# Subroutine: print_waiters_XML 
#  $_[0] => reference to hashmap of all sessions 
#
#  return waiters in XML format.
#
sub print_waiters_XML
{

  my $all_sessions_ref = shift;
  my $fblkrs_arr_ref   = shift;

  my $xml_str = "";

  $xml_str .= "<top_waiters>";   

  foreach my $fblkr_id(@$fblkrs_arr_ref) 
  {

    # Get the sessions below final blocker
    my $waiters_ref = $wtrs_below_fb{$fblkr_id};

    foreach my $wtrid(keys %$waiters_ref) 
    {
      ::EMAGENT_PERL_DEBUG(" Printing top waiter $wtrid ");

      my $wtr_ref = $all_sessions_ref->{$wtrid};
  
      # start the traversal here
      $xml_str .= "<wter ref='$wtrid'>"; 

      $xml_str .= "<blkr_chain>";

      my $pos = 1;
     
      my $next_blkr_id = $all_sessions_ref->{$wtrid}{$BLOCKER_SID}."_".
                         $all_sessions_ref->{$wtrid}{$BLOCKER_SESS_SERIAL_NO}."_".
                         $all_sessions_ref->{$wtrid}{$BLOCKER_INST_ID};

      my $immed_blkr_id = $next_blkr_id; # We have only one immed blkr in 10g

      my $next_blkr_ref = $all_sessions_ref->{$next_blkr_id};

      while($next_blkr_id != $fblkr_id)
      {
        $xml_str .= "<sess_ref ref='$next_blkr_id' pos='$pos'/>";

        $next_blkr_id = $next_blkr_ref->{$BLOCKER_SID}."_".
                        $next_blkr_ref->{$BLOCKER_SESS_SERIAL_NO}."_".
                        $next_blkr_ref->{$BLOCKER_INST_ID};

        $next_blkr_ref = $all_sessions_ref->{$next_blkr_id};
        $pos++;
      }

      $xml_str .= "<sess_ref ref='$fblkr_id' pos='$pos'/>";
      $xml_str .= "</blkr_chain>";
      $xml_str .= "<immed_blkrs><sess_ref ref='$immed_blkr_id'/></immed_blkrs>";
      $xml_str .= "</wter>";
    }
  }

 $xml_str .= "</top_waiters>";
}


################################
# Functions used for debugging only.
################################

sub tostring_hashmap_of_hashmaps
{
  my $hashmap_ref = shift;
  my %hashmap = %$hashmap_ref;
  my $str = "";

  for my $key (keys %hashmap)
  {
    $str .= "( $key => ( "; 
    $str .= tostring_hash($hashmap{$key});
    $str .= ")) \n";
  }
  return $str;
}

sub tostring_hash
{
  my $h_ref = shift;
  my %h = %$h_ref;
  my $key;
  my $str = "";
  $str .= $SESSION_ATTS[0].":".$h{$SESSION_ATTS[0]};
	
  for(my $i=1;$i<=$#SESSION_ATTS;$i++)
  {
    $str .= "," . $SESSION_ATTS[$i].":".$h{$SESSION_ATTS[$i]};
  }
	
  return $str;
}

1;
