#!/usr/local/bin/perl
#
# $Header: emdb/sysman/admin/scripts/db/net/listenerUtil.pl /st_emgc_pt-12.1.0.4pg/8 2012/12/19 03:47:30 shasingh Exp $
#
# listenerUtil.pl
#
# Copyright (c) 2003, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      listenerUtil.pl - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    shasingh     12/17/12   - bug 15895724
#    tpalgudi     10/25/12   - Bug#14508395 - removing the dependcy of esaUtils
#    tpalgudi     10/16/12   - Bug#14096840 - removing the esaDbUtils dependecy
#                              as we are not using any methods of it
#    rsamaved     09/14/12   - return accurate listener types
#    mappusam     07/10/12   - bug-14221636 fix
#    prjaiswa     01/11/12  -  dyn prop fix
#    prjaiswa     01/06/12 - fix scan listener to be determined by scanName=machineName instead of port
#    prjaiswa     09/22/11 - adding method resolveAddressesFromNames
#    prjaiswa     05/24/11 - bug 12561317
#    prjaiswa     05/24/11 - bug 12319814
#    glavash      04/08/11 - add OCM shortnanme check to isSameFileSystemEntity
#    mappusam     09/16/10 - bug-9934831 fix
#    rsamaved     01/20/10 - remove internal identifier - bug 9302375
#    nasounda     12/15/09 - Bug: 9209729
#    mappusam     08/07/09 - bug-8241651 Fix
#    tpalgudi     11/13/08 - Bug#4067458 fix
#    prjaiswa     09/05/08 -
#    vivsharm     12/22/06 - ifile fix had got rolled back after strict
#                            checking for listener parameter file
#    vivsharm     07/26/07 - Backport vivsharm_bug-5714860 from main
#    mappusam     06/20/07 - Backport mappusam_bug-5948073 from main
#    mkiran       03/09/07 - 4774824: restore shared lib path
#    vivsharm     07/04/05 - for 4433810
#    vivsharm     01/26/05 - for fix of bug 4133345
#    vivsharm     12/20/04 - take care of warnings
#    rmadampa     11/29/04 - getResult to take ref to array of commands
#    usinha       09/10/04 - BugFix # 3798597 -- IFILE issue resolution
#    usinha       05/26/04 - BugFix # 3639863 -- do file symlink comparison
#    dkapoor     03/08/04 - workaround for net manager password in
#    dkapoor     01/30/04 - use quotemeta
#    dkapoor     12/18/03 - remove file in result
#    dkapoor     12/09/03 - revert NLS_LANG
#    dkapoor     12/02/03 - convert path to canonical before comparing
#    dkapoor     11/03/03 - fix listener response
#    dkapoor     10/28/03 - fix bug3221435
#    dkapoor     10/03/03 - add method to check if param has quotes
#    rasundar    09/12/03 - use own temp files for NT
#    dkapoor     05/28/03 - use upper case param names
#    dkapoor     05/14/03 - make params uc before accessing contents
#    dkapoor     05/08/03 - remove trace
#    dkapoor     05/07/03 - add host to db detail
#    dkapoor     04/23/03 - dkapoor_fix_2846571
#    dkapoor     04/23/03 - add method to run lsnrctl command
#    dkapoor     04/08/03 - Creation
#

use strict;
require 5.6.1;
require "semd_common.pl";
require has::Common;

use File::Spec::Functions;
use File::Temp qw(tempfile);

my $OSNAME = get_osType();
if ($OSNAME eq "WIN")
{
    require "db/net/winLsnrUtils.pl";
}

# Global constant for file
my $LSNR_VER_11_2 =  "11.2.0.0.0";
my $LSNR_SI_TYPE = "SI";
my $LSNR_RAC_TYPE = "RAC";
my $LSNR_SCAN_TYPE = "SCAN";
my $LSNR_SIHA_TYPE = "SIHA";
my $LSNR_HUB_TYPE = "HUB";
my $LSNR_RIM_TYPE = "RIM";
my $LSNR_ASM_TYPE = "ASM";
my $LSNR_SSL_TYPE = "SSL";
my $LSNR_UNKNOWN_TYPE = "UNKNOWN";
my $LSNR_CONTEXT = $ENV{LSNR_NAME};
my $LOG_PREFIX = "";
if($LSNR_CONTEXT)
{
  $LOG_PREFIX = "listenerUtil : $LSNR_CONTEXT :" ;
}
else
{
  $LOG_PREFIX = "listenerUtil: " ;
}


# The method takes an array of namedAdresses or tnsaliases and tnsadmin location
# Based on input it parses tnsnames.ora and look for alias one by one
# and returns array of addresses for all aliases passed as input.
# each adress in turn is again represented as an array of hashes like
# (ADDRESS_LIST =    (ADDRESS = (PROTOCOL = TCP)(HOST = abc)(PORT = 1527))    (ADDRESS = (PROTOCOL = TCP)(HOST = xyz)(PORT = 1528))  )
# Above adress is represemnted as
# (
#   {
#      PROTOCOL => TCP
#      HOST => abc
#      PORT => 1527
#    },
#    {
#      PROTOCOL => TCP,
#      HOST => xyz
#      PORT => 1528
#    },
# )

sub resolveAddressesFromNames
{
 my ( $tnsAdminDir, $namedAddresses ) = @_;
 my $addresses;
 my $tnsnamesFile;
 my $tnsnamesInfoRef;

 my $sqlnetFile;
 my $sqlnetInfo;
 my $addrValue;
 my $defaultDomain = "";
 my @outputArray;

 #print "TNSADMIN = [$tnsAdminDir]\n";

 if ( -d $tnsAdminDir )
 {
  $tnsnamesFile = $tnsAdminDir . "/tnsnames.ora";

  #print "tnsnames.ora file = [$tnsnamesFile]\n";
  if ( -e $tnsnamesFile )
  {
   $sqlnetFile = $tnsAdminDir . "/sqlnet.ora";
   $sqlnetInfo = parseOracleConfigFile($sqlnetFile);
   if ( defined $sqlnetInfo )
   {
    $defaultDomain = trim( $sqlnetInfo->{"NAMES.DEFAULT_DOMAIN"} );

    #print "setting default domain = [$defaultDomain]\n";
   }
  }

  $tnsnamesInfoRef = parseOracleConfigFile($tnsnamesFile);

  if ( defined $tnsnamesInfoRef )
  {
   foreach my $namedAddr (@$namedAddresses)
   {
    if ( $namedAddr !~ /\./ )
    {

     #Does not contain domain, add domain
     if ( $defaultDomain ne "" )
     {
      $namedAddr .= "." . $defaultDomain;

      #print "Domain added [$defaultDomain]\n";
     }
    }

    #print "Check named address [$namedAddr]\n";

    $addrValue = $tnsnamesInfoRef->{ uc($namedAddr) };
    EMD_PERL_DEBUG("  $LOG_PREFIX addrValue  = $addrValue  \n");

    #print "Value = [$addrValue]\n";
    if ( defined $addrValue )
    {
     $addresses = getAddresses($addrValue);
     push( @outputArray, $addresses );

     EMD_PERL_DEBUG(" $LOG_PREFIX addresses  = $addresses  \n");

    }    #End of if(defined $addrValue)
   }    # End of foreach $namedAddr (@$namedAddresses)
  }    # End of if(defined $tnsnamesInfo)
 }    # End of if ( -d $tnsAdminDir )
 return @outputArray;
}

# Return: an array of words separated by zero or more given chars
sub getCharsSeparatedWords
{
 my ( $paramValue, $charList ) = @_;
 my $wd;
 my @words;

 #print "paramValue =[$paramValue],  charList =[$charList] \n";
 while ( $paramValue =~ /(\s*\w+\s*)[$charList]*/g )
 {
  $wd = $1;

  #print "wd =[$wd]\n";
  $wd =~ s/^\s*|\s*$//;
  push @words, $wd if ( $wd ne "" );
 }
 return @words;
}

#Parse listener.ora file
# and return a hashtable containing
#LHS and RHS
sub parseOracleConfigFile
{
 my ($configFile) = @_;
 my ( $paramOrder, $hashtable ) = parseOracleConfigFileWithOrder($configFile);
 return $hashtable;
}

#Parse listener.ora file
# and return a hashtable containing
#LHS and RHS
sub parseOracleConfigFileWithOrder
{
 my ($configFile) = @_;
 my %hashtable;
 my @paramOrder;
 my $paramName;
 my $config_fh;
 if ( !-e $configFile )
 {

  #print "File does not exist[$configFile]\n";
  return ( @paramOrder, %hashtable );
 }
 open( $config_fh, $configFile );
 if ( !( defined $config_fh ) )
 {

  #print "File cound not be open for reading [$configFile]\n";
  return ( @paramOrder, %hashtable );
 }
 my @lines;
 my $line;
 while ( $line = <$config_fh> )
 {
  chomp($line);
  push( @lines, $line );
 }

 close $config_fh;

 my $nvElem = "";
 foreach $line (@lines)
 {
  if ( length($line) == 0
       || $line =~ /^[#]/ )    #ignore empty lines and comment lines
  {
   next;
  }
  elsif ( $line =~ /^[ |\t|)]/ )    #continued input on new line
  {
   if ( length($nvElem) == 0 )
   {

    #Eat WS(White Space) characters
    $line = EatNLPWS($line);
   }

   #Before creating or appending to NV Element check strip off any comments.
   $line =~ s/\s*\#.*//;

   #Now create/append to NV elem.
   $nvElem = $nvElem . $line;
  }
  else    # new NV Element starting here
  {

   #encountered a new parameter
   # There are 4 cases
   if ( length($nvElem) == 0 )
   {
    $line =~ s/\s*\#.*//;
    $nvElem = $nvElem . $line;
   }
   elsif ( length($nvElem) != 0 )
   {
    $paramName =
      addNLPListElement( $nvElem, \%hashtable );    # Add Parameter to Hashtable
    if ( defined $paramName )
    {
     push( @paramOrder, $paramName );
    }

    $nvElem = "";    #Clear first, before storing current line
    $line =~ s/\s*\#.*//;
    $nvElem = $nvElem . $line;
   }
  }
 }

 #$len = length($nvElem);
 if ( length($nvElem) != 0 )    # at eof, still one more parameter to read
 {
  $paramName =
    addNLPListElement( $nvElem, \%hashtable );    # Add Parameter to Hashtable
  if ( defined $paramName )
  {
   push( @paramOrder, $paramName );
  }
 }
 return ( \@paramOrder, \%hashtable );
}

#Adds an NV string to the Hashtable.
sub addNLPListElement
{
 my ( $elem, $hashtable ) = @_;
 $_ = $elem;
 my $paramName;
 /[=]/;
 if ( length($&) != 0 )
 {
  $paramName = uc( trim_lst($`) );
  $hashtable->{$paramName} = $';
 }
 return $paramName;
}

#Given a string, this method checks for the first non-whitespace
#character. If EOL(End of Line) is reached before a valid character,
#a null is returned, else the rest of the line from the non-WS
#character is returned.
sub EatNLPWS
{
 my ($str) = @_;
 if ( $str =~ /#/ )
 {
  return $';
 }
 return $str;
}

#Get the list of listeners in the specified listenersInfo hashtable
#A key is a listener is it contains any entry of type
# DESCRIPTION_LIST
# DESCRIPTION
# ADDRESS_LIST
# ADDRESS
sub getListenerNames
{
 my ($listenersInfo) = @_;
 my @listeners;
 while ( ( my $param, my $value ) = each(%$listenersInfo) )
 {
  if ( $value =~ /DESCRIPTION_LIST|DESCRIPTION|ADDRESS_LIST|ADDRESS/i )
  {
   push( @listeners, $param );
  }
 }
 return @listeners;
}

#Get addresses from a NV String
sub getAddresses
{
 my ($line) = @_;
 return getParamValueListFor( $line, "ADDRESS" );
}

#Get sids from a NV String
sub getSIDS
{
 my ($line) = @_;
 return getParamValueListFor( $line, "SID_DESC" );
}

#Gets the parameter values for a quoted string
#For example:
# Given $argument as below:
#$argument = "'oracle_home=/oraclehome,oracle_sid=sid'"
#OR, $argument = "'(oracle_home=/oraclehome,oracle_sid=sid)'"
#OR, $argument = "(oracle_home=/oraclehome,oracle_sid=sid)"
#OR replace single quote (') with double quote (").
#
# then this routine will return a Hashtable like
# (
#      oracle_home => /oraclehome,
#      oracle_sid => sid
# )
#
#Special Case:
# Given $argument as below:
#$argument = "'oracle_home,oracle_sid'"
#Or, its variations.
#
#
# then this routine will return a Hashtable like
# (
#      VALUE0=> oracle_home,
#      VALUE1=> oracle_sid,
# )
sub getParamValues
{
 my ($argument) = @_;
 if ( defined($argument) && $argument ne "" )
 {
  if ( $argument =~ /\s*\'(.*)\'\s*/ )
  {
   $argument = $1;
  }
  if ( $argument =~ /\s*\"(.*)\"\s*/ )
  {
   $argument = $1;
  }
  if ( $argument =~ /\s*\((.*)\)\s*/ )
  {
   $argument = $1;
  }
 }
 my %params;
 my @delimiters = ( ',', '=' );
 my @toknizedLiterals = tokenize( $argument, @delimiters );
 my $currWord;
 my $curPoint       = 0;
 my $paramName      = "";
 my $added          = 0;
 my $wildParamCount = 0;

 while ( ( $currWord = getNextWord( $curPoint++, @toknizedLiterals ) ) ne "" )
 {
  if ( $currWord eq "=" )
  {
   $currWord = getNextWord( $curPoint++, @toknizedLiterals );
   if ( $currWord ne "" )
   {
    $params{ uc $paramName } = $currWord;
    $paramName               = "";
    $added                   = 1;
   }
  }
  elsif ( $currWord eq "," )
  {
   if ($added)
   {
    $added = 0;
   }
   else
   {
    $params{ "VALUE" . $wildParamCount++ } = $paramName;
   }
   next;
  }
  else
  {
   $paramName = $currWord;
  }
 }
 if ( $paramName ne "" )
 {
  $params{ "VALUE" . $wildParamCount++ } = $paramName;
 }
 return \%params;
}

#Gets the parameter values for a give parameter and given NV String
#For example:
# Given $line as below:
#$line = "SID_LIST_LISTENER12 =  (SID_LIST =    (SID_DESC =
# (SID_NAME = emdw1)   (ORACLE_HOME = home2) )
# (SID_DESC = (GLOBAL_DBNAME = gdbname) (SID_NAME = orcl) (ORACLE_HOME = home1)))
# And $paramName = SID_DESC,
# then this routine will return an array of two Hashes like
# (
#   {
#      SID_NAME => emdw1,
#      ORACLE_HOME => home2
#    },
#    {
#      GLOBAL_DBNAME => gdbname,
#      SID_NAME => orcl
#      ORACLE_HOME => home1
#    },
# )
sub getParamValueListFor
{
 my ( $line, $paramName ) = @_;
 my @delimiters = ( '(', '=', ')' );
 my @toknizedLiterals = tokenize( $line, @delimiters );
 $paramName = uc($paramName);
 my $expectingParam  = 0;
 my $expectingValue  = 0;
 my $paramStartFound = 0;
 my $currWord;
 my $paramFound       = 0;
 my $paramValueMarker = 0;
 my %tempParamValue;
 my @retParamValueList;
 my $curPoint = 0;
 my $tempParamName;
 my $bError;

 while ( ( $currWord = getNextWord( $curPoint++, @toknizedLiterals ) ) ne "" )
 {
  if ( $currWord eq "(" )
  {
   $expectingParam = 1;
   $expectingValue = 0;
   if ($paramFound)
   {
    $paramFound++;
    $paramStartFound = 1;
   }
  }
  elsif ( $currWord eq ")" )
  {
   $expectingParam = 0;
   $expectingValue = 0;
   if ( $tempParamName ne "" )
   {

    # Need to add empty paramvalue
    $tempParamValue{ uc $tempParamName } = "";
    $tempParamName = "";
   }
   if ($paramFound)
   {
    $paramFound--;

    #if($paramStartFound eq 0)
    if ( !$paramFound )
    {

     #end of param found;
     push( @retParamValueList, {%tempParamValue} );
    }
    else
    {
     $paramStartFound = 0;
    }
   }
  }
  elsif ( $currWord eq "=" )
  {
   $expectingValue = 1;
   $expectingParam = 0;
  }
  else
  {
   if ($paramFound)
   {
    if ($expectingParam)
    {
     $tempParamName = $currWord;
    }
    else
    {
     $tempParamValue{ uc $tempParamName } = $currWord;
     $tempParamName = "";
    }
   }
   elsif ( $expectingParam && $paramName eq uc($currWord) )
   {
    $paramFound     = 1;
    %tempParamValue = ();
    foreach my $key ( keys %tempParamValue )
    {
     delete $tempParamValue{$key};
    }
   }
  }    #End of Else
 }
 return \@retParamValueList;
}

#This subroutine tokenizes the given line based on the specified delimiters
# For the following delimiters
# ( , =  and )
# and
#$line = "(SID_LIST =    (SID_DESC =
# (SID_NAME = emdw1)   (ORACLE_HOME = home2) )
# (SID_DESC = (GLOBAL_DBNAME = gdbname) (SID_NAME = orcl) (ORACLE_HOME = home1)))
# the return value will be and array as follows
# (
#"(", "SID_LIST", "=", "(" , "SID_DESC", "=" ,
#"(" , "SID_NAME", "=","emdw1", ")", "(" , "ORACLE_HOME", "=" , "home2" , ")", ")",
#"(", "SID_DESC", "=", "(" , "GLOBAL_DBNAME", "=" , "gdbname", ")",
#"(" , "SID_NAME", "=","orcl", ")", "(" , "ORACLE_HOME", "=" , "home1" , ")", ")", ")" )
sub tokenize
{
 my ( $line, @delimiters ) = @_;
 if ( $#delimiters == -1 )
 {
  @delimiters = ( '(', '=', ')' );
 }
 my $quotedLiteral = 0;
 my $quoteChar;
 my $currChar;
 my @parsedChars = parseStrToChars($line);
 my $curPoint    = 0;
 my @paramValue;
 my $tempLiteral;
 my $literalFound;
 my $backSlashFound;

 while ( ( $currChar = getNextWord( $curPoint++, @parsedChars ) ) ne "" )
 {
  if ( $currChar =~ /\s/ )
  {
   next;
  }
  elsif ( contains( $currChar, @delimiters ) )
  {
   push( @paramValue, $currChar );
  }
  else
  {
   $tempLiteral = $currChar;
   if ( $currChar eq "\"" || $currChar eq "'" )
   {
    $quotedLiteral = 1;
    $quoteChar     = $currChar;
   }
   else
   {
    $quotedLiteral = 0;
   }

   $backSlashFound = 0;
   $literalFound   = 0;
   while ( ( $currChar = getNextWord( $curPoint++, @parsedChars ) ) ne "" )
   {

    #On a backslash (escaped character), save the backslash and
    #following character into the literal.
    if ( $currChar eq "\\" )
    {
     $tempLiteral .= $currChar;
     $backSlashFound = 1;
     next;
    }

    if ($backSlashFound)
    {

     #don't process this char and reset the back slash found flag
     $backSlashFound = 0;
    }
    else
    {
     if ($quotedLiteral)    # literal wrapped with quotes
     {
      if ( $currChar eq $quoteChar )    # quote terminator found
      {
       $tempLiteral .= $currChar;
       $literalFound = 1;
       last;
      }
     }
     else
     {    #did we hit unescaped meta character ( ) or =
      if ( contains( $currChar, @delimiters ) )
      {

       #terminate string - do NOT increment POS, or it will
       #swallow the metacharacter into the literal
       $curPoint--;

       # if($currChar eq ")" || $currChar eq "=")
       {
        $literalFound = 1;
       }
       last;
      }
     }
    }    #End of if($backSlashFound)
    $tempLiteral .= $currChar;
   }  #End of while ( ($currChar = getNextChar($curPoint++,@parsedChars)) ne "")
   if ( $currChar eq "" )
   {
    $literalFound = 1;
   }

   #String.substring() is exclusive for end (does not include end
   # character.
   if ($literalFound)
   {
    push( @paramValue, trim_lst($tempLiteral) );
   }
  }    #else
 }    #while
 return @paramValue;
}

#parse line into characters
#for $line=(SID_NAME = emdw1)
# the return value will be and array as follows
# (
#"(", "S", "I" , "D", "_" , "N" , "A" , "M" , "E", " ", "=", " " ,"e","m","d","w","1",")"
# ).
sub parseStrToChars
{
 my ($parseStr) = @_;
 my @parsedChars;

 #  #print "In parseStrToChars\n";
 if ( defined($parseStr) && $parseStr ne "" )
 {
  while ( $parseStr =~ /(.)/g )
  {

   #    #print "[$1]\n";
   push( @parsedChars, $1 );
  }
 }
 return @parsedChars;
}

#Gets the next word in an array of words or strings
#if end of array is reached empty char "" is returned.
sub getNextWord
{
 my ( $curPoint, @parsedChars ) = @_;
 my $char = "";

 if ( $curPoint < @parsedChars )
 {
  $char = $parsedChars[$curPoint];
 }
 return $char;
}

#Strips leading and trailing spaces and returns the string
sub trim_lst
{
 my $origStr = $_[0];

 #Strip trailing and leading
 $origStr =~ s/^\s*|\s*$//g;
 return $origStr;
}

# Given the output from the services command in raw mode
# from the lsnrctl, this subroutine
# returns a hashtable for sids found and itss correponding details
# For lsnrctl services output in raw mode like:
#LSNRCTL> services (ADDRESS=(PROTOCOL=TCP)(HOST=dkapoor-pc3)(PORT=1234))
#Connecting to (ADDRESS=(PROTOCOL=TCP)(HOST=dkapoor-pc3)(PORT=1234))
#Services Summary...
#(SERVICE=(SERVICE_NAME=orclServiceName)(INSTANCE=(INSTANCE_NAME=orcloid)(NUM=1)(IN
#STANCE_STATUS=UNKNOWN)(HANDLER=(HANDLER_DISPLAY=DEDICATED SERVER)(HANDLER_INFO=L
#OCAL SERVER)(HANDLER_MAXLOAD=0)(HANDLER_LOAD=0)(ESTABLISHED=0)(REFUSED=0)(HANDLE
#R_ID=B94B489DEB84-11D6-B6FD-0002A517EED1)(PRE=any)(HANDLER_NAME=DEDICATED)(SESSI
#ON=NS)(ADDRESS=(PROTOCOL=beq)(PROGRAM=extproc)(ENVS='ORACLE_HOME=d:\oracle92\test
#,ORACLE_SID=orcloid')(ARGV0=extprocPLSExtProc)(ARGS='(LOCAL=NO)')))(NUMREL=
#1)))\n"
#(SERVICE=(SERVICE_NAME=orcl92.company.com)(INSTANCE=(INSTANCE_NAME=orcl92)(NUM
#=1)(INSTANCE_STATUS=UNKNOWN)(HANDLER=(HANDLER_DISPLAY=DEDICATED SERVER)(HANDLER_
#INFO=LOCAL SERVER)(HANDLER_MAXLOAD=0)(HANDLER_LOAD=0)(ESTABLISHED=0)(REFUSED=0)(
#HANDLER_ID=B94B489EEB84-11D6-B6FD-0002A517EED1)(PRE=any)(HANDLER_NAME=DEDICATED)
#(SESSION=NS)(ADDRESS=(PROTOCOL=beq)(PROGRAM=oracle)(ENVS='ORACLE_HOME=d:\oracle9
#2\ora92,ORACLE_SID=orcl92')(ARGV0='oracleorcl92')(ARGS='(LOCAL=NO)')))(NUMREL=1)))
#
# Following hashtable is returned:
# (
#   orcl92 =>
#   {
#      SERVICE_NAME => orcl92.company.com,
#      ORACLE_HOME => d:\oracle92\ora92,
#      PORT => 1234 (obtained from the first address description)
#    },
#   orcloid =>
#    {
#      SERVICE_NAME => orclServiceName,
#      ORACLE_HOME => d:\oracle92\test,
#      PORT => 1234
#    },
# )
#
#Assuptions:
# 0. The entry for INSTANCE_NAME = *extproc* (ignoring case) is not returned.
# 1. If no ORACLE_SID or ORACLE_HOME found , the entry is not returned
#
sub getDBDetails
{
 my ( $result, $port, $lsnrHost, @dynamicDiscoveredSids ) = @_;
 my %dbDetails;

 if ( $result !~ /(ORACLE_HOME)|(ORACLE_SID)/i
      || ( !defined $port || $port eq "" ) )
 {
  return \%dbDetails;
 }
 my @resultArray = split /\n/, $result;

 foreach my $line (@resultArray)
 {
  if ( $line !~ /^\s*\(SERVICE/i )
  {
   next;
  }
  if ( $line !~ /(ORACLE_HOME)|(ORACLE_SID)/i )
  {
   next;
  }
  my $serviceName;
  my $oracleHome;
  my $oracleSid;
  my $instanceName;
  if ( $line =~ /\(ENVS='.*ORACLE_SID=(.*?)(,|')/i )
  {
   $oracleSid = convertSIDForOS($1);
  }
  if ( $oracleSid =~ /extproc/i )
  {

   #filter out sids containing with extproc
   next;
  }
  if ( contains( $oracleSid, @dynamicDiscoveredSids ) )
  {
   next;
  }
  if ( $line =~ /\(ENVS='.*ORACLE_HOME=(.*?)(,|')/i )
  {
   $oracleHome = $1;
  }
  if ( $line =~ /\(SERVICE_NAME=(.*?)\)/i )
  {
   $serviceName = $1;
  }
  if (    defined $oracleSid
       && $oracleSid ne ""
       && defined $oracleHome
       && $oracleHome ne "" )
  {
   my $sidDetails = {
                      ORACLE_HOME  => $oracleHome,
                      SERVICE_NAME => $serviceName,
                      PORT         => $port,
                      HOST         => $lsnrHost
   };
   $dbDetails{$oracleSid} = $sidDetails;
  }
 }
 return \%dbDetails;
}

#Returns the listener endpoint string
sub getListenerAddresses
{
 my ( $listener, $listenersInfo ) = @_;
 my $addressesStr;
 while ( ( my $param, my $value ) = each(%$listenersInfo) )
 {
  if ( $param eq $listener )
  {
   $addressesStr = $value;
   last;
  }
 }
 return getAddresses($addressesStr);
}

#Returns the listener static registered SIDs
sub getStaticSIDs
{
 my ( $listener, $listenersInfo ) = @_;
 my $sidListStr = getParamValue( $listenersInfo, "SID_LIST_" . $listener );
 return getSIDS($sidListStr);
}

#Get the array of password from the listener.ora file
sub getPasswordsFromFile
{
 my ( $listener, $listenerFile ) = @_;
 my @passwordList = ();
 if ( -e $listenerFile )
 {
  my $listenersInfo = parseOracleConfigFile($listenerFile);
  if ( defined $listenersInfo )
  {
   @passwordList = getPasswords( $listener, $listenersInfo );
  }
 }
 return @passwordList;
}

#Get the array of password for the spwcified listener
sub getPasswords
{
 my ( $listener, $listenersInfo ) = @_;
 my $passwords = getParamValue( $listenersInfo, "PASSWORDS_" . $listener );
 my $nvPairs = getParamValues($passwords);
 my @passwords;
 while ( ( my $param, my $value ) = each %$nvPairs )
 {

  #bug#3473397: Net Manager sets listener
  #password with trailing null characters ("\0").
  #need to remove that character before using it.
  if ( $value =~ /\0$/ )
  {
   $value =~ s/\0*$|$//g;
  }
  push( @passwords, $value );
 }
 return @passwords;
}

#Get the parameter value for listener
sub getParamValue
{
 my ( $listenersInfo, $parameterName ) = @_;
 return $listenersInfo->{ uc($parameterName) };
}

# unset the *LD_* env variable for proper function of lsnrctl
sub unset_lib_path_env
{
 EMD_PERL_DEBUG(" $LOG_PREFIX unset_lib_path_env called \n");
 my %envHashpMap = ();
 if (    $OSNAME eq "SOL"
      || $OSNAME eq "LNX"
      || $OSNAME eq "OSF1" )
 {

  $envHashpMap{"LD_LIBRARY_PATH"} = $ENV{LD_LIBRARY_PATH}
    if defined( $ENV{LD_LIBRARY_PATH} );
  $envHashpMap{"LD_LIBRARY_PATH_64"} = $ENV{LD_LIBRARY_PATH_64}
    if defined( $ENV{LD_LIBRARY_PATH_64} );
  delete $ENV{LD_LIBRARY_PATH};
  delete $ENV{LD_LIBRARY_PATH_64};
 }
 elsif ( $OSNAME eq "HP" )
 {
  $envHashpMap{"SHLIB_PATH"} = $ENV{SHLIB_PATH}
    if defined( $ENV{SHLIB_PATH} );
  delete $ENV{SHLIB_PATH};
 }
 elsif ( $OSNAME eq "AIX" )
 {
  $envHashpMap{"LIBPATH"} = $ENV{LIBPATH} if defined( $ENV{LIBPATH} );
  delete $ENV{LIBPATH};
 }
 elsif ( $OSNAME eq "MAC OS X" )
 {
  $envHashpMap{"DYLD_LIBRARY_PATH"} = $ENV{DYLD_LIBRARY_PATH}
    if defined( $ENV{DYLD_LIBRARY_PATH} );
  delete $ENV{DYLD_LIBRARY_PATH};
 }
 else
 {

  # Unsupported Operating System
  # Do nothing
 }
 return %envHashpMap;
}

# set *LD* Env related variables
sub revert_original_lib_path_env
{
 my %envHashpMap = @_;
 my $envKeys ="";
 while ( ( my $key, my $value ) = each %envHashpMap )
 {
  $envKeys = $envKeys."   ".$key;
  $ENV{$key} = $value;
 }
 EMD_PERL_DEBUG(" $LOG_PREFIX : revert_original_lib_path_env : $envKeys \n");
}

# The method dumps all env in debug mode : useful for triaging in customer env
sub dump_lib_path_env
{
 EMD_PERL_DEBUG("$LOG_PREFIX : dump_lib_path_env :LD_LIBRARY_PATH = $ENV{LD_LIBRARY_PATH} \n LD_LIBRARY_PATH_64 = $ENV{LD_LIBRARY_PATH_64} \n LIBPATH = $ENV{LIBPATH} \n DYLD_LIBRARY_PATH = $ENV{DYLD_LIBRARY_PATH} \n");
}

sub getRunCommandOutput
{
 EMD_PERL_DEBUG(" $LOG_PREFIX :: getRunCommandOutput");
 my ( $oracleHome, $executable, @commands ) = @_;

 $ENV{ORACLE_HOME} = $oracleHome;

 #bug 12561317
 my %bkup_path = unset_lib_path_env();

 set_lib_path($oracleHome);
 dump_lib_path_env();

 my $filename;
 my $fh;
 if ( $OSNAME eq 'WIN' )
 {
  my $TEMP = $ENV{SYSTEMDRIVE} . "\\TEMP";

  #A temp solution
  &mkDir_lst($TEMP);
  $filename = "$TEMP\\" . "net.$$";
 }
 else
 {
  ( $fh, $filename ) = tempfile( UNLINK => 1 );
 }
 my $output_string;
 my $oldNLS_LANG = $ENV{NLS_LANG};

 #Unset ORA_NLS variables if in the environment
 my $oldORA_NLS = $ENV{ORA_NLS};
 if ( defined $oldORA_NLS )
 {
  delete( $ENV{ORA_NLS} );
 }
 my $oldORA_NLS32 = $ENV{ORA_NLS32};
 if ( defined $oldORA_NLS32 )
 {
  delete( $ENV{ORA_NLS32} );
 }
 my $oldORA_NLS33 = $ENV{ORA_NLS33};
 if ( defined $oldORA_NLS33 )
 {
  delete( $ENV{ORA_NLS33} );
 }

 # Bug   12319814
 my $old_nls_date_lang = $ENV{NLS_DATE_LANGUAGE};
 if ( defined $old_nls_date_lang )
 {
  delete( $ENV{NLS_DATE_LANGUAGE} );
 }

 #  set the NLS_LANG  American always for the bug 8241651
 $ENV{NLS_LANG} = "american_america.utf8";
 if ( open( EXEC_WRITER, "|$executable > $filename" ) )
 {
  my $cmd;
  foreach $cmd (@commands)
  {
   print EXEC_WRITER "$cmd\n";
  }
  close EXEC_WRITER;
  {
   if ( open( OUT_PUT, "$filename" ) )
   {
    my @output_content = <OUT_PUT>;
    $output_string = "@output_content";
    close OUT_PUT;
   }
  }
 }
 if ( defined $oldNLS_LANG )
 {
  $ENV{NLS_LANG} = $oldNLS_LANG;
 }
 else
 {
  delete( $ENV{NLS_LANG} );
 }

 if ( defined $oldORA_NLS )
 {
  $ENV{ORA_NLS} = $oldORA_NLS;
 }
 if ( defined $oldORA_NLS32 )
 {
  $ENV{ORA_NLS32} = $oldORA_NLS32;
 }
 if ( defined $oldORA_NLS33 )
 {
  $ENV{ORA_NLS33} = $oldORA_NLS33;
 }

 if ( defined $old_nls_date_lang )
 {
  $ENV{NLS_DATE_LANGUAGE} = $old_nls_date_lang;
 }

 #close($fh);
 if ( $OSNAME eq 'WIN' )
 {
  unlink $filename;
 }

 revert_original_lib_path_env(%bkup_path);
 EMD_PERL_DEBUG(
        " $LOG_PREFIX :: getRunCommandOutput : output_string =$output_string");
 return $output_string;
}

#checks if an array contains an item
sub contains
{
 my ( $item, @array ) = @_;
 foreach my $anItem (@array)
 {
  if ( $anItem eq $item )
  {
   return 1;
   last;
  }
 }
 return 0;
}

#Check if the passed file is present in the given list of files
sub containsFile
{
 my ( $file, @listOfFiles ) = @_;
 foreach my $aFile (@listOfFiles)
 {
  if ( isSameFileSystemEntity( $aFile, $file ) )
  {
   return 1;
  }
 }
 return 0;
}

sub containsIgnoreCase
{
 my ( $item, @array ) = @_;
 my $quotedItem = quotemeta($item);
 foreach my $anItem (@array)
 {
  if ( $anItem =~ /^\s*$quotedItem\s*$/i )
  {
   return 1;
  }
 }
 return 0;
}

#Get the result of running a lsnrctl command
#if dontUsePswdFirst is set
sub getResult
{
 my ( $executable, $command, $listenerFile, $name, $dontUsePswdFirst ) = @_;
 if ( !defined $dontUsePswdFirst )
 {
  $dontUsePswdFirst = 0;
 }
 my $r;
 my @passwordList = ();

 #read the password from the file only is dontUsePwd is not set
 if ( !$dontUsePswdFirst )
 {
  @passwordList = getPasswordsFromFile( $name, $listenerFile );
 }

 my @commands;
 my $totalPswd    = @passwordList;
 my $currPswdInex = 0;
 do
 {
  @commands = ();
  my $pswd;

  #Add password if present
  if ( $currPswdInex < $totalPswd )
  {
   push( @commands, "set password " . $passwordList[ $currPswdInex++ ] );
  }

  #Adding capability to take multiple commands also
  if ( ref($command) eq "ARRAY" )
  {
   push( @commands, @{$command} );
  }
  else
  {
   push( @commands, $command );
  }
  push( @commands, "exit" );
  $r = getRunCommandOutput( $ENV{LSNR_ORACLE_HOME}, $executable, @commands );

  #if there is a password error try the next password,
  #TNS-01169: The listener has not recognized the password
  if ( $r =~ /TNS-01169/i )
  {

   #Got password error, get passwords if dontUsePswdFirst is set
   if ($dontUsePswdFirst)
   {
    @passwordList     = getPasswordsFromFile( $name, $listenerFile );
    $totalPswd        = @passwordList;
    $dontUsePswdFirst = 0;
    $currPswdInex     = 0;
   }
  }
 } while ( $r =~ /TNS-01169/i && $currPswdInex < $totalPswd );
 return $r;
}

sub getResultNew
{
 EMD_PERL_DEBUG(" $LOG_PREFIX :: getResultNew called .... ");
 my ( $executable, $command, $listenerFile, $name, $pswd ) = @_;
 my $r;
 my @commands;
 @commands = ();
 if ( !( $pswd eq "" ) )
 {
  EMD_PERL_DEBUG(" $LOG_PREFIX :: push passwrd ");
  push( @commands, "set password " . $pswd );
 }

 #Adding capability to take multiple commands also
 if ( ref($command) eq "ARRAY" )
 {
  EMD_PERL_DEBUG(" $LOG_PREFIX :: command is ARRAY ref : @{$command}");
  push( @commands, @{$command} );
 }
 else
 {
  EMD_PERL_DEBUG(" $LOG_PREFIX :: command is not ARRAY ref : $command");
  push( @commands, $command );
 }
 push( @commands, "exit" );
 $r = getRunCommandOutput( $ENV{LSNR_ORACLE_HOME}, $executable, @commands );

 EMD_PERL_DEBUG(" $LOG_PREFIX :: r = $r");
 return $r;
}

#return hostName is listener host is "localhost", empty or local ip
#127.0.0.1
sub resolveLsnrHostName
{
 my ( $lsnrHost, $hostName ) = @_;
 $lsnrHost = trim_lst($lsnrHost);
 if (    uc($lsnrHost) eq uc("localhost")
      || $lsnrHost eq ""
      || $lsnrHost eq "127.0.0.1" )
 {
  return $hostName;
 }
 return $lsnrHost;
}

#Get the default initialization file location
sub getDefaultInitFileLocation
{
 my ($OracleHome) = @_;

 my $initDir;
 if ( ( get_osType() eq 'WIN' ) )
 {
  $initDir = $OracleHome . "/database";
 }
 else
 {
  $initDir = $OracleHome . "/dbs";
 }

 return $initDir;
}

#Returns 1 if the argument is a quoted string like
# 'param1' or "param".
sub isQuoted
{
 my ($argument) = @_;
 if ( $argument =~ /\s*\'(.*)\'\s*/ || $argument =~ /\s*\"(.*)\"\s*/ )
 {
  return 1;
 }
 return 0;
}

#Returns an upper cased value if the argument is not quoated
sub convertToUcIfNotQuoted
{
 my ($argument) = @_;
 if ( !isQuoted($argument) )
 {
  if ( $argument =~ /\s*\'(.*)\'\s*/ )
  {
   $argument = $1;
  }
  if ( $argument =~ /\s*\"(.*)\"\s*/ )
  {
   $argument = $1;
  }
  $argument = uc $argument;
 }
 return $argument;
}

sub isThisListenerRunning
{
 my ( $statusResult, $listenerOraDir, $name ) = @_;

 my $line;
 my $nameMatches = 0;

 #due to IFILE entries , the listener might not o/p parameter file entries
 #for such cases we default it to UP status

# No more keeping the DEFAULT as UP here, because in case the Listener at port 1521
# is started without a listener.ora file, we need to show that as down. At the same time
# our Alert message will say that although there is a listener at port 1521, it does not use
# any "listener.ora" file. Please make it use some "listener.ora".

# ideal fix would be to open the listener.ora and look for IFILE entry there
# but in case the listener.ora does not exist at this location then  this check will fail
# even if the listener.ora exists and we dont even have read permissions then
# also this check will fail.
# hence setting the default back to 1 - this will fix IFILE issue, but will also report
# other kind of listeners started without a listener.ora on the same host-port
 my $tnsadminMatches = 1;

 my @info                 = split( /\n/, $statusResult );
 my $quotedName           = quotemeta($name);
 my $quotedListenerOraDir = quotemeta($listenerOraDir);

 foreach $line (@info)
 {

  #match the listener Alias
  if ( $line =~ /^(.*)\s+$quotedName\s*$/i )
  {
   $nameMatches = 1;
  }

  #look for a line which contains listener.ora
  if ( $line =~ /^(.*)\s+(.*)(.)listener.ora\s*$/i )
  {

   #check if the listener ora dir is same as the one in the pattern
   if ( $line =~ /^(.*)\s+$quotedListenerOraDir(.)listener.ora\s*$/i )
   {
    $tnsadminMatches = 1;
   }
   else
   {

    #BugFix  3639863 : append "listener.ora" to allow file symlink comparison
    my $sym_file1 = $2 . "/listener.ora";
    my $sym_file2 = $listenerOraDir . "/listener.ora";
    if ( isSameFileSystemEntity( $sym_file1, $sym_file2 ) )
    {
     $tnsadminMatches = 1;
    }
    else
    {
     $tnsadminMatches = 0;
    }
   }
  }
 }
 if ( $nameMatches && $tnsadminMatches )
 {
  return 1;
 }
 return 0;
}

#Checks if the two files are same.
#In Windows, match the files.
#In Unix, match the device and inode of the files.
sub isSameFileSystemEntity
{
 my ( $file1, $file2 ) = @_;
 if ( !( get_osType() eq 'WIN' ) )
 {
  my ( $dev,  $ino )  = stat $file1;
  my ( $dev1, $ino1 ) = stat $file2;
  if ( $dev == $dev1 && $ino == $ino1 )
  {
   return 1;
  }
 }
 else
 {

  #Convert to windows slashes
  $file2 =~ s/\//\\/g;
  $file1 =~ s/\//\\/g;
  my $quotedFile2 = quotemeta($file2);
  if ( defined $ENV{LL_ORACLE_HOME} )
  {

   # compare shortened paths
   if (
        lc( Win32::GetShortPathName($file1) ) eq
        lc( Win32::GetShortPathName($file2) ) )
   {
    return 1;
   }
  }
  else
  {
   if ( $file1 =~ /^\s*$quotedFile2\s*$/i )
   {
    return 1;
   }
  }
 }
 return 0;
}

sub getErrorIfAny
{
 my ($lsnrctlResult) = @_;
 my $errorMsg;
 if ( $lsnrctlResult =~ /^.*TNS-[0-9].*/mi )
 {
  my @info = split( /\n/, $lsnrctlResult );
  my $line;
  foreach $line (@info)
  {
   if ( $line =~ /^.*TNS-[0-9].*/i )
   {
    $errorMsg = $line;
    last;
   }
  }
 }
 return $errorMsg;
}

# Create a specified directory.
# Return OK if succeed, otherwise, return NOK.
# OK is returned if the specified directory already exists.
# mkDir_lst(dirName)
sub mkDir_lst
{
 my ($dirName) = @_;
 my $dirExist = &dirExists_lst($dirName);
 if ( $dirExist eq "OK" )
 {
  return "OK";
 }

 my (@create);
 push( @create, $dirName );

 #create parent directories if necessary
 my ($parent) = dirname($dirName);
 while ( !-e "$parent" )
 {
  push( @create, $parent );
  $parent = dirname($parent);
 }

 while ( $dirName = pop(@create) )
 {
  if ( !mkdir( $dirName, 0755 ) )
  {
   ## Actually want this error to come out in the job output
   return "NOK";
  }
 }

 return "OK";
}

# Check if a specified directory exists
# Return OK if the directory exists, otherwise, return NOK.
# dirExists_lst(dirName)
sub dirExists_lst
{
 my ($dirName) = @_;
 if ( !-e "$dirName" )
 {
  return "NOK";
 }
 elsif ( !-d "$dirName" )
 {
  return "NOK";
 }

 return "OK";
}

# We work with a result in a format as below:
# ######################################################################################
# lsnrctl status '(ADDRESS=(PROTOCOL=TCP)(HOST=host)(PORT=2449))'
#
# LSNRCTL for Linux: Version 10.2.0.0.0 - Beta on 04-JUL-2005 01:37:27
#
# Copyright (c) 1991, 2012, Oracle and/or its affiliates. All rights reserved. 
#
# Connecting to (ADDRESS=(PROTOCOL=TCP)(HOST=stadm18)(PORT=2449))
# STATUS of the LISTENER
# ------------------------
# Alias                     LISTENER_NAME
# Version                   TNSLSNR for Linux: Version 10.2.0.0.0 - Beta
# Start Date                04-JUL-2005 01:36:48
# Uptime                    0 days 0 hr. 0 min. 38 sec
# Trace Level               user
# Security                  ON: Local OS Authentication
# SNMP                      OFF
# Listener Parameter File   /scratch/private/oracle/ora10g/network/admin/listener.ora
# Listener Trace File       /scratch/private/oracle/ora10g/network/trace/listener5.trc
# Listening Endpoints Summary...
#  (DESCRIPTION=(ADDRESS=(PROTOCOL=tcp)(HOST=host.domain)(PORT=2249)))
#  (DESCRIPTION=(ADDRESS=(PROTOCOL=tcp)(HOST=host.domain)(PORT=2349)))
#  (DESCRIPTION=(ADDRESS=(PROTOCOL=tcp)(HOST=host.domain)(PORT=2449)))
#  (DESCRIPTION=(ADDRESS=(PROTOCOL=tcp)(HOST=host.domain)(PORT=2549)))
# The listener supports no services
# The command completed successfully
#
# #########################################################################################

sub isAnyListenerRunning
{
 my ( $statusResult, $lsnrHost, $lsnrPort, $name ) = @_;

 my $line;
 my $nameMatches     = 0;
 my $hostPortMatches = 0;
 my @info            = split( /\n/, $statusResult );
 my $quotedName      = quotemeta($name);

 foreach $line (@info)
 {

  #match the listener Alias
  if ( $line =~ /^(.*)\s+$quotedName\s*$/i )
  {
   $nameMatches = 1;

   #print "\n NAME MATCHED \n";
  }

  #look for a line which contains:
  # (DESCRIPTION=(ADDRESS=(PROTOCOL=tcp)(HOST=host.domain)(PORT=2249)))
  # $1 will contain host name "host.domain"
  # $2 will contain port "2249"
  if ( ( $nameMatches eq 1 ) && ( $hostPortMatches eq 0 ) )
  {
   if ( $line =~
/^\s+\(DESCRIPTION=\(ADDRESS=\(PROTOCOL=.*\(HOST=(.*)\)\(PORT=(.*)\)\)\)\s*$/
     )
   {
    if ( $2 eq $lsnrPort )
    {
     $hostPortMatches = matchHost( $1, $lsnrHost );

     if ( $hostPortMatches eq
          0 )    #one more try - In case we got an IP instead of host name
     {
      my $convertedHostName = ip2host($lsnrHost);
      if ( !( $convertedHostName eq "" ) )
      {
       $hostPortMatches = matchHost( $1, $convertedHostName );
      }
     }

     #one last try - In case RESULT displays an IP instead of host name
     if ( $hostPortMatches eq 0 )
     {

      # This should never happen.
      my $convertedHostName = ip2host($1);
      if ( !( $convertedHostName eq "" ) )
      {
       $hostPortMatches = matchHost( $convertedHostName, $lsnrHost );
      }
     }
    }
   }
  }
 }
 if ( $nameMatches && $hostPortMatches )
 {
  return 1;
 }
 return 0;
}

sub matchHost
{
 my ( $hostInResult, $lsnrHost ) = @_;

 $hostInResult = trimwhitespace($hostInResult);
 $lsnrHost     = trimwhitespace($lsnrHost);

 my $hostMatches = 0;

 if ( $hostInResult eq $lsnrHost )
 {
  $hostMatches = 1;
 }
 else #There is still hope, maybe $1 is "host.domain.com", but $quotedHost = "host"
 {
  if ( $hostInResult =~ /^$lsnrHost\..*/i )
  {
   $hostMatches = 1;
  }
  else #REMOTE CHANCE OF THIS: $1 is "host", but $quotedHost = "host.domain.com"
  {
   if ( $lsnrHost =~ /^$hostInResult\..*/i )
   {
    $hostMatches = 1;
   }
  }
 }
 return $hostMatches;
}

# Remove whitespace from the start and end of the string
sub trimwhitespace
{
 my ($mstring) = @_;

 my $retString = $mstring;

 $retString =~ s/^\s+//;
 $retString =~ s/\s+$//;
 return $retString;
}

# In case we got am IP addr, but the output of "lsnrctl status" shows FQDN etc...
sub ip2host
{
 my ($ip) = @_;

 my $hostName = trimwhitespace($ip);

 my @numbers = split( /\./, $ip );
 my $ip_number = pack( "C4", @numbers );

 $hostName = ( gethostbyaddr( $ip_number, 2 ) )[0];

 return $hostName;
}

# This method does not check the HOST and PORT, it assumes you already did that by calling:
# isAnyListenerRunning()
# Here we just check for a matching LISTENER name and then check for a line saying:
# 'Listener Parameter File SPACE <listener.ora directory>'
# we return the listener.ora location from this method.
sub getCurrentListenerOraFile
{
 my ( $statusResult, $name ) = @_;

 my $line;
 my $nameMatches      = 0;
 my $listenerOraFound = 0;
 my $listenerOraFile  = "";

 my @info = split( /\n/, $statusResult );
 my $quotedName = quotemeta($name);

 foreach $line (@info)
 {

  #match the listener Alias
  if ( $line =~ /^(.*)\s+$quotedName\s*$/i )
  {
   $nameMatches = 1;
  }

#look for a line which contains:
# Listener Parameter File   /scratch/private/oracle/ora10g/network/admin/listener.ora
  if ( $nameMatches eq 1 )
  {
   if ( $line =~ /^(.*)\s+(.*)(.)listener.ora\s*$/i )
   {
    if ( !( get_osType() eq 'WIN' ) )
    {
     $listenerOraFile = $2 . "/listener.ora";
    }
    else
    {
     $listenerOraFile = $2 . "\\listener.ora";
    }
    return $listenerOraFile;
   }
  }
 }
 return $listenerOraFile;
}

# check for scan listener
sub isScanListener
{

(my $lsnrhome , my $lsnrname , my $machineName , my $lsnrVersion) = @_;

 EMD_PERL_DEBUG("$LOG_PREFIX : is scan listener Inputs -$lsnrname- : -$lsnrhome- : -$machineName- : -$lsnrVersion-");

 my $isScan = "FALSE";
 eval {

  local $SIG{ALRM} = sub {die "alarm\n"};
  # we are expecting scan check to be over in 45 seconds
  # in case it hangs , this method will time out and execution
  # resumes ....
  alarm 45;
  # To test time out use case : uncomment below line
  # my $sleep = `sleep 60`;

  #bug 12561317
  my %bkup_path = unset_lib_path_env();
  set_lib_path($lsnrhome);
  dump_lib_path_env();

  # lsnr version check
  EMD_PERL_DEBUG("$LOG_PREFIX: isScanListener : $lsnrname : lsnrctl version is  [$lsnrVersion]");

  if ( compareVersions( $lsnrVersion, $LSNR_VER_11_2 ) < 0 )
  {
   EMD_PERL_DEBUG(" $LOG_PREFIX : isScanListener  $lsnrname : version is less than 11.2" );
   $isScan = "FALSE";
   return "FALSE";
  }

  #emcrsp executable check
  unless ( has::Common::hasCheckForEmcrsp($lsnrhome) )
  {
   EMD_PERL_DEBUG("$LOG_PREFIX : isScanListener  $lsnrname : emcrsp is not present in $lsnrhome");
   $isScan = "FALSE";
   return "FALSE";
  }

  #get cluster scan name from emcrsp api
  my $scanRef = has::Common::hasGetScanInformation($lsnrhome);

  EMD_PERL_DEBUG("$LOG_PREFIX : isScanListener  $lsnrname : scanRef = $scanRef");

  revert_original_lib_path_env(%bkup_path);

  if (     $scanRef
       and ref($scanRef)
       and ref($scanRef) =~ /HASH/i
       and keys %{$scanRef} )
  {

   my $clusterScanName = $scanRef->{SCAN_NAME}
     if defined $scanRef->{SCAN_NAME};
   EMD_PERL_DEBUG(" $LOG_PREFIX : isScanListener  $lsnrname : cluster scan name is -$clusterScanName-  "   );

   if ( trim($clusterScanName) eq "" )
   {

    # TODO - check if error was captured by has api hasGetScanInformation ,
    # if error , set isScan to "none"
    # if no error , set isScan to false
    $isScan = "FALSE";

   }
   elsif ( equalHosts( $machineName, $clusterScanName ) )
   {

    $isScan = "TRUE";
    EMD_PERL_DEBUG(" $LOG_PREFIX : isScanListener  $lsnrname : isScan is TRUE" );
   }
  }
  alarm 0;
 };
 if ($@)
 {
  $isScan = "NONE";
   if ($@ eq "alarm\n")
   {
     EMD_PERL_INFO(" $LOG_PREFIX : isScanListener : timed out limit 45 seconds ");
   }

  # log the message to the log file
  EMD_PERL_ERROR("$LOG_PREFIX Exception in isScanListener: $lsnrname : WARN: $@->getErrorMessage()" );


 }

 EMD_PERL_DEBUG(" $LOG_PREFIX:isScanListener :listener name= $lsnrname : isScan= $isScan" );
 return $isScan;

}

# check for listener type
# it should return
# SI , if it is Single Instance
# SCAN ,if SCAN listener
# RAC , if it is CRS resource  
# SIHA , if it is Single instance HAS managed resource  (TODO)
# RIM , in future 
# HUB , in future (TODO)
# UNKNOWN , if error while getting type .
# As this method will be called from lsnr metrics we need to pass version here .
# This method will be invoked as part of dynamic property . We need to ensure
# the calls should be quick before it times out .
# TODO -IMPORTANT-  Check with HAS module Api, if we can also captures error along with cached command result

sub getListenerType
{
(my $lsnrHome , my $lsnrName , my $machineName , my $lsnrVersion) = @_;

 EMD_PERL_DEBUG("$LOG_PREFIX : getListenerType Inputs -$lsnrName- : -$lsnrHome- : -$machineName- : -$lsnrVersion-");

 my $lsnrType = "SI";

 eval {

  # check for srvctl executable , it is present in siha/crs home  and rac home
  # but not in SI home .
  unless ( has::Common::hasCheckForExecutableFile( 'srvctl', $lsnrHome ) )
  {
   EMD_PERL_DEBUG("$LOG_PREFIX : getListenerType  $lsnrName : srvctl is not present in $lsnrHome");
   $lsnrType = $LSNR_SI_TYPE;
   return $lsnrType;
  }

# check for other listener types

  my $db_ref = has::Common::hasGetAllListenersInformation($lsnrHome);
  
  EMD_PERL_DEBUG("$LOG_PREFIX : getListenerType  $lsnrName : db_ref=$db_ref ");

  if ( $db_ref and keys %{$db_ref} )
  {

     EMD_PERL_DEBUG("$LOG_PREFIX : getListenerType  $lsnrName: db_ref has some keys  ");

     for my $dbres ( keys %{$db_ref} )
     {
        EMD_PERL_DEBUG("$LOG_PREFIX : getListenerType  $lsnrName Inside for  Key = $dbres  and value map = $db_ref->{$dbres} ");

        if ( $db_ref->{$dbres} )
        {
          EMD_PERL_DEBUG("$LOG_PREFIX : getListenerType  $lsnrName : Got TYPE from map = $db_ref->{$dbres}{TYPE}");
          EMD_PERL_DEBUG("$LOG_PREFIX : getListenerType  $lsnrName : Got NAME from map = $db_ref->{$dbres}{NAME}");

          if ( $db_ref->{$dbres}{TYPE} and 
               $db_ref->{$dbres}{NAME} and 
               $db_ref->{$dbres}{NAME} =~ /^ora.$lsnrName.lsnr$/ )
          {
            $lsnrType = $db_ref->{$dbres}{TYPE};

            if ($lsnrType =~ /^ora.rim_listener.type$/)
            {
              $lsnrType = $LSNR_RIM_TYPE;
            }
            elsif ($lsnrType =~ /^ora.asm_listener.type$/)
            {
              $lsnrType = $LSNR_ASM_TYPE;
            }
            elsif ($lsnrType =~ /^ora.scan_listener.type$/)
            {
              $lsnrType = $LSNR_SCAN_TYPE;
            }
            else
            {
              $lsnrType = $LSNR_RAC_TYPE;
            }

            EMD_PERL_DEBUG(" $LOG_PREFIX :$lsnrName: calculated getListenerType : $lsnrType");
            return $lsnrType;
          }
        }
     }
  }

  # Check for scan listener in case has api map do not have scan lsnr name 
  my $isScan = isScanListener( $lsnrHome , $lsnrName , $machineName ,  $lsnrVersion);

  if ( $isScan eq "TRUE" )
  {
   $lsnrType = $LSNR_SCAN_TYPE;
   return $lsnrType;

  }

  
 };
 if ($@)
 {
  $lsnrType = $LSNR_UNKNOWN_TYPE;
  # in case of error thrown treat it as unknown
  # this status will force metric to  re-evaluate listener type again and again .
  EMD_PERL_DEBUG(" $LOG_PREFIX :$lsnrName: getListenerType : WARN: $@->getErrorMessage()");
 }


 EMD_PERL_DEBUG(" $LOG_PREFIX:getListenerType :listener name= $lsnrName : lsnrType= $lsnrType" );
 return $lsnrType;
}

# Usage: equalHosts($host1, $host2)
# return 1 if an ip address of $host1 equals an ip address of $host2,
# otherwise return 0

sub equalHosts
{

 if ( $_[0] eq "" || $_[1] eq "" )
 {
  return 0;
 }

 my ( $name1, $aliases1, $addrtype1, $length1, @addrs1 ) =
   gethostbyname( $_[0] );
 my ( $name2, $aliases2, $addrtype2, $length2, @addrs2 ) =
   gethostbyname( $_[1] );

 foreach my $a1 (@addrs1)
 {
  foreach my $a2 (@addrs2)
  {
   return 1 if ( $a1 eq $a2 );
  }
 }

 0;
}


#------------------------------------------------------------------------------
# FUNCTION :    compareVersions
#
#
# ARGUMENTS
# v1 - version 1
# v2 - version 2
#
# RETURNS
# -1 , is v1 less than v2
# 0 , if v1 = v2
# 1 , if v1 > v2
#
#
#------------------------------------------------------------------------------
sub compareVersions
{
 my ( $v1, $v2 ) = @_;
 my @subv1;
 my @subv2;
 my $name = $ENV{LSNR_NAME};

 while ( $v1 =~ /\d+/g )
 {
  push @subv1, $&;
 }

 while ( $v2 =~ /\d+/g )
 {
  push @subv2, $&;
 }

 my $size = ( @subv1 > @subv2 ) ? @subv1 : @subv2;

 my $i;
 for ( $i = 0 ; $i < $size ; $i++ )
 {
  EMD_PERL_DEBUG(
      " $LOG_PREFIX : compareVersions  : -$subv1[$i]- , -$subv2[$i]- " );
  if ( $subv1[$i] > $subv2[$i] )
  {
   EMD_PERL_DEBUG("$LOG_PREFIX : compareVersions  : 1");
   return 1;
  }

  if ( $subv1[$i] < $subv2[$i] )
  {
   EMD_PERL_DEBUG("$LOG_PREFIX : compareVersions  : -1");
   return -1;
  }
 }
 EMD_PERL_DEBUG(" $LOG_PREFIX : compareVersions  : 0");
 return 0;
}

sub isExecutionPossible
{
 my $lsnrname = $ENV{LSNR_NAME};
 EMD_PERL_DEBUG(" $LOG_PREFIX :: isExecutionPossible");
 my ($pcommand) = @_;

 if ( isPriviligeCommand($pcommand) )
 {
  if ( isPriviligeUser() )
  {
   return 1;
  }
  else
  {
   if ( isPasswordSet() )
   {
    return 1;
   }
   else
   {
    EMD_PERL_DEBUG(
"$LOG_PREFIX.isExecutionPossible :: Skipping execution of privilege command @{$pcommand} :: $pcommand as the listener  $lsnrname is not password protected and user emagent is not a listener oracle home owner "
    );
    return 0;
   }
  }
 }
 else
 {
  return 1;
 }

}

sub isPasswordSet
{

 my $paswd = $ENV{'LSNR_PASSWORD'};
 if ( !( $paswd eq "" ) )
 {
  EMD_PERL_DEBUG(" $LOG_PREFIX :: isPasswordSet :: password is set");
  return 1;
 }
 else
 {
  EMD_PERL_DEBUG(" $LOG_PREFIX :: isPasswordSet :: password is not set");
  return 0;
 }
}

sub isPriviligeCommand
{
 EMD_PERL_DEBUG(" $LOG_PREFIX :: isPriviligeCommand");
 my ($pcommand) = @_;

 EMD_PERL_DEBUG(
            " $LOG_PREFIX :: isPriviligeCommand :: @{$pcommand} :: $pcommand");

 #List of privilige commands
 my @priviligeComands = (
                          "save_config",
                          "stop",
                          "trace",
                          "spawn",
                          "reload",
                          "set log_file",
                          "set log_status",
                          "set inbound_connect_timeout",
                          "set save_config_stop_on",
                          "set trc_file",
                          "set trc_level",
                          "set log_directory",
                          "set startup_waittime",
                          "show rules",
                          "show trc_directory",
                          "show log_file",
                          "show log_status",
                          "show inbound_connect_timeout",
                          "show snmp_visible",
                          "show trc_file",
                          "show trc_level",
                          "show log_directory",
                          "show startup_waittime",
                          "show save_config_stop_on"
 );

 #Adding capability to take multiple commands also
 if ( ref($pcommand) eq "ARRAY" )
 {
  foreach my $val2 ( @{$pcommand} )
  {
   foreach my $val1 (@priviligeComands)
   {
    $val2 = trim($val2);

    #use case insensitive match
    if ( $val1 =~ m/$val2/i )
    {
     EMD_PERL_DEBUG(
                " $LOG_PREFIX :: isPriviligeCommand :: Command val :: $val1" );
     return 1;
    }
   }
  }
 }
 else
 {
  foreach my $val (@priviligeComands)
  {
   $pcommand = trim($pcommand);

   #use case insensitive match
   if ( $val =~ m/$pcommand/i )
   {
    EMD_PERL_DEBUG(
                 " $LOG_PREFIX :: isPriviligeCommand :: Command val :: $val" );
    return 1;
   }
  }
 }

}

sub isPriviligeUser
{
 my $lsnrhome = $ENV{'LSNR_ORACLE_HOME'};

 #my $agenthome = $ENV{'EMDROOT'};
 my $agenthome = $ENV{'PLUGIN_ROOT'};

 my $listeneruser = file_owner($lsnrhome);
 my $agentuser    = file_owner($agenthome);

 if ( $listeneruser eq $agentuser )
 {
  EMD_PERL_DEBUG(
" $LOG_PREFIX :: isPriviligeUser :: agent user  $agentuser and listener user $listeneruser matches"
  );
  return 1;
 }
 else
 {
  EMD_PERL_DEBUG(
" $LOG_PREFIX :: isPriviligeUser :: agent user  $agentuser and listener user $listeneruser does not match"
  );
  return 0;
 }
}

sub trim
{
 my $string = shift;
 $string =~ s/^\s+//;
 $string =~ s/\s+$//;
 return $string;
}

#Gets listener control version.
#if not found returns 0.
sub getLsnrctlVersion
{
 my ($oracleHome) = @_;
 my $lsnrctl = getListenerControl($oracleHome);
 if ( -e $lsnrctl )
 {
  my @commands = ( "version", "exit" );
  my $result = getRunCommandOutput( $oracleHome, $lsnrctl, @commands );
  my @resultArray = split /\n/, $result;

  foreach my $ln (@resultArray)
  {
   if ( $ln =~ /\s*LSNRCTL (.+) Version ([0-9.a-zA-Z]+?) (.+)/i )
   {
    EMD_PERL_INFO("$LOG_PREFIX :The version of the $lsnrctl=[$2]");
    return $2;
   }
  }
 }
 EMD_PERL_ERROR("$LOG_PREFIX : Could not get version of $lsnrctl");
 return 0;
}

sub getListenerControl
{
 my ($oracleHome) = @_;
 if ( get_osType() eq 'WIN' )
 {
  return $oracleHome . "\\bin\\lsnrctl.exe";
 }
 return $oracleHome . "/bin/lsnrctl";
}

sub isLsnrCommand
{
 EMD_PERL_DEBUG(" $LOG_PREFIX :: isLsnrCommand");
 my ($pcommand) = @_;

  EMD_PERL_DEBUG(
            " $LOG_PREFIX :: isLsnrCommand ::  $pcommand");

  #List of lsnr commands
   my @lsnrComands = (
                          "start",
                          "stop",
                          "status",
                          "services",
                          "version",
                          "reload",
                          "save_config",
                          "trace",
                          "spawn",
                          "change_password",
                          "quit",
                          "exit",
                          "password",
                          "rawmode",
                          "displaymode",
                          "trc_file",
                          "trc_directory",
                          "trc_level",
                          "log_file",
                          "log_directory",
                          "log_status",
                          "current_listener",
                          "inbound_connect_timeout",
                          "startup_waittime",
                          "save_config_on_stop",
                          "dynamic_registration",
                          "rules",
                          "snmp_visible"
 );

 
  foreach my $val (@lsnrComands)
  {
   $pcommand = trim($pcommand);

   #use case insensitive match
   if ( $val =~ m/$pcommand/i )
   {
      EMD_PERL_DEBUG(
              " $LOG_PREFIX :: isLsnrCommand :: Command val :: $val" );
	return 1;
   }
  }

}

#This is to return the owner name as string, of the fpath
##This has been modified to make it work for Window also
sub file_owner{
    my $fileName = shift ;
    if($OSNAME ne "WIN"){
        if ($fileName eq "")
        {
            print "File Name Passed To The function Was A Null String \n";
            return "";
        }
        my  $uid =file_owner_uid($fileName);
        if($uid == -1){
            return "";
        }
        else{
            my  $name  = getpwuid($uid); #This would return Name string of the corresponding UID
            return $name;
        }
    }
    elsif($OSNAME eq "WIN"){
        return win32_file_owner($fileName);
    }
    else{
        #code for systems neither mswin32, nor linux
    }
}

#This is to return UID for a given file Fpath
sub file_owner_uid {
    my $fileName = shift;
    my $info=stat($fileName);
    if($info)
    {
        return $info->uid;
    }
    return -1;
}

1;
