#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/db/oradebug_dump.pl /st_emgc_pt-12.1.0.4pg/1 2012/09/09 03:06:52 pbhogara Exp $
#
# oradebug_dump.pl
# 
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      oradebug_dump.pl - execute oradebug dump command
#
#    DESCRIPTION
#      
#    This script provies API that executes an "oradebug" dump command and 
#    returns the file name containing the dump contents.
# 
#   
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    pbhogara    08/22/12 - use set_env_var() for setting the env variables
#    pbhogara    03/19/12 - fix tracefile path on windows
#    pbhogara    03/04/12 - on windows chomp does not remove ^M
#    pbhogara    07/20/11 - use temp file utility for creating temp files
#    pbhogara    02/15/11 - print TZ and local time before spawning sqlplus
#    pbhogara    04/14/10 - Creation
# 


require "emd_common.pl";
require "semd_common.pl";
require "db/emergPerfUtil.pl";
require "db/db_common.pl";

use vars qw($NT);

package OradebugDump;

use Time::Local;
use IPC::Open2;
use File::Copy;

use strict;

################################
## context attributes
################################

my $oracle_home;
my $oracle_sid;
my $connect_string;
my $enable_prelim = 1; # True by default


my $SQLPLUS_OUTPUT;     # sqlplus process output handle
my $SQLPLUS_INPUT;      # sqlplus process input handle
my $spawn_sqlplus = 1;  # set to 0 when sqlplus process i/o handles
                        # are passed to this script

my $PRELIM     = 1;
my $NON_PRELIM = 0;

#
# Subroutine: setContextAttributes
#  $_[0] => oracle_home
#  $_[1] => oracle_sid
#  $_[2] => connect string (should be as SYSDBA)
#
sub setContextAttributes
{
  $oracle_home    = shift;
  $oracle_sid     = shift;
  $connect_string = shift;
  $enable_prelim  = shift;

  $enable_prelim = 1 if !(defined $enable_prelim);

  ::EMAGENT_PERL_DEBUG("oracle_home: $oracle_home");
  ::EMAGENT_PERL_DEBUG("oracle_sid: $oracle_sid");
  ::EMAGENT_PERL_DEBUG("connect string: $connect_string");
  ::EMAGENT_PERL_DEBUG("prelim mode: ".($enable_prelim?"enabled":"disabled"));
}

#
# Subroutine: setSqlPlusProcessHandles
#  $_[0] => Output handle 
#  $_[1] => Input handle 
# 
# This function is called before calling the generate_dump
# routine.
#
sub setSqlPlusProcessHandles
{
  $SQLPLUS_OUTPUT = shift;
  $SQLPLUS_INPUT  = shift;  
 
  $spawn_sqlplus  = 0;
}

our $ERROR_CODE=0;  # error codes are as follows:
                    #     0 - no error
                    #   750 - unable to spawn SQL*Plus
                    #  1017 - invalid SYSDBA username/password
                    #  0042 - Invalid sql command
                    # 00070 - Invalid oradebug command.
                    # 00073 - Invalid dump command: DUMP takes
                    #             between 2 and 3 argument(s)
                    # 32730 - Command cannot be executed on remote instance.
                    # 01031 - insufficient privileges
                    #    -1 - global dump option cannot be used in space saving mode.
                    #    -2 - Error while parsing trace file header. 


our %modes = ( NORMAL => 0,
               SPACE_SAVING => 1,
             );


# no-op like command used to get the trace file name before executing
# the actual dump command
my $NOOP_EQUIV_ORADEBUG_DUMP_COMMAND_10G =
                               "oradebug dump record_callstack 1000001;\n";


my $PRINT_TRACEFILE_NAME = "oradebug tracefile_name;\n";


my $TEMP_TRC_FILE_PREFIX = "oradebug_dump_temp_file_$$"."_";


#::EMAGENT_PERL_INFO(" Trace file generated by oradebug command is ".
#                      "$tracefile_name_with_path ");

#
# Subroutine: generateDump
#  $_[1] => oradebug command
#  $_[2] => cleanup flag - 0/1
#
#  Returns: Returns the trace file name with full path.
#
#  Notes:
#
#  This API supports two modes through cleanup flag
#    a) normal mode(0): it executes the oradebug dump command and returns the
#       trace file name containing the dump contents.
#
#    b) space saving mode(1):
#
#       - If dump command generated a new trace file then 
#         this file is discarded and a temp trace file with these contents
#         is returned.
#
#       - If an existing trace file was used for dump then it returns the
#         temp trace file with the latest dump contents. (Ideally, It is 
#         good to wipe off the latest dump contents from the trace file but
#         we dont do it here)    
#
#       The temporary tracefile should be deleted by the caller.
#
#       This mode is specifically used by Emergency Performance page to 
#       generate Hang Analysis and ASH dump every 15/30secs. This might 
#       generate a lot of trace files unnecessarily. To avoid this, they 
#       are cleaned up after use.
#
#  Set the context OR sqlplus process handles before calling this
#  function.
#  If sqlplus process handles are passed to this function then this
#  function will use those handles instead of spawning a new sqlplus process.
#

sub generateDump
{
  my($oradebug_command,$dump_mode) = @_;

  ::EMAGENT_PERL_DEBUG("Executing oradebug command: $oradebug_command: $dump_mode");  

  my $prelim_mode = $enable_prelim?"-prelim":"";

  # prompt to identify the line in o/p containing the trace file name
  my $prompt1 = "Trace file name";

  # prompt to identify end of dump command output
  my $prompt2 = "End of dump command";

  # Locally instantiate the relevant pieces of the ENV array so that SQL*Plus
  # is spawned in the correct context.
  # Override the ORACLE_HOME, ORACLE_SID, LD_LIBRARY_PATH env variables.
  ::set_env_var($oracle_home,$oracle_sid,"TRUE"); 

  # log the value of TZ env variable and local time
  ::EMAGENT_PERL_DEBUG(" Value of ENV{TZ} is ".$ENV{'TZ'});
  my $now_string = localtime; 
  ::EMAGENT_PERL_DEBUG(" Local time is $now_string "); 
  
  # Note the current time before generating the dump command
  # this is epoch time
  my $time_before_dump = time();

  my $tracefile_name_with_path="";
  my $metadatafile_name_with_path="";
  my $sql_process_id="";
  my $return_val= "";


  if(($oradebug_command =~ /oradebug -g/) and
     ($dump_mode == $modes{SPACE_SAVING}))
  {
    # Global option is not allowed in space saving mode because trace file
    # cannot be cleaned up.
    ::EMAGENT_PERL_ERROR(" Global dump option cannot be used along with ".
                         " space saving mode. Please use normal mode");
    $ERROR_CODE = -1;
    return;
  }
 
  #if(!(defined $SQLPLUS_OUTPUT) && !(defined $SQLPLUS_INPUT))
  #{
    eval
    {
      # Open a bi-directional pipe for input/output to/from a sqlplus process.
      $sql_process_id = open2(*SQLPLUS_OUTPUT,*SQLPLUS_INPUT,
                            "$oracle_home/bin/sqlplus $prelim_mode -S \"$connect_string\"");
      $SQLPLUS_OUTPUT = *SQLPLUS_OUTPUT;
      $SQLPLUS_INPUT  = *SQLPLUS_INPUT;
    };
    if ($@)
    {
      ::EMAGENT_PERL_ERROR(" oradebug_dump: unexpected error $@");
      $ERROR_CODE= $?;
      goto EXIT; 
    }

    # Execute the sqlplus commands
    print $SQLPLUS_INPUT <<"EOS";
set heading off;
set echo off;
set feedback off;
set pagesize 0;
EOS
  #}

  print $SQLPLUS_INPUT "oradebug setmypid;\n";

  # execute the "oradebug tracefile_name" command to get
  # the trace file name. In 10G, this command does not
  # return the tracefile name until a dump is generated.
  # Therefore, we first use a light-weight no-op like dump command.
  print $SQLPLUS_INPUT $NOOP_EQUIV_ORADEBUG_DUMP_COMMAND_10G;

  # prompt to know that the next line is trace file name";
  print $SQLPLUS_INPUT "prompt $prompt1;\n";

  # Print the trace file name
  print $SQLPLUS_INPUT $PRINT_TRACEFILE_NAME;

  my $sqlout;
  my $error_occurred = 0;
  while($sqlout = <$SQLPLUS_OUTPUT>)
  {
    chomp $sqlout;
    # on windows chomp may not work
    $sqlout = EmergPerfUtil::trim_new_line($sqlout);
    if($sqlout =~ /$prompt1/)
    {
      last;
    }
    elsif($error_occurred){
      next; # continue until prompt1 OR end of output stream
    }
    else
    {
      $ERROR_CODE = EmergPerfUtil::findErrorCode($sqlout);
      if ($ERROR_CODE)
      {
        ::EMAGENT_PERL_ERROR("oradebug dump script - Error occurred: $sqlout");
        $error_occurred = 1;
      }
    }
  }
  if($error_occurred){
    goto EXIT;
  }

  $tracefile_name_with_path = <$SQLPLUS_OUTPUT>;
  chomp $tracefile_name_with_path;
  # on windows chomp may not work
  $tracefile_name_with_path = EmergPerfUtil::trim_new_line($tracefile_name_with_path);
  if($::NT){
    $tracefile_name_with_path = join '/', split /\\/, $tracefile_name_with_path;
  }

  ::EMAGENT_PERL_DEBUG(" oradebug command will dump to trace file ".
                           "$tracefile_name_with_path");

  # get the ".trm" file corresponsing to trace file
  my @tracefile_path_info = split(/\./,$tracefile_name_with_path);
  $#tracefile_path_info = $#tracefile_path_info - 1;
  $metadatafile_name_with_path = $tracefile_path_info[0].".trm";

  # Note the position of the end of trace file before executing the dump command
  if(!open(TRCFILE_RDR,"<",$tracefile_name_with_path)){
     goto EXIT;
  }

  seek(TRCFILE_RDR,0,2);
  my $pos_before_dump = tell TRCFILE_RDR;
  close(TRCFILE_RDR);


  # Execute the dump command.
  print $SQLPLUS_INPUT $oradebug_command.";\n";

  # prompt to know that the end of dump command output
  # This is useful when output has multiple lines
  print $SQLPLUS_INPUT "prompt $prompt2;\n";
 
  while($sqlout = <$SQLPLUS_OUTPUT>)
  {
    chomp $sqlout;
    # on windows chomp may not work
    $sqlout = EmergPerfUtil::trim_new_line($sqlout);
    if($sqlout eq $prompt2)
    {
      last;
    }
    else
    {
      $ERROR_CODE = EmergPerfUtil::findErrorCode($sqlout);
      if ($ERROR_CODE)
      {
        ::EMAGENT_PERL_ERROR("oradebug dump script - Error occurred: $sqlout");
        goto EXIT;
      }
    }
  }

  print $SQLPLUS_INPUT "oradebug flush;\n";

  if($dump_mode == $modes{NORMAL}) # Normal mode
  {
    ::EMAGENT_PERL_DEBUG("Mode: Normal");
    # In normal mode we don't have to clean up the trace file
    $return_val = $tracefile_name_with_path;
    ::EMAGENT_PERL_DEBUG(" returning trace file $return_val");
    goto EXIT;
  }
  elsif($dump_mode == $modes{SPACE_SAVING}) # space saving mode
  {
    ::EMAGENT_PERL_DEBUG("Mode: Space saving");
    # In space saving mode generate the dump and clean up the trace file
    # if it already exists and return a temporary trace file with
    # latest dump contents.

    my $existsBeforeDump =  existsTraceFile($time_before_dump,
                                            $tracefile_name_with_path);

    if($existsBeforeDump < 0)
    {
      $ERROR_CODE=$existsBeforeDump;
      goto EXIT;
    }

    my $temp_trace_file = createTempTraceFile($tracefile_name_with_path,
                                              $pos_before_dump);

    if( $existsBeforeDump == 1 )
    {
      # This method is for cleaning up the most recent dump contents.
      # For now, this is disabled. We can uncomments this when it is
      # really necessary. 

      # cleanUpExistingTraceFile($tracefile_name_with_path,$pos_before_dump);
    }
    else
    {
      deleteNewTraceFile($tracefile_name_with_path,$metadatafile_name_with_path);
    }

    $return_val = $temp_trace_file;
    ::EMAGENT_PERL_DEBUG(" returning trace file $return_val");
  }

  EXIT:
    if($spawn_sqlplus) # If this function has not spawned a sqlplus proces then 
                       # don't close the handles.
    {
      print $SQLPLUS_INPUT "exit;\n";
      # close STDIN and STDOUT handles of the sqlplus process
      close($SQLPLUS_INPUT);
      close($SQLPLUS_OUTPUT);
    }

  return $return_val;
}


#
# Subroutine: createTempTraceFile
#  $_[0] => name of trace file generated by oradebug command
#
# Returns: Creates a temporary copy of the existing trace file
#          and returns the name of this temp file.
#
sub createTempTraceFile()
{
  my $tracefile_name_with_path = EmergPerfUtil::trim(shift);

  my ($temp_trace_handle, $temp_trace_file) = ::create_temp_file(".trc");

  if($::NT){
    $temp_trace_file = join '/', split /\\/, $temp_trace_file;
  }

  #create a copy of the current trace file
  copy($tracefile_name_with_path,$temp_trace_file);

  return $temp_trace_file;
}

#
# Subroutine: deleteNewTraceFile
#  $_[0] => trace file name
#  $_[1] => metadaa file name
#
sub deleteNewTraceFile()
{
  my($tracefile,$metadata_file) = @_;
  ::EMAGENT_PERL_DEBUG(" removing trace file $tracefile");
  my $count = 0;

  if(-e "$tracefile"){
    if(unlink($tracefile)){
      ::EMAGENT_PERL_DEBUG("Trace file $tracefile unlinked ");
      $count++;
    }
    else{
      ::EMAGENT_PERL_DEBUG("Could not unlink $tracefile: $!");
    }
  }
  else{
     ::EMAGENT_PERL_DEBUG("$tracefile does not exist: $!");
  }
  if(-e $metadata_file){
    if(unlink($metadata_file)){
      ::EMAGENT_PERL_DEBUG("File $metadata_file unlinked ");
      $count++;
    }
    else{
      ::EMAGENT_PERL_DEBUG("Could not unlink $metadata_file: $!");
    }
  }
  else{
    ::EMAGENT_PERL_DEBUG("File $metadata_file does not exist ");
  }
  return $count;
}


#
# Subroutine: cleanUpExisingTraceFile
#  $_[0] => Name of trace file with complete path
#  $_[1] => byte offset from the the begining of the file.
#
#  Note: Need to deal with cleaning up .trm file also.
sub cleanUpExistingTraceFile()
{
  my $tracefile_name_with_path = shift;
  my $pos_before_dump = shift;

  # truncate the file so that the latest contents are discarded
  truncate $tracefile_name_with_path,$pos_before_dump;
}


#
# Subroutine: existsTraceFile
#  $_[0] => Time noted before executing the dump command.
#  $_[1] => Trace file name
#
# Returns: returns "1" if trace file existed before the dump.
#          Returns 0 if trace file did not exists.
#          Return value <0 on error.
sub existsTraceFile
{
  my $time_before_dump = shift;
  my $tracefile_name = shift;
  my $tracefile_gen_time;

  open TRACEFILE_RDR,"<",$tracefile_name;

  # Read the trace file header until "unix process id" entry.
  while(<TRACEFILE_RDR>)
  {
    if((/(Unix process pid.*)/) || (/Windows thread id.*/))
    { last; }
  }

  # Skip all empty lines and then read the line with time stamp of
  # trace file creation.
  while(<TRACEFILE_RDR>)
  {
    my $temp_line = EmergPerfUtil::trim($_);
    if($temp_line eq "")
    {
      next;
    }
    elsif( $temp_line =~ m{\A\*\*\*\s*  # the line starts with "***"
            (\d\d\d\d)-(\d\d)-(\d\d)\s* # Year-Month-Day followed by space
            (\d\d):(\d\d):(\d\d)\.(\d\d\d)}xm ) # Hours-minutes-secs
    {
       # convert to epoch time
      $tracefile_gen_time = timelocal($6,$5,$4,$3,$2-1,$1);
      last;
    }
  }
  close(TRACEFILE_RDR);

  # If time stamp was not found then log an error
  if(!defined($tracefile_gen_time))
  {
    ::EMAGENT_PERL_INFO("Unexpected trace file ".
                        "header format - ".$tracefile_name);
     return 1; # we are not sure if file exists so return 
               # true to be safe.
  }

  ::EMAGENT_PERL_DEBUG("Comparing times ".
                       "traceFileTime: $tracefile_gen_time and ".
                       "time_before_dump: $time_before_dump");

  # Numerically compare the epoch times
  if($tracefile_gen_time < $time_before_dump)
  {
   return 1; # trace file already exists
  }
  else
  {
   return 0;
  }
}

1;
