#!/usr/local/bin/perl
#
# $Header: emdb/sysman/admin/scripts/db/waitChains.pl /main/13 2011/06/29 00:45:50 pbhogara Exp $
#
# waitChains.pl
#
# Copyright (c) 2006, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      waitChains.pl - Get the wait chains data from the server.
#
#    DESCRIPTION
#      Use the direct_access API to retrieve wait chain data from the server.
#
#    NOTES
#      The following conventions must be followed.
#      1) SYSDBA user/password are to be passed on stdin.
#         If the password is missing, that signals the script to attempt
#         OS authentication using BEQ protocol.
#      2) The TNS descriptor is passed as an environment variable.
#         This is unused if password is missing (use "/ as sysdba").
#      3) Command-line parameters are $ORACLE_HOME $ORACLE_SID [global]
#         If global is present, this gets the chains for all instances.
#
#    MODIFIED   (MM/DD/YY)
#    pbhogara    06/23/11 - dont query x$ksuse
#    pbhogara    10/21/10 - Use x$ksdhng_chains.osid instead of
#                           x$ksuse.ksusepid
#    pbhogara    04/14/10 - Rewriting the script to support identifying
#                           chains(open/cycles) and retrieving session
#                           attributes from x$ksuse
#    jsoule      10/04/06 - better tracing
#    jsoule      09/15/06 - Creation
#


use strict;

require "emd_common.pl";
require "db/db_common.pl";

require "db/direct_access.pl";
require "db/sessionDetails.pl";
require "db/emergPerfUtil.pl";

package WaitChains;

our $ERROR_CODE;

# A session is stored as an hashmap of its attribute values.
# Following is the list of attributes.
my $SID                    = "SID";
my $SESS_SERIAL_NO         = "SESS_SERIAL#";

#In case of RAC, INST_ID and INSTANCE can be different.
#INST_ID is the instance on which hanganalysis query is executed.
#INSTANCE is the instance to which the session belongs.
my $INST_ID                = "INST_ID";
my $INSTANCE               = "INSTANCE";
my $P1                     = "P1";
my $P1_TEXT                = "P1_TEXT";
my $P2                     = "P2";
my $P2_TEXT                = "P2_TEXT";
my $P3                     = "P3";
my $P3_TEXT                = "P3_TEXT";
my $IN_WAIT                = "IN_WAIT";
my $IN_WAIT_SECS           = "IN_WAIT_SECS";
my $TIME_REMAINING_SECS    = "TIME_REMAINING_SECS";
my $WAIT_EVENT             = "WAIT_EVENT";
my $WAIT_EVENT_TEXT        = "WAIT_EVENT_TEXT";
my $NUM_WAITERS            = "NUM_WAITERS";
my $CHAIN_ID               = "CHAIN_ID";
my $CHAIN_IS_CYCLE         = "CHAIN_IS_CYCLE";
my $BLOCKER_SID            = "BLOCKER_SID";
my $BLOCKER_SESS_SERIAL_NO = "BLOCKER_SESS_SERIAL#";
my $BLOCKER_INSTANCE       = "BLOCKER_INSTANCE";
my $BLOCKER_INSTANCE_ID    = "BLOCKER_INSTANCE_ID";
my $BLOCKER_CHAIN_ID       = "BLOCKER_CHAIN_ID";
my $BLOCKER_IS_VALID       = "BLOCKER_IS_VALID";
my $OSID                   = "OSID";

my $ID                     = "ID"; # SID_SERIAL#_INSTID
my $BLOCKER_ID             = "BLOCKER_ID"; # SID_SERIAL#_INSTID of blocker
my $CUML_BLOCK_TIME        = "CUML_BLOCK_TIME";
my $WAIT_EVENT_NUM         = "WAIT_EVENT_NUM";
my $IMMED_BLOCKERS         = "IMMED_BLOCKERS";
my $USER_NAME              = "USER_NAME";
my $USER_ID                = "USER_ID";
my $PROGRAM_NAME           = "PROGRAM";
my $MODULE                 = "MODULE";
my $ACTION                 = "ACTION";
my $SERVICE                = "SERVICE";
my $SQL_ID                 = "SQL_ID";

# Projection list used for querying x$ksdhng_chains
my @QUERY_KSDHNG_CHAINS    = ($SID,
                              $SESS_SERIAL_NO,
                              $INST_ID,
                              $INSTANCE,
                              $P1,
                              $P1_TEXT,
                              $P2,
                              $P2_TEXT,
                              $P3,
                              $P3_TEXT,
                              $IN_WAIT,
                              $IN_WAIT_SECS,
                              $TIME_REMAINING_SECS,
                              $WAIT_EVENT,
                              $WAIT_EVENT_TEXT,
                              $NUM_WAITERS,
                              $OSID,
                              $CHAIN_ID,
                              $CHAIN_IS_CYCLE,
                              $BLOCKER_SID,
                              $BLOCKER_SESS_SERIAL_NO,
                              $BLOCKER_INSTANCE,
                              $BLOCKER_CHAIN_ID,
                              $BLOCKER_IS_VALID,);
 
# Projection list used for querying x$ksuse
my $INDX      = "INDX";
my $KSUSESER  = "KSUSESER";
my $KSUUDLNA  = "KSUUDLNA"; #user name
my $KSUUDLUI  = "KSUUDLUI"; #user id
my $KSUSEPNM  = "KSUSEPNM"; #program
my $KSUSEAPP  = "KSUSEAPP"; #module
my $KSUSEACT  = "KSUSEACT"; #action 
my $KSUSESVC  = "KSUSESVC"; #service
my $KSUSESQI  = "KSUSESQI"; #sqlid
my $KSUSEPID  = "KSUSEPID"; #ospid

my @QUERY_KSUSE  = ($INDX,
                    $KSUSESER,
                    $INST_ID,
                    $KSUUDLNA,
                    $KSUUDLUI,
                    $KSUSEPNM,
                    $KSUSEAPP,
                    $KSUSEACT,
                    $KSUSESVC,
                    $KSUSESQI,
                    $KSUSEPID,);

# Projection list used for querying x$ksdhng_session_blockers >=11gR2
my @QUERY_SESSION_BLOCKERS = ($SID,
                              $SESS_SERIAL_NO,
                              $INST_ID,
                              $WAIT_EVENT,
                              $WAIT_EVENT_TEXT,
                              $BLOCKER_SID,
                              $BLOCKER_SESS_SERIAL_NO,
                              $BLOCKER_INSTANCE_ID,);

# list of session attributes used for output 
our @OUTPUT_COLUMNS        = ($P1,
                              $P1_TEXT,
                              $P2,
                              $P2_TEXT,
                              $P3,
                              $P3_TEXT,
                              $IN_WAIT,
                              $IN_WAIT_SECS,
                              $TIME_REMAINING_SECS,
                              $WAIT_EVENT_NUM,
                              $WAIT_EVENT_TEXT,
                              $NUM_WAITERS,
                              $OSID,
                              $CUML_BLOCK_TIME,
                              $BLOCKER_ID, #SID_SERIAL#_INSTID
                              $BLOCKER_CHAIN_ID,
                              $CHAIN_ID,
                              $CHAIN_IS_CYCLE,
                              $USER_NAME,
                              $USER_ID,
                              $PROGRAM_NAME,
                              $MODULE,
                              $ACTION,
                              $SERVICE,
                              $SQL_ID,);

my $target_version;

sub setContext
{
  my $oracle_home    = shift;
  my $oracle_sid     = shift;
  my $connect_string = shift;
  $target_version    = shift;
  my $use_prelim     = shift;

  DirectAccess::setConnectionContext($oracle_home,
                                     $oracle_sid,
                                     $connect_string,
                                     $target_version,
                                     $use_prelim,
                                    );

  SessionDetails::setContext($oracle_home,
                             $oracle_sid,
                             $connect_string,
                             $target_version,
                             $use_prelim);
}



# Global list of blocked sessions - top 50 waiters
# + 1 immediate blocker + [ next non-critical
# blocker if first blocker is critical ] + 
# [ 1 non-critical blocker before final blocker
#  if the final blocker is critical ]
my %blocked_sessions = ();

# global list of all immediate blockers
my %immed_blockers   = ();

# global list of final blockers (fb => cuml block time)
my %fb_block_time = ();

# cycle id => final blocker
my %cycle_fb = ();

# final blocker => hash of waiter ids below it
my %wtrs_below_fb = ();

# visited node used for cuml blocking time calculation
my %visited = ();


# Subroutine: identify_waitchains
#
# $_[0] => Local/global hang analysis
#
# Returns: Reference to a hashmap containing all sessions.
#          The key used to identify a session is "SID_SERIAL#_INSTID".
#          Each session is an associative array with attribute names
#          as keys.
#
# Queries x$ksdhng_chains, x$ksdhng_session_blockers(>=11gR2)
# and combines the information retrieved from these tables.
#
sub identify_waitchains
{

  my $analysisArea = shift;

  %visited       = ();
  %cycle_fb      = ();
  %fb_block_time = ();   


  ################################
  ## get wait chain data from x$ksdhng_chains
  ################################

  DirectAccess::setQueryContext($DirectAccess::SAFE_MODE,
                                $DirectAccess::DISABLE_TRACE,
                                $DirectAccess::OUTPUT_XML);

  my @allcolumns_arr = ("*");
  # We need about 24 cols from the X$KSDHNG_CHAINS table. Some times the direct_access query fails when there are too many columns. Observed this in 11.1.0.7 db. Hence use "*" instead of querying all the columns.
  my @ksdhng_chains_sessions = DirectAccess::getFixedTable('X$KSDHNG_CHAINS',
                                                           \@allcolumns_arr);
  if ($DirectAccess::ERROR_CODE)
  {
    ::EMAGENT_PERL_ERROR("received error code $DirectAccess::ERROR_CODE while querying for x\$ksdhng_chains");
    $ERROR_CODE = $DirectAccess::ERROR_CODE;
    return;
  }


  # Filter the output such that we have only waiters and blockers
  @ksdhng_chains_sessions = filter_chains($analysisArea ne "global",
                                          @ksdhng_chains_sessions);

  # hashmap to map a unique session id => session info hashmap.
  my %ksdhng_chains_sess_hashmap = ();

  # Iterate on the ksdhng_chains_sessions array and create a hashmap that maps
  # the unique session identifier "sid_serial#_instid" to session info hashmap.
  # For example:
  # (129_1223_1 => ('SID' => 129, 'SESS_SERIAL#' => 1223, 'INST_ID' => 1,
  #   'INSTANCE' => 1, 'P1' => 14143342424))
  # This will help performing a join on x$ksuse result
  %ksdhng_chains_sess_hashmap = _create_hashmap(\@ksdhng_chains_sessions,
                                                ($SID,
                                                 $SESS_SERIAL_NO,
                                                 $INSTANCE),
                                               );

  ::EMAGENT_PERL_DEBUG(" *********** KSDHNG_CHAINS START *************");
  ::EMAGENT_PERL_DEBUG(tostring_hashmap_of_hashmaps(\%ksdhng_chains_sess_hashmap));
  ::EMAGENT_PERL_DEBUG(" *********** KSDHNG_CHAINS END *************");

  ################################
  ## Join x$ksdhng_chains and x$ksdhng_session_blockers(>=11.2)
  ###############################
  if(::compareVer($target_version,"11.2")>=0)
  {
    # x$ksdhng_session_blockers exists from 11gR2 onwards
    add_immed_blocker_info(\%ksdhng_chains_sess_hashmap);  
  }

  # Add the blockers found in x$ksdhng_chains as immed blockers.
  foreach my $sid (keys %ksdhng_chains_sess_hashmap)
  {
    my $row = $ksdhng_chains_sess_hashmap{$sid};

    my $blkr_unique_id = get_blkr_unique_id($row);

    if($row->{$BLOCKER_SID} != 0 && $row->{$BLOCKER_SID} != "")
    {
      $ksdhng_chains_sess_hashmap{$sid}{$IMMED_BLOCKERS} = "[".$blkr_unique_id."]";
    }
  }

  #%ksdhng_chains_sess_hashmap = %{read_chains(\%ksdhng_chains_sess_hashmap)};

  #Compute cumulative wait time
  compute_cuml_block_time(\%ksdhng_chains_sess_hashmap);

  ::EMAGENT_PERL_DEBUG(" Printing all the sessions after processsing ");

  ::EMAGENT_PERL_DEBUG(" *********** KSDHNG_CHAINS START *************");
  ::EMAGENT_PERL_DEBUG(tostring_hashmap_of_hashmaps(\%ksdhng_chains_sess_hashmap));
  ::EMAGENT_PERL_DEBUG(" *********** KSDHNG_CHAINS END *************");

  return \%ksdhng_chains_sess_hashmap;
}


#
# Subroutine: compute_cuml_block_time
#
# $_[0] => Reference to hashmap of sessions in x$ksdhng_chains
#
# cumulative wait time is the sum of in_wait_secs of all
# waiters behind a blocker. 
#
sub compute_cuml_block_time
{
  my $ksdhng_chains_sess_hash_ref = shift; 

  ::EMAGENT_PERL_DEBUG("Computing cuml blocking time");
  
  for my $unique_id (keys %$ksdhng_chains_sess_hash_ref)
  {
    my $sess_ref = $ksdhng_chains_sess_hash_ref->{$unique_id};

    # if session is in cycle
    if($sess_ref->{$CHAIN_IS_CYCLE} && !$visited{$unique_id})
    {
      if(!exists $visited{$unique_id})
      {
        _traverse_cycle($unique_id,
                        1,
                        $ksdhng_chains_sess_hash_ref,
                        0);
      }
      else
      {
        next;
      }
    }
    # If session is non-blocking waiter then chain starts here
    elsif($sess_ref->{$NUM_WAITERS} == 0 &&
          $sess_ref->{$BLOCKER_SID} != 0)
    {

      my $wait_time = $sess_ref->{$IN_WAIT_SECS};

      # If this flag is set then we add the blocking time 
      # cumulatively as we move up the chain.
      my $cuml_sum_flg = 1;

      ::EMAGENT_PERL_DEBUG(" starting traversal from $unique_id");

      my $tmp_blkr_id = get_blkr_unique_id($sess_ref); 
      my $tmp_wtr_id  = $unique_id;

      my $cuml_block_time = 0;

      # traverse the chain until the final blocker
      LOOP1: while($tmp_blkr_id ne "0_0_0" && $tmp_blkr_id ne "")
      {
        my $tmp_wtr_sref  = $ksdhng_chains_sess_hash_ref->{$tmp_wtr_id};
        my $tmp_blkr_sref = $ksdhng_chains_sess_hash_ref->{$tmp_blkr_id};

        ::EMAGENT_PERL_DEBUG("edge $tmp_wtr_id -- $tmp_blkr_id ");


        if($cuml_sum_flg) 
        {
          $cuml_block_time  = $tmp_wtr_sref->{$IN_WAIT_SECS} +
                              $tmp_wtr_sref->{$CUML_BLOCK_TIME};

          ::EMAGENT_PERL_DEBUG(" Adding $cuml_block_time to cuml block time of $tmp_blkr_id");

          $tmp_blkr_sref->{$CUML_BLOCK_TIME} += $cuml_block_time; 

          $visited{$tmp_wtr_id} = 1; # It means the wait time has been
                                     # propagated up.
        }
        else
        {
          ::EMAGENT_PERL_DEBUG(" Adding $cuml_block_time to cuml block time of $tmp_blkr_id");
          $tmp_blkr_sref->{$CUML_BLOCK_TIME} += $cuml_block_time;
          ::EMAGENT_PERL_DEBUG(" visited node found ");
        }
        ::EMAGENT_PERL_DEBUG(" = $tmp_blkr_sref->{$CUML_BLOCK_TIME}");



        if(exists $visited{$tmp_blkr_id})
        {
          # When we hit a visited node, it means that
          # we need not add the blocking time cumulatively
          # from now. This typically happens when we hit a 
          # intersecting chain.
          $cuml_sum_flg = 0;
          ::EMAGENT_PERL_DEBUG(" disabled cuml flag ");
        }
        else
        {
          $cuml_sum_flg = 1;
        }
     
        if($tmp_blkr_sref->{$CHAIN_IS_CYCLE})
        {
          _traverse_cycle($tmp_blkr_id,
                          $cuml_sum_flg,
                          $ksdhng_chains_sess_hash_ref,
                          $cuml_block_time);
          last;
        }

        # Next blocker
        $tmp_wtr_id = $tmp_blkr_id;
        $tmp_blkr_id = get_blkr_unique_id($tmp_blkr_sref);
      }

      # save the final blocker and cuml block time for later use
      $fb_block_time{$tmp_wtr_id} = 
                 $ksdhng_chains_sess_hash_ref->{$tmp_wtr_id}{$CUML_BLOCK_TIME};
    }
  }    
} 



#
# Subroutine: _traverse_cycle 
#
# $_[0] => cycle node from where traversal begins 
# $_[1] => should blocking time be cumulatively added
# $_[2] => reference to ksdhng chains 
# $_[3] => cuml blocking time of node below the start node 
#          This will be added to all nodes in cycle
#
sub _traverse_cycle
{

  my $cyc_node_id = shift;
  my $cuml_sum_flg = shift;
  my $ksdhng_chains_sess_hash_ref = shift;
  my $cuml_block_time = shift;

  my $cycle_id = $ksdhng_chains_sess_hash_ref->{$cyc_node_id}{$CHAIN_ID};

  my @cycle_members   = ();

  my $prev_cyc_node_id = $cyc_node_id;
  my $prev_cyc_sref    = $ksdhng_chains_sess_hash_ref->{$cyc_node_id};

  ::EMAGENT_PERL_DEBUG("Traversing cycle starting from $cyc_node_id");

  # note the members in cycle
  push(@cycle_members,$prev_cyc_node_id);

  my $next_cyc_node_id = get_blkr_unique_id($prev_cyc_sref);

  # traverse the cycle
  while($next_cyc_node_id ne $cyc_node_id)
  {
    push(@cycle_members,$next_cyc_node_id);

    ::EMAGENT_PERL_DEBUG("cycle edge $prev_cyc_node_id -- $next_cyc_node_id");

    my $next_cyc_node_sref = $ksdhng_chains_sess_hash_ref->{$next_cyc_node_id};

    if($cuml_sum_flg)
    {
      $cuml_block_time += $prev_cyc_sref->{$IN_WAIT_SECS};
    }
    else
    {
      ::EMAGENT_PERL_DEBUG(" Adding $cuml_block_time to cuml blocking time of $next_cyc_node_id ");
      $next_cyc_node_sref->{$CUML_BLOCK_TIME} += $cuml_block_time;
      ::EMAGENT_PERL_DEBUG(" = $next_cyc_node_sref->{$CUML_BLOCK_TIME}");
    }

    $prev_cyc_node_id = $next_cyc_node_id;
    $prev_cyc_sref    = $next_cyc_node_sref;

    $next_cyc_node_id = get_blkr_unique_id($next_cyc_node_sref); 
  }

  if($cuml_sum_flg)
  {
    # update members in cycle
    foreach my $id(@cycle_members)
    {
      ::EMAGENT_PERL_DEBUG(" Adding $cuml_block_time to cuml blocking time of $id");
      $ksdhng_chains_sess_hash_ref->{$id}{$CUML_BLOCK_TIME} = $cuml_block_time;
      ::EMAGENT_PERL_DEBUG(" = $ksdhng_chains_sess_hash_ref->{$id}{$CUML_BLOCK_TIME}");
      $visited{$id} = 1;
    }

    # One of the nodes in the cycle is chosen as 
    # final blocker to be displayed in UI.
    # cuml blocking time is computed based on this
    # node. And all nodes in cycle are updated with 
    # this value.(it is similar to num_waiters calculation
    # in a cycle which is number of nodes in cycle-1).

    # note the FB of the cycle 
    $cycle_fb{$cycle_id} = $prev_cyc_node_id;

    ::EMAGENT_PERL_DEBUG(" Designating $prev_cyc_node_id as final blocker for cycle $cycle_id");

    # save the cuml blocking time of FB 
    $fb_block_time{$prev_cyc_node_id} = $cuml_block_time;
  }
  else
  {
    # update the cuml blocking time of FB
    my $cyc_fb_id = $cycle_fb{$cycle_id};
    $fb_block_time{$cyc_fb_id} = $ksdhng_chains_sess_hash_ref->{$cyc_fb_id}{$CUML_BLOCK_TIME};
  }
}

#
# Subroutine: add_immed_blocker_info 
#
# $_[0] => Reference to hashmap of sessions in x$ksdhng_chains 
#
# Find the immediate blockers of sessions in x$ksdhng_chains
# from x$ksdhng_session_blockers.
#
sub add_immed_blocker_info
{

  my $ksdhng_chains_sess_hashmap_ref = shift;

  ################################
  # Get immediate blockers from x$ksdhng_session_blockers (>=11gR2)
  ################################

  DirectAccess::setQueryContext($DirectAccess::UNSAFE_MODE,
                                $DirectAccess::DISABLE_TRACE,
                                $DirectAccess::OUTPUT_XML);

  my @immed_blockers = DirectAccess::getFixedTable('X$KSDHNG_SESSION_BLOCKERS',
                                                   \@QUERY_SESSION_BLOCKERS,
                                                  );
  if ($DirectAccess::ERROR_CODE)
  {
    ::EMAGENT_PERL_ERROR("received error code $DirectAccess::ERROR_CODE while querying for x\$ksdhng_session_blockers");
    $ERROR_CODE = $DirectAccess::ERROR_CODE;
    return;
  }

  ::EMAGENT_PERL_DEBUG(" *********** SESSION BLOCKERS START *************");
  ::EMAGENT_PERL_DEBUG(tostring_array_of_hashmaps(\@immed_blockers));
  ::EMAGENT_PERL_DEBUG(" *********** SESSION BLOCKERS END *************");


  ################################
  # Join results from x$ksdhng_chains with x$ksdhng_session_blockers
  ################################


  # Associative array that maps uniqueid(SID_SESSION#_INSTID) of a session
  # to a list of uniqueids of its immed blockers. Example:
  # (
  #   123_4356_1 => [[144_4562_1],[654_2345_1]],
  #   156_1234_1 => [[122_3234_3],[132_1235_1][111_1123_1]],
  # )
  my %immed_blockers_hashmap = ();

  ::EMAGENT_PERL_DEBUG("Adding Immediate Blocker info");

  foreach my $row (@immed_blockers)
  {
    my $sess_unique_id = $row->{$SID}."_".
                         $row->{$SESS_SERIAL_NO}."_".
                         $row->{$INST_ID};

    my $blkr_unique_id = $row->{$BLOCKER_SID}."_".
                         $row->{$BLOCKER_SESS_SERIAL_NO}."_".
                         $row->{$BLOCKER_INSTANCE_ID};

    $ksdhng_chains_sess_hashmap_ref->{$sess_unique_id}{$IMMED_BLOCKERS} .= "[".$blkr_unique_id."]";

    my $blkr_sess_ref = {};
    if(exists $ksdhng_chains_sess_hashmap_ref->{$blkr_unique_id})
    {
      $blkr_sess_ref = $ksdhng_chains_sess_hashmap_ref->{$blkr_unique_id};
    } 
    else
    {
      $blkr_sess_ref->{$SID} = $row->{$BLOCKER_SID};
      $blkr_sess_ref->{$SESS_SERIAL_NO} = $row->{$BLOCKER_SESS_SERIAL_NO};
      $blkr_sess_ref->{$INSTANCE} = $row->{$BLOCKER_INSTANCE_ID};

      $ksdhng_chains_sess_hashmap_ref->{$blkr_unique_id} = $blkr_sess_ref;
    }
  }
}




#
# Subroutine: filter_chains
#  $_[0] => true iff single-instance
#  $_[*] => rows in wait chains
#
# Returns: the set of filtered rows
#
sub filter_chains
{
  my $instance_only = shift;
  my @rows          = @_;

  # instance filtering
  if ($instance_only)
  {
    ::EMAGENT_PERL_DEBUG("keeping only this instance");
    @rows = grep { %{$_}->{$INST_ID} == %{$_}->{$INSTANCE} } @rows;
  }
  else
  {
    ::EMAGENT_PERL_DEBUG("keeping all instances");
  }

   ::EMAGENT_PERL_INFO("keeping ".(@rows+0)." sessions from instance(s)");

  ################################
  # trivial chain filtering
  ################################

  #
  # blocked sessions have BLOCKER_IS_VALID == 1; keep all of those
  #
  my @blocked = grep { %{$_}->{$BLOCKER_IS_VALID} } @rows;
  #::EMAGENT_PERL_INFO("found ".(@blocked+0)." blocked sessions");
  ::EMAGENT_PERL_INFO("found ".($#blocked+1)." blocked sessions");

  #
  # find unblocked sessions which are blockers
  #
  my @unblocked_blockers = ();
  my @blocker_ids = ();
  for my $hash (@blocked)
  {
    my $instance = %{$hash}->{$BLOCKER_INSTANCE};
    my $sid = %{$hash}->{$BLOCKER_SID};
    my $serialno = %{$hash}->{$BLOCKER_SESS_SERIAL_NO};
    my $blocker_id = $sid."#".$serialno."@".$instance;
    foreach (@rows)
    {
      if (!%{$_}->{$BLOCKER_IS_VALID} &&
          %{$_}->{$INSTANCE} eq $instance &&
          %{$_}->{$SID} eq $sid &&
          %{$_}->{$SESS_SERIAL_NO} eq $serialno)
      {
        if (!grep { /$blocker_id/ } @blocker_ids)
        {
          # only add each once
          ::EMAGENT_PERL_DEBUG("adding BLOCKER ".$blocker_id);
          push @blocker_ids, $blocker_id;
          push @unblocked_blockers, $_;
        }
        else
        {
          ::EMAGENT_PERL_DEBUG("skipping BLOCKER ".$blocker_id.
                             ": already added");
        }
      }
    }
  }

  #
  # return blocked + unblocked_blockers
  #
  my @waiters = ();
  push @waiters, @blocked;
  push @waiters, @unblocked_blockers;
  return @waiters;
}



################################
# subroutines used for printing data in XML format 
################################


#
# Subroutine: print_metric_data 
#
# $_[0] => Reference to hashmap of sessions in x$ksdhng_chains 
#
# retuns hanganalysis data in XML format.
#
sub print_metric_data
{
  my $ksdhng_chains_sess_hash_ref = shift;

  my $xml_str;

  ::EMAGENT_PERL_DEBUG("Generating XML output");
 
  $xml_str .= "<hanganalysis_data dbversion='".$target_version."'>";

  # print top waiters
  $xml_str .= print_waiters_XML($ksdhng_chains_sess_hash_ref);

  # print top final blockers
  $xml_str .= print_fb_XML($ksdhng_chains_sess_hash_ref); 

  # print the full session attributes of all sessions
  $xml_str .= print_all_sessions_XML($ksdhng_chains_sess_hash_ref);

  $xml_str .= "</hanganalysis_data>";

  ::EMAGENT_PERL_DEBUG(" XML output: $xml_str");
  return _encode_delimiters($xml_str);
}



#
# Subroutine: print_waiters_XML 
#
# $_[0] => Reference to hashmap of sessions in x$ksdhng_chains 
#
# return top waiters info XML
#
sub print_waiters_XML
{
  my $ksdhng_chains_sess_hash_ref = shift;

  # get the ids waiters sorted by in_wait_secs in desc order
  my @blkd_sessions = sort {$ksdhng_chains_sess_hash_ref->{$b}{$IN_WAIT_SECS} 
                            <=> $ksdhng_chains_sess_hash_ref->{$a}{$IN_WAIT_SECS}
                           } keys %{$ksdhng_chains_sess_hash_ref}; 

  my $xml_str = "";

  $xml_str .= "<top_waiters>";

  # Get the top 50 waiters
  my $count = 0;
  for(my $it=0;$it<@blkd_sessions;$it++)
  {
    my $id = $blkd_sessions[$it];

    my $sess_ref = $ksdhng_chains_sess_hash_ref->{$id};
    my $cycle_id = $sess_ref->{$CHAIN_IS_CYCLE}?$sess_ref->{$CHAIN_ID}:undef;

    # Skip final blockers
    if(($sess_ref->{$BLOCKER_SID} eq 0) || ($sess_ref->{$BLOCKER_SID} eq "") ||
       ($cycle_fb{$cycle_id} == $id)) # this is final blocker in cycle
    {
      next;
    }
    else
    {
      if($count > 50)
      {
        last;
      }
      ::EMAGENT_PERL_DEBUG("Printing waiter $id");
      $count++;

      $xml_str .= "<wter ref='".$id."'>";
       $xml_str .= print_blocker_chain_XML($ksdhng_chains_sess_hash_ref,$id);
       $xml_str .= print_immed_blocker_refs_XML($ksdhng_chains_sess_hash_ref,$id);
      $xml_str .= "</wter>";
    }
  }
  $xml_str .= "</top_waiters>";

  return $xml_str;
}



#
# Subroutine: print_blocker_chain_XML 
#
# $_[0] => Reference to hashmap of sessions in x$ksdhng_chains 
# $_[1] => Id of the session for which we need blocker chain
#
# Not all sessions in blocker chain are chosen for
# display in UI. 
# We choose:
# One blocker next to waiter and if it is critical 
# then the next non-critical blocker. 
# The final blocker, if it is critical then the next 
# non-critical pre-final blocker. 
# So, we choose a max of 4 sessions in blocker chain.

sub print_blocker_chain_XML
{

  my $ksdhng_chains_sess_hash_ref = shift;
  my $id = shift;

  # update global list of all waiters
  $blocked_sessions{$id} = 1;

  my @full_chain_trace = ();
  my $xml_str = "";
  my $fb_in_cycle = "";
  my $pos = 1;
  my $non_critical_ses_found = 0;

  # this keeps track of sessions in the chain 
  # chosen for display in UI. 
  my @selected_blkrs   = ();

  my $sess_ref = $ksdhng_chains_sess_hash_ref->{$id};

  ::EMAGENT_PERL_DEBUG("Printing blocker chain of $id");

  $xml_str .= "<blkr_chain>";

  # Get the first blocker
  my $next_blkr_id = get_blkr_unique_id($sess_ref);
  my $next_blkr = $ksdhng_chains_sess_hash_ref->{$next_blkr_id};

  push(@full_chain_trace,$next_blkr_id);
  push(@selected_blkrs,$next_blkr_id);

  $xml_str .= "<sess_ref ref='$next_blkr_id' pos='$pos' />";

  if(!is_instance_critical($next_blkr_id))
  {
    $non_critical_ses_found = 1; 
  }

  # If the node is in a cycle then get the designated fb of cycle
  $fb_in_cycle = $cycle_fb{$ksdhng_chains_sess_hash_ref->{$next_blkr_id}{$CHAIN_ID}};

  if($fb_in_cycle eq $next_blkr_id)
  {
    $next_blkr_id="";
  }
  else
  {
    $next_blkr_id = get_blkr_unique_id($next_blkr);
  }

  ::EMAGENT_PERL_DEBUG(" blocker is .... .$next_blkr_id.");
  # Follow the chain(this may hit intersecting chains/cycles).
  # Follow until we hit a final blocker in chain/cycle.
  while($next_blkr_id ne '0_0_0' && $next_blkr_id ne "" && 
        $next_blkr_id ne $fb_in_cycle)
  {
    $pos++;

    my $next_blkr = $ksdhng_chains_sess_hash_ref->{$next_blkr_id};

    push(@full_chain_trace,$next_blkr_id);

    if(!$non_critical_ses_found)
    {

      if(!is_instance_critical($next_blkr_id))
      {
        push(@selected_blkrs,$next_blkr_id);

        $xml_str .= "<sess_ref ref='$next_blkr_id' pos='$pos' />"; 
        $non_critical_ses_found = 1;
      }
    }
    $next_blkr_id = get_blkr_unique_id($next_blkr);
    $next_blkr = $ksdhng_chains_sess_hash_ref->{$next_blkr_id};

    # If we hit a cycle then get the final blocker
    $fb_in_cycle = $cycle_fb{$next_blkr->{$CHAIN_ID}};
  }

  if($next_blkr_id eq $fb_in_cycle && $next_blkr_id ne "")
  {
    push(@full_chain_trace,$next_blkr_id);
  }

  # Trace back from final blocker. If fb is a critical session
  # then get the first non-critical blocker below fb.
  for(my $i=$#full_chain_trace;$i>=0;$i--)
  {
    my $sess_id = $full_chain_trace[$i];

    if(!is_instance_critical($sess_id) || $i==$#full_chain_trace)
    {
      if(!(grep { $_ eq $sess_id } @selected_blkrs))
      {
        $xml_str .= "<sess_ref ref='$sess_id' pos='$pos' />";        
       
        push(@selected_blkrs,$sess_id); 
      }
    }
    if(!is_instance_critical($sess_id)) 
    {
      last; 
    }
    $pos--;
  }

  $xml_str .= "</blkr_chain>";

  
  # add the selected sessions to the wtrs_below_fb

  my $fb_id = $full_chain_trace[$#full_chain_trace];

  my $wtrs_hash_ref;
  if(exists $wtrs_below_fb{$fb_id})
  {
    $wtrs_hash_ref = $wtrs_below_fb{$fb_id};
  }
  else
  {
    $wtrs_hash_ref = {};
    $wtrs_below_fb{$fb_id} = $wtrs_hash_ref;
  }

  $wtrs_hash_ref->{$id} = 1;
 
  # remove the fb from selected sessions and
  # add the rest to the waiters list of fb 
  my @arr = grep { $_ != $fb_id } @selected_blkrs;
  foreach my $bid(@arr)
  {
    $wtrs_hash_ref->{$bid} = 1;
    $blocked_sessions{$bid} = 1;
  }

  return $xml_str;
}


#
# Subroutine: print_immediate_blocker_refs_XML 
#
# $_[0] => Reference to hashmap of sessions in x$ksdhng_chains 
# 
sub print_immed_blocker_refs_XML
{
  my $ksdhng_chains_sess_hash_ref = shift;
  my $id = shift;

  my $immed_blkrs = $ksdhng_chains_sess_hash_ref->{$id}{$IMMED_BLOCKERS};
  my $str = "<immed_blkrs>";

  while ($immed_blkrs =~ m/\[(\d+_\d+_\d+)\]/g) 
  {
    my $blkr_id = $1;
    $str .= "<sess_ref ref='".$blkr_id."' />";

    $immed_blockers{$blkr_id} = 1;;
  } 
  $str .= "</immed_blkrs>";

  return $str;
}


#
# Subroutine: print_fbXML 
#
# $_[0] => Reference to hashmap of sessions in x$ksdhng_chains 
#
# return the top final blockers in XML format.
# 
sub print_fb_XML
{

  my $ksdhng_chains_sess_hash_ref = shift;

  my @fblkrs = sort { $fb_block_time{$b} <=> $fb_block_time{$a} } keys %fb_block_time;

  my $xml_str = "";

  $xml_str .= "<top_final_blockers>";

  my $count = 0;
  for(my $it=0;$it<=$#fblkrs && $count<50;$it++)
  {
    my $fb_id = $fblkrs[$it];

    if(!exists $wtrs_below_fb{$fb_id})
    {
      next;
    } 

    $xml_str .= "<fblkr ref='$fb_id' >";

    my $wtrs_hash_ref = $wtrs_below_fb{$fb_id};

    $xml_str .= "<waiters>";
  
    foreach my $wid(keys %{$wtrs_hash_ref})
    {
      $xml_str .= "<sess_ref ref='$wid'/>";
    }
    $xml_str .= "</waiters>";
    $xml_str .= "</fblkr>";

    $count++;
  }

 $xml_str .= "</top_final_blockers>";
}

#
# Subroutine: print_all_sessions_XML 
#
# $_[0] => Reference to hashmap of sessions in x$ksdhng_chains 
# 
sub print_all_sessions_XML
{
  my $ksdhng_chains_sess_hash_ref = shift;

  my %all_sessions = (%blocked_sessions,
                      %immed_blockers,
                      %fb_block_time);

  my $xml_str = "";

  $xml_str .= "<session_details>";

  # print the att info first
  #$xml_str .= "<att_info>";
  #foreach(my $it=0;$it<=$#;$it++)
  #{
    # reserve 1,2,3 ids for sid,serialno,instid
  #  $xml_str .= "<att id='".($it + 4)."' name='".lc($OUTPUT_COLUMNS[$it])."'/>";
  #}
  #$xml_str .= "</att_info>"; 

  my @session_ids = sort keys %all_sessions;

  foreach my $id(@session_ids)
  {
    my $sess_ref = $ksdhng_chains_sess_hash_ref->{$id};

    #$xml_str .= "<session ".lc($ID)."='$id'>";
    $xml_str .= "<session ".lc($ID)."='$id'";

    foreach(my $it=0;$it<=$#OUTPUT_COLUMNS;$it++)
    {
      my $col = $OUTPUT_COLUMNS[$it];
      if($col eq $BLOCKER_ID)
      {
        $sess_ref->{$col} = get_blkr_unique_id($sess_ref);
      }
      if($col eq $WAIT_EVENT)
      {
        $sess_ref->{$col} = EmergPerfUtil::use_XML_entityrefs($sess_ref->{$WAIT_EVENT});
      }
      if(exists $sess_ref->{$col})
      {
        #$xml_str .= "<att_ref ref='".($it + 4)."' val='".$sess_ref->{$col}."' />";
        $xml_str .= " ".lc($col)."='".EmergPerfUtil::use_XML_entityrefs($sess_ref->{$col})."'";
      }
    }
    #$xml_str .= "</session>";
    $xml_str .= "/>";
  }

  $xml_str .= "</session_details>";

  return $xml_str;
  
}


sub is_instance_critical
{
  my $unique_id = shift;

  return 0; # TODO: use x$ksupr
}



#
# Subroutine:
#  $_[0] => reference to an array of hashmaps
#  $_[*] => list of keys which will be used to
#           create the new hashmap.
#
#  It creates a hashmap of hashmaps passed to this
#  routine. The key of the new hashmap is generated
#  by combining the values of keys passed as the
#  second argument.
#
#  Suppose the input array is as follows:
#  ( sid => 40, serial_num => 12321, wait_event => "enq -TM contention" )
#  ( sid => 22, serial_num => 4333, wait_event => "SQL*Net message from client")
#    ...
#    ...
#
#  Then the output hashmap with key generated using sid and serial_num is:
#  ( 40_12321 => ( sid => 40,
#                  serial_num => 12321,
#                  wait_event => "enq -TM contention"),
#    22_43533 => ( sid => 22,
#                  serial_num => 4333,
#                  wait_event => "SQL*Net message from client")
#    ...
#    ...
#   )

sub _create_hashmap
{
  my $arr_ref = shift;
  my @arr = @$arr_ref;
  my @key_cols = @_;

  my %new_hashmap = (); # new hashmap to be returned

  if (@key_cols <= 0)
  { return; }

  for (my $row = 0; $row <= $#arr; $row++)
  {
    my $unique_key = $arr[$row]{$key_cols[0]};
    my $it=0;
    for ($it = 1; $it <= $#key_cols; $it++)
    {
      $unique_key .= "_".$arr[$row]{$key_cols[$it]};
    }
    $new_hashmap{$unique_key} = $arr[$row];
  }

  return %new_hashmap;
}

sub get_unique_id
{
  my $sess_ref = shift;
  my $unique_id = $sess_ref->{$SID}."_".
                  $sess_ref->{$SESS_SERIAL_NO}."_".
                  $sess_ref->{$INSTANCE};
  return $unique_id;
}

sub get_blkr_unique_id
{
  my $sess_ref = shift;
  if(!defined $sess_ref->{$BLOCKER_SID} ||  
     $sess_ref->{$BLOCKER_SID} eq ""    ||
     !$sess_ref->{$BLOCKER_SID}
    )
  {
    return "";
  }

  my $unique_id = $sess_ref->{$BLOCKER_SID}."_".
                  $sess_ref->{$BLOCKER_SESS_SERIAL_NO}."_";
  if(defined $sess_ref->{$BLOCKER_INSTANCE})
  {
    $unique_id .= $sess_ref->{$BLOCKER_INSTANCE};
  }
  else
  {
    $unique_id .= $sess_ref->{$BLOCKER_INSTANCE_ID};
  }
  return $unique_id;
}

sub _encode_delimiters
{
  my $value = shift;
  $value =~ s/\|/_x007C_/g;
  return $value;
}

################################
# Functions used for debugging only.
################################

sub tostring_hash
{
  my $h_ref = shift;
  my %h = %$h_ref;
  my $key;
  my $str = "";
  for $key (keys %h)
  {
    $str .= $key.":".$h{$key}.",";
  }
  return $str;
}

sub tostring_array_of_hashmaps
{
  my $hash_arr_ref = shift;
  my @hash_arr = @$hash_arr_ref;
  my $str = "";
  foreach my $item (@hash_arr)
  {
    $str .= tostring_hash($item) . "\n";
  }
  return $str;
}

sub tostring_hashmap_of_hashmaps
{
  my $hashmap_ref = shift;
  my %hashmap = %$hashmap_ref;
  my $str = "(\n";
  for my $key (keys %hashmap)
  {
    $str .= "( $key => ( "; 
    $str .= tostring_hash($hashmap{$key});
    $str .= ") \n";
  }
  $str .= ")\n";

  return $str;
}

1;
