#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/db/workload/workload_capture.pl /main/14 2010/11/04 12:40:53 keiwong Exp $
#
# workload_capture.pl
#
# Copyright (c) 2006, 2010, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      workload_capture.pl
#
#    DESCRIPTION
#      This script will start a workload capture. It may involve restarting the
#      database before the capture.
#
#    NOTES
#      When restarting RAC, all nodes are first shut down and only one node is
#      started up. After capture begins, all the nodes will be started up.
#
#    MODIFIED   (MM/DD/YY)
#       keiwong  11/12/09 - support STS capture
#       keiwong  03/27/09 - include force shutdown option for RAC
#       keiwong  04/24/09 - don't use srvctl to start up db in restricted mode
#       keiwong  10/24/08 - handle non-ASCII value
#       keiwong  10/24/08 - execute sqlplus with optional connect_descriptor
#       jsoule   06/18/08 - 
#       keiwong  04/24/08 - add connect_descriptor
#       jsoule   12/10/07 - factor out common workload utilities
#       jsoule   12/03/07 - pass dir_path_shared
#       jsoule   11/19/07 - write job output
#       keiwong  10/05/06 - Fix bug 5558757
#       keiwong  07/06/06 - Creation
# 

require "db/db_common.pl";
require "db/workload/workload_utils.pl";

use strict;
use vars qw($DBSTATE_SUCCESS_CODE $TARGET_TYPE_ORACLE_DATABASE $TARGET_TYPE_RAC_DATABASE $RESTRICTED_MODE $STATE_OPEN @SQLCMDS);
use vars qw($oracle_home $oracle_sid $db_name $username $password $role $target_type $is_rac $connect_descriptor);

# Start workload capture
#
# Parameter:
# $capture_name: name of the capture
# $capture_directory: name of directory object for capture
# $duration: length of capture in seconds
# $filter_type: "INCLUDE" or "EXCLUDE"
# $filters: name:attribute:value:type,name:attribute:value:type,...
# $restart_database: 0/1 (no/yes)
# $shutdown_options: "NORMAL" or "IMMEDIATE" or ...
# $startup_options: "NOMOUNT RESTRICT FORCE ..."
# $force_shutdown: 0/1 (no/yes)
# $create_sts: 0/1 (no/yes)
#
# Return value:
# Success: 0
# Failure: !0
sub capture_workload
{
    my $redirect_file = make_temp_file();

    # save stdout
    open OLD_STDOUT, ">& STDOUT";
    my $old_stdout = *OLD_STDOUT;

    # redirect stdout and stderr
    open STDOUT, "> $redirect_file";
    open STDERR, ">& STDOUT";

    my ($capture_name, $capture_directory, $duration, $filter_type, $filters, $restart_database, $shutdown_options, $startup_options, $force_shutdown, $create_sts) = @_;

    my $shutdown_status = 0;
    my $startup_status = 0;
    my $startup_all_status = 0;
    my $capture_status = 0;

    EMD_PERL_DEBUG("workload_capture:capture_workload restart_database=$restart_database");

    # check if capture is running
    my $validation_status = is_workload_running("capture");

    if ($validation_status > 0)
    {
        printError("Failed to start workload capture: capture is already running.");
        return dump_file($validation_status, $redirect_file, $old_stdout, 1);
    }
    elsif ($validation_status < 0)
    {
        return dump_file($validation_status, $redirect_file, $old_stdout, 1);
    }

    # check if capture name already exists
    $validation_status = workload_exists("capture", "name", $capture_name);

    if ($validation_status > 0)
    {
        printError("Failed to start workload capture: capture $capture_name already exists.");
        return dump_file($validation_status, $redirect_file, $old_stdout, 1);
    }
    elsif ($validation_status < 0)
    {
        return dump_file($validation_status, $redirect_file, $old_stdout, 1);
    }

    # make sure capture directory object exists
    $validation_status = dir_object_exists($capture_directory);

    if ($validation_status == 0)
    {
        printError("Failed to start workload capture: directory object $capture_directory does not exist.");
        return dump_file(!$validation_status, $redirect_file, $old_stdout, 1);
    }
    elsif ($validation_status < 0)
    {
        return dump_file($validation_status, $redirect_file, $old_stdout, 1);
    }

    if ($restart_database)
    {
        $shutdown_status = shutdown_database($shutdown_options, $force_shutdown);

        if (!$shutdown_status)
        {
            my $startup_options_restricted = $startup_options;

            if ($startup_options_restricted !~ /$RESTRICTED_MODE/i)
            {
                # must be first started up in restricted mode
                $startup_options_restricted .= " $RESTRICTED_MODE";
            }

            if ($startup_options_restricted !~ /$STATE_OPEN/i)
            {
                $startup_options_restricted .= " $STATE_OPEN";
            }

            # bring up only one instance before capture
            $startup_status =
                startup_database($startup_options_restricted, $oracle_sid);

            if ($startup_status)
            {
                printError("Failed to start up database instance $oracle_sid.");
            }
        }
        else
        {
            printError("Failed to shut down database $db_name.");
        }
    }

    if (!$shutdown_status && !$startup_status)
    {
        $capture_status = start_capture($capture_name, $capture_directory,
            $duration, $filter_type, $filters, $restart_database, $create_sts);
        if (!$capture_status)
        {
          # A successful capture start will dump the details (to STDOUT).

          my %whereMap = 
            ( "name",      $capture_name,
              "directory", $capture_directory
            );
          dump_workload_output($old_stdout, "capture", %whereMap);
        }
    }

    if ($restart_database && $is_rac =~ /true/i)
    {
        # bring up all instances for RAC
        $startup_all_status = startup_database($startup_options);

        if ($startup_all_status)
        {
            printError("Failed to start up RAC database $db_name.");
        }
    }

    return dump_file($shutdown_status + $startup_status + $startup_all_status +
        $capture_status, $redirect_file, $old_stdout, 1);
}

# Shut down database
# set_db_parameters should be called first
#
# Parameter:
# $shutdown_options: "NORMAL" or "IMMEDIATE" or ...
# $force_shutdown: 0/1 (no/yes)
# $instance_list: list of instances names
#
# Return value:
# Success: 0
# Failure: not 0
sub shutdown_database
{
    my ($shutdown_options, $force_shutdown, $instance_list) = @_;

    EMD_PERL_DEBUG("workload_capture:shutdown_database shutdown_options=$shutdown_options force_shutdown=$force_shutdown target_type=$target_type instance_list=$instance_list");

    my $tns = "";
    my $is_db_10i = 1;
    my $restore_db_state = 1;
    my $init_state = "";
    my @pre_shutdown_sql;
    my $shutdown_target_type =
        ($is_rac =~ /true/i) ? $TARGET_TYPE_RAC_DATABASE : $target_type;
    my $shutdown_options_formatted =
        ($is_rac =~ /true/i && $shutdown_options) ?
            '"'.$shutdown_options.'"' : $shutdown_options;

    if ($is_rac =~ /true/i && $force_shutdown)
    {
        $shutdown_options_formatted .= " -f";
    }

    my $return_value =
        shutdown_db($shutdown_target_type, $oracle_home, $oracle_sid, $db_name,
            $username, $password, $role, $tns, $instance_list, $is_db_10i,
            $shutdown_options_formatted, $restore_db_state, $init_state,
            @pre_shutdown_sql);

    return $return_value;
}

# Start up database
# set_db_parameters should be called first
#
# Parameter:
# $startup_options: "NOMOUNT RESTRICT FORCE ..."
# $instance_list: list of instances names
#
# Return value:
# Success: 0
# Failure: !0
sub startup_database
{
    my ($startup_options, $instance_list) = @_;

    EMD_PERL_DEBUG("workload_capture:startup_database startup_options=$startup_options target_type=$target_type instance_list=$instance_list");

    my $tns = "";
    my $is_db_10i = 1;
    my $sql_run_state = "";
    my $restore_db_state = 0;
    my $init_state = "";
    my $bounce_after_post_sql = 0;
    my @post_startup_sql;
    my $startup_target_type =
        ($startup_options =~ /$RESTRICTED_MODE/i) ?
            $TARGET_TYPE_ORACLE_DATABASE :
            ($is_rac =~ /true/i) ? $TARGET_TYPE_RAC_DATABASE : $target_type;
    my $startup_options_formatted = $startup_options;

    if ($startup_target_type =~ /$TARGET_TYPE_RAC_DATABASE/i &&
        $startup_options_formatted)
    {
        # trim leading and trailing spaces
        $startup_options_formatted =~ s/^\s+//;
        $startup_options_formatted =~ s/\s+$//;

        $startup_options_formatted = '"'.$startup_options_formatted.'"';
    }

    my $return_value =
        startup_db($startup_target_type, $oracle_home, $oracle_sid, $db_name,
            $username, $password, $role, $tns, $instance_list, $is_db_10i,
            $startup_options_formatted, $sql_run_state, $restore_db_state,
            $init_state, $bounce_after_post_sql, @post_startup_sql);

    return $return_value;
}

# Start capture
#
# Parameter:
# $capture_name: name of the capture
# $capture_directory: name of directory object for capture
# $duration: length of capture in seconds
# $filter_type: "INCLUDE" or "EXCLUDE"
# $filters: name:attribute:value:type,name:attribute:value:type,...
# $restart_database: 0/1 (no/yes)
# $create_sts: 0/1 (no/yes)
#
# Return value:
# Success: 0
# Failure: !0
sub start_capture
{
    my ($capture_name, $capture_directory, $duration, $filter_type, $filters, $restart_database, $create_sts) = @_;

    EMD_PERL_DEBUG("workload_capture:start_capture capture_name=$capture_name capture_directory=$capture_directory duration=$duration filter_type=$filter_type filters=$filters restart_database=$restart_database create_sts=$create_sts");

    # clear any current filters
    my $clear_filters_status = clear_current_filters();

    if ($clear_filters_status)
    {
        return $clear_filters_status;
    }

    # add workload filters
    my $add_filters_status = add_filters($filters);
    my $return_value = $add_filters_status;

    if (!$add_filters_status)
    {
        my $auto_unrestrict = $restart_database ? "TRUE" : "FALSE";

        # @SQLCMDS is defined in dbstate.sql
        @SQLCMDS =
            $create_sts ?
                "exec dbms_workload_capture.start_capture(UNISTR('$capture_name'), UNISTR('$capture_directory'), $duration, '$filter_type', $auto_unrestrict, TRUE)" :
                "exec dbms_workload_capture.start_capture(UNISTR('$capture_name'), UNISTR('$capture_directory'), $duration, '$filter_type', $auto_unrestrict)";

        my $sql_status =
            execute_sqlplus($username, $password, $role, $connect_descriptor);

        if ($sql_status != $DBSTATE_SUCCESS_CODE)
        {
            printError("Failed to start workload capture $capture_name.");
            $return_value = 1;
        }
    }

    return $return_value;
}

# Add workload filters
#
# Parameter:
# $filters: name:attribute:value:type,name:attribute:value:type,...
#
# Return value:
# Success: 0
# Failure: !0
sub add_filters
{
    EMD_PERL_DEBUG("workload_capture:add_filters filters=@_[0]");

    my @filters = split / *, */, @_[0];

    my $return_value = 0;

    while (@filters)
    {
        my ($name, $attribute, $value, $type) = split / *: */, shift @filters;

        # type can be "string" or "number"
        # @SQLCMDS is defined in dbstate.sql
        if ($type =~ /string/i)
        {
            @SQLCMDS = "exec dbms_workload_capture.add_filter(UNISTR('$name'), '$attribute', UNISTR('$value'))";
        }
        else
        {
            @SQLCMDS = "exec dbms_workload_capture.add_filter(UNISTR('$name'), '$attribute', $value)";
        }

        my $sql_status =
            execute_sqlplus($username, $password, $role, $connect_descriptor);

        if ($sql_status != $DBSTATE_SUCCESS_CODE)
        {
            printError("Failed to add workload capture filter $name.");
            $return_value = 1;
        }
    }

    return $return_value;
}

# Check if a directory object exists
#
# Parameter:
# $dir_object_name: name of directory object
#
# Return value:
# True: > 0
# False: 0
# Error: < 0
sub dir_object_exists
{
    my ($dir_object_name) = @_;

    EMD_PERL_DEBUG("workload_capture:dir_object_exists dir_object_name=$dir_object_name");

    my $return_value = -1;

    if (! $dir_object_name)
    {
        return $return_value;
    }

    my $sql = "SET ECHO OFF;\n";
    $sql .= "SET SERVEROUTPUT ON;\n";
    $sql .= "VARIABLE result NUMBER;\n";
    $sql .= "BEGIN\n";
    $sql .= " SELECT COUNT(*) INTO :result FROM all_directories WHERE ASCIISTR(directory_name) = '$dir_object_name';\n";
    $sql .= " dbms_output.put_line('result=' || :result);\n";
    $sql .= "END;\n";
    $sql .= "/\n";

    @SQLCMDS = $sql;

    my $temp_file = make_temp_file();

    my $sql_status =
        execute_sqlplus($username, $password, $role, $connect_descriptor,
            $temp_file);

    if ($sql_status != $DBSTATE_SUCCESS_CODE)
    {
        printError("Failed to check if directory object exists.");
    }

    open(SQL_OUTPUT, $temp_file);

    while (<SQL_OUTPUT>)
    {
        if ($sql_status != $DBSTATE_SUCCESS_CODE)
        {
            printError($_);
        }
        elsif (/=/)
        {
            my @output = split /=/;
            $return_value = $output[1];
            last;
        }
    }

    close(SQL_OUTPUT);

    if ($return_value > -1)
    {
        removeFile($temp_file);
    }

    EMD_PERL_DEBUG("workload_capture:dir_object_exists return_value=$return_value");

    return $return_value;
}

# Clear all current filters
#
# Return value:
# Success: 0
# Failure: !0
sub clear_current_filters
{
    EMD_PERL_DEBUG("workload_capture:clear_current_filters");

    my $return_value = 0;

    my $sql = "SET ECHO OFF;\n";
    $sql .= "BEGIN\n";
    $sql .= " FOR i in (SELECT name FROM dba_workload_filters WHERE type = 'CAPTURE' AND status = 'NEW')\n";
    $sql .= " LOOP\n";
    $sql .= "   dbms_workload_capture.delete_filter(i.name);\n";
    $sql .= " END LOOP;\n";
    $sql .= "END;\n";
    $sql .= "/\n";

    @SQLCMDS = $sql;

    my $temp_file = make_temp_file();

    my $sql_status =
        execute_sqlplus($username, $password, $role, $connect_descriptor,
            $temp_file);

    if ($sql_status != $DBSTATE_SUCCESS_CODE)
    {
        printError("Failed to clear current filters.");
        $return_value = 1;

        open(SQL_OUTPUT, $temp_file);

        while (<SQL_OUTPUT>)
        {
            printError($_);
        }

        close SQL_OUTPUT;
    }

    if ($return_value == 0)
    {
        removeFile($temp_file);
    }

    return $return_value;
}

1;
