#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/db/workload/workload_utils.pl /st_emgc_pt-12.1.0.4pg/1 2011/11/07 21:11:34 keiwong Exp $
#
# workload_utils.pl
# 
# Copyright (c) 2007, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      workload_utils.pl - Utilities used by WCR jobs.
#
#    DESCRIPTION
#      Standard utilities for WCR jobs.
#
#    NOTES
#      
#
#    MODIFIED   (MM/DD/YY)
#    keiwong     10/30/11 - add dbtimezone to capture details
#    keiwong     10/24/08 - handle non-ASCII value
#    keiwong     10/24/08 - add db_version parameter
#    jsoule      06/18/08 - 
#    keiwong     04/24/08 - add connect_descriptor
#    jsoule      12/11/07 - Creation
# 

require "emd_common.pl";
require "db/dbstate.pl";

use strict;
use vars qw($DBSTATE_SUCCESS_CODE $DBROLE_NORMAL @SQLCMDS $NT);
use vars qw($oracle_home $oracle_sid $db_name $username $password $role $target_type $is_rac $db_version $connect_descriptor);

# Set up global database parameters
#
# Parameter:
# $oracle_home: Oracle home
# $oracle_sid: Oracle SID
# $db_name: DB name
# $username: DB user name
# $password: DB user password
# $role: DB role
# $target_type: target type
# $is_rac: true for RAC target
# $db_version: DB version
# $connect_descriptor: connect descriptor
#
# Return value:
# None
sub set_db_parameters
{
    ($oracle_home, $oracle_sid, $db_name, $username, $password, $role, $target_type, $is_rac, $db_version, $connect_descriptor) = @_;

    EMD_PERL_DEBUG("workload_utils:set_db_parameters oracle_home=$oracle_home oracle_sid=$oracle_sid db_name=$db_name username=$username role=$role target_type=$target_type is_rac=$is_rac db_version=$db_version connect_descriptor=$connect_descriptor");

    set_db_var($oracle_home, $oracle_sid, 1);
}

# Generate SQL to get capture info for a capture
#
# Parameter:
# %whereMap - where clause conditions (not empty)
#
# Return value:
# The generated SQL
sub capture_details_sql
{
  my (%whereMap) = @_;

  my $capture_details_sql = '';

  #   gc relation         / db relation transform
  my %selectMap =
    ( "capture_end_time",  "TO_CHAR(c.end_time,'yyyyMMddHH24MISS')",
      "capture_name",      "c.name",
      "capture_start_time","TO_CHAR(c.start_time,'yyyyMMddHH24MISS')",
      "capture_status",    "c.status",
      "dbid",              "TO_CHAR(c.dbid,'FM9999999999999999')",
      "dbname",            "c.dbname",
      "dbversion",         "c.dbversion",
      "dir_path",          "c.dir_path",
      "dir_path_shared",   "c.dir_path_shared",
      "directory",         "c.directory",
      "parallel",          "c.parallel",
      "start_scn",         "TO_CHAR(c.start_scn,'FM9999999999999999')",
      "capture_dbtimezone","(select dbtimezone from dual)"
    );
  my $itemSeparator = "||CHR(10)||";
  my @selectItems   = ();
  my $gc_name;
  my $db_column;
  while (($gc_name, $db_column) = each %selectMap)
  {
    push(@selectItems, "'$gc_name='||$db_column");
  }

  my @whereConditions = ();
  my $gc_value;
  while (($db_column, $gc_value) = each %whereMap)
  {
    push(@whereConditions, "ASCIISTR(c.$db_column) = '$gc_value'");
  }

  $capture_details_sql =
    "select ".join($itemSeparator, @selectItems)." capture_attrs ".
      "from dba_workload_captures c ".
     "where ".join(" and ", @whereConditions);

  EMD_PERL_DEBUG("workload_utils:capture_details_sql=$capture_details_sql");

  return $capture_details_sql;
}

# Generate SQL to get replay info for a replay
#
# Parameter:
# %whereMap - where clause conditions (not empty)
#
# Return value:
# The generated SQL
sub replay_details_sql
{
  my (%whereMap) = @_;

  my $replay_details_sql = '';

  #   gc relation           / db relation transform
  my %selectMap =
    ( "capture_end_time",    "TO_CHAR(c.end_time,'yyyyMMddHH24MISS')",
      "capture_name",        "c.name",
      "capture_start_time",  "TO_CHAR(c.start_time,'yyyyMMddHH24MISS')",
      "capture_status",      "c.status",
      "dbid",                "c.dbid",
      "dbname",              "c.dbname",
      "dir_path",            "c.dir_path",
      "directory",           "c.directory",
      "num_clients",         "TO_CHAR(r.num_clients,'FM9999999999999999')",
      "parallel",            "c.parallel",
      "preprocessed_version","c.last_processed_version",
      "replay_end_time",     "TO_CHAR(r.end_time,'yyyyMMddHH24MISS')",
      "replay_name",         "r.name",
      "replay_start_time",   "TO_CHAR(r.start_time,'yyyyMMddHH24MISS')",
      "replay_status",       "r.status"
    );
  my $itemSeparator = "||CHR(10)||";
  my @selectItems   = ();
  my $gc_name;
  my $db_column;
  while (($gc_name, $db_column) = each %selectMap)
  {
    push(@selectItems, "'$gc_name='||$db_column");
  }

  my @whereConditions = ();
  my $gc_value;
  while (($db_column, $gc_value) = each %whereMap)
  {
    push(@whereConditions, "ASCIISTR(r.$db_column) = '$gc_value'");
  }

  $replay_details_sql =
    "select ".join($itemSeparator, @selectItems)." replay_attrs ".
      "from dba_workload_captures c, dba_workload_replays r ".
     "where r.capture_id = c.id and ".join(" and ", @whereConditions);

  EMD_PERL_DEBUG("workload_utils:replay_details_sql=$replay_details_sql");

  return $replay_details_sql;
}

# Check if workload is running
#
# Parameters:
# $workload_type - the type of workload (capture vs. replay)
#
# Return value:
# True: > 0
# False: 0
# Error: < 0
sub is_workload_running
{
    my ($workload_type) = @_;

    EMD_PERL_DEBUG("workload_utils:is_".$workload_type."_running");

    my $return_value = -1;

    my $sql = "SET ECHO OFF;\n";
    $sql .= "SET SERVEROUTPUT ON;\n";
    $sql .= "VARIABLE result NUMBER;\n";
    $sql .= "BEGIN\n";
    $sql .= " SELECT COUNT(*) INTO :result FROM dba_workload_".$workload_type."s WHERE status = 'IN PROGRESS';\n";
    $sql .= " dbms_output.put_line('result=' || :result);\n";
    $sql .= "END;\n";
    $sql .= "/\n";

    @SQLCMDS = $sql;

    my $temp_file = make_temp_file();

    my $sql_status =
        execute_sqlplus($username, $password, $role, $connect_descriptor,
            $temp_file);

    if ($sql_status != $DBSTATE_SUCCESS_CODE)
    {
        printError("Failed to check if $workload_type is running.");
    }

    open(SQL_OUTPUT, $temp_file);

    while (<SQL_OUTPUT>)
    {
        if ($sql_status != $DBSTATE_SUCCESS_CODE)
        {
            printError($_);
        }
        elsif (/=/)
        {
            my @output = split /=/;
            $return_value = $output[1];
            last;
        }
    }

    close(SQL_OUTPUT);

    if ($return_value > -1)
    {
        removeFile($temp_file);
    }

    EMD_PERL_DEBUG("workload_utils:is_".$workload_type."_running ".
                   "return_value=$return_value");

    return $return_value;
}

# Check if a workload exists
#
# Parameter:
# $workload_type: type of workload (capture vs. replay)
# $workload_column: column to check
# $workload_value: value sought
#
# Return value:
# True: > 0
# False: 0
# Error: < 0
sub workload_exists
{
    my ($workload_type, $workload_column, $workload_value) = @_;

    EMD_PERL_DEBUG("workload_utils:workload_exists ".
                   "workload_type=$workload_type,".
                   "$workload_column='$workload_value'");

    my $return_value = -1;

    if (!$workload_column || !$workload_value)
    {
        return $return_value;
    }

    my $sql = "SET ECHO OFF;\n";
    $sql .= "SET SERVEROUTPUT ON;\n";
    $sql .= "VARIABLE result NUMBER;\n";
    $sql .= "BEGIN\n";
    $sql .= " SELECT COUNT(*) INTO :result FROM dba_workload_".$workload_type."s WHERE ASCIISTR($workload_column) = '$workload_value';\n";
    $sql .= " dbms_output.put_line('result=' || :result);\n";
    $sql .= "END;\n";
    $sql .= "/\n";

    @SQLCMDS = $sql;

    my $temp_file = make_temp_file();

    my $sql_status =
        execute_sqlplus($username, $password, $role, $connect_descriptor,
            $temp_file);

    if ($sql_status != $DBSTATE_SUCCESS_CODE)
    {
        printError("Failed to check if $workload_type exists.");
    }

    open(SQL_OUTPUT, $temp_file);

    while (<SQL_OUTPUT>)
    {
        if ($sql_status != $DBSTATE_SUCCESS_CODE)
        {
            printError($_);
        }
        elsif (/=/)
        {
            my @output = split /=/;
            $return_value = $output[1];
            last;
        }
    }

    close(SQL_OUTPUT);

    if ($return_value > -1)
    {
        removeFile($temp_file);
    }

    EMD_PERL_DEBUG("workload_utils:workload_exists ".
                   "return_value=$return_value");

    return $return_value;
}

# Make a temporary file
#
# Return value:
# Temporary file name
sub make_temp_file
{
    EMD_PERL_DEBUG("workload_utils:make_temp_file");

    my $dir = tempdir(CLEANUP => 1);
    my ($temp_handle, $temp_file);

    if (!$NT)
    {
        ($temp_handle, $temp_file) = tempfile(DIR => $dir);
    }
    else
    {
        $temp_file = "$dir\\"."wcapture.$$";
    }

    return $temp_file;
}

# Return status and optionally send content of a file to a file handle if
# status is not zero.
#
# Parameter:
# $status: return status
# $file: name of file to dump
# $output: file handle for output
# $remove: optionally remove file afterward
#
# Return value:
# Return status
sub dump_file
{
    my ($status, $file, $output, $remove) = @_;

    # send content of file to $output if status is not 0
    if ($status && $file && $output)
    {
        open FILE, $file;
        print $output <FILE>;
        close FILE;
    }

    if ($remove && $file)
    {
        removeFile($file);
    }

    return $status;
}

# Dump the output of a capture
#
# Parameters:
# $output - file handle for STDOUT
# $workload_type - the type of workload (capture vs. replay)
# %whereMap - map for where conditions
#             where clause becomes key1 = value1 and ... and keyN = valueN
sub dump_workload_output
{
  my ($output, $workload_type, %whereMap) = @_;

  if ("capture" eq $workload_type)
  {
    @SQLCMDS = capture_details_sql(%whereMap);
  }
  elsif ("replay" eq $workload_type)
  {
    @SQLCMDS = replay_details_sql(%whereMap);
  }
  else
  {
    EMD_PERL_DEBUG("Unknown workload_type: $workload_type; dumping nothing.");
    return;
  }

  my $temp_file = make_temp_file();

  execute_sqlplus($username, $password, $role, $connect_descriptor, $temp_file);

  open(SQL_OUTPUT, $temp_file);
  while (<SQL_OUTPUT>)
  {
    if (/.*=.*/)
    {
      print $output $_;
    }
  }
  close SQL_OUTPUT;
}

# Execute sqlplus
#
# Parameter:
# dbUsername: The username to log onto the database with.
# dbPassword: The password for the username.
# dbRole: The role for the username.
# tns The TNS descriptor. Optional.
# outputFile Redirect output to the named file. Optional.
#
# Return value: 
# $DBSTATE_SUCCESS_CODE or $DBSTATE_ERROR_CODE with the output of the 
# SQLPLUS session written to standard output or to the file passed in.
sub execute_sqlplus
{
    my ($dbUsername, $dbPassword, $dbRole, $tns, $outputFile) = @_;
    my $connStr;

    if (defined($tns) && $tns ne "")
    {
        if (!($tns =~ /^'/ || $tns =~ /^"/))
        {
            $tns = '"'.$tns.'"';
        }

        $connStr = "${dbUsername}/${dbPassword}\@${tns}";
    }
    else
    {
        $connStr = "${dbUsername}/${dbPassword}";
    }

    if (!($dbRole =~ /$DBROLE_NORMAL/i || $dbRole eq ''))
    {
        $connStr .= " as $dbRole";
    }

    return (&executeSQLPlus($connStr, $outputFile));
}

# Check if a db version is equal to or higher than a given version
#
# Parameter:
# $version1: version in question
# $version2: version to compare with
#
# Return value:
# True: > 0
# False: 0
# Error: < 0
sub isVersionEqualOrHigher
{
    my ($version1, $version2) = @_;

    EMD_PERL_DEBUG("workload_capture:isVersionEqualOrHigher version1=$version1 version2=$version2");

    if (!$version1 || !$version2)
    {
        return -1;
    }

    my @parts1 = split('\.', $version1);
    my @parts2 = split('\.', $version2);

    # make lengths equal with '0'
    while (@parts1 < @parts2)
    {
        push(@parts1, 0);
    }

    while (@parts2 < @parts1)
    {
        push(@parts2, 0);
    }

    for (my $i = 0; @parts1 > $i && @parts2 > $i; $i++)
    {
        if ($parts1[$i] < $parts2[$i])
        {
            return 0;
        }
        elsif ($parts1[$i] > $parts2[$i])
        {
            return 1;
        }
    }

    # equal
    return 1;
}

1;
