#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/has/ClusterHasDiscovery.pm /st_emgc_pt-12.1.0.4pg/1 2012/10/29 13:17:53 ajdsouza Exp $
#
# ClusterHasDiscovery1.pl
# 
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      ClusterHasDiscovery.pm - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    ajdsouza    09/28/12 - Use getcrshome to get crshome from inventory
#    shasingh    01/30/12 - bug 13613078: error handling
#    nismehta    11/02/11 - include both olsnodes and inventory for nodelist
#    ajdsouza    04/06/11 - add function hasGetClusterNodeListPre11g to get nodelist
#    ajdsouza    04/06/11 - cash the discovered inventory for a run
#    rsamaved    03/02/11 - include get_osType function locally
#    rsamaved    11/27/10 - include sCommon with absolute path for discovery
#    ajdsouza    07/02/10 - check for emcrsp before execution in getOcrType
#                            in checkForEmcrsp separate the EMD_PERL debug line from return
#    shasingh    04/01/10 - Perl module for cluster and has taret discovery
#    shasingh    04/01/10 - Creation
# 

package ClusterHasDiscovery;

use Exporter;
use strict;
use warnings;
use File::Spec::Functions;
use File::Basename;
use File::Path;
#use has::sCommon;
use XML::Parser;
use emdcommon;
use Net::Domain qw(hostdomain);
use Sys::Hostname;
use Data::Dumper;

use hostOSD;


sub hasGetCRSHomeFromBin($;);
sub hasIsDir($;);


my $discovery_root = $ENV{DISC_ROOT};
require "$discovery_root/discover/has/sCommon.pm"; 

our @ISA = qw(Exporter);
our @EXPORT = qw( );

#Discovery LOG CATEGORY
my $LOG_CATEGORY = "CLUSTER_HAS_DISCOVERY: ";
# FUNCTION : clusterHasTargetDiscovery
#
# DESC
# return the cluster name, crsHome and type(has|cluster)
#     
# ARGUMENTS
# emdRoot 
# crsHome if known
#----------------------------------------------------------------------------
sub clusterHasTargetDiscovery($;$;$)
{
  my ($emdRoot,$hostName,$crsHome)  = @_;
      
  my $hascluster = 'cluster';
  my $clusterName;

  undef $crsHome unless $crsHome;

  emdcommon::emdcommon::EMD_PERL_ERROR("$LOG_CATEGORY EMDROOT is not passed as an argument, failed discovery")
    and return unless $emdRoot;

  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY emdRoot=$emdRoot");

  if (not $crsHome )
  {
     if (defined($ENV{CRS_HOME}))
     {
       if ($ENV{CRS_HOME} ne "#CRS_HOME#")
       {
          $crsHome = $ENV{CRS_HOME};
          emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY crsHome is picked from the environment CRS_HOME as $crsHome");
       }
     }
  }
  else
  {
    emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY crsHome passed as argument=$crsHome");
  }
  
  $clusterName = '';
  # a crsHome has been passed for discovery
  if ($crsHome)
  {

    if ( checkForEmcrsp($crsHome) and getOcrType($crsHome) =~ /^has$/  )
    {
      $hascluster = 'has';
      $clusterName = getHasName($crsHome,$hostName);
      emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY hasName=$clusterName");
    }
    else
    {
      $hascluster = 'cluster';
      $clusterName = getCemutloClusterName($crsHome);
      emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY clusertName=$clusterName") if $clusterName;
      emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY clusertName is null from cemutlo ") unless $clusterName;
    }

  }

   # Attempt to search OUI inventory for the CRS Home if we still could not get a clsName
   if ( not $clusterName )
   {
       my $hasDiscref;
       # bug 13613078, catch runtime exception
       my $dieh = $SIG{__DIE__} if $SIG{__DIE__};
       $SIG{__DIE__}='';
       eval { 
         $hasDiscref = getClusterDataFromInventory()};
       $SIG{__DIE__} = $dieh if $dieh;
 
        if ( $hasDiscref and ref($hasDiscref) and ref($hasDiscref) =~ /HASH/ )
       {

        for my $ch ( keys %{$hasDiscref} )
        {
          next unless $hasDiscref->{$ch}{discover};
          emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY Checking $ch ...");

          if ( checkForEmcrsp($ch) and getOcrType($ch) =~ /^has$/ )
          {
            $hascluster = 'has';
            $clusterName = getHasName($ch, $hostName);
            emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY hasName=$clusterName");
          }
          else
          {
            $hascluster = 'cluster';
            $clusterName = getCemutloClusterName($ch);
            emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY clusertName=$clusterName");
          }

          $clusterName =~ s/^\s+|\s+$// if $clusterName;

          if ($clusterName)
          {
            emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY cemutlo works fine in $ch. Found ($clusterName, $ch)");
            $crsHome = $ch;
            last;
          }
        }
      }
    }
    
    
    # TODO check if this stack supported NG agent with 9.2 cluster

   if ( not $clusterName )
   {
        # still empty, check cemutls in case of 92 cluster
        $hascluster = 'cluster';
        $clusterName = getCemutlsClusterName($emdRoot);
        $clusterName =~ s/^\s+|\s+$// if $clusterName;

        emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY clusertName=$clusterName") if $clusterName;

        if ( not $clusterName )
        {
          $hascluster = 'cluster';
          # it's a 9.2 cluster, setting crsHome to emtpy.
          $crsHome = '';
        }
    }
     $clusterName = '' unless $clusterName;
     emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY Returning clusterName:$clusterName");

    # append first node name if cluster name is crs and target type is cluster to get a distinct targetName
    if ( $hascluster =~ /^cluster$/ )
    {
   #   $clusterName = has::Common::appendNodeNameToClusterName($clusterName,$crsHome) if $clusterName and $crsHome;
    }
   return ($clusterName, $crsHome, $hascluster);

}


#------------------------------------------------------------------------------
# FUNCTION :    hasGetCRSHomeFromBin
#
# DESC
# return the crs_home returned by the getCrsHome binary in that oracle home
#
# ARGUMENTS
#  oracle_home
#
#  RETURNS
#  crsHome from getcrshome binary
#
#------------------------------------------------------------------------------
sub hasGetCRSHomeFromBin($;)
{
  my ($oracleHome) =@_;

  emdcommon::EMD_PERL_DEBUG("has::Common::hasGetCRSHomeFromBin: oracle home directory passed is null") and return unless $oracleHome;

  emdcommon::EMD_PERL_DEBUG("has::Common::hasGetCRSHomeFromBin: oracle home directory $oracleHome is not accessible") and return 
    unless ClusterHasDiscovery::hasIsDir($oracleHome);

  # from the oracle Home passed get the full path for getcrsHome if it exists
  my $execname = 'getcrshome'; 
  my $execpath;
  my $srvmadmin = catdir('srvm','admin');

  # this is only in dev env
  my $hasbin = catdir('has','bin');

  for my $bin ( ( 'bin', $srvmadmin, $hasbin ) )
  {

    my $binpath = catdir($oracleHome,$bin);

    for my $exe ( ($execname,"$execname.exe","$execname.bat") )
    {
      $execpath = catfile($binpath,$exe);

      if ( ClusterHasDiscovery::isReadable($execpath) )
      {
        last;
      }
      else
      {
       undef $execpath;
      }

    }

     last if $execpath;

  }

  emdcommon::EMD_PERL_DEBUG("has::Common::hasGetCRSHomeFromBin: Failed to get getcrshome under directory $oracleHome") and return unless $execpath;

  # get the results from getCrsHome
  my %command_args = (exit_failure_list => [()]);

  my $oldOH = $ENV{ORACLE_HOME} if $ENV{ORACLE_HOME};
  my $oldCRSHome = $ENV{CRS_HOME} if $ENV{CRS_HOME};
  
  $ENV{ORACLE_HOME} = $oracleHome if $oracleHome;
  $ENV{CRS_HOME} = $oracleHome if $oracleHome;

  my $crsHome = runsystemcommand($execpath,'',\%command_args);

  if ( $oldOH )
  {
    $ENV{ORACLE_HOME} = $oldOH;
  }
  else
  {
    delete $ENV{ORACLE_HOME} if $ENV{ORACLE_HOME};
  }

  if ( $oldCRSHome )
  {
    $ENV{CRS_HOME} = $oldCRSHome;
  }
  else
  {
    delete $ENV{CRS_HOME} if $ENV{CRS_HOME};
  }

  emdcommon::EMD_PERL_DEBUG("has::Common::hasGetCRSHomeFromBin Error getting crsHome name from $execpath") and return if $command_args{command_return_status};

  chomp($crsHome) if $crsHome;
  $crsHome =~ s/\n/ /g if $crsHome;
  $crsHome =~ s/^\s+|\s+$// if $crsHome;

  return $crsHome if $crsHome;

  emdcommon::EMD_PERL_DEBUG("has::Common::hasGetCRSHomeFromBin: Failed to get crshome under directory $oracleHome from $execpath "); 

}


#------------------------------------------------------------------------------
# FUNCTION : getHasName
#
# DESC
#  generate a name for has ( has_<hostname> )
#
# ARGUMENTS
#   crsHome for vendor or EMDROOR
# RETURN
#  cluster name 
#------------------------------------------------------------------------------
sub getHasName(;$$)
{
   my ( $crsHome,$hostName ) = @_;
   my $clsName;
  
   $clsName = "has_$hostName" if $hostName;
   $clsName = "has_" unless $clsName;
  
   return $clsName;
}


#------------------------------------------------------------------------------
# FUNCTION :  checkForEmcrsp
#
# DESC
#  checks if emcrsp is present
#
# ARGUMENTS
# crsHome if known
#
# RETRUNS
#  1 for true
#------------------------------------------------------------------------------
sub checkForEmcrsp($)
{
  my ( $crsHome ) = @_;


  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY cluster home directory passed is null") and return unless $crsHome;

  # check if emcrsp exists in the crs bin dir
  my $emcrsppath;
  my %dirlist;

  # this is only in dev env
  my $hasbin = catdir('has','bin');

  %dirlist = ( 1=>$crsHome );

  for my $order ( sort {$a<=>$b} keys %dirlist )
  {

    my $dir = $dirlist{$order};
    next unless $dir;

    for my $bin ( ( 'bin',$hasbin ) )
    {
      my $binpath = catdir($dir,$bin);

      for my $exe ( ('emcrsp','emcrsp.exe','emcrsp.bat') )
      {
        $emcrsppath = catfile($binpath,$exe);

        if ( isReadable($emcrsppath) )
        {
          last;
        }
        else
        {
         undef $emcrsppath;
        }
      }
       last if $emcrsppath;
    }
    last if $emcrsppath;
  }


  # use emcrsp to get node name if it exists
  unless ( $emcrsppath ) {
   emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY emcrsp is not present in cluster home $crsHome");
   return;
  }

  return 1;
}

#------------------------------------------------------------------------------
# FUNCTION :    isReadable
#
# DESC
# true if path is readable false otherwise
#
# ARGUMENTS
# path
#
#  RETURNS
#
#------------------------------------------------------------------------------
sub isReadable($;)
{
  my ( $path ) = @_;
  return unless $path;
  stat $path;
  return 1 if -e $path and -r $path;
  return;
}

#------------------------------------------------------------------------------
# FUNCTION :    hasIsDir
#
# DESC
# true if dir false otherwise
#
# ARGUMENTS
# path
#
#  RETURNS
#
#------------------------------------------------------------------------------
sub hasIsDir($;)
{
  my ( $path ) = @_;

  return unless $path;

  stat $path;

  return 1 if -e $path and -r $path and -d $path;

  return;
}



#------------------------------------------------------------------------------
# FUNCTION :    getCemutloClusterName
#
# DESC
#  get the clustername from cemutlo for oracle clusters
#
# ARGUMENTS
#   crsHome 
# RETURN
#  cluster name
#------------------------------------------------------------------------------
sub getCemutloClusterName($)
{
  my ( $crsHome) =  @_;
  my $clusterName;

  my %cemutlo_command_args = (exit_failure_list => [()]);
  
  #setCRSEnv($crsHome);
  my $cmdfull = catfile("$crsHome",'bin','cemutlo');
  $clusterName = executeCommand($crsHome,"$cmdfull -n 2>&1",'',\%cemutlo_command_args);
  #restoreCRSEnv();
    
  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY Error getting cluster name from cemutlo") 
     and return if $cemutlo_command_args{command_return_status};
  
  chomp($clusterName) if $clusterName;
  $clusterName =~ s/\n/ /g if $clusterName;
  $clusterName =~ s/^\s+|\s+$// if $clusterName;

  if ( $clusterName )
  {
      if ( $clusterName =~ /[^a-zA-Z0-9\-_]+/ )
      {
        # crs may be shutdown, to avoid descrepency between discovery on different nodes, do not 
        # return a clustername
        #$clusterName = 'crs';
        undef $clusterName;
      }
  }

  return $clusterName if $clusterName;

  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY Failed to get cluster name from cemutlo") and return;
}

#------------------------------------------------------------------------------
# FUNCTION :    hasCemutlsClusterName
#
# DESC
#  get the clustername from cemutls for non oracle clusters
#
# ARGUMENTS
#   crsHome for vendor or EMDROOR
# RETURN
#  cluster name
#------------------------------------------------------------------------------
sub getCemutlsClusterName($)
{
  my ($dir) = @_;
 
  # bug 4667678
  # TBD is solaris 64 bit run the 64 bit cemutls
  my $cemutls;
  my %cemutls_command_args = (exit_failure_list => [()]);
      
  if (get_osType() eq "SOL")
  { 
    # bug fix 8459125
    my $o = runsystemcommand("file /opt/ORCLcluster/lib/libskgxn2.so 2>&1",'',\%cemutls_command_args);
    if ($o and $o =~ /64-bit/)
    {
       my $cemutls64path;
       $cemutls64path = catfile($dir,'bin') if $dir;
       $cemutls64path = catfile($cemutls64path,'cemutls64') if $cemutls64path;
       stat $cemutls64path if $cemutls64path;
       $cemutls = "cemutls64" if $cemutls64path and -e $cemutls64path;
    }

  }

  $cemutls = "cemutls" unless $cemutls;

  %cemutls_command_args = (exit_failure_list => [()]);
  #setCRSEnv($dir);
  my $cmdfull = catfile("$dir",'bin','cemutls');
  
  my $vo;
  if (-f $cmdfull)
  {
      $vo = executeCommand($dir,"$cmdfull -w 2>&1",'',\%cemutls_command_args);
  }
  #restoreCRSEnv();
  
  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY Error getting cluster name from $cemutls") 
    and return if $cemutls_command_args{command_return_status};

  chomp($vo) if $vo;
  $vo =~ s/\n/ /g if $vo;
  $vo =~ s/^\s+|\s+$// if $vo;

  return $vo if $vo;

  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY Failed to get cluster name from $cemutls") and return;
}

sub executeCommand($;$;$$)
{
	my ($oh,$cmd,$args,$argref) = @_;
	
	my($oldORACLEHOme, $oldCRSHome) = ($ENV{ORACLE_HOME}, $ENV{CRS_HOME});
	
	$ENV{ORACLE_HOME} =  $oh;
    $ENV{CRS_HOME} = $oh;
    my $results = runsystemcommand($cmd,$args,$argref);
    
    $ENV{ORACLE_HOME} =  $oldORACLEHOme;
    $ENV{CRS_HOME} = $oldCRSHome;
     
    return $results;	
}

#------------------------------------------------------------------------------
# FUNCTION : runsystemcommand($;$$)
#
# DESC 
# Run a system command , retry n times if it times out
#
# ARGUMENTS
# command to be executed ( e.g. crsctl start crs )
# variable args to the command ( eg nodename)
#  a hash ref with
#  {timeout} in seconds, default 120
#  {tries} no of tries , default 2. 
#  {timeout_return} flag 1 to indicate if function should return 
#   in case of timeout, default is to die
#  {exit_failure_list}
#  {exit_success_list}
#   The lists to indicate failure and sucess exit status if they are other
#   than 0=Success all other exit status are =Fail
#   You can define one of the list of both the lists
#    e.g. for 'crsctl start crs' 
#    {exit_failure_list}=[(1,3,5,6)]
#    {exit_success_list}=[(0,2)]
#
# RETURNS:
#  result set either an array or a arg
#  the $? for the os command as {command_return_status} of the hash ref arg
#  {command_error_message} returns $! $@
#
#------------------------------------------------------------------------------
sub runsystemcommand( $;$$ )
{
  my ($fullcmd,$args,$argref) = @_;

  my $cmd;
  my $cmdargs;

  #split any args from command
  ($cmd,$cmdargs) = ( $fullcmd =~ /^([^\s]+) (.*)/ );
  $cmd = $fullcmd and undef $cmdargs unless $cmd;

  $cmd =~ s/^\s+//g if $cmd;

  # $cmd is asolute path , invoke runsystemcommand
  return has::sCommon::runsystemcommand($fullcmd,$args,$argref)
}

sub getClusterDataFromInventory(;$)
{
#<HOME NAME="OraCrs11g_home" LOC="/scratch/11.1.0.6/crs" TYPE="O" IDX="1" CRS="true">
#   <NODE_LIST>
#      <NODE NAME="stbdq16"/>
#      <NODE NAME="stbdq17"/>
#   </NODE_LIST>
#</HOME>

  sub inventory_fn(\%\%)
  {

    my ( $elref, $rsref ) = @_;
    my $rsrefptr;

    return 1 unless $elref->{element} and $elref->{element} =~ /^(HOME)$/i;

    # get crsHome using getcrshome bin from the Oracle Home in inventory
    # save this value , so it is executed only once
    if (  $elref->{attrs} and $elref->{attrs}{LOC} and not $rsref->{from_getcrshome} ) 
    {
      my $tempOh = $elref->{attrs}{LOC};
      $tempOh =~ s/^\s+|\s+$// if $tempOh;

      my $crsHomeFromBin = ClusterHasDiscovery::hasGetCRSHomeFromBin($tempOh) if $tempOh;

      if  ( $crsHomeFromBin )
      {
        $rsref->{from_getcrshome}=$crsHomeFromBin;
      }
     
    }

    return 1
     if $elref->{attrs} and $elref->{attrs}{remote} and $elref->{attrs}{remote} =~ /^true$/i;

    return 1
     if $elref->{attrs} and $elref->{attrs}{REMOTE} and $elref->{attrs}{REMOTE} =~ /^true$/i;

    emdcommon::EMD_PERL_INFO("$LOG_CATEGORY LOC is NULL for CRS in Oracle Inventory file")
    and return 1 unless $elref->{attrs} and $elref->{attrs}{LOC};

    # save oracle homes, add nodes later on
    my $ch = $elref->{attrs}{LOC};
    $rsrefptr->{$ch}{discover}{crs_home}=$ch;

    my $idx = $elref->{attrs}{IDX} if exists $elref->{attrs}{IDX};
    $idx = 0 unless defined $idx;
    $idx++ if $idx =~ /\d+/;
    $rsrefptr->{$ch}{discover}{IDX}=$idx;
    $rsrefptr->{$ch}{discover}{IDX}=-1 unless $rsrefptr->{$ch}{discover}{IDX};

    $rsrefptr->{$ch}{discover}{TYPE}=$elref->{attrs}{TYPE} if $elref->{attrs}{TYPE};
    $rsrefptr->{$ch}{discover}{TYPE}=0 unless $rsrefptr->{$ch}{discover}{TYPE};

    # get NODE NAME from children
    return 1 unless  $elref->{children} and @{$elref->{children}};

    for my $nodeListRef ( @{$elref->{children}} )
    {
      next unless $nodeListRef and $nodeListRef->{element} and $nodeListRef->{element} =~ /^NODE_LIST$/i;

      next unless $nodeListRef->{children} and @{$nodeListRef->{children}};

      for my $noderef ( @{$nodeListRef->{children}} )
      {
         next unless $noderef and $noderef->{element} and $noderef->{element} =~ /^NODE$/i;

          emdcommon::EMD_PERL_INFO("$LOG_CATEGORY NODE name is null in Oracle Inventory file")
          and next unless $noderef->{attrs} and $noderef->{attrs}{NAME};

         $rsrefptr->{$ch}{discover}{nodes}{$noderef->{attrs}{NAME}}=1;
      }
      last;
    }
    
    # inventory has crs=true so save this as crs_home for discovery
    if ( $elref->{attrs} and $elref->{attrs}{CRS} and $elref->{attrs}{CRS} =~ /^true$/i )
    {
     $rsref->{$ch}= $rsrefptr->{$ch} if $rsrefptr->{$ch};
    }
    # this is not a crs_home as it does nto have a crs=true so it is saved as a discovered oracel home
    else
    {
      $rsref->{other_oracle_homes}{$ch} = $rsrefptr->{$ch} if $rsrefptr->{$ch};
    }

    return 1;
  }

  my $invPath;
  my $invXMLContent;
  my %xmlVar;

  my %clusref = ();
  $invPath = emdcommon::getInventoryXmlPath();

  emdcommon::EMD_PERL_INFO("$LOG_CATEGORY Failed to find Oracle Inventory File ") and return unless $invPath;

  emdcommon::EMD_PERL_INFO("$LOG_CATEGORY Failed to read Oracle Inventory File $invPath") 
   and return unless isReadable($invPath);

  $invXMLContent = getFileContents($invPath);

  emdcommon::EMD_PERL_INFO("$LOG_CATEGORY Oracle Inventory file $invPath has no content") 
   and return unless $invXMLContent;

  %xmlVar = parse_xml($invXMLContent);

  traverse_xml(\%xmlVar,\&log_error,\&inventory_fn,\%clusref) 
    or warn "WARN:Failed to traverse the Oracle Inventory xml $invXMLContent" and return;
  
  # check to see if the crs_home from getcrshome matches the one from inventory
  if ( keys %clusref and $clusref{from_getcrshome} ) 
  {
    # save the crshome from getcrshomebin
    my $crshomefrombin = $clusref{from_getcrshome};
    delete $clusref{from_getcrshome};

    # save the other oracle_homes in inventory without the crs=true
    my $otheroraclehomesref = $clusref{other_oracle_homes} if $clusref{other_oracle_homes};
    delete $clusref{other_oracle_homes} if $clusref{other_oracle_homes};

    # save the crs_homes from clusref
    my %cpclusref = %clusref;
    %clusref=(); 

    # if there is a crshome from getcrshomebin and it is different from the crshome from inventory with crs=true then use the previous
    if ( keys %cpclusref ) 
    {
     if ( $cpclusref{$crshomefrombin} )
     {
       $clusref{$crshomefrombin} = $cpclusref{$crshomefrombin};
     }
     else 
     {
      for my $ch ( keys %cpclusref ) 
      {
       $clusref{$crshomefrombin} = $cpclusref{$ch};
      }
     }
    }
    # this is the case where crs_home is in inventory but true is not set
    elsif ( $otheroraclehomesref and ref($otheroraclehomesref) and ref($otheroraclehomesref) =~ /hash/i
               and keys %{$otheroraclehomesref} )
    {
       if ( $otheroraclehomesref->{$crshomefrombin} ) 
       {
          $clusref{$crshomefrombin} = $otheroraclehomesref->{$crshomefrombin};
       }
    }

  }

  return \%clusref;

}

#------------------------------------------------------------------------------
# FUNCTION :    hasReturnFileContents
#
# DESC
# print file contents to stdout
#
# ARGUMENTS
# dir
# file name
#
# RETURNS
#  return contents dumped to string
#------------------------------------------------------------------------------
sub getFileContents($;$)
{

  my ($flnm) = @_;
  my $pstring;

  # Open the file for reading , if it fails dont return an error,
  # log an error and return gracefully, so metric is blank but with errror
  open(CSHFH,"$flnm") or emdcommon::EMD_PERL_INFO("$LOG_CATEGORY Failed to open the cached file $flnm") and return;

  my @cols = <CSHFH>;

  close(CSHFH) or emdcommon::EMD_PERL_INFO("$LOG_CATEGORY Failed to close the file $flnm");

  for my $row ( @cols )
  {
    chomp $row if $row;

    $row =~ s/^\s+|\s+$//g if $row;

    $pstring =  "$pstring$row\n" and next if $pstring and $row;

    $pstring =  "$row\n" and next if $row;
  }

  return $pstring if $pstring;

  return;

}

# variables for xml related parsing
my $has_xref;
my $has_fref;


# name : has_start_handler
# desc : handler to be invoked by perl parser when starting an element
#
# arg :
#  to be passed by the perl parser
sub xml_elem_start_handler
{
  my ($pr,$el,%attrs) = @_;

  $pr->{cdata_buffer} = '';

  my %ehash = (element=>$el);

  $has_fref = \%ehash unless  $has_fref;

  for my $name ( keys %attrs )
  {
    $ehash{attrs}{$name}=$attrs{$name};
  }

  $ehash{parent} = $has_xref if $has_xref;
  $ehash{depth} = $ehash{parent}->{depth}+1 if $ehash{parent};
  $ehash{depth} = 0 unless $ehash{depth};

  push @{$has_xref->{child_elements}{$el}},\%ehash if $has_xref;
  push @{$has_xref->{children}},\%ehash if $has_xref;

  $has_xref = \%ehash;
}

# name : has_end_handler
# desc : handler function for perl parser when element closes
#
# arg : 
#  passed by perl parser
sub xml_elem_end_handler
{
  my ($pr,$el) = @_;

  $has_xref = $has_xref->{parent};

}

# name : has_char_handler
# desc : handler function for perl parser for char
#
# arg 
#  passed by parser
#
sub xml_char_handler
{
  my ($pr,$tag) = @_;

  $has_xref->{name}=$tag unless $has_xref->{name};

  $has_xref->{name} =~ s/^\s|\s+$//g;

  chomp $has_xref->{name};

#  delete $has_xref->{name} unless $has_xref->{name};

}

# name : has_parse_xml
# desc :  parse a xml string to a perl variable
#
# arg  : 
#  xml string to be parsed
#
# return:
#  hash of parsed perl variable
#

sub parse_xml($)
{

 my ( $result ) = @_;

 emdcommon::EMD_PERL_INFO("$LOG_CATEGORY No XML content to parse") and return unless $result;

 my $p = new XML::Parser(ErrorContext => 2,
                        ProtocolEncoding => 'UTF-8',
                        );

 $p->setHandlers(Start => \&xml_elem_start_handler,
	        End => \&xml_elem_end_handler,
                Char  => \&xml_char_handler);

 undef $has_fref;
 undef $has_xref;

 # save the signal handler defined for die
 my $diesh = $SIG{__DIE__} if $SIG{__DIE__};

 # remove any signal handler defined for die
 $SIG{__DIE__}='';
 
 eval{  $p->parse($result) };

 # restore back the original die signal handler
 $SIG{__DIE__} = $diesh if $diesh;

 die  "parse_xml $@ Failed to Parse $result\n" if $@ and $result;
 die  "parse_xml $@ Failed to Parse \n" if $@;

 return %$has_fref;

}



# name : traverse_xml
# desc : traverse the xml tree, execute specificed function for each element
#
# args :
#  ref to hash of root of xml
#  ref to error handlig function
#  ref to traverse function
#  ref to list of args to function
#
#sub traverse_xml(\%\&\&@)
sub traverse_xml($$$@)
{
  my ( $xmlref,$fnerrhndl,$fnref,@args) = @_;

  my @stack;

  # to print the array depth first
  push @stack, $xmlref if $xmlref;

  while ( my $xref = pop @stack )
  {

   next unless $xref;

   # keep error messages in the error stack
   #<errors>
   #  <error>
   #   <type>error|warn</type>
   #   <message>error_message</message>
   #  </error>
   #</errors>
   if ( $xref->{element} and $xref->{element} =~ /^error$/i  and $xref->{children} )
   {

     my $mtype;
     my $message;

     for my $ec ( @{$xref->{children}} )
     {
       $mtype = $ec->{name} if $ec->{element} =~ /^type$/i and $ec->{name};

       $message = $ec->{name} if $ec->{element} =~ /^message$/i and $ec->{name};
     }

     &{$fnerrhndl}("$mtype:$message") if $mtype and $message and $fnerrhndl;

   }
   # keep error messages in the error stack for opcode != 0
   #<opcode>
   # <code>0</code>
   # <message>Succcess</message>
   #</opcode>
   elsif ( $xref->{element} and $xref->{element} =~ /^opcode$/i  and $xref->{children} )
   {
     my $code;
     my $mtype;
     my $message;
     my $emessage;

     for my $ec ( @{$xref->{children}} )
     {
       $code = $ec->{name} if $ec->{element} =~ /^code$/i and $ec->{name};

       $message = $ec->{name} if $ec->{element} =~ /^message$/i and $ec->{name};
     }

     # if code != 0 indicates a failure for that element
     if ( $code )
     {
        $mtype = 'WARN';  #a failure of a step is a warning as other steps might have succeeded

        my $entity_name;
        my $entity_type;

        my $parent_ref = $xref->{parent} if $xref->{parent};
        
        if ( $parent_ref and $parent_ref->{attrs} )
        {
          $entity_name = $parent_ref->{attrs}{entity_name} if $parent_ref->{attrs}{entity_name};
          $entity_type = $parent_ref->{attrs}{entity_type} if $parent_ref->{attrs}{entity_type};

          $emessage = "for $entity_type $entity_name" if $entity_type and $entity_name;
          $emessage = "for $entity_type" if $entity_type and not $emessage;
          $emessage = "for $entity_name" if $entity_name and not $emessage;

        }

        $message = "$message $emessage" if $message and $emessage;
        $message = "Failed $emessage" if $emessage and not $message;
        $message = "Failed" unless $message;

        &{$fnerrhndl}("$mtype:$message") if $mtype and $message and $fnerrhndl;
     }

   }
   # keep error messages in the error stack for return code != 0
   #<returncode>
   # <code>0</code>
   # <mesg>Succcess</mesg>
   #</returncode>
   elsif ( $xref->{element} and $xref->{element} =~ /^returncode$/i  and $xref->{children} )
   {

     my $code;
     my $mtype;
     my $message;

     for my $ec ( @{$xref->{children}} )
    {
       $code = $ec->{name} if $ec->{element} =~ /^code$/i and $ec->{name};

       $message = $ec->{name} if $ec->{element} =~ /^mesg$/i and $ec->{name};
     }

     # if code != 0 indicates a failure for that element
     if ( $code )
     {
        $mtype = 'ERROR';  #a failure overall is an ERROR

        $message = "Failed" unless $message;
        
        &{$fnerrhndl}("$mtype:$message") if $mtype and $message and $fnerrhndl;
     }

   }

   # execute function for each element
   &{$fnref}($xref,@args) or die "traverse_xml Failed to execute function\n";

   next if $xref->{ignore};

   push @stack, reverse @{$xref->{children}} if $xref->{children};

  }

  return 1;
}

#------------------------------------------------------------------------------
# FUNCTION :    hasCheckAndReturnEmcrspResults
#
# DESC:
#  checks he command results to make sure they do not have anything
#  other than the stdout printed from emcrsp
#
# arg  :
#  cmdresults from emcrsp
#
# return:
#  return the ckecked cmd results
#------------------------------------------------------------------------------
sub checkAndReturnEmcrspResults($)
{

  my ( $cmdresults ) = @_;

  return unless $cmdresults;

  my $temp_res;
  $temp_res = $cmdresults;
  #bug fix 9212119
  $temp_res =~ s/\n\s*[^\s|^<]+[^<]*<[^\n]+//g if $temp_res;
  $temp_res =~ s/^\s*[^\s|^<]+[^<]*<[^\n]+//g if $temp_res;

  return $temp_res if $temp_res;
  return;
}

#------------------------------------------------------------------------------
# FUNCTION :    hasGetScanInformation
#
# DESC
#  return the scan Information
#
# ARGUMENTS
# crsHome if known
#
# RETURNS
# hash for results
#------------------------------------------------------------------------------
sub getEonsPort(;$)
{

  my ( $crsHome) = @_;
 
  # get the scan information
  my %meta; 
  my $cmdfull = catfile("$crsHome",'bin','emcrsp');
  $meta{cmd}="$cmdfull em config -e resource_instance -p ora.eons.type";
  $meta{element_name}='entity';
  $meta{entity}{attrs}{entity_name}=1;
  $meta{entity}{name_value}{PORT}=1;
  
  my $eonsref;
  $eonsref = getEntityInformation(\%meta,$crsHome);

  my $eonsport;
  
  # if we have eons data then farm the fields to make sure we have reqd data
  
  if ( $eonsref and ref($eonsref) and ref($eonsref) =~ /HASH/ and keys  %{$eonsref}  )
  {
    for my $id ( keys %{$eonsref} )
    {

      $eonsport = $eonsref->{$id}{PORT} if defined $eonsref->{$id}{PORT};
      last if $eonsport;
    }
  }
   
  return $eonsport;

  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY Failed to get Information for Scan resource") and return;
}

#------------------------------------------------------------------------------
# FUNCTION :    hasGetScanInformation
#
# DESC
#  return the scan Information
#
# ARGUMENTS
# crsHome if known
#
# RETURNS
# hash for results
#------------------------------------------------------------------------------
sub getScanNameAndPort(;$)
{

  my ( $crsHome) = @_;
 
  # get the scan information
  my %meta;
  my $cmdfull = catfile("$crsHome",'bin','emcrsp'); 
  $meta{cmd}="$cmdfull em config -e resource -p ora.scan_listener.type";
  $meta{element_name}='entity';
  $meta{entity}{attrs}{entity_name}=1;
  $meta{entity}{name_value}{PORT}=1;
  
  my $scanlistref;
  $scanlistref = getEntityInformation(\%meta,$crsHome);

  # get the scan vip information
  %meta = ();
  $cmdfull = catfile("$crsHome",'bin','emcrsp');
  $meta{cmd}="$cmdfull em config -e resource -p ora.scan_vip.type";
  $meta{element_name}='entity';
  $meta{entity}{attrs}{entity_name}=1;
  $meta{entity}{name_value}{SCAN_NAME}=1;
  $meta{entity}{name_value}{USR_ORA_VIP}=1;

  my $scanvipref;
  $scanvipref = getEntityInformation(\%meta,$crsHome);
 
  my %scaninfo;
  
  if ( $scanlistref and ref($scanlistref) and ref($scanlistref) =~ /HASH/ and keys  %{$scanlistref}  )
  {
    for my $id ( keys %{$scanlistref} )
    {

      $scaninfo{SCAN_PORT} = $scanlistref->{$id}{PORT} if defined $scanlistref->{$id}{PORT};
      last if $scaninfo{SCAN_PORT};

    }

  }

  if ( $scanvipref and ref($scanvipref) and ref($scanvipref) =~ /HASH/ and keys  %{$scanvipref}  )
  {
    for my $id ( keys %{$scanvipref} )
    {

      $scaninfo{SCAN_NAME} = $scanvipref->{$id}{SCAN_NAME} if $scanvipref->{$id}{SCAN_NAME};
      last if $scaninfo{SCAN_NAME};

    }

  }

  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY Failed to get SCAN PORT for scan listener") unless $scaninfo{SCAN_PORT};
  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY Failed to get NAME for scan") unless $scaninfo{SCAN_NAME};

  $scaninfo{SCAN_PORT} = 0 unless $scaninfo{SCAN_PORT};
  
  return \%scaninfo if keys %scaninfo;

  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY Failed to get Information for Scan resource") and return;
}

#------------------------------------------------------------------------------
# FUNCTION :    hasGetEntityInformation
#
# DESC
# return the required entity information for the cluster 
#
# ARGUMENTS
#  ref to the metadata to be used to extract data
#  crsHome if known
#  fn pointer or null if you want to use the default fn
#
#  meta should have
#   ->{cmd}
#   ->{element_name}
#   ->{attrs}{[attr_name]}
#   ->{name_value}{[name]}
# RETURN
#  ref to the hash to be filled in with values
#------------------------------------------------------------------------------
sub getEntityInformation($;$$)
{
  # name : hasGetEntityInformation_fn
  # desc : it filters our the required information
  #
  # arg  :
  #  ref to xml element to be filtered
  #  ref to the result hass entities  array
  #  ref to the metadata hash to be used to extract data
  # 
  sub getEntityInformation_fn($$$)
  {
    my ( $elref, $rsref, $mtref ) = @_;

    # return unless meta data is provided
    return 1 unless $mtref and $mtref->{element_name};

    return 1 unless $elref->{element} and $elref->{element} =~ /^$mtref->{element_name}$/i;

    # get the name for this entity
    my $thisval;
    if ( $elref->{attrs} )
    {
      for my $hdval ( keys %{$elref->{attrs}} )
      {
        $thisval =  $elref->{attrs}{$hdval} if $hdval =~ /^entity_name$/;
      }
    }

   $thisval =~ s/^\s+|\s+$//g;

   emdcommon::EMD_PERL_INFO("$LOG_CATEGORY failed to get entity_name from $elref->{element}") and return 1 unless $thisval;

    # if there are attrs to be read read them
    if ( $mtref->{$mtref->{element_name}}{attrs} )
    {
       for my $hdval ( keys %{$elref->{attrs}} )
       {
         $hdval =~ s/^\s+|\s+$//g;
         next unless $hdval;

         next unless $mtref->{$mtref->{element_name}}{attrs}{$hdval};
 
         $rsref->{$thisval}{$hdval} = $elref->{attrs}{$hdval} if $elref->{attrs}{$hdval};

         $rsref->{$thisval}{$hdval} =~ s/^\s+|\s+$//g if $rsref->{$thisval}{$hdval};
        }
    }


    # return unless there are name value paris to be read from attributes
    return 1 unless $mtref->{$mtref->{element_name}}{name_value};

    # read all name=value pairs from the attributes of the node
    for my $elref1 (   @{$elref->{children}} )
    {
      next unless $elref1->{element} =~ /^attributes$/;

      for my $elref2 (   @{$elref1->{children}} )
      {
        next unless $elref2->{element} =~ /^attribute$/;

        my $name;

        my $value;

        for my $elref3 (   @{$elref2->{children}} )
        {
          next unless $elref3->{element} =~ /^(name|value)$/;

          $name = $elref3->{name} if $elref3->{element} =~ /^name$/;
          $value = $elref3->{name} if $elref3->{element} =~ /^value$/;

          $name =~ s/^\s+|\s+$//g if $name;
          $value =~ s/^\s+|\s+$//g if $value;
        }

        next unless $name;

        # if this name is not required skip it
        next unless $mtref->{$mtref->{element_name}}{name_value}{$name};

        $value = '' unless $value;

        $rsref->{$thisval}{$name}=$value;

      }
    }

    return 1;

  }


  my ( $metaref,$crsHome,$fn_ptr ) = @_;

  $fn_ptr = \&getEntityInformation_fn unless $fn_ptr;

  my $reslistref;

  return 1 unless $metaref and $metaref->{cmd};

  unless ($crsHome ) {
   emdcommon::EMD_PERL_INFO("$LOG_CATEGORY home is not passed or not set in env");
   return;
  }

  unless ( checkForEmcrsp($crsHome) ) {
   emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY emcrsp is not present in $crsHome");
   return;
  }

  my $cmdresults = executeCommand($crsHome,$metaref->{cmd});
 
  $cmdresults = checkAndReturnEmcrspResults($cmdresults);

  unless ( $cmdresults ) {
   emdcommon::EMD_PERL_INFO("$LOG_CATEGORY Failed to get results from command - $metaref->{cmd}");
   return ;
  }

  my  %xmlvar =  parse_xml($cmdresults);

  my %resulthash;
  traverse_xml(\%xmlvar,\&log_error,$fn_ptr,\%resulthash,$metaref);

  return \%resulthash;
}

#------------------------------------------------------------------------------
# FUNCTION :    hasGetClusterConfigEmcrsp
#
# DESC
# return the config information for the cluster using emcrsp for 11gR2 clusters
#   and higher
#
# ARGUMENTS
#  crsHome if known
#  force recompute if already computed
# RETURN
#  ref to the hash to be filled in with values
#------------------------------------------------------------------------------
sub getOcrType(;$$)
{
  # name : hasGetClusterConfigEmcrsp_fn
  # desc : it filters our the nodename
  #
  # arg  :
  #  ref to xml element to be filtered
  #``ref to the result hass entities  nodearray
  # 
  sub getOcrConfigEmcrsp_fn(\%\%)
  {
    my ( $elref, $rsref ) = @_;

    return 1 unless $elref->{element} and $elref->{element} =~ /^(header)$/i and $elref->{attrs};
    
    for my $hdval ( keys %{$elref->{attrs}} )
    {
       $rsref->{$hdval} = $elref->{attrs}{$hdval};
    }
    
    return 1;
  }
  my ($crsHome) = @_;
  
  my $ocr_type ='cluster'; # default

  my $cmdfull = catfile("$crsHome",'bin','emcrsp');

  my %command_args = (exit_success_list => [(0,1)]);

  my $cmdresults = executeCommand($crsHome, "$cmdfull", 'em config -e node',\%command_args);

  $cmdresults = checkAndReturnEmcrspResults($cmdresults);

  unless ( $cmdresults ) {
   emdcommon::EMD_PERL_INFO("$LOG_CATEGORY Failed to get results from command - emcrsp em config -e node");
   return;
  }

  my  %xmlvar = parse_xml($cmdresults);
  my %headerref;
  traverse_xml(\%xmlvar,\&log_error,\&getOcrConfigEmcrsp_fn,\%headerref);
  
  return $ocr_type unless %headerref and keys %headerref;
  return $ocr_type unless $headerref{ocr_configured} and $headerref{ocr_location};
  
  my $ocr_location = $headerref{ocr_location};
  
  if ( $ocr_location =~ /CLUSTER/ )
  {
    $ocr_type = 'cluster';
  }
  elsif ( $ocr_location =~ /LOCAL/ )
  {
    $ocr_type = 'has';
  }
  
  return $ocr_type;
}

#------------------------------------------------------------------------------
# FUNCTION :    log_error
#
# DESC
#  log the message
#
# ARGUMENTS
#  message
sub log_error(@)
{ 
   my ( $message ) = @_; 
   
   chomp $message if $message; 
   return unless $message;

   emdcommon::EMD_PERL_INFO("$LOG_CATEGORY $message");
   return 1;
}

sub get_osType
{
  my $os = $^O;

  if (( $os eq "Windows_NT") ||
      ( $os eq "MSWin32")) {
      return "WIN";
  }
  return $os;
}




#------------------------------------------------------------------------------
# FUNCTION :    hasGetClusterNodeListPre11g
#
# DESC
# return a array list of nodes in a cluster
#
# ARGUMENTS
# crsHome 
# local host name
#
# RETURNS
# array list of nodes
#------------------------------------------------------------------------------
sub hasGetClusterNodeListPre11g($;$)
{
  my ($crsHome, $hostName ) = @_; # CRSHome
  my $nodes;
  my @nodearray;
  my $ncrsHome;
  my @invnodearray;
  my @olsnodearray;

  $crsHome =~ s/^\s+|\s+$//g if $crsHome;
  undef $crsHome if $crsHome and $crsHome =~ /^\s*$/;

  # the order of search directories for olsnode/lsnodes
  my %olsnodeorder;

  if ( $crsHome )
  {
    $olsnodeorder{0}{path}=$crsHome;
    $olsnodeorder{0}{cmd}='olsnodes';
  }

  $olsnodeorder{2}{path}='CRS_HOME';
  $olsnodeorder{2}{cmd}='olsnodes';

  $olsnodeorder{3}{path}='EM_CRS_HOME';
  $olsnodeorder{3}{cmd}='olsnodes';

  $olsnodeorder{4}{path}='ORACLE_HOME';
  $olsnodeorder{4}{cmd}='olsnodes';

  $olsnodeorder{5}{path}='EMDROOT';
  $olsnodeorder{5}{cmd}='lsnodes';

  for my $order ( sort  {$a <=> $b} keys %olsnodeorder )
  {
    my $path;
    my $cmd;
    undef $ncrsHome if $ncrsHome;
    undef $nodes if $nodes;

    next unless $olsnodeorder{$order} and $olsnodeorder{$order}{path} and $olsnodeorder{$order}{cmd};

    $path = $olsnodeorder{$order}{path};
    $path = $ENV{$olsnodeorder{$order}{path}} if $ENV{$olsnodeorder{$order}{path}};

    next unless $path;
    next if $path =~ /#CRS_HOME#|EM_CRS_HOME|CRS_HOME|ORACLE_HOME|EMDROOT/;

    next unless isReadable($path);

    $cmd = $olsnodeorder{$order}{cmd};

    #get the list of nodes using olsnodes
    my %command_args = (exit_failure_list => [()]);

    my $fullcmd;

    for my $ext ( ( 'exe', 'bat' , undef) )
    {
      my $ecmd;

      $ecmd = "$cmd\.$ext" if $ext;
      $ecmd = $cmd unless $ecmd;

      $fullcmd = catfile($path,'bin',$ecmd);

      stat $fullcmd;

      last if isReadable($fullcmd);

      undef $fullcmd;
   
    }

    next unless $fullcmd;

    #bug fix 11932967 extended
    undef $nodes if $nodes;

    $nodes = executeCommand($path,$fullcmd.' 2>&1','',\%command_args);

    #bug fix 11932967 extended
    if ( $command_args{command_return_status} ) {
      emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY DEBUG:has::ClusterHasDiscovery::hasGetClusterNodeListPre11g:Failed executing command **$fullcmd") ;
      undef $nodes if $nodes;
      next;
    }

    if ( $nodes )
    {
      # bug 4667678
      $ncrsHome = $path unless $path =~ /EMDROOT/;

      emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY DEBUG:has::ClusterHasDiscovery::hasGetClusterNodeListPre11g:Getting node list from $cmd at $path") ;

      last;

    } 

  }

  emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY DEBUG:has::ClusterHasDiscovery::hasGetClusterNodeListPre11g:Failed to get nodelist from cluster binaries olsnodes and cluvfy") unless $nodes;

  # if hostName is not provided get local hostName 
  unless ( $hostName ) {
    $hostName = hostOSD::getHostName;
    $hostName =~ s/^\s+|\s+$// if $hostName;
    $hostName =~ s/\n//g if $hostName;

    unless ( $hostName ) {
     $hostName = hostname;
     $hostName =~ s/^\s+|\s+$// if $hostName;
     $hostName =~ s/\n//g if $hostName;
    }

  }


  # get the domain name to append to nodenames to build fully qual host names
  my $hostdomain = Net::Domain::hostdomain();
  $hostdomain =~ s/^\s+|\s+$// if $hostdomain;
  $hostdomain =~ s/\n//g if $hostdomain;
  $hostdomain =~ s/^\.// if $hostdomain and $hostdomain =~ /^\./;

  unless ( $hostdomain ) {
   emdcommon::EMD_PERL_DEBUG("$LOG_CATEGORY DEBUG:has::Common::hasGetDomainName:Failed to get the Domain name for the local host from net::Domain ");

   # if there is a hostName get domain name from hostName
   if ( $hostName ) {
     $hostdomain = hostOSD::getDomainName($hostName);

     $hostdomain =~ s/^\s+|\s+$// if $hostdomain;
     $hostdomain =~ s/\n//g if $hostdomain;
     $hostdomain =~ s/^\.// if $hostdomain and $hostdomain =~ /^\./;
   }

  }

  warn "DEBUG:has::Common::hasGetDomainName:Failed to get the Domain name for the node" unless $hostdomain;

  chomp($nodes) if $nodes;
  $nodes=~ s/^\s+|\s+$//g if $nodes;
  $nodes =~ s/\n/,/g if $nodes;
  $nodes =~ s/,\s+,/,,/g if $nodes;
  $nodes =~ s/,+/,/g if $nodes;
  $nodes=~ s/^,|,$//g if $nodes;

  @olsnodearray = split/,/,$nodes if $nodes;
  
  # try to get the nodelist from inventory
  my $invref;
  # bug 13613078, catch runtime exception
  my $dieh = $SIG{__DIE__} if $SIG{__DIE__};
  $SIG{__DIE__}='';
  eval {  
    $invref = getClusterDataFromInventory()};
  $SIG{__DIE__} = $dieh if $dieh;
 
  if ( 
   $crsHome 
   and keys %{$invref}
   and $invref->{$crsHome}
   and $invref->{$crsHome}{discover}
   and $invref->{$crsHome}{discover}{nodes}
   and ref($invref->{$crsHome}{discover}{nodes})
   and ref($invref->{$crsHome}{discover}{nodes}) =~ /HASH/i
  )
  {
     # build the node list for this crsHome
     for my $discnode ( keys %{$invref->{$crsHome}{discover}{nodes}} )
     {
        next unless $discnode;
        push @invnodearray,$discnode;

     }
  }

  # merge the two node arrays to get unique node list
  my @allnodes=(@olsnodearray, @invnodearray);
  my %seenHash   = ();

  foreach my $elem ( @allnodes )
  {
    next if $seenHash{ $elem }++;
    push @nodearray, $elem;
  }

  # build the host list by appending domain to node name
  my @clusterHostList;

  if ( $hostdomain and @nodearray and @nodearray > 0 ) {
   
   for my $nnode ( @nodearray ) {

      $nnode =~ s/^\s+|\s+$// if $nnode;
      $nnode =~ s/\.$// if $nnode;
      next unless $nnode;

      my $fullQualHostName = "$nnode\.$hostdomain";

      # skip local node
      next if $hostName and $nnode =~ /^$hostName$/;
      next if $hostName and $fullQualHostName =~ /^$hostName$/;

      push @clusterHostList,$fullQualHostName;
   }
    
  }

  return \@clusterHostList if @clusterHostList and @clusterHostList > 0;

  # failed to get nodelist from both discovery inventory and olsnodes
  emdcommon::EMD_PERL_WARN("WARN:has::ClusterHasDiscovery::hasGetClusterNodeListPre11g:Failed to get nodelist from cluster binaries olsnodes and discovery") and return;

}

1;

