#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/hostUtil.pl /main/2 2010/07/18 00:53:22 prapatil Exp $
#
# hostUtil.pl
# 
# Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      hostUtil.pl - determine if given host name or IP address 
#                    represents local host or not.
#                    Compare with IPs of all NICs.
#
#    DESCRIPTION
#      Introduced during fix to bug#6320617.
#      Contains sub-routines that help to...
#        [a] equalHosts()      : GENERAL PURPOSE UTILITY FUNCTION that compares 2
#                                given hostnames and tells whether they match or not.
#        [b] computeLocalIPs() : Generate IP addresses of all NICs on local host. 
#        [c] exec_computeLocalIPs_with_timeout
#                              : Call computeLocalIPs() after defining timeout and
#                                code to execute upon hitting timeout.
#        [d] checkAllNICs()    : GENERAL PURPOSE UTILITY FUNCTION that checks if
#                                given host belongs to local host or not by comparing 
#                                with IP addresses of all NICs of local host.
#
#    NOTES
#     NEW FOR 11.2 -- hostUtil.pl is now used for discovery and cannot reference any other scripts (other than emd_common.pl) 
#
#    MODIFIED   (MM/DD/YY)
#    mkiran      02/04/10 - Creation
# 

my $TIMEOUT_FOR_COMPUTELOCALIPS = 150; # = 2.5 mins

# NOTE: Moved from sysman/admin/discover/oracledb.pl
# Usage: equalHosts($host1, $host2)
# return 1 if an ip address of $host1 equals an ip address of $host2,
# otherwise return 0
sub equalHosts
{
    
    if ($_[0] eq "" || $_[1] eq "")
    {
        return 0;
    }
    
    my ($name1, $aliases1, $addrtype1, $length1, @addrs1) = gethostbyname($_[0]);
    my ($name2, $aliases2, $addrtype2, $length2, @addrs2) = gethostbyname($_[1]);
    

    foreach my $a1 (@addrs1)
    {
        foreach my $a2 (@addrs2)
        {
            return 1 if ($a1 eq $a2);
        }
    }

    0;
}

# 6320617: Generate list of IP addresses of local host.
#          If local host has multiple NICs configured,
#          then each NIC will have unique IP address
#          Upon SUCCESS, return the list of IP addresses
#          Upon FAILURE, return an empty list.
sub computeLocalIPs
{
    my @valarr = ();

    my $isSuccess = 1;
    my $cmd = "";
    my $cmd1 = "";

    # Just declare old_LC_ALL but don't define it, even by 
    # initializing to "". If initialized, that would define
    # it and will break the overall logic
    my $old_LC_ALL;

    # Take backup of env var $LC_ALL and define it.
    $old_LC_ALL = $ENV{LC_ALL} if defined $ENV{LC_ALL};
    $ENV{LC_ALL} = "C";

    my $OSNAME = get_osType();

    if ($OSNAME eq "LNX") {
        if ( (-e "/sbin/ifconfig") && (-e "/bin/grep") && 
             (-e "/bin/awk") && (-e "/bin/cut") ) {
            $cmd = "/sbin/ifconfig -a | /bin/grep \"inet addr\" | /bin/awk \'{print \$2}\' | /bin/cut -d\":\" -f2";
        } else {
            $isSuccess = 0;
        }
    } elsif ($OSNAME eq "SOL") {
        if (-e "/sbin/ifconfig") {
            $cmd = "/sbin/ifconfig";
        } elsif (-e "/usr/sbin/ifconfig") {
            $cmd = "/usr/sbin/ifconfig";
        } else {
            $isSuccess = 0;
        }

        if (-e "/usr/bin/grep") {
            $cmd1 = "/usr/bin/grep";
        } elsif (-e "/usr/local/bin/grep") {
            $cmd1 = "/usr/local/bin/grep";
        } else {
            $isSuccess = 0;
        }

        if (-e "/usr/bin/awk") {
            $cmd = $cmd . " -a | " . $cmd1 . " inet | /usr/bin/awk \'{print \$2}\'";
        } else {
            $isSuccess = 0;
        }
    } elsif ($OSNAME eq "AIX") {
        if (-e "/etc/ifconfig") {
            $cmd = "/etc/ifconfig";
        } elsif (-e "/usr/sbin/ifconfig") {
            $cmd = "/usr/sbin/ifconfig";
        } else {
            $isSuccess = 0;
        }

        if ( (-e "/usr/bin/grep") && (-e "/usr/bin/awk") ) {
            $cmd = $cmd . " -a | /usr/bin/grep -w inet | /usr/bin/awk \'{print \$2}\'";
        } else {
            $isSuccess = 0;
        }
    } elsif ($OSNAME eq "WIN") {
        if ( (-e "\%SystemRoot\%\\system32\\ipconfig.exe") && 
             (-e "\%SystemRoot\%\\system32\\findstr.exe") ) {
            # ipconfig and findstr will always be in the standard location 
            # %SystemRoot%\system32\ on all flavours of Windows (32bit, 
            # 64bit, 2003, etc). So, no need to provide absolute path.
            $cmd = "ipconfig /all | findstr /C:\"IP Address\"";
        } else {
            $isSuccess = 0;
        }
    } elsif ($OSNAME eq "HP") {
        if (-e "/usr/bin/awk") {
            $cmd = "/usr/bin/awk"; 
        } elsif (-e "/sbin/awk") {
            $cmd = "/sbin/awk"; 
        } else {
            $isSuccess = 0;
        }

        if ( (-e "/usr/bin/netstat") && (-e "/usr/bin/grep") ) {
            $cmd = "/usr/bin/netstat -i -n | " . $cmd . " \'{print \$4}\' | /usr/bin/grep -v Address";
        } else {
            $isSuccess = 0;
        }
    } elsif ($OSNAME eq "OSF1") {
        if (-e "/sbin/ifconfig") {
            $cmd = "/sbin/ifconfig";
        } elsif (-e "/usr/sbin/ifconfig") {
            $cmd = "/usr/sbin/ifconfig";
        } else {
            $isSuccess = 0;
        }

        if (-e "/sbin/grep") {
            $cmd1 = "/sbin/grep";
        } elsif (-e "/usr/bin/grep") {
            $cmd1 = "/usr/bin/grep";
        } else {
            $isSuccess = 0;
        }

        if (-e "/usr/bin/awk") {
            $cmd = $cmd . " -a | " . $cmd1 . " inet | /usr/bin/awk \'{print \$2}\'";
        } else {
            $isSuccess = 0;
        }
    } elsif ($OSNAME eq "MAC OS X") {
        if ( (-e "/sbin/ifconfig") && (-e "/usr/bin/grep") && (-e "/usr/bin/awk") ) {
            $cmd = "/sbin/ifconfig -a | /usr/bin/grep -w inet | /usr/bin/awk \'{print \$2}\'";
        } else {
            $isSuccess = 0;
        }
    } else {
        # computeLocalIPs() isn't implemented for current OS.
        $isSuccess = 0;
    }

    # Restore the original value of env var $LC_ALL
    delete $ENV{LC_ALL};
    $ENV{LC_ALL} = $old_LC_ALL if defined $old_LC_ALL;

    if ($isSuccess == 0) {
        # Either computeLocalIPs() isn't implemented for current OS
        # OR the required command doesn't exist on current OS
        # So, signal failure of computeLocalIPs() -- return empty list
        @valarr = ();
        return @valarr;
    }

    my $value = "";
    chomp( $value = `$cmd` );

    # computeLocalIPs() is implemented for
    # current OS, but it's evaluation failed!
    $isSuccess = 0 if ($? != 0);

    if ($isSuccess == 0) {
        # signal failure of computeLocalIPs() -- return empty list
        @valarr = ();
        return @valarr;
    }

    @valarr = split("\n", $value);
    chomp(@valarr);

    # Here is WINDOWS implementation to get the IP list
    if ($OSNAME eq "WIN") {
        my @tempValArr = ();
	my ($myIP, $tempIP) = ("", "");
        foreach $myIP (@valarr) {
            $tempIP = trim( right(":", $myIP) );
	    push @tempValArr, $tempIP;
        }
	@valarr = @tempValArr;
    }

    # @valarr will contain the IP addresses of all NICs of local host
    return @valarr;
}

# 6320617: On some platform, the command executed to return 
#          IP addresses' info might hang. To prevent that 
#          affect discovery action, let's execute computeLocalIPs()
#          with a pre-defined timeout [$TIMEOUT_FOR_COMPUTELOCALIPS()]
#          This sub-routine will return an empty list upon timeout,
#          signalling failure of computeLocalIPs().
sub exec_computeLocalIPs_with_timeout()
{
    local $SIG{ALRM} =
        sub {
            alarm 0;
            EMD_PERL_ERROR("exec_computeLocalIPs_with_timeout(): Timed Out");
            EMD_PERL_ERROR("returning empty list, signalling failure of computeLocalIPs()");
            die "Timed Out";
        };

    alarm $TIMEOUT_FOR_COMPUTELOCALIPS;
    my @retList = computeLocalIPs();
    alarm 0;

    return @retList;
}

# 6320617: Return FALSE (i.e. 0) if the given Host doesn't 
#          represent any IP address of local host, wherein
#          we obtain the IP addresses of all NICs by calling 
#          exec_computeLocalIPs_with_timeout(), which inturn
#          calls computeLocalIPs() after setting a timeout 
#          value and defining code to execute upon hitting 
#          timeout. Return TRUE (i.e. 1) if any IP address
#          matches that of the given Host.
sub checkAllNICs
{
    my ($remoteHost) = @_;

    return 0 if ($remoteHost eq "");

    my @localIPs = eval {
                          return exec_computeLocalIPs_with_timeout();
                        };

    if ($@)
    {
        EMD_PERL_ERROR("checkAllNICs(): Following exception is hit...");
        EMD_PERL_ERROR("checkAllNICs(): $@");
        @localIPs = ();
    };

    # @localIPs will be an empty list if either 
    # computeLocalIPs() wasn't implemented for current OS
    # OR implemented but it's evaluation had failed.
    return 0 if (@localIPs == ());

    # Compare given host with all IPs in @localIPs.
    # Return 1 upon hitting a match. In the worst 
    # case, return 0 at the end, signalling failure.
    foreach my $localIPAddr (@localIPs) {
        if ($localIPAddr ne "") {
            return 1 if (&equalHosts($remoteHost, $localIPAddr));
        }
    }

    0;
}

1;

