#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/jobDiagUtil.pl /main/4 2011/05/26 09:44:11 mohitver Exp $
#
# tvmrjdgp.pl
# 
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      tvmrjdgp.pl - test perl diag helper JobDiagUtil
#
#    DESCRIPTION
#      test perl diag helper JobDiagUtil
#
#    NOTES
#      call into the perl helper
#      If called from a job, the job must accept commend blocks:
#      remoteOp must have a param like:
#          <param name="outputType">command</param>
#
#    MODIFIED   (MM/DD/YY)
#    shynaray    03/08/11 - Adding conditions for incident reporting
#    swbalasu    10/29/10 - redirect STDERR to STDOUT
#    rdabbott    07/06/10 - default to user visible for logging apis
#    rdabbott    06/25/10 - sys_incident -> log
#    rdabbott    03/12/10 - file times
#    rdabbott    03/09/10 - impl logging apis
#    rdabbott    03/09/10 - Creation
# 

use strict;

use File::Copy;
no strict "refs";
use FindBin();
use lib "$FindBin::Bin";
require "jobutil/JobDiagUtil.pm";
##
## Include the incident workdirectory for using ProvCommon module
##
eval "require \"$ENV{'DBCONFIG_WORK_DIRECTORY'}/incident/ProvCommon.pm\"";
#Prints if eval throws exception and silently continues with perl execution
#print "$@";

my $MY_NAME = 'jobDiagUtil.pl';

sub usage 
{
    print ("\nERROR: ", @_ , "\n") if ( @_ );

    print <<EOF

Usage: $MY_NAME [options] 
    options:           Options to $MY_NAME
                       One of proc, logfile, logdir must be specified
                       All params after that would be associated ith the option
                           These must be specified before the command block proc
    -h                     Print help message
    -pre                   String to print before running the command block
    -post                  String to print after running the command block
    -proc cbProc [cbops]   The command block pl/sql procedure to run
                           The step_id is always the first parameter
        cbops                  Options to the command block procedure
                               These may be specified recursively
        record, #, elems       Create a record with # elements
                               # Elements must follow this record
        array, #, elems        Create an array with # elements
                               # Elements must follow this array
        value                  Scalar value
    -logfile <file> [...]  The logfile(s) to associate with this step
                               NOTE: <file> will be replaced by full path
    -logdir  <dir> [ldops] The log dir from which to associate files
        <dir>                  The directory to look in (required)
        ldops                  Options to associateRemoteLogDirFiles (ordered)
            <pattern>              The (grep) file pattern to match in <dir>
            <startTime>            Pick files modified after start 'time()'
            <endTime>              Pick files modified before end  'time()'
                                   Start and end time should be en Epoch time;
                                   the vaule returned by a call to time().
                                   They can also be specified as an offset 
                                   (+/-)<num_seconds> of the current time.
    -log <args>            CommandBlock to log a message 
                               Logs to the object logging table
                               Logs to the system error table for VISIBLE_SYSTEM
        <error_code>           The (ORA) error code, if any, or 'null' if none
        <logLevel>             The log level for this message
                                  INCIDENT_LEVEL
                                  SEVERE_LEVEL
                                  WARNING_LEVEL
                                  INFO_LEVEL
                                  FINE_LEVEL
        <visible>              Who can see this message? 
                                  VISIBLE_SUPPORT_ONLY
                                  VISIBLE_USER
                                  VISIBLE_SYSTEM
        <msg_info>             The log message is requried.
                               Optional: resource id, bundle, replacement args.
    -log<Type> <msg_info>  CommandBlock to log a message in the OMS log table(s)
        <Type>                 Which log level?
                               Eg: Incident, Severe, Warning, Info, Fine
        <msg_info>             See above.
    -api <perl_api> <args> Run a perl API from JobDiagUtil with its args
    -incident              Triggers an incident to em. Uses ProvCommon.pm from 
                           provisioning. Takes two parameters logfiles and 
                           correlationKey, in that order. If correlationKey is
                           'NULL' then the environment variable is looked up and 
                           used. If logFiles is 'NULL' then just the general_diagnose_data.txt
                           with the host current status details are sent. logFiles can be comma
                           seperated.
EOF
;

    exit 1 if (scalar @_);

    print <<EOF

Examples:

COMMAND BLOCK:
    $MY_NAME -pre "before trunc" -post "after trunc" -proc MGMT_JOBS.truncate_step_output
        This would show just "after trunc" in the output of a job.
    $MY_NAME -proc MY_PKG.my_proc a b c
        Run my_proc at the repository with 3 more args: %step_id%, a, b, c
    $MY_NAME -proc MY_PKG.my_proc1 array 3 a b c
        Run my_proc at the repository with an added array: %step_id%, [a, b, c]
    $MY_NAME -proc MY_PKG.my_proc2 record 2 1 array 3 a b c d e
        Run my_proc2 at the repository with 4 args: 
        ... a record with an array in it and 2 scalars:
                                    %step_id%,
                                    record(1, [a, b, c]),
                                    b,
                                    c
LOGFILE:
    $MY_NAME -logfile filepath1 filepath2
        associate filepath1, filepath2 with this step
    $MY_NAME -logdir dir '*.log'
        associate the *.log files in dir with this step
        single quote *.log if called from the command line
    $MY_NAME -logdir dir *.log
        same as above, called from perl

LOGGING:
    $MY_NAME -log 123  SEVERE_LEVEL VISIBLE_SYSTEM 'Error 123 occurred'
        log a system error message for ora error 123
    $MY_NAME -log null SEVERE_LEVEL VISIBLE_SYSTEM 'Some error occurred' SOME_ERR oracle.sysman.resources.myPkg.MyBundle
        log a generic translated system error message w/o ora code
    $MY_NAME -logInfo 'Finished accessing target files'
        log a user visible info message
    $MY_NAME -logWarning 'Unable to find file foo.txt on target1' ERR_MISSING_FILE oracle.sysman.resources.myPkg.MyBundle foo.txt target1
        log a warning message with translation info, including 2 replacements
    $MY_NAME -log null WARNING_LEVEL VISIBLE_SUPPORT_ONLY 'Access denied on foo.txt'
        log a support warning message with internal details

API:
    $MY_NAME -api triggerIncidentByID ecid incident_id
        trigger an incident at the agent
    $MY_NAME -api getIncidentDumpFileDir 
        print the dump file dir
        
Incident:
    $MY_NAME -incident correlationKey logFiles
        triggers an incident at the agent (Using ProvCommon api)
EOF
;
    exit 0;
}

my $preMsg;
my $postMsg;
my $api;
my $arg1;
my $printResult;
my $isIncident = 0;
my $logFiles;
my $correlationKey;

#EMD_PERL_DEBUG("jobDiagUtil.pl()MAIN: *** START ***");

while ( ! $api )
{
    $_ = shift or &usage("must specify some operation");

    if    ( /^-/ )
    {
        $_ = substr($_, 1);
    }

    if    ( /^h/ )
    {
        &usage;
    }
    elsif ( /^pre/)
    {
        $preMsg    = shift or &usage("Supply message for $_");
    }
    elsif ( /^post/ )
    {
        $postMsg   = shift or &usage("Supply message for $_");
    }
    elsif ( /^proc/ )
    {
        $api  = "printCommandBlock";
        $arg1 = shift or &usage("Supply commandBlockProc for $_");
    }
    elsif ( /^logf/ )
    {
        $api  = "associateRemoteLogFiles";
        $arg1 = shift or &usage("Supply file for $_");
    }
    elsif ( /^logd/ )
    {
        $api  = "associateRemoteLogDirFiles";
        $arg1 = shift or &usage("Supply dir for $_");
    }
    elsif ( /^log/ )
    {
        $api  = $_;
        $arg1 = shift;
    }
    elsif ( /^api/ )
    {
        $_    = shift or &usage("Supply api name");
        $api  = $_;
        $arg1 = shift; # some APIs do not take parameters
        $printResult = $api if ( /get/ );
    }
    elsif ( /^inci/ )
    {
        $api = "triggerIncidentWithFiles";
        $logFiles = shift or &usage("Please specify log files (comma seperated): $_");
        $correlationKey =shift or &usage("Please spcify correlation key: $_");
        $isIncident = 1;
    }
    else              
    {
        &usage("Unsupported Option $_");
    }
}

#Job API in JobDiagUtil prints command block output to STDERR to avoid data garbling.
#We need to redirect STDERR to STDOUT to capture the same
open STDERR, ">>&STDOUT" or die "Can't redirect STDERR: $!";

if(!$isIncident) {
    $api = "JobDiagUtil::$api";

    print "$preMsg\n"            if ($preMsg);
    $_ = &$api($arg1, @ARGV);
    print "$printResult = $_ \n" if ($printResult);
    print "$postMsg\n"           if ($postMsg);
}
else 
{

    #Dump directory - this is zipped and sent with incident.
    my $dmpDir =  ProvCommon::getIncidentZipFileDir();
    
    #State of the host is recovered and is sent along with logs
    my $generalDiag = ProvCommon::dumpDiagInfo($dmpDir);
    
    #Files to be dumped to the dump directory while an incident is generated
    if($logFiles ne "NULL"){
        my @dmpFiles = split(',',$logFiles);
        #Copy the log files in to dump file
        copy ($_, $dmpDir) foreach(@dmpFiles);
    }
    #Call the prov common api that generates incident with log files    
    $api = "ProvCommon::$api";
    
    #If the correlation_key is not sent via the caller then look for it in the environment
    if($correlationKey == "NULL"){
      $correlationKey = $ENV{CORRELATION_KEY};
    }
    
    print "Correlation Key: $correlationKey\n";
    
    $_ = &$api($correlationKey, $dmpDir);
    
    print "$printResult = $_ \n" if ($printResult);
}
