#!/usr/local/bin/perl
#
# $Header: emdb/sysman/admin/scripts/lsnr_log_parse.pl /st_emgc_pt-12.1.0.4pg/5 2012/10/12 04:00:41 tpalgudi Exp $
#
# lsnr_log_parse.pl
#
# Copyright (c) 2008, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      lsnr_log_parse.pl - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    tpalgudi    07/26/12 - Deduplication changes
#    mappusam    06/22/12 - bug-14221636 fix (Parse all TNS errors)
#    prjaiswa    01/11/12 - dyn prop fix
#    prjaiswa    01/11/12 - scan check fix :use machine check instead of port
#    prjaiswa    06/27/11 - bug 12658693
#    prjaiswa    05/24/11 - bug 12561317
#    mkiran      04/06/11 - 9930490: Multiple issues fixed
#    mappusam    03/14/11 - bug-10258842 Fix
#    prjaiswa    04/08/10 - moving scan check to listenerUtil
#    prjaiswa    09/15/09 - lrg 4156608
#    prjaiswa    08/21/09 - bug 8686293 : adding check for scan listener
#    prjaiswa    12/18/08 - bug 7653876
#    prjaiswa    08/07/08 - IPC key added
#    prjaiswa    05/21/08 - The scripts read location of listener log file from
#                           lsnrctl status command output and parses it to
#                           check for TNS errors
#    prjaiswa    05/21/08 - Creation
#

use strict;
use warnings;

require "semd_common.pl";
require has::HasCluster;
require "db/net/listenerUtil.pl";

use File::stat;
use vars qw/ $NT /;

sub handle_error
{
 my ($msg) = @_;

 # remove new line character at end if present
 chomp $msg if $msg;

 # remove spaces at begining and at end
 $msg =~ s/^\s+|\s+$// if $msg;

 # remove new line at beginning
 $msg =~ s/^\n//g if $msg;

 # remove em_error if present in mesg
 $msg =~ s/^em_error=// if $msg;

 print "em_error=$msg\n";
 exit(1);
}

# handle die message by printing me_error
$SIG{'__DIE__'} = sub { handle_error(@_); };

$ENV{ORACLE_HOME} = $ENV{LSNR_ORACLE_HOME};
$ENV{TNS_ADMIN}   = $ENV{LSNR_ORA_DIR};

my $SLASH = "/";
if ($NT)
{
 $SLASH = "\\";
}

my @errcode;
my @message;
my $state_root   = $ENV{EM_AGENT_STATE_DIR};
my $targetGuid   = $ENV{EM_TARGET_GUID};
my $machine      = $ENV{LSNR_MACHINE};
my $port         = $ENV{LSNR_PORT};
my $name         = $ENV{LSNR_NAME};
my $listenerFile = $ENV{LSNR_ORA_DIR} . $SLASH . "listener.ora";
my $executable   = $ENV{LSNR_ORACLE_HOME} . $SLASH . "bin" . $SLASH . "lsnrctl";
my $key          = $ENV{LSNR_KEY};
my $pswd         = $ENV{LSNR_PASSWORD};
my $lsnrType     = $ENV{LSNR_TARGET_SUBTYPE};
my $lsnrVersion  = $ENV{LSNR_TARGET_VERSION};

EMD_PERL_DEBUG(" $name :: listener subtype is $lsnrType");
EMD_PERL_DEBUG(" $name :: listener version is $lsnrVersion");

# check for listener type
# retry only if type info was unavailable at agent start up
if ( $lsnrType eq "UNKNOWN" )
{
 $lsnrType = getListenerType($ENV{LSNR_ORACLE_HOME} , $name ,$machine ,$lsnrVersion);
 EMD_PERL_INFO(" $name : retry listener type check : lsnrType =$lsnrType");
}

my $address = "";
if ($key)
{
 EMD_PERL_DEBUG(" $name :: IPC listener");
 $address = "(ADDRESS=(PROTOCOL=IPC)(KEY=$key))";
}
else
{
 EMD_PERL_DEBUG(" $name :: TCP listener");
 $address = "(ADDRESS=(PROTOCOL=TCP)(HOST=$machine)(PORT=$port))";
}

my $command = "";
if ( $lsnrType eq "SCAN" )
{
 $command = "status $name";
}
else
{
 $command = "status $address";
}

EMD_PERL_DEBUG("command = $command ");

my $result;
eval {
 $result = getResultNew( $executable, $command, $listenerFile, $name, $pswd );
};
if ($@)
{
 print STDERR "em_error=Failed to run lsnrctl\n";
 exit(-1);
}
EMD_PERL_DEBUG(" $name : getResultNew = $result ");

my $listener_log = " ";

## parse result
my @info = split( /\n/, $result );
my $line;
foreach $line (@info)
{

 if ( $line =~ /^\s*Listener Log File\s+(.*)/i )
 {
  $listener_log = $1;
 }

}
EMD_PERL_DEBUG("$name : listener_log = $listener_log ");

# 9930490: listener.log will not be generated if listener starts with
# LOGGING_LISTENER=OFF set in listener.ora. Proceeding further is
# disastrous as it will cause incorrect file size computation and
# improper file scanning. So, just log a message about
# non-availability of listener.log and exit normally.
if ( ( $listener_log eq " " ) || ( !-e $listener_log ) )
{
 EMD_PERL_DEBUG("listener.log does not exist\n");
 exit(0);
}

# If emdstate or  targetguid is not found  -> exit

if ( !( defined($state_root) && defined($targetGuid) ) )
{
 die(
"ERROR : unable to get targetguid ($targetGuid) or state($state_root) directory \n"
 );
}

my $listener_log_size =
    $state_root
  . $SLASH
  . "sysman"
  . $SLASH . "emd"
  . $SLASH . "state"
  . $SLASH
  . "$targetGuid"
  . "_lsnr_log_size.log";

if ( EMAGENT_isPerlDebugEnabled() )
{
 EMD_PERL_DEBUG("$name:Debug state_root : $state_root");
 EMD_PERL_DEBUG("$name:Debug targetGuid : $targetGuid");
 EMD_PERL_DEBUG("$name:Debug listener_log_size     : $listener_log_size");

}

# Implementing Incremental reading of log file

my $NEW_FILE;
if ( -e $listener_log_size )
{
 EMD_PERL_DEBUG(" $name : File exists \n");
 $NEW_FILE = 0;
}
else
{
 EMD_PERL_DEBUG(" $name : File does not exist \n");
 $NEW_FILE = 1;
}

# 9930490: Open listener.log in read-only mode, to scan for TNS errors.
open( LOGFILE, "<", "$listener_log" )
  || die "Can't open listener log file: $listener_log for reading : $! \n";

if ( $NEW_FILE == 1 )
{

 # check whether file can be created under the directory or not
 # Exit if file cannot be created
 #do scan the full log and create new size file
 # +> mode is read write mode with create/override existing file
 open( MYFILE, "+>", "$listener_log_size" )
   || die
   "Can't open NEW listener log size tracking file: $listener_log_size : $!\n";

 # 9930490: Use size attribute of File::stat to compute
 # listener.log file size.
 my $filesize = ( stat($listener_log) )->size;
 EMD_PERL_DEBUG("$name : Size of newly available listener.log = $filesize");

 print MYFILE "$filesize\n";
 close(MYFILE);

 # 9930490: This is the first ever scan of listener.log. So, let's
 # scan from the top of the file.
 seek( LOGFILE, 0, 0 );
}

# open the size file and calculate the tailing bytes to read from log
if ( $NEW_FILE == 0 )
{

# always open state file in read/write mode. This will ensure to abort even if somebody played with dir/file permissions at backend after state size file is created .
# +< mode is read write mode without create/override existing file
 open( MYFILE, "+<", "$listener_log_size" )
   || die
"Can't open EXISTING listener log size tracking file: $listener_log_size in read/write mode : $! \n";
 my @lines = <MYFILE>;

 my $oldsize = $lines[0];
 EMD_PERL_DEBUG("$name :old size = $oldsize");

 # 9930490: Use size attribute of File::stat to compute
 # listener.log file size.
 my $newsize = ( stat($listener_log) )->size;
 EMD_PERL_DEBUG("$name : Size of listener.log = $newsize");

 # 9930490: listener.log was already scanned for TNS errors till
 # $oldsize position. Let's now scan from $oldsize position onwards.
 #check whether read size is a valid digit
 if ( $oldsize =~ /\d+/ )
 {
  seek( LOGFILE, $oldsize, 0 )
    || die("Couldn't seek to $oldsize position in log \n");
 }
 else
 {
  die(" $name : Invalid entry in $listener_log_size file");
 }

 # override always 1st line in size file
 seek( MYFILE, 0, 0 )
   || die(" $name : Couldn't seek to start position in size log \n");
 print MYFILE "$newsize\n";
 close(MYFILE);
}

# Patterns to search which indicate security breaches

my $logcommand, my $logcommandcode, my $tmp1, my $tmp2, my $tmp3, my $tmp4,
  my $lastErrorTimeStamp;
#14221636 bug fix parse all tns errors in the listener log file

# To parse all TNS pattern TNS-(.*)
my $searchTNSerrpattern  = 'TNS-(.*)';

my $previous_line = "";
my $readline = "";
my $tmpcode= "";
my $tnserrorcodemsg= "";
my $tnscode= "";
my %first_occur_timestamp;
my %last_occur_timestamp;
my %error_code_msg;
my %error_count;
my $msg = "";

my $xmllog =0;
# check the log file has .xml/.log extn 
if ($listener_log =~ /.*.xml/)
{
   $xmllog = 1;
}

#read all TNSerror code in the listener log and parse  previous line to get command and code
while (<LOGFILE>)
{

      $readline = $_;     
      # check the log file has .xml/.log extn 
      if ( $xmllog eq 1)
      {
          #If it is xml read only <txt> tags others ignore
          if( $readline !~ m/\<txt\>/)
          {
              next;
          }
      }
     # Search the TNS error code in the listener log
      if(($readline=~/$searchTNSerrpattern/))
      {
          $tmpcode =$1;
          # remove leading zero in the tns error code
          if($tmpcode=~/^0(.*)/)
          {
             $tnserrorcodemsg = $1;
          }
          else
          {
            $tnserrorcodemsg = $tmpcode;
          }
          # $tnserrorcodemsg = 12528: TNS:listener: all appropriate instances are blocking new connections
          my @temptnscode = split(':', $tnserrorcodemsg);
          $tnscode = $temptnscode[0]; 
          EMD_PERL_DEBUG(" TNS ERROR Found in listener log = $tnscode");
          # To parse the previouse line to get the command which introduced the tns code in the log.
         if($previous_line=~/$tnscode/)
          {
              #Remove all xml tag from the previous line
              $previous_line =~ s/<.*>// if $previous_line;
              my @tmp5 = split(/\*/i,$previous_line);
              
              my $size = @tmp5;
              $lastErrorTimeStamp = $tmp5[0];

              # size value will be 4 if the line has lsnrcommand
              if ( $size == 4 )
              {
                  #get data from log file columns
                  $logcommandcode     = $tmp5[ $size - 1 ];
                  $logcommand         = $tmp5[ $size - 2 ];
              }
              else
              {
                  $logcommandcode = $tnscode;
                  $logcommand = "";
              }
             # remove spaces at begining and at end
             $logcommandcode     = trim($logcommandcode);
             $logcommand         = trim($logcommand);
             $lastErrorTimeStamp = trim($lastErrorTimeStamp);
             $lastErrorTimeStamp = formatDatenTime($lastErrorTimeStamp);
              
             if ( ($logcommandcode) && isLsnrCommand($logcommand) )
             {
               $logcommandcode = "TNS-" . "$logcommandcode";
               $msg = "$logcommandcode" . "($logcommand)";
             }
             else # if there is no logcommand then add logcommandcode to the message
             {
               $logcommandcode = "TNS-" . "$logcommandcode";
               $msg = $logcommandcode;
             }
            
             #insert error code and message into hash, if error code already exists then update the message
             if( !exists $error_code_msg{$logcommandcode} )
             {
               $error_code_msg{$logcommandcode} = $msg;
             }
             else
             {
               my $mesg = $error_code_msg{$logcommandcode};
               $error_code_msg{$logcommandcode}  = "$mesg,"."$msg";
             }

             #update the count
             if( !exists $error_count{$logcommandcode} )
             {
               $error_count{$logcommandcode} = 1;
             }
             else
             {
               my $count = $error_count{$logcommandcode};
               $error_count{$logcommandcode} = $count+1;
             }

             #update first occurance timestamp
             if( !exists $first_occur_timestamp{$logcommandcode} )
             {
               $first_occur_timestamp{$logcommandcode} = $lastErrorTimeStamp;
             }
             #update last occured timestamp
             $last_occur_timestamp{$logcommandcode} = $lastErrorTimeStamp;
         }
     }
     $previous_line = $readline;
}

close(LOGFILE);

foreach $key (keys %error_code_msg)
{
  #remove the duplicates in the message
  $msg = $error_code_msg{$key};
  my @umsg = split(',', $msg);
  my %umhash   = map { $_, 1 } @umsg;
  my @uniq_msgs = keys %umhash;
  $msg = "@uniq_msgs";
  # restrict the message to 128 characters
  $msg = substr $msg, 0, 128;
  print "em_result=$key|$msg|$error_count{$key}|$first_occur_timestamp{$key}|$last_occur_timestamp{$key}\n";
  EMD_PERL_DEBUG("em_result=$key|$msg|$error_count{$key}|$first_occur_timestamp{$key}|$last_occur_timestamp{$key}\n");
}

sub formatDatenTime
{
    #time stamp will come in the format  = "18-JUN-2012 22:59:03";
    (my $timeStamp) = @_;
    my %month = ( 'JAN' => '01', 'FEB' => '02', 'MAR' => '03', 'APR' => '04',
                  'MAY' => '05', 'JUN' => '06', 'JUL' => '07', 'AUG' => '08',
                  'SEP' => '09', 'OCT' => '10', 'NOV' => '11', 'DEC' => '12');

    my @dateTime = split(' ',$timeStamp);
    my @dat = split('-', $dateTime[0]);
    my $date = "$dat[2]-$month{$dat[1]}-$dat[0]";
    my $time = $dateTime[1];
    $timeStamp = "$date $time";
    # it will be converted to the form yyyy-mm-dd hh:mm:ss i.e 2012-06-18 22:59:03
    return $timeStamp;
}

exit 0;
