#
# Copyright (c) 2001, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#  $Id: oracle_db_files.pl /st_emgc_pt-12.1.0.4pg/4 2012/10/09 19:40:05 yozhang Exp $ 
#
#
# NAME
#   oracle_db_files.pl
#
# DESC
#     Subroutine for getting the file metrics for a Oracle DB files along with the filesystem or raw device
#
#
# FUNCTIONS
#
#
# NOTES
#           Target Adress expected to be in the standard format
#      (DESCRIPTION=(ADDRESS=(PROTOCOL=TCP)(HOST=dlsun1170)(PORT=1531))(CONNECT_DATA=(SID=emd)))
#
# MODIFIED   (MM/DD/YY)
# yozhang       10/05/12 - Fix query errors
# yozhang       09/17/12 - Bug 14564972: replace services with containers
# yozhang       09/17/12 - Bug 14572116: archive
# yozhang       03/02/12 - Avoid duplicate records
# yozhang       01/06/12 - Add CDB query for datafile
# glavash       02/01/11 - add type to redo logs ocm convergence
# skatdare      07/30/10 - Fix bug 9956393: perf optimizing
# yozhang       09/22/09 - Fix bug 8266595: error handling
# yozhang       03/06/09 - Fix RFI bug 8242774: perf issues
# mnihalan      11/18/08 - Fix datafiles temp files
# mnihalan      11/17/08 - Fix bug 7514751
# yozhang       10/03/08 - change sutilities back to register
# yozhang       09/24/08 - Use sUtilies directly
# kganapat      11/29/07 - XbranchMerge kganapat_filter_ora_errs from main
# ajdsouza      12/02/05 - fix bug 4739322, iload storage modules after db io
#                           to prevent nls issues
# xshen         01/12/05 - modify redolog query 
# ajdsouza      10/13/04 - Add hint for v$datafile
# ajdsouza      08/19/04 - Added locale
# ajdsouza      07/23/04 - Commented the column os_storage_entity
# ajdsouza      06/25/04 - storage reporting sources 
# ajdsouza      04/16/04 - Creation, Replicated the sqls from oracle_database.xml
#
#

require v5.6.1;
require 'emd_common.pl';

use strict;
use warnings;
use DBI;

my $fptr_get_os_storage_entity_path;
my %list_of_sqls;
my %order_of_fields;
my %os_file_name;

# define the other configuration variables
%list_of_sqls = (
       db_datafiles => 
       [
        "
    SELECT REPLACE(REPLACE(ddf.file_name, chr(10), ''), chr(13), '') \"file_name\",
           vdf.status    \"status\",
           ddf.tablespace_name \"tablespace_name\",
           NVL(ddf.bytes,0) \"bytes\",
           ddf.autoextensible \"autoextensible\",
           ddf.increment_by \"increment_by\",
           ddf.maxbytes \"max_bytes\",
           vdf.create_bytes  \"create_bytes\" 
      FROM sys.dba_data_files ddf,
           v\$datafile vdf 
     WHERE (ddf.file_id = vdf.file#)",      
        "
    SELECT REPLACE(REPLACE(dtf.file_name, chr(10), ''), chr(13), '') \"file_name\",
           vtf.status     \"status\",
           dtf.tablespace_name \"tablespace_name\",
           NVL(dtf.bytes,0) \"bytes\",
           dtf.autoextensible \"autoextensible\",
           dtf.increment_by \"increment_by\",
           dtf.maxbytes \"max_bytes\",
           vtf.create_bytes \"create_bytes\"
      FROM sys.dba_temp_files dtf,
           v\$tempfile vtf
     WHERE (dtf.file_id = vtf.file#)" 
       ],
       db_datafiles_10gR2 => 
       [
        "
    SELECT REPLACE(REPLACE(ddf.file_name, chr(10), ''), chr(13), '') \"file_name\",
           ddf.online_status    \"status\",
           ddf.tablespace_name \"tablespace_name\",
           NVL(ddf.bytes,0) \"bytes\",
           ddf.autoextensible \"autoextensible\",
           ddf.increment_by \"increment_by\",
           ddf.maxbytes \"max_bytes\", 
           vdf.create_bytes \"create_bytes\"
      FROM sys.dba_data_files ddf,
           v\$datafile vdf /*+ all_rows use_concat */ 
      WHERE (ddf.file_id = vdf.file#) ",      
        "
    SELECT REPLACE(REPLACE(dtf.file_name, chr(10), ''), chr(13), '') \"file_name\",
           vtf.status     \"status\",
           dtf.tablespace_name \"tablespace_name\",
           NVL(dtf.bytes,0) \"bytes\",
           dtf.autoextensible \"autoextensible\",
           dtf.increment_by \"increment_by\",
           dtf.maxbytes \"max_bytes\",
           vtf.create_bytes \"create_bytes\"
      FROM sys.dba_temp_files dtf,
           v\$tempfile vtf
     WHERE (dtf.file_id = vtf.file#)" 
       ],
       db_datafiles_11gR2 => 
       [
        "
    	 SELECT REPLACE(REPLACE(ddf.file_name, chr(10), ''), chr(13), '') \"file_name\", 
           ddf.online_status    \"status\",
           ddf.tablespace_name \"tablespace_name\",
           NVL(ddf.bytes,0) \"bytes\",
           ddf.autoextensible \"autoextensible\",
           ddf.increment_by \"increment_by\",
           ddf.maxbytes \"max_bytes\",
           vdf.create_bytes \"create_bytes\"
      FROM sys.dba_data_files ddf, v\$datafile vdf
          WHERE (vdf.file# = ddf.file_id) ",      
        "
    	SELECT REPLACE(REPLACE(dtf.file_name, chr(10), ''), chr(13), '') \"file_name\", 
           dtf.status     \"status\",
           dtf.tablespace_name \"tablespace_name\",
           NVL(dtf.bytes,0) \"bytes\",
           dtf.autoextensible \"autoextensible\",
           dtf.increment_by \"increment_by\",
           dtf.maxbytes \"max_bytes\",
           vtf.create_bytes \"create_bytes\"
      FROM sys.dba_temp_files dtf, v\$tempfile vtf
         WHERE (dtf.file_id = vtf.file#) " 
       ],
       db_datafiles_12cdb => 
       [
        "
    	 SELECT pdb.name \"pdb_name\",
           REPLACE(REPLACE(ddf.file_name, chr(10), ''), chr(13), '') \"file_name\", 
           ddf.online_status    \"status\",
           ddf.tablespace_name \"tablespace_name\",
           NVL(ddf.bytes,0) \"bytes\",
           ddf.autoextensible \"autoextensible\",
           ddf.increment_by \"increment_by\",
           ddf.maxbytes \"max_bytes\",
           vdf.create_bytes \"create_bytes\"
      FROM sys.cdb_data_files ddf, v\$datafile vdf, (select con_id, name from v\$containers where con_id != 2) pdb
          WHERE (vdf.file# = ddf.file_id) and ddf.con_id = pdb.con_id and vdf.con_id = pdb.con_id ",      
        "
    	SELECT pdb.name \"pdb_name\",
           REPLACE(REPLACE(dtf.file_name, chr(10), ''), chr(13), '') \"file_name\", 
           dtf.status     \"status\",
           dtf.tablespace_name \"tablespace_name\",
           NVL(dtf.bytes,0) \"bytes\",
           dtf.autoextensible \"autoextensible\",
           dtf.increment_by \"increment_by\",
           dtf.maxbytes \"max_bytes\",
           vtf.create_bytes \"create_bytes\"
      FROM sys.cdb_temp_files dtf, v\$tempfile vtf, (select con_id, name from v\$containers where con_id != 2) pdb
         WHERE (dtf.file_id = vtf.file#) and dtf.con_id = pdb.con_id and vtf.con_id = pdb.con_id " 
       ],
       db_controlfiles => 
       [
        "
   SELECT   cf.name \"file_name\",
            db.controlfile_type \"status\",
            to_char(db.controlfile_created,'YYYY-MM-DD HH24:MI:SS') \"creation_date\",
            db.controlfile_sequence# \"sequence_no\",
            db.controlfile_change# \"last_change_scn\",
            to_char(db.controlfile_time,'YYYY-MM-DD HH24:MI:SS') \"last_modification_date\"
    FROM   v\$controlfile cf,
            v\$database db"
        ],
       db_redologs =>
       [
        "
       SELECT  l.group# \"group_no\",
       NLS_INITCAP(l.status) \"status\",
       l.members \"no_of_members\",
       lf.member \"file_name\",
       'N/A' \"archived\",
          l.bytes \"size_k\",
          l.sequence# \"sequence_no\",
          l.first_change# \"first_change_scn\",
          l.thread# \"thread_num\" ,
          lf.type \"type\" 
  FROM      v\$log l,
          v\$logfile lf
  WHERE   l.group# = lf.group#"
       ]
      );

%order_of_fields = (
          db_datafiles => {
                   file_name=>1,
                   status=>2,
                   tablespace_name=>3,
                   bytes=>4,
                   autoextensible=>5,
                   increment_by=>6,
                   max_bytes=>7,
                   create_bytes=>8,
                   os_storage_entity_identifier=>9
                  },
          db_datafiles_10gR2 => {
                   file_name=>1,
                   status=>2,
                   tablespace_name=>3,
                   bytes=>4,
                   autoextensible=>5,
                   increment_by=>6,
                   max_bytes=>7,
                   create_bytes=>8,
                   os_storage_entity_identifier=>9
                  },
          db_datafiles_11gR2 => {
                   file_name=>1,
                   status=>2,
                   tablespace_name=>3,
                   bytes=>4,
                   autoextensible=>5,
                   increment_by=>6,
                   max_bytes=>7,
                   create_bytes=>8,
                   os_storage_entity_identifier=>9
                  },
          db_datafiles_12cdb => {
                   pdb_name=>1,
                   file_name=>2,
                   status=>3,
                   tablespace_name=>4,
                   bytes=>5,
                   autoextensible=>6,
                   increment_by=>7,
                   max_bytes=>8,
                   create_bytes=>9,
                   os_storage_entity_identifier=>10
                  },
          db_controlfiles => {
               file_name=>1,
               status=>2,
               creation_date=>3,
               sequence_no=>4,
               last_change_scn=>5,
               last_modification_date=>6,
               os_storage_entity_identifier=>7
                   },
          db_redologs => {
                group_no=>1,
                status=>2,
                no_of_members=>3,
                file_name=>4,
                archived=>5,
                size_k=>6,
                sequence_no=>7,
                first_change_scn=>8,
                thread_num=>9,
                type=>10,
                os_storage_entity_identifier=>11
               }
         );

%os_file_name = (
       db_datafiles =>'file_name',      
       db_datafiles_10gR2 =>'file_name',
       db_datafiles_11gR2 =>'file_name',
       db_datafiles_12cdb =>'file_name',
       db_controlfiles=>'file_name',
       db_redologs=>'file_name'
      );

#-----------------------------------------------------------------------------------------
# subs declared
my %stdinArgs;
my $password;
my $username;
my $address;
my $role;
my $mode;
my @results;

# DB connection timeout and retry values in secs
my $db_timeout  = 10;   #timeout secs
my $db_retry    = 3;    #number of retries
my $db_waittime = 60;   #wait time before retry

# Get the username,password and target database base address from
# fetchlet environment
%stdinArgs = get_stdinvars();

$password = $stdinArgs{EM_TARGET_PASSWORD} if $stdinArgs{EM_TARGET_PASSWORD};
$username = $stdinArgs{EM_TARGET_USERNAME} if $stdinArgs{EM_TARGET_USERNAME};
$address = $ENV{EM_TARGET_ADDRESS} if $ENV{EM_TARGET_ADDRESS};
$role = $ENV{EM_TARGET_ROLE} if $ENV{EM_TARGET_ROLE};
$mode = 0;

$mode = 2 if $role and $role =~ /SYSDBA/i;
$mode = 4 if $role and $role =~ /SYSOPER/i;

die "ERROR: Target database credentials not available \n" unless $username and $password and $address;

EMD_PERL_DEBUG ("Connecting to $username $address \n");

#--------------------------------
# Get connection to the database
#--------------------------------
my %dbh;

$dbh{dbh}= DBI->connect('dbi:Oracle:', "$username@".$address , "$password", {ora_session_mode => $mode, PrintError => 0, RaiseError => 0}) or die (filterOraError("Could not connect to $username/$address: $DBI::errstr\n", $DBI::err));

$dbh{dbh}->{FetchHashKeyName} = 'NAME_lc';

#---------------------------------------------------------------------------------
# Execute the sql ad fetch the results as an array of hashes
#---------------------------------------------------------------------------------
die "Usage perl oracle_database.pl <metric_name>" unless $ARGV[0];

die "The metric name $ARGV[0] is invalid" unless $list_of_sqls{$ARGV[0]} and @{$list_of_sqls{$ARGV[0]}};

for  my $sql ( @{$list_of_sqls{$ARGV[0]}} )
{

    my $sth = $dbh{dbh}->prepare($sql) or die (filterOraError("ERROR preparing $sql : $DBI::errstr\n", $DBI::err));
    
    $sth->execute or $sth->finish and die (filterOraError("ERROR executing $sql : $DBI::errstr\n", $DBI::err));
    
    my $array_ref = $sth->fetchall_arrayref( $order_of_fields{$ARGV[0]} ) or $sth->finish and die (filterOraError("ERROR fetching $sql : $DBI::errstr\n", $DBI::err));
    
    die (filterOraError("ERROR : $sql fetch :: ".$sth->errstr." \n", $sth->err)) and $sth->finish and $dbh{dbh}->disconnect if $sth->err;
    
    # this check is not required refer bug# 4764143, there can be queries that fetch no results
    # die "ERROR : $sql No rows found \n" and $sth->finish and $dbh{dbh}->disconnect unless $sth->rows and $array_ref and @{$array_ref};

    next unless  $sth->rows and $array_ref and @{$array_ref};

    # storage the results in a global
    for my $row ( @{$array_ref} )
    {
      my %rowhsh = %{$row};
      push @results,\%rowhsh;
    }

    $sth->finish or die (filterOraError("Failed to disconnect from the database $address : $DBI::errstr",  $DBI::err));   
    
}
    
$dbh{dbh}->disconnect or return;
    
#---------------------------------------------------------------------------------
# Load the storage libraries after DB I/O is completed
#  This will prevent any issues with DBI NLS
#---------------------------------------------------------------------------------
# Get a pinter to the storage::Register::get_os_storage_entity_path
# function, dont fail if module is absent

$SIG{'__WARN__'} = sub {};

eval '
use storage::Register; 
$fptr_get_os_storage_entity_path = \&storage::Register::get_os_storage_entity_path 
';

if (  $@ )
{
 # In case of failure setting the environment
  undef $fptr_get_os_storage_entity_path 
   if $fptr_get_os_storage_entity_path; 

  # Do no use locale, in the absence of use locale the namespace of this
  # script will use the default locale
  # but the storage modules will use the locale set in them

  # Set environment to C locale
  # OS binaries can see this locale
  my $clocale='C';

  for ( qw ( LC_ALL LC_COLLATE LC_CTYPE LC_TIME LC_NUMERIC LC_MESSAGES LC_MONETARY LANG LANGUAGE ) )
  {
    $ENV{$_}=$clocale;
  }
}

$SIG{'__WARN__'} = sub { warn "$_[0]"; }; 

# Shouldnt we set locale here if it was not set in Register



#---------------------------------------------------------------------------------
# Get the filesystem or the raw device used by the oracle file
#---------------------------------------------------------------------------------
if ( $os_file_name{$ARGV[0]} )
{
   
   for my $row ( @results )
   {   
       
       # Initialize
       $row->{os_storage_entity_identifier}='NA';
       
       next unless $row->{$os_file_name{$ARGV[0]}};

        next unless $fptr_get_os_storage_entity_path;

       # Do not show warn messaegs from storage reporting module
       $SIG{'__WARN__'} = sub {};

       # Get the filesystem or raw device for the oracle file
       $row->{os_storage_entity_identifier} = $fptr_get_os_storage_entity_path->( $row->{$os_file_name{$ARGV[0]}} );

       $SIG{'__WARN__'} = sub { warn "$_[0]"; }; 

       # if the field is blanked out set it to NA
       $row->{os_storage_entity_identifier}='NA' unless $row->{os_storage_entity_identifier};

       
    }

}

#---------------------------------------------------------------------------------
# Print the metrics to STDOUT
#---------------------------------------------------------------------------------
for my $row ( @results )
{
   print "em_result=";
   
   for my $key ( sort { $order_of_fields{$ARGV[0]}{$a} <=> $order_of_fields{$ARGV[0]}{$b} } keys %{$order_of_fields{$ARGV[0]}} )
      {
       
       warn " field $key is not in the list of quieried fields for ARGV[0] \n" unless exists $row->{$key};
       
       print "$row->{$key}" if $row->{$key};
       
       print "|" unless $order_of_fields{$ARGV[0]}{$key} == keys %{$order_of_fields{$ARGV[0]}};
       
      }
   
   print "\n";
}
    

exit 0;
