#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/rac/rac_db_metrics.pl /st_emgc_pt-12.1.0.4pg/3 2012/08/16 10:27:55 mcouturi Exp $
#
# rac_db_metrics.pl
# 
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      rac_db_metrics.pl - RAC database configuration and server pool metrics
#
#    DESCRIPTION
#      RAC database configuration and server pool metrics.
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    mcouturi    07/30/12 - bug 14386000 - added null checking before string
#                           split
#    mcouturi    03/20/12 - Added srvpool type support to handle PQ pools
#    ajdsouza    04/15/11 - bug#11826053 
#    rsamaved    03/18/11 - handle pre 11.2 databases
#    rsamaved    06/05/10 - replace oracle_unqnname
#    pardutta    03/26/10 - RAC database configuration and server pool metrics
#    pardutta    03/26/10 - Creation
#
use strict;
use warnings;
use Switch;
use File::Spec::Functions;
use File::Path;
use Storable;

require "semd_common.pl";

$SIG{__WARN__} = 'WARN_handler';
$SIG{__DIE__}  = 'DIE_handler';

my $em_target_name = $ENV{EM_TARGET_NAME} if $ENV{EM_TARGET_NAME};
die "Target Name not set in metric scripts environment variable EM_TARGET_NAME"
and return unless $em_target_name;

my $db_name = $ENV{EM_TARGET_DBNAME} if  $ENV{EM_TARGET_DBNAME};
die "Database Name not set in metric script's environment variable EM_TARGET_DBNAME"
and return unless $db_name;

my $oh = $ENV{EM_TARGET_ORACLE_HOME} if $ENV{EM_TARGET_ORACLE_HOME};
die "Oracle home not set in metric script's environment variable EM_TARGET_ORACLE_HOME"
and return unless $oh;

my $agent_state_dir = $ENV{AGENT_STATE_DIR} if $ENV{AGENT_STATE_DIR};
warn "Agent state directory not set in metric script's environment variable AGENT_STATE_DIR"
and return unless $agent_state_dir;

my $osm_instance = $ENV{OSM_INSTANCE} if $ENV{OSM_INSTANCE};
my $pre11gr2 = $ENV{PRE_11GR2} if $ENV{PRE_11GR2};

switch($ARGV[$#ARGV]) {
        case "RAC_DATABASE_CONFIG" { configuration($osm_instance,$pre11gr2); last;}
        case "RAC_DATABASE_SRVPOOLS" { server_pools(); last;}
}


# Subroutines:

sub configuration {
        EMD_PERL_DEBUG("RAC_DATABASE_CONFIG metric collection started.");
    my ($osm_instance,$pre11gr2) = @_;
        my $is_on_asm = 'N';

    if(defined($osm_instance) && $osm_instance ne "")
    {
           $is_on_asm = 'Y';
        }

    if (defined($pre11gr2) && $pre11gr2 eq "TRUE")
    {
        # For pre 11gR2 vesions return hard coded values
        # database type - RAC
        # config type   - empty
        # resource name - empty
        # asm           - actual value
        # bug# 11826053 - resource name is not null in table MGMT_RACDB_CONFIG_ECM
        my $upload_rec = "em_result=RAC|ADMIN_MANAGED|NA|$is_on_asm\n";
        print $upload_rec;
        EMD_PERL_DEBUG("RAC_DATABASE_CONFIG metric data: $upload_rec");
    }
    else
    {
        build_database_config_hash();
        my $href = load_cache();
        my $upload_rec = "em_result=$href->{db_category}|$href->{db_type}".
                         "|$href->{res_name}|$is_on_asm\n";
        print $upload_rec;
        EMD_PERL_DEBUG("RAC_DATABASE_CONFIG metric data: $upload_rec");
    }

}

sub server_pools {
 EMD_PERL_DEBUG("RAC_DATABASE_SRVPOOL metric collection started.");

 #define the different types of server pools
        my $normal = "NORMAL_POOL";
        my $pq = "PARALLEL_QUERY_POOL";

        #get all pools
        my $href = load_cache();
        
	my $srvpools = $href->{srvpool};
	my @srvpool_list = ();
	#making sure href for srvpools is not null
	if(defined($srvpools)){
		@srvpool_list = split(",", $srvpools);
	}

        #get pq pools
	my $pqpools = $href->{pq_srvpool};
        my @pqpool_list = ();
	#making sure the href for pqpools is not null
	if(defined($pqpools)){
		@pqpool_list = split(",", $pqpools);
	}
	my %hash;
	if(@pqpool_list){
		@hash{@pqpool_list} = ();
	}
        
	EMD_PERL_DEBUG("parsed pqpools");

        #write normal pools
        foreach (@srvpool_list) {
                #ignore pq-entries
                unless (%hash && exists $hash{$_}){       
	                my $upload_rec ="em_result=ora.$_|$normal\n";
        	        print $upload_rec;
                	EMD_PERL_DEBUG("RAC_DATABASE_SRVPOOLS metric data: ".
                                "$upload_rec");
		}
        }

        #write pqpools
        foreach (@pqpool_list) {
                my $upload_rec ="em_result=ora.$_|$pq\n";
                print $upload_rec;
                EMD_PERL_DEBUG("RAC_DATABASE_PQPOOLS metric data: ".
                                "$upload_rec");
        }

 }

# Run srvctl command
sub build_database_config_hash() {
        EMD_PERL_DEBUG("build_database_config_hash(): Preparing to ".
                        "collect database attributes.\n");

        my $dbname = $ENV{EM_TARGET_DBNAME};
        die "Database name not set in metric scripts ".
            "environment variable 'DBNAME'\n" and return unless $dbname;

    my $saveOH = $ENV{ORACLE_HOME};
    $ENV{ORACLE_HOME} = $oh;
        my $cmd = $oh."/bin/srvctl config database -d ".$dbname." -S 1";
        my $result = run_command($cmd);

	EMD_PERL_DEBUG("result of dbconfig: $result");	
        die "$cmd did not run successfully. Refer agent's ".
            "perl log files for details.\n" and return unless $result;

    $ENV{ORACLE_HOME} = $saveOH;
        my @reqd_tokens = ("res_name", "db_type", "db_category", "srvpool", "pq_srvpool");
        my $db_config_ref = parse_srvctl_result($result, \@reqd_tokens);

        die "Failed to parse database configuation output. ".
            "Refer agent's perl log files for details.\n"
        and return unless $db_config_ref;

        save_cache($db_config_ref);
}


# Given one line of srvctl output that has <token, value> pairs,
# this function will return a hash of <key,value> pairs where key=token.
# Parameters: 1. srvctl output like
#               "#@=result[0]: res_name={ora.r1.db} dbunique_name={r1} ..."
#             2. array of tokens to extract like [res_name, dbunique_name]
sub parse_srvctl_result {

        my ($srvctl_result, $reqd_tokens) = @_;

        EMD_PERL_DEBUG("parse_srvctl_result: Input : $srvctl_result");

        my $parse_result_href = {};
        my @token_list = split(" ", $srvctl_result);
        foreach (@$reqd_tokens) {
                return undef unless extract_attr($_, $parse_result_href,
                                                        \@token_list) == 0;
        }

        EMD_PERL_DEBUG("parse_srvctl_result: Database attribute hash dump: ".
                        "@{[%$parse_result_href]}");

        return $parse_result_href;
}

sub extract_attr {
        my ($ip, $href, $aref) = @_;

        my @tokens = @$aref;
        EMD_PERL_DEBUG("extract_attr: Attribute: $ip.");

        my @tok_list = grep /^$ip={.*}/i, @tokens;
        my $no_matches = @tok_list;
        if($no_matches == 1) {
                $tok_list[0] =~ /$ip={(.*)}/;
                $href->{$ip} = $1;
                EMD_PERL_DEBUG("extract_attr: Value: $1.");
        } else {
                # some versions of databases may not have some attributes
                # for example, db_category is applicable in 11.2.0.2 and
                # later only
                if ($ip =~ /db_category/)
                {
                    $href->{$ip} = 'RAC';
                }
                EMD_PERL_DEBUG("extract_attr: Number of tokens mathing $ip ".
                                "attribute is $no_matches.");
        }

        return 0;
}

sub load_cache {
        EMD_PERL_DEBUG("load_cache(): Loading cache from disk.\n");

        my $cache_file = locate_cache_file($em_target_name);

        EMD_PERL_DEBUG("load_cache(): Path to cache $cache_file \n");

        return retrieve($cache_file);
}

sub save_cache() {
        EMD_PERL_DEBUG("save_cache(): Saving cache to disk.\n");


        my ($href) = @_;
        my %attr_hash = %$href;
        my $cache_file = locate_cache_file($em_target_name);
        store $href, $cache_file;

        EMD_PERL_DEBUG("save_cache(): Path to cache file $cache_file\n");
}

sub locate_cache_file {

        my ($target_name) = @_;

        my $cache_dir = $agent_state_dir;

        EMD_PERL_DEBUG("locate_cache_file(): Target Name: $target_name, ".
                       "Agent State Dir: $agent_state_dir\n");

        # if the em state dir doesnt work use temp dir
        $cache_dir =  File::Spec->tmpdir() unless isDirUsable($cache_dir);

        # if the em state dir doesnt work use temp dir
        $cache_dir =  Cwd::abs_path() unless isDirUsable($cache_dir);

        # terminate if it still doesnt work
        die "Unable to use cache directory $cache_dir. Check if directory ".
            "exist and is writable." and return unless isDirUsable($cache_dir);

        # build the path to the em state directory
        $cache_dir =  catfile($cache_dir,'sysman');
        $cache_dir =  catfile($cache_dir,'emd');
        $cache_dir =  catfile($cache_dir,'state');
        $cache_dir =  catfile($cache_dir,'rac_database');
        $cache_dir =  catfile($cache_dir, $target_name);
        unless( -e $cache_dir) {
                mkpath($cache_dir,0,0777);
        }
        return catfile($cache_dir, 'racdb.txt');
}

sub isDirUsable
{
  my ( $path ) = @_;

  return unless $path;

  return 1 if -e $path and -r $path and -w $path and -d $path;

  return;
}

sub run_command
{
    my ($cmd) = @_;

    my $result = '';

    chomp ($result = `$cmd`);

    EMD_PERL_DEBUG( "run_command: Command $cmd returned $result.\n");

    if ($? != 0)
    {
        EMD_PERL_ERROR("Failed to run command ".$cmd." in run_command(). Return value - $?.\n");
        return;
    }

    my $err= has_errcode($result);
    if ($err != 0)
    {
        EMD_PERL_ERROR("run_command(): error codes found in the command result : $result \n");
        return;
    }

    return $result;
}

# Purpose: find following error codes in input string:
#          "PRKP"
#          "PRKH"
#          "PRKO"
#          "CRS"
#          "ORA"
sub has_errcode {
    my $in_str = $_[0];

    if ($in_str =~ /PRKP-/ || $in_str =~ /PRKH-/ || $in_str =~ /PRKO-/
        || $in_str =~ /CRS-/ || $in_str =~ /ORA-/) {
        return(-1);  # error mesg
   } else {
        return(0);
    }
}

sub WARN_handler {
        my ($msg) = @_;
        EMD_PERL_DEBUG("DEBUG.......".$msg);
        print "em_warning=".$msg;
}

sub DIE_handler {
        my ($msg) = @_;
        EMD_PERL_ERROR($msg);
        print "em_error=".$msg;
}




