#!/usr/local/bin/perl
# 
# $Header: emdb/sysman/admin/scripts/segAdv.pl /st_emgc_pt-12.1.0.4pg/2 2012/05/25 06:49:18 mpawelko Exp $
#
# segAdv.pl
# 
# Copyright (c) 2005, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      segAdv.pl - Get the Segment Advisor Recommendation count
#
#    DESCRIPTION
#       This script retrieves the timestamp when the last Segment Advisor
#       job ran [per PDB]. It compares this timestamp with the timestamp written
#       to the agent state directory file. If a Segment Advisor job was run since the 
#       last time this script (metric) ran, this script retrieves the latest count of
#       Segment Advisor recommendations and updates the agent state directory file with 
#       the new timestamp and new count(s) [per PDB]. If no Segment Advisor jobs have
#       run since the last time this script (metric) ran, this script returns the previous count(s)
#       [per PDB].  This is implemented this way because the PL/SQL dbms_space.asa_recommendations
#       call is costly (slow).
#
#    OUTPUT:
#      This script generates this output:
#      1) Segment Advisor Recommendation Count - the count of Segment Advisor recommendations
#         [per PDB, if the target is a CDB-enabled database].
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    mpawelko    05/14/12 - CDB: use v$containers (exclude PDB$SEED)
#    mpawelko    03/07/12 - 13794652: if CDB, return counts per PDB
#    kganapat    12/03/07 - XbranchMerge kganapat_filter_ora_errs from main
#    mnihalan    02/21/07 - Backport mnihalan_bug-5573975 from main
#    mnihalan    12/14/06 - Fix bug 5573975
#    mnihalan    03/17/05 - mnihalan_segment_adv_workflow_change
#    mnihalan    03/07/05 - Creation
# 

use strict;
use DBI;

require "emd_common.pl";

if (EMAGENT_isPerlDebugEnabled())
{
  EMD_PERL_DEBUG ("segAdv:Debug --> ++++  Welcome to Segment Advisor Recommendation Debugging+
      ++++ ");
}

# --------------------------------------------------------------------
# +++ VARIABLES
# --------------------------------------------------------------------

# GENERAL
my @fetch_row;
my $seg_adv_cur;
my $seg_adv_sql;
my $timestamp_last_run="";
my $current_time="";
my $seg_adv_last_run_cur;
my $seg_adv_last_run_sql;
my $lda;
# CDB support
my $pdb_name="";
my $set_container_sql;
my $set_container_cur;
my @all_con_ids;
my @all_pdbs;
my @all_last_runs;
my @all_counts;
my %pdb_last_count_hash = ();

# OUTPUT
my $seg_adv_count = 0;

# INPUT
# The DB Connection info is passed in by the metric definition

my %stdinArgs = get_stdinvars();
my $username = $stdinArgs{"EM_TARGET_USERNAME"};
my $password = $stdinArgs{"EM_TARGET_PASSWORD"};
my $address = $ENV{EM_TARGET_ADDRESS};
my $role = $ENV{EM_TARGET_ROLE};
my $targetGuid = $ENV{EM_TARGET_GUID};
my $oracle_home = $ENV{EM_TARGET_ORACLE_HOME};
my $state_root = $ENV{EM_AGENT_STATE_DIR};
my $mode = 0;
if($role =~ /SYSDBA/i)
{
  $mode = 2;
}
elsif($role =~ /SYSOPER/i)
{
  $mode = 4;
}

my $cdb = $ENV{CDB};

my $separator = $^O =~ m/MSWin32/ ? "\\" : "\/";

#
# Location of the offsets state file is
#  $EMDROOT/sysman/emd/state/<TARGET_GUID>.<filename>
#
my $scannedFile = $state_root.$separator."sysman".$separator."emd".$separator."state".$separator."$targetGuid"."_seg_adv_count.log";



# ------------------------------------------------------------------
# Open scanned file for reading (if scanned file does not exist, create it.)
# ------------------------------------------------------------------
if (!open(SCANNED, "+>> $scannedFile"))
{
    EMAGENT_PERL_ERROR("target: $targetGuid; Cannot open $scannedFile for read/write.");
    exit 1;
}

# seek to top of file

seek(SCANNED, 0, 0);

# If not CDB, the state file contains a single line containing the timestamp and count.
# If CDB, the state file contains the timestamp on 1 line, followed by n lines, 
# each containing the container id and count for a particular PDB.
if ($cdb =~ /YES/i)
{
  my $first = 1;
  while (my $text = <SCANNED>)
  {
    chomp($text);
    if ($first eq 1)
    {
     $timestamp_last_run = $text;
     $first = 0;
    }
    else
    {  
      my @pos = split('~', $text);
      my $pos = \@pos;
      $pdb_last_count_hash{ $pos->[0] } = $pos->[1];
    }
  }
}
else
{
# file only contains a single line with two numbers:  a timestamp and a count
# get these and close the scanned file

  while (<SCANNED>)
  {
    my @pos = split('~', $_);
    my $pos = \@pos;
    
    $timestamp_last_run =$pos->[0];
    $seg_adv_count = $pos->[1];
  }
}

close(SCANNED);

if (EMAGENT_isPerlDebugEnabled())
{
    EMD_PERL_DEBUG ("segAdv:Debug  timestamp last run  : $timestamp_last_run");
    if ($cdb =~ /YES/i)
    {
      while ( my ($key, $value) = each(%pdb_last_count_hash) ) {
	EMD_PERL_DEBUG ("segAdv(CDB):Debug container_id => count: $key => $value ");
      }
    }
    else
    {
      EMD_PERL_DEBUG ("segAdv:Debug  Seg Adv Count  : $seg_adv_count");
    }
}
# --------------------------------------------------------------------
# +++ Establish Target DB Connection
# --------------------------------------------------------------------

    EMD_PERL_DEBUG ("segAdv:Debug DB Address : $address"); 
$lda = DBI->connect('dbi:Oracle:', "$username@".$address, "$password",
    {ora_session_mode => $mode, PrintError => 0, RaiseError => 0})
    or die (filterOraError("em_error=Could not connect to $username/$address: $DBI::errstr\n", $DBI::err));
register_metric_call($lda);

# --------------------------------------------------------------------
# +++ Segment Advisor Recommendation Count
# --------------------------------------------------------------------
# If CDB, we must query CDB_* views to get data for all containers.
if ($cdb =~ /YES/i)
{
  if ($timestamp_last_run eq "")
  {
      $seg_adv_last_run_sql = "SELECT p.con_id, p.name, TO_CHAR(MAX(CAST(l.execution_end AS TIMESTAMP)), ".
                  "'DD-MON-YYYY HH24:MI:SSxFF'), ".
                  "TO_CHAR(CAST(sysdate AS TIMESTAMP), 'DD-MON-YYYY HH24:MI:SSxFF') ".
                  "FROM (SELECT con_id, name FROM v\$containers where con_id != 2) p ".
                  "LEFT OUTER JOIN cdb_advisor_tasks l ON ".
                  "l.advisor_name = 'Segment Advisor' ".
                  "AND p.con_id = l.con_id GROUP BY p.con_id, p.name";
  }
  else
  {
      $seg_adv_last_run_sql = "SELECT p.con_id, p.name, TO_CHAR(MAX(CAST(l.execution_end AS TIMESTAMP)), ".
                  "'DD-MON-YYYY HH24:MI:SSxFF'), ".
                  "TO_CHAR(CAST(sysdate AS TIMESTAMP), 'DD-MON-YYYY HH24:MI:SSxFF') ".
                  "FROM (SELECT con_id, name FROM v\$containers where con_id != 2) p ".
                  "LEFT OUTER JOIN cdb_advisor_tasks l ON ".
                  "l.advisor_name = 'Segment Advisor' ".
                  "AND TO_CHAR(CAST(l.execution_end AS TIMESTAMP), 'DD-MON-YYYY HH24:MI:SSxFF') > '". $timestamp_last_run. "' ".
                  "AND p.con_id = l.con_id GROUP BY p.con_id, p.name";
  }
}
else
{
  if ($timestamp_last_run eq "")
  {
      $seg_adv_last_run_sql = "SELECT TO_CHAR(MAX(CAST(l.execution_end AS TIMESTAMP)), 'DD-MON-YYYY HH24:MI:SSxFF'), 
TO_CHAR(CAST(sysdate AS TIMESTAMP), 'DD-MON-YYYY HH24:MI:SSxFF') ".
                  " from dba_advisor_tasks l where ".
                  " l.advisor_name='Segment Advisor'";
  }
  else
  {
      $seg_adv_last_run_sql = "SELECT TO_CHAR(MAX(CAST(l.execution_end AS TIMESTAMP)), 'DD-MON-YYYY HH24:MI:SSxFF'),
TO_CHAR(CAST(sysdate AS TIMESTAMP), 'DD-MON-YYYY HH24:MI:SSxFF') ".
		  " from dba_advisor_tasks l where ".
                  " l.advisor_name='Segment Advisor' and ".
                  " TO_CHAR(CAST(l.execution_end AS TIMESTAMP), 'DD-MON-YYYY HH24:MI:SSxFF') > '". $timestamp_last_run. "'";    
  }
}
if (EMAGENT_isPerlDebugEnabled())
{
  EMD_PERL_DEBUG ("segAdv:Debug  ***** Seg Adv Last Run SQL  : $seg_adv_last_run_sql");
}

$seg_adv_last_run_cur = $lda->prepare($seg_adv_last_run_sql)
    or die (filterOraError("em_error=prepare($seg_adv_last_run_sql): $DBI::errstr\n", $DBI::err));
$seg_adv_last_run_cur->execute()
    or die (filterOraError("em_error=seg_adv_last_run_cur->execute(): $DBI::errstr\n", $DBI::err));

# If CDB, we store the last time a Segment Advisor job ran per PDB.
if ($cdb =~ /YES/i)
{
  while ( @fetch_row = $seg_adv_last_run_cur->fetchrow_array() ) 
  {
    push (@all_con_ids, $fetch_row[0]);
    push (@all_pdbs, $fetch_row[1]);
    push (@all_last_runs, $fetch_row[2]);
# Current time should be the same for all rows, so just choose last.
    $current_time = $fetch_row[3];
  }
}
else
{
  @fetch_row = $seg_adv_last_run_cur->fetchrow_array();
   
  $timestamp_last_run = $fetch_row[0];
  $current_time = $fetch_row[1];
}
 
# --------------------------------------------------------------------
# +++ Segment Advisor Recommendation Count
# --------------------------------------------------------------------
if ($cdb =~ /YES/i)
{
  $seg_adv_sql = "SELECT count(*) ".
                   "FROM table(dbms_space.asa_recommendations('TRUE', 'TRUE', 'FALSE'))";

  for ( my $i = 0; $i <= $#all_last_runs; $i++ )
  {
    if ( $all_last_runs[$i] ne "")
    {
      if (EMAGENT_isPerlDebugEnabled())
      {
        EMD_PERL_DEBUG ("segAdv(CDB):Debug  Retrieving Segment Advisor Recommendations for $all_pdbs[$i]");
      }
      $set_container_sql = "ALTER SESSION SET CONTAINER=". $all_pdbs[$i];
    
      $set_container_cur = $lda->prepare($set_container_sql)
          or die (filterOraError("em_error=prepare($set_container_sql): $DBI::errstr\n", $DBI::err));
      $set_container_cur->execute()
          or die (filterOraError("em_error=set_container_cur->execute(): $DBI::errstr\n", $DBI::err));

      $seg_adv_cur = $lda->prepare($seg_adv_sql)
          or die (filterOraError("em_error=prepare($seg_adv_sql): $DBI::errstr\n", $DBI::err));
      $seg_adv_cur->execute()
          or die (filterOraError("em_error=seg_adv_cur->execute(): $DBI::errstr\n", $DBI::err));
    
      @fetch_row = $seg_adv_cur->fetchrow_array();
      push (@all_counts, $fetch_row[0]);
      $set_container_cur->finish();
      $seg_adv_cur->finish();
    }
    else
    {
      # If no new Segment Advisor runs exist for this PDB since the last time this metric ran,
      # return the previous count if it exists, or 0 if this is the first time we've seen this
      # PDB. [PDBs can be plugged and unplugged, so they may come and go.]
      if (exists $pdb_last_count_hash{ $all_con_ids[$i] })
      {
	push (@all_counts, $pdb_last_count_hash{ $all_con_ids[$i] });
      }
      else
      {
	push (@all_counts, 0);
      }
    }
  }
}
else
{
  if ($timestamp_last_run ne "")
  {
    if (EMAGENT_isPerlDebugEnabled())
    {
        EMD_PERL_DEBUG ("segAdv:Debug  Retrieving Segment Advisor Recommendations ");
    }
    $seg_adv_sql = "SELECT count(*) ".
                       "FROM table(dbms_space.asa_recommendations('TRUE', 'TRUE', 'FALSE'))";
    
    $seg_adv_cur = $lda->prepare($seg_adv_sql)
        or die (filterOraError("em_error=prepare($seg_adv_sql): $DBI::errstr\n", $DBI::err));
    $seg_adv_cur->execute()
        or die (filterOraError("em_error=seg_adv_cur->execute(): $DBI::errstr\n", $DBI::err));
    
    @fetch_row = $seg_adv_cur->fetchrow_array();
    $seg_adv_count = $fetch_row[0];
  }
}

# --------------------------------------------------------------------
# Reopen scanned file and write out the new timestamp when this
# script ran and the new recommendation count(s) [per PDB].
# --------------------------------------------------------------------
open (SCANNED, "+> $scannedFile") or die "em_error=Cannot open $scannedFile";
if ($cdb =~ /YES/i)
{
  print SCANNED "$current_time\n";
  for ( my $i = 0; $i <= $#all_con_ids; $i++ )
  {
    print SCANNED "$all_con_ids[$i]~$all_counts[$i]\n";
  }
}
else
{
  print SCANNED "$current_time~$seg_adv_count";
}
close(SCANNED);

# --------------------------------------------------------------------
# +++ Print Results
#
# This returns the results to the agent, which will look for a
# standard output line that starts with em_result. The '|' character
# is the delimiter between values.
# --------------------------------------------------------------------
if ($cdb =~ /YES/i)
{
  for ( my $i = 0; $i <= $#all_pdbs; $i++ )
  {
    print "em_result=$all_pdbs[$i]|$all_counts[$i]\n";
  }
}
else
{ 
  print "em_result=$seg_adv_count\n";
}

# --------------------------------------------------------------------
# +++ Disconnect from the Target DB
# --------------------------------------------------------------------

$lda->disconnect
    or warn "disconnect $DBI::errstr\n";

exit 0;

