# 
# $Header: emas/sysman/admin/scripts/ias.10g/Apache_confFileParser.pm /main/2 2010/10/03 18:52:31 jsmoler Exp $
#
# Apache_confFileParser.pm
# 
# Copyright (c) 2004, 2010, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      Apache_confFileParser.pm - Apache .conf file parser
#
#    DESCRIPTION
#      Parses an Apache .conf file.
#
#      Entry points:
#       parseConfFile
#       peek
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    jsmoler     08/05/10 - inheritance independent of directive order
#    jsmoler     09/18/09 - fix syntax for Perl 5.10
#    jsmoler     04/23/08 - Backport jsmoler_bug-6319688 from main
#    jsmoler     07/18/07 - Backport jsmoler_bug-5736476_test_070712 from
#                           st_emasgc_10.2.0.1.0
#    jsmoler     12/28/06 - handle absolute and relative paths (bug 5736476)
#    jsmoler     08/07/06 - Backport jsmoler_bug-5224028 from main
#    jsmoler     06/26/06 - don't die on all errors 
#    jsmoler     08/07/06 - Backport jsmoler_bug-5295826 from
#                           st_emasgc_10.2.0.1.0
#    jsmoler     04/06/06 - fix bug 5143034
#    jsmoler     08/02/06 - Backport jsmoler_bug-5143034 from main
#    jsmoler     08/02/06 - Backport jsmoler_bug-5046383 from main
#    jsmoler     11/01/05 - XbranchMerge jsmoler_bug-4613189 from main 
#    jsmoler     03/06/06 - add package statement 
#    jsmoler     09/29/05 - change module handling
#    jsmoler     02/28/05 - inherit simple directives
#    jsmoler     02/10/05 - handle simple directive with no arguments
#    jsmoler     10/22/04 - move "sub contains" to asecm 
#    jsmoler     10/07/04 - some support for default values
#    jsmoler     10/04/04 - jsmoler_ecm01
#    jsmoler     08/18/04 - Creation
# 

#
# Custom hash for storing Apache directives.
#
# Apache directive names are case insensitive. Since they are stored in a hash by
# the parser, use a tied hash that converts all keys to lower case.
#
# Also, directives may be inherited from a parent scope to a child scope (a virtual
# host will inherit directives from the server scope).
#
{
    package ias::Apache_confFileParser::DirectiveHash;

    sub TIEHASH
    {
        my ( $className, $parent ) = @_;
        my $node = { local => {}, parent => $parent };
        return bless $node, $className;
    }
    sub FETCH
    {
        my ( $self, $key ) = @_;
        my $key = lc($key);
        if (defined($self->{parent}) && exists($self->{parent}{$key}))
        {
            my $parentVal = $self->{parent}{$key};
            if (exists($self->{local}{$key}))
            {
                my $localVal = $self->{local}{$key};
                my @combinedVal;
                push(@combinedVal, @{$parentVal});
                push(@combinedVal, @{$localVal});
                return \@combinedVal;
            }
            return $parentVal;
        }
        return $self->{local}{$key};
    }
    sub STORE
    {
        my ( $self, $key, $val ) = @_;
        $self->{local}{lc($key)} = $val;
    }
    sub EXISTS
    {
        my ( $self, $key ) = @_;
        return exists($self->{local}{lc($key)});
    }
    sub DELETE
    {
        my ( $self, $key ) = @_;
        return delete $self->{local}{lc($key)};
    }
    sub CLEAR
    {
        my ( $self ) = @_;
        $self->{local} = {};
    }
    sub SCALAR
    {
        my ( $self ) = @_;
        my $count = 0;
        my $key = FIRSTKEY($self);
        while (defined($key))
        {
            $count++;
            $key = NEXTKEY($self);
        }
        return $count;
    }
    sub FIRSTKEY
    {
        my ( $self ) = @_;
        $self->{localIter} = 1;
        keys %{$self->{local}}; # reset hash iterator
        return NEXTKEY($self);
    }
    sub NEXTKEY
    {
        my ( $self ) = @_;
        if ($self->{localIter})
        {
            my $key = each %{$self->{local}};
            if (defined($key))
            {
                return $key;
            }
            $self->{localIter} = 0;
            if (defined($self->{parent}))
            {
                keys %{$self->{parent}}; # reset hash iterator
            }
        }
        if (defined($self->{parent}))
        {
            my $key = each %{$self->{parent}};
            while (defined{$key} && exists($self->{local}{$key}))
            {
                $key = each %{$self->{parent}};
            }
            return $key;
        }
        return undef;
    }
    sub DESTROY {}
}

BEGIN
{
    require "emd_common.pl";
}

package ias::Apache_confFileParser;

use strict;
use Exporter 'import';
use File::Spec;
use File::Glob ':glob';

our @EXPORT = qw(
    parseConfFile
    peek
);


#
# Parses an Apache .conf file. Returns a data structure modeling the contents
# of the file.
#
# The data structure returned is a hash, mapping directive names to references to
# subordinate data structures. Let us refer to this hash as a type A data structure.
# The form of each subordinate data structure depends on the type of the associated
# directive. These may be:
#   1) A non-nesting directive with or without an argument. The resulting data structure
#      is a list of the values of all arguments to the directive, in the order
#      they appear in the configuration file. If the directive appears without
#      an argument, it is treated as thought it has an argument ''.
#   2) A nesting directive with an argument. The resulting data structure is a
#      hash mapping each argument value (argument values are assumed to be unqiue)
#      to a reference to a subordinate type A data structure. This subordinate
#      type A data structure represents the nested contents of the directive.
#   3) A nesting directive with no argument. The resulting data structure is a
#      list of references to subordinate type A data structures. Each subordinate
#      type A data structure represents the nested contents of a directive.
# Values of non-nesting directives are inherited by nested directives.
#
# Arguments:
#   0: The name of the configuration file to parse.
#   1: Optional: A reference to a list of defined parameters.
#   2: Optional: A reference to a list of compiled-in modules. Each item should be
#      in the form "<module-name>.c".
#   3: Optional: Reference to a list that will get filled in with names of all 
#      included files.
#   4: Optional: Default server root directory
#   5: Optional: Function that reads the contents of a file based on its path,
#      and returns a reference to a list of lines. If not specified, a default
#      will be used that reads a file from the local file system.
#   6: Optional: Function that lists file paths which match a file name expression
#      and a context path. Result paths will be passed to the previous argument 5.
#
# Returns: A data structure modeling the configuration file, as described above.
#
sub parseConfFile
{
    my $confFileName = $_[0];
    my $definedParamsRef = $_[1];
    my $compiledModulesRef = $_[2];
    my $fileNamesRef = $_[3];
    my $defaultServerRoot = $_[4];
    my $readFileSub = $_[5];
    my $resolveFilesSub = $_[6];
    
    my %modules;
    for my $mod (@{$compiledModulesRef})
    {
        if ($mod)
        {
            $modules{$mod} = 1;
        }
    }
    
    my %defines;
    for my $def (@{$definedParamsRef})
    {
        if ($def)
        {
            $defines{$def} = 1;
        }
    }

    if (!defined($readFileSub))
    {
        $readFileSub = sub
        {
            my ( $fileName ) = @_;
            
            if ( ! -e $fileName)
            {
                main::EMD_PERL_ERROR("File does not exist: '$fileName'");
                return undef;
            }
            my $FILE;
            open($FILE, $fileName);
            my @lines = readline($FILE);
            close($FILE);
            return \@lines;
        }
    }
    if (!defined($resolveFilesSub))
    {
        $resolveFilesSub = sub
        {
            my ( $fileName, $contextPath ) = @_;
            
            if (defined($contextPath))
            {
                $fileName = File::Spec->rel2abs($fileName, $contextPath);
            }
            return glob($fileName);
        }
    }

    my %rootHash;
    tie %rootHash, 'ias::Apache_confFileParser::DirectiveHash';
    my @context;
    my %simpleHash;
    tie %simpleHash, 'ias::Apache_confFileParser::DirectiveHash';
    
    my @fileNames = &$resolveFilesSub($confFileName, $defaultServerRoot);
    for my $fileName (@fileNames)
    {
        parseConfFileImpl($fileName,
                          $fileNamesRef,
                          \%modules,
                          \%defines,
                          \%rootHash,
                          \%simpleHash,
                          \@context,
                          [],
                          $defaultServerRoot,
                          $readFileSub,
                          $resolveFilesSub);
    }

    return \%rootHash;
}

sub parseConfFileImpl
{
    my $confFileName = $_[0];
    my $fileNamesRef = $_[1];
    my $modulesRef = $_[2];
    my $definesRef = $_[3];
    my $hashRef = $_[4];
    my $simpleHashRef = $_[5];
    my $contextRef = $_[6];
    my $simpleContextRef = $_[7];
    my $defaultServerRoot = $_[8];
    my $readFileSub = $_[9];
    my $resolveFilesSub = $_[10];
    
    if (defined($fileNamesRef))
    {
        push(@{$fileNamesRef}, $confFileName);
    }

    my $lines = &$readFileSub($confFileName);
    if (!defined($lines))
    {
        return;
    }

    while (1)
    {
        my $line = nextLine($lines);
        if (length($line) == 0)
        {
            last;
        }
        if ($line =~ /^<\/(.*)>$/)
        {
            # directive is an end tag
            
            my $directive = lc($1);
            if ($directive ne 'ifmodule' && $directive ne 'ifdefine')
            {
                # Return to the context of the parent directive
                $hashRef = pop(@{$contextRef});
                $simpleHashRef = pop(@{$simpleContextRef});
            }
        }
        elsif ($line =~ /^<(.*)>$/)
        {
            # directive is a start tag
        
            my $directive = $1;
            my $argument;

            if ($directive =~ /^(\S*)\s+(.*)$/)
            {
                # directive is a start tag with arguments
                $directive = $1;
                $argument = $2;
            }

            $directive = lc($directive);
            
            if ($directive eq 'ifmodule')
            {
                my $neg = 0;
                if ($argument =~ /^!(.*)$/)
                {
                    $argument = $1;
                    $neg = 1;
                }
                my $shouldSkip = !$$modulesRef{$argument};
                if ($neg)
                {
                    $shouldSkip = !$shouldSkip;
                }
                if ($shouldSkip)
                {
                    skipToEndTag($lines, 'ifmodule');
                }
            }
            elsif ($directive eq 'ifdefine')
            {
                my $neg = 0;
                if ($argument =~ /^!(.*)$/)
                {
                    $argument = $1;
                    $neg = 1;
                }
                my $shouldSkip = !$$definesRef{$argument};
                if ($neg)
                {
                    $shouldSkip = !$shouldSkip;
                }
                if ($shouldSkip)
                {
                    skipToEndTag($lines, 'ifdefine');
                }
            }
            else
            {
                # The current context is either the top level of the file,
                # or some nesting directive. Call this the base context.
                # We are entering a new nesting directive, so we need to create
                # a new context.
                # Context is represented by two hash variables:
                # hashRef, which contains all directives, and simpleHashRef,
                # which contains only non-nesting directives to be inherited
                # by any child nesting directives.
                # We must create a new hashRef and simpleHashRef for the new
                # context. Both inherit from simpleHashRef in base context.
                
                # Create new hashes that inherit from the existing simple hash.
                my %newSimpleHash;
                tie %newSimpleHash, 'ias::Apache_confFileParser::DirectiveHash', $simpleHashRef;
                my %newHash;
                tie %newHash, 'ias::Apache_confFileParser::DirectiveHash', $simpleHashRef;

                # Store a reference to the new hash in the base hash.
                if (defined($argument))
                {
                    my $argHash = $$hashRef{$directive};
                    if ($argHash)
                    {
                        $$argHash{$argument} = \%newHash;
                    }
                    else
                    {
                        $argHash = { $argument => \%newHash };
                        $$hashRef{$directive} = $argHash;
                    }
                }
                else
                {
                    my $listRef = $$hashRef{$directive};
                    if ($listRef)
                    {
                        push(@{$listRef}, \%newHash);
                    }
                    else
                    {
                        $$hashRef{$directive} = [ \%newHash ];
                    }
                }
                
                # Save the base context for when we need to exit the new context.
                push(@{$contextRef}, $hashRef);
                push(@{$simpleContextRef}, $simpleHashRef);

                # Update hash references for the new context.
                $hashRef = \%newHash;
                $simpleHashRef = \%newSimpleHash;
            }
        }
        elsif ($line =~ /^(\S+)\s+(.*)$/)
        {
            # directive is a simple directive with arguments
            
            my $directive = lc($1);
            my $argument = $2;

            # strip quotes from argument
            if ($argument =~ /^"(.*)"$/)
            {
                $argument = $1;
            }

            if ($directive eq 'loadmodule')
            {
                if ($argument =~ /^(\S+)_module\s/)
                {
                    my $moduleName = "mod_$1.c";
                    $$modulesRef{$moduleName} = 1;
                }
                else
                {
                    main::EMD_PERL_ERROR("Cannot parse line: '$line'");
                }
            }
            elsif ($directive eq 'addmodule')
            {
                $$modulesRef{$argument} = 1;
            }
            elsif ($directive eq 'include')
            {
                # Look up the most recent ServerRoot directive, if any
                my $serverRoot = peek($$hashRef{'ServerRoot'});
                if (!defined($serverRoot))
                {
                    $serverRoot = $defaultServerRoot;
                }
                my @fileNames = &$resolveFilesSub($argument, $serverRoot);
                for my $fileName (@fileNames)
                {
                    parseConfFileImpl($fileName,
                                      $fileNamesRef,
                                      $modulesRef,
                                      $definesRef,
                                      $hashRef,
                                      $simpleHashRef,
                                      $contextRef,
                                      $simpleContextRef,
                                      $defaultServerRoot,
                                      $readFileSub,
                                      $resolveFilesSub);
                }
            }
            else
            {
                my @list;
                if (exists($$hashRef{$directive}))
                {
                    my $oldListRef = $$hashRef{$directive};
                    push(@list, @{$oldListRef});
                }
                push(@list, $argument);
                $$hashRef{$directive} = \@list;
                $$simpleHashRef{$directive} = \@list;
            }
        }
        elsif ($line =~ /^(\S+)$/)
        {
            # directive is a simple directive with no arguments
            my $directive = lc($1);

            if ($directive eq 'clearmodulelist')
            {
                %{$modulesRef} = ();
            }
            else
            {
                my @list;
                if (exists($$hashRef{$directive}))
                {
                    my $oldListRef = $$hashRef{$directive};
                    push(@list, @{$oldListRef});
                }
                push(@list, '');
                $$hashRef{$directive} = \@list;
                $$simpleHashRef{$directive} = \@list;
            }
        }
        else
        {
            # unknown case
            main::EMD_PERL_ERROR("Cannot parse line: '$line'");
        }
    }
}

sub nextLine
{
    my $linesRef = $_[0];
    my $line = '';
    while (@{$linesRef} > 0)
    {
        $line = shift(@{$linesRef});

        # Skip comments
        if ($line =~ /^\s*\#.*$/)
        {
            $line = '';
            next;
        }
        # Trim whitespace
        if ($line =~ /^\s*(.*?)\s*$/)
        {
            $line = $1;
        }
        # Skip blank lines
        if (length($line) == 0)
        {
            next;
        }
        last;
    }
    return $line;
}

sub skipToEndTag
{
    my $lines = $_[0];
    my $tagName = $_[1];
    my $depth = 1;
    my $re = qr/<$tagName\s.*>/;
    while ($depth > 0 && @{$lines} > 0)
    {
        my $line = lc(nextLine($lines));
        if ($line =~ $re)
        {
            $depth = $depth + 1;
        }
        if ($line eq "</$tagName>")
        {
            $depth = $depth - 1;
        }
    }
    if ($depth > 0)
    {
        main::EMD_PERL_ERROR("End tag not found: '$tagName'");
    }
}

#
# Returns the last item in an array.
#
# Arguments
#   0: A reference to an array, or undef.
#
# Returns: The last item in the array, or the default value if the array reference is not defined.
#
sub peek
{
    my $ref = $_[0];
    if (!defined($ref))
    {
        return undef;
    }
    return $$ref[-1];
}

return 1;
