#!/usr/local/bin/perl
# 
# $Header: emas/sysman/admin/scripts/ias.10g/OC4J_getDataSources.pl /main/1 2009/10/12 04:39:07 djoly Exp $
#
# OC4J_getDataSources.pl
# 
# Copyright (c) 2004, 2009, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      OC4J_getDataSources.pl - OC4J ECM integration: get data sources
#
#    DESCRIPTION
#      Command-line utility called by the Agent to retrieve a list of data
#      sources for an OC4J instances, and configuration information about each
#      data source.
#
#      Implements the DataSources metric defined in oc4j.xml.
#
#      Command-line arguments: <oracle home> <instance name> <default app only>
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    jsmoler     08/07/06 - Backport jsmoler_bug-5295826 from
#                           st_emasgc_10.2.0.1.0
#    jsmoler     06/27/06 - use new output functions 
#    jsmoler     01/16/06 - add flag to only collect data sources of default 
#                           app, for backward compatibility 
#    jsmoler     12/09/05 - Backport jsmoler_bug-4769271 from main 
#    jsmoler     11/29/05 - dynamically determine path of XML file,
#                           get data sources for all applications
#                           (bug 4769271) 
#    jsmoler     12/07/05 - Backport jsmoler_bug-4765772 from main 
#    jsmoler     11/28/05 - fix inactivity-timeout typo (bug 4765772) 
#    jsmoler     11/12/04 - add JDBC URL
#    jsmoler     10/04/04 - jsmoler_ecm01
#    jsmoler     08/26/04 - Creation
# 

use strict;
use ias::simpleXPath;
use ias::asecm;

my $oracleHome = $ARGV[0];
setOracleHome($oracleHome);
my $instanceName = $ARGV[1];

my $defaultAppOnly = defined($ARGV[2]) && $ARGV[2] eq 'default-app-only';

if ($defaultAppOnly)
{
    # Metric: DataSources 10.2.0.0.0
    setMetricFormat(
        { 'type' => 'string', 'key' => 1 }, # name
        # no application column -- default app only
        { 'type' => 'string', 'key' => 1 }, # location
        { 'type' => 'string' },             # class
        { 'type' => 'string' },             # driver
        { 'type' => 'string' },             # URL
        { 'type' => 'number' },             # max connections
        { 'type' => 'number' },             # min connections
        { 'type' => 'number' },             # wait timeout
        { 'type' => 'number' },             # inactivity timeout
    );
}
else
{
    # Metric: DataSources 10.2.0.2.0
    setMetricFormat(
        { 'type' => 'string', 'key' => 1 }, # name
        { 'type' => 'string', 'key' => 1 }, # application
        { 'type' => 'string', 'key' => 1 }, # location
        { 'type' => 'string' },             # class
        { 'type' => 'string' },             # driver
        { 'type' => 'string' },             # URL
        { 'type' => 'number' },             # max connections
        { 'type' => 'number' },             # min connections
        { 'type' => 'number' },             # wait timeout
        { 'type' => 'number' },             # inactivity timeout
    );
}

my @dataSourceFiles = listOC4JDataSourceFiles($instanceName);
for my $appInfo (@dataSourceFiles)
{
    if ($defaultAppOnly && $$appInfo{'application'} ne 'default')
    {
        next;
    }
    my @dataSources = simpleXPathQuery(
        $$appInfo{'file'},
        'data-sources/data-source'
    );
    my %jndiLocHash;
    for my $dataSource (@dataSources)
    {
        $jndiLocHash{$$dataSource{'location'}} = $dataSource;
    }
    for my $jndiLoc (keys %jndiLocHash)
    {
        my $dataSource = $jndiLocHash{$jndiLoc};
        if ($defaultAppOnly)
        {
            printMetricResult(
                $$dataSource{'name'},
                # no application column -- default app only
                $$dataSource{'location'},
                $$dataSource{'class'},
                $$dataSource{'connection-driver'},
                $$dataSource{'url'},
                $$dataSource{'max-connections'},
                $$dataSource{'min-connections'},
                $$dataSource{'wait-timeout'},
                $$dataSource{'inactivity-timeout'},
            );
        }
        else
        {
            printMetricResult(
                $$dataSource{'name'},
                $$appInfo{'application'},
                $$dataSource{'location'},
                $$dataSource{'class'},
                $$dataSource{'connection-driver'},
                $$dataSource{'url'},
                $$dataSource{'max-connections'},
                $$dataSource{'min-connections'},
                $$dataSource{'wait-timeout'},
                $$dataSource{'inactivity-timeout'},
            );
        }
    }
}
