# 
# $Header: emas/sysman/admin/scripts/ias.10g/weblogic_j2eeserver/ServerConfigUtil.pm /main/3 2011/11/14 03:37:48 gisivaku Exp $
#
# ServerConfigUtil.pm
# 
# Copyright (c) 2004, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      ServerConfigUtil.pm - Server configuration utilities
#
#    DESCRIPTION
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    gisivaku    10/28/11 - Added check for OS while getting the hash of mounts
#                           in the getMounts subroutine as the current logic is
#                           only applicable to Linux.
#    gisivaku    10/26/11 - for refresh to EMGC_MAIN_LINUX.X64_111025
#    gisivaku    10/26/11 - Added new subroutine getMounts to retrieve a hash
#                           of mount points or volumes vs the mount source or
#                           device to fix bug 13070354.
#    gisivaku    09/14/11 - XbranchMerge
#                           gisivaku_new_ocm_collections_for_10.3.7 from main
#    gisivaku    09/06/11 - Refactored part of subroutine
#                           getRefServersComponentTargets to create new
#                           reusable subroutines isTargetted and
#                           isIntersectionPresent
#    gisivaku    09/01/11 - Refactored subroutine getServerComponentNames by
#                           spliting into getServerComponentTargets,
#                           getServersComponentTargets, getServerCluster and
#                           getRefServersComponentTargets for retrieving the
#                           components and deployed targets from a set of
#                           multiple targets.
#    jsmoler     04/06/10 - fix cluster name query
#    jsmoler     11/09/08 - Creation
# 

package ias::weblogic_j2eeserver::ServerConfigUtil;

use strict;
use Exporter 'import';
use ias::simpleXPath;
use biutil;

our @EXPORT = qw(
    getServerComponentNames
    getServerComponentTargets
    getServerCluster
    isTargetted
    isIntersectionPresent
    getMounts
);

#Returns a list of the names of the components whose Xpath is specified that are 
#targetted to the specified server.
#
#Params:
# 0: Reference to the parsed XML file Node within which the specified XPath is 
#    to be looked up.
# 1: The XPath of the 'DeploymentType' component whose names is to be retrived 
#    if targeted to the specified target.
# 3: The name of the server the components targeted to which are to be retrieved.
sub getServerComponentNames
{
    return keys(%{+{ &getServerComponentTargets }});
}

#Returns a hash of the names of the components whose Xpath is specified mapped 
#to the targets that they are targetted to as csv if targetted to the specified 
#server.
#
#Params:
# 0: Reference to the parsed XML file Node within which the specified XPath is 
#    to be looked up.
# 1: The XPath of the 'DeploymentType' component whose names is to be retrived 
#    if targeted to the specified target.
# 3: The name of the server the components targeted to which are to be retrieved.
sub getServerComponentTargets
{
    my ( $configFile, $componentPath, $serverName ) = @_;
    
    my $clusterName = getServerCluster($configFile, $serverName);

    return getServersComponentTargets($configFile, $componentPath, $serverName, $clusterName);
}

#Returns a hash of the names of the components whose Xpath is specified mapped 
#to the targets that they are targetted to as csv if targetted to any of the 
#specified servers/deployment targets. This is a conveniance wrapper to 
#getRefServersComponentTargets that takes care of calling 
#getRefServersComponentTargets with a refernce to an array containing the 
#server/deployment target names following the 1st 2 arguments.
#
#Params:
# 0 : Reference to the parsed XML file Node within which the specified XPath is 
#     to be looked up.
# 1 : The XPath of the 'DeploymentType' component whose names is to be retrived 
#     if targeted to the specified target.
# 2+: The name of the servers/deployment targets the components targeted to 
#     which are to be retrieved.
sub getServersComponentTargets
{
    my @args = @_;
    my $configFile = shift @args;
    my $componentPath = shift @args;
    return getRefServersComponentTargets($configFile, $componentPath, \@args);
}

#Returns the cluster name of the cluster to which the server specified belongs 
#to if it is a part of a cluster.
#
#Params:
# 0: Reference to the parsed XML file Node within which the specified XPath is 
#    to be looked up.
# 1: The name of the server the cluster in which it is deployed is to be retrieved.
sub getServerCluster
{
    my ( $configFile, $serverName ) = @_;
    
    my $clusterName = (ias::simpleXPath::queryForText($configFile,
        'domain/server[name="' . $serverName . '"]/cluster'))[0];
        
    return $clusterName;
}

#Returns a hash of the names of the components whose Xpath is specified mapped 
#to the targets that they are targetted to as csv if targetted to any the specified 
#servers/deployment targets.
#
#Params:
# 0 : Reference to the parsed XML file Node within which the specified XPath is 
#     to be looked up.
# 1 : The XPath of the 'DeploymentType' component whose names is to be retrived 
#     if targeted to the specified target.
# 2 : A reference toa an array containing the names of the servers/deployment 
#     targets the components targeted to which are to be retrieved.
sub getRefServersComponentTargets
{
    my ( $configFile, $componentPath, $serverNamesRef ) = @_;

    my @components = ias::simpleXPath::queryForNodes($configFile, "domain/$componentPath");
    my $componentTag = (split('/', $componentPath))[-1];

    my %matchingNames = ();
    for my $component (@components)
    {
        my $name = (ias::simpleXPath::queryForText($component, "$componentTag/name"))[0];
        my $targetStr = (ias::simpleXPath::queryForText($component, "$componentTag/target"))[0];
        $matchingNames{$name} = $targetStr if (isTargetted($targetStr, $serverNamesRef));
    }
    return %matchingNames;
}

#Determines if any of the targets present in the target String csv is present in 
#the array referred by the provided reference. If one such target is present 1 
#is returned else 0 is returned.
#
#Params:
# 0: The string of comma seperated targets.
# 1: A reference to an array of strings against which the targets specified have 
#    to bo compared against.
sub isTargetted
{
    my ( $targetStr, $serverNamesRef ) = @_;
    
    if(defined($targetStr))
    {    
        my @targets = split(',', $targetStr);
        return isIntersectionPresent(\@targets, $serverNamesRef);
    }
    else
    {
        return 0;
    }
}

#Returns 1 if there is atleast one element common among the 2 arrays refferred by 
#the 2 parameters else returns 0
sub isIntersectionPresent
{
    my ( $array1Ref, $array2Ref ) = @_;
    
    if(defined($array1Ref) && defined($array2Ref) && 
       (ref($array1Ref) eq 'ARRAY') && (ref($array2Ref) eq 'ARRAY') && 
       $#{$array1Ref} >= 0 && $#{$array2Ref} >= 0)
    {
        for my $target (@{$array1Ref})
        {
            if(defined($target))
            {
                for my $server (@{$array2Ref})
                {
                    if (defined($server) && $target eq $server)
                    {
                        return 1;
                    }
                }
            }
        }
    }

    return 0;
}

# Returns a hash of mount points or volumes vs the mount source or device.
sub getMounts
{
    my %mounts;
    my $osType = biutil::get_osType;
    
    if ($osType eq "LNX")
    {
        my @mountOutput = `mount`;
        
        foreach (@mountOutput)
        {
            my @mountDetails = split ;
            $mounts{$mountDetails[2]} = $mountDetails[0];
        }
    }
    else
    {
        #Other OSes not yet supported.
    }
    
    return %mounts;
}

return 1;
