# 
# $Header: Apache_confFileParser.pm 07-aug-2006.15:16:01 jsmoler Exp $
#
# Apache_confFileParser.pm
# 
# Copyright (c) 2004, 2006, Oracle. All rights reserved.  
#
#    NAME
#      Apache_confFileParser.pm - Apache .conf file parser
#
#    DESCRIPTION
#      Parses an Apache .conf file.
#
#      Entry points:
#       parseConfFile
#       peek
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    jsmoler     08/07/06 - Backport jsmoler_bug-5224028 from main
#    jsmoler     06/26/06 - don't die on all errors 
#    jsmoler     08/07/06 - Backport jsmoler_bug-5295826 from
#                           st_emasgc_10.2.0.1.0
#    jsmoler     04/06/06 - fix bug 5143034
#    jsmoler     08/02/06 - Backport jsmoler_bug-5143034 from main
#    jsmoler     08/02/06 - Backport jsmoler_bug-5046383 from main
#    jsmoler     11/01/05 - XbranchMerge jsmoler_bug-4613189 from main 
#    jsmoler     03/06/06 - add package statement 
#    jsmoler     09/29/05 - change module handling
#    jsmoler     02/28/05 - inherit simple directives
#    jsmoler     02/10/05 - handle simple directive with no arguments
#    jsmoler     10/22/04 - move "sub contains" to asecm 
#    jsmoler     10/07/04 - some support for default values
#    jsmoler     10/04/04 - jsmoler_ecm01
#    jsmoler     08/18/04 - Creation
# 

BEGIN
{
    require "emd_common.pl";
}

package ias::Apache_confFileParser;

use strict;
use Exporter 'import';

our @EXPORT = qw(
    parseConfFile
    peek
);

#
# Parses an Apache .conf file. Returns a data structure modeling the contents
# of the file.
#
# The data structure returned is a hash, mapping directive names to references to
# subordinate data structures. Let us refer to this hash as a type A data structure.
# The form of each subordinate data structure depends on the type of the associated
# directive. These may be:
#   1) A non-nesting directive with or without an argument. The resulting data structure
#      is a list of the values of all arguments to the directive, in the order
#      they appear in the configuration file. If the directive appears without
#      an argument, it is treated as thought it has an argument ''.
#   2) A nesting directive with an argument. The resulting data structure is a
#      hash mapping each argument value (argument values are assumed to be unqiue)
#      to a reference to a subordinate type A data structure. This subordinate
#      type A data structure represents the nested contents of the directive.
#   3) A nesting directive with no argument. The resulting data structure is a
#      list of references to subordinate type A data structures. Each subordinate
#      type A data structure represents the nested contents of a directive.
# Values of non-nesting directives are inherited by nested directives.
#
# Arguments:
#   0: The name of the configuration file to parse
#   1: A reference to a list of defined parameters.
#   2: A reference to a list of compiled-in modules. Each item should be in
#      the form "<module-name>.c".
#   3: Reference to a list that will get filled in with names of all included files.
#
# Returns: A data structure modeling the configuration file, as described above.
#
sub parseConfFile
{
    my $confFileName = $_[0];
    my $definedParamsRef = $_[1];
    my $compiledModulesRef = $_[2];
    my $filesRef = $_[3];

    if (! -e $confFileName)
    {
        die "No such file: $confFileName";
    }
    
    my %modules;
    for my $mod (@{$compiledModulesRef})
    {
        if ($mod)
        {
            $modules{$mod} = 1;
        }
    }
    
    my %defines;
    for my $def (@{$definedParamsRef})
    {
        if ($def)
        {
            $defines{$def} = 1;
        }
    }
    
    my %rootHash;
    my @context;
    
    parseConfFileImpl($confFileName,
                      $filesRef,
                      \%modules,
                      \%defines,
                      \%rootHash,
                      {},
                      \@context,
                      []);

    return %rootHash;
}

sub parseConfFileImpl
{
    my $confFileName = $_[0];
    my $filesRef = $_[1];
    my $modulesRef = $_[2];
    my $definesRef = $_[3];
    my $hashRef = $_[4];
    my $simpleHashRef = $_[5];
    my $contextRef = $_[6];
    my $simpleContextRef = $_[7];
    
    if (defined($filesRef))
    {
        push(@{$filesRef}, $confFileName);
    }

    my $FILE;
    open($FILE, $confFileName);
    my @lines = readline($FILE);
    close($FILE);

    while (1)
    {
        my $line = nextLine(\@lines);
        if (length($line) == 0)
        {
            last;
        }
        if ($line =~ /^<\/(.*)>$/)
        {
            # directive is an end tag
            
            my $directive = $1;
            if ($directive ne "IfModule" && $directive ne "IfDefine")
            {
                # Return to the context of the parent directive
                $hashRef = pop(@{$contextRef});
                $simpleHashRef = pop(@{$simpleContextRef});
            }
        }
        elsif ($line =~ /^<(.*)>$/)
        {
            # directive is a start tag
        
            my $directive = $1;
            my $argument;

            if ($directive =~ /^(\S*)\s+(.*)$/)
            {
                # directive is a start tag with arguments
                $directive = $1;
                $argument = $2;
            }
            
            if ($directive eq "IfModule")
            {
                my $neg = 0;
                if ($argument =~ /^!(.*)$/)
                {
                    $argument = $1;
                    $neg = 1;
                }
                my $shouldSkip = !$$modulesRef{$argument};
                if ($neg)
                {
                    $shouldSkip = !$shouldSkip;
                }
                if ($shouldSkip)
                {
                    skipToEndTag(\@lines, "IfModule");
                }
            }
            elsif ($directive eq "IfDefine")
            {
                my $neg = 0;
                if ($argument =~ /^!(.*)$/)
                {
                    $argument = $1;
                    $neg = 1;
                }
                my $shouldSkip = !$$definesRef{$argument};
                if ($neg)
                {
                    $shouldSkip = !$shouldSkip;
                }
                if ($shouldSkip)
                {
                    skipToEndTag(\@lines, "IfDefine");
                }
            }
            else
            {
                # The current context is either the top level of the file,
                # or some nesting directive. Call this the base context.
                # We are entering a new nesting directive, so we need to create
                # a new context.
                # Context is represented by two hash variables:
                # hashRef, which contains all directives, and simpleHashRef,
                # which contains only non-nesting directives to be inherited
                # by any child nesting directives.
                # We must create a new hashRef and simpleHashRef for the new
                # context. Both should be copies of simpleHashRef from the base
                # context.
                
                # Create one copy of the base simpleHashRef, which will become
                # the new hashRef.
                my %newHash = %{$simpleHashRef};

                # Store a reference to the new hash in the base hash.
                if (defined($argument))
                {
                    my $argHash = $$hashRef{$directive};
                    if ($argHash)
                    {
                        $$argHash{$argument} = \%newHash;
                    }
                    else
                    {
                        $argHash = { $argument => \%newHash };
                        $$hashRef{$directive} = $argHash;
                    }
                }
                else
                {
                    my $listRef = $$hashRef{$directive};
                    if ($listRef)
                    {
                        push(@{$listRef}, \%newHash);
                    }
                    else
                    {
                        $$hashRef{$directive} = [ \%newHash ];
                    }
                }
                
                # Save the base context for when we need to exit the new context.
                push(@{$contextRef}, $hashRef);
                push(@{$simpleContextRef}, $simpleHashRef);

                # Update hashRef for the new context.
                $hashRef = \%newHash;
                
                # Copy the base simpleHashRef again, and update simpleHashRef
                # for the new context.
                my %newHash1 = %{$simpleHashRef};
                $simpleHashRef = \%newHash1;
            }
        }
        elsif ($line =~ /^(\S+)\s+(.*)$/)
        {
            # directive is a simple directive with arguments
            
            my $directive = $1;
            my $argument = $2;

            # strip quotes from argument
            if ($argument =~ /^"(.*)"$/)
            {
                $argument = $1;
            }

            if ($directive eq 'LoadModule')
            {
                if ($argument =~ /^(\S+)_module\s/)
                {
                    my $moduleName = "mod_$1.c";
                    $$modulesRef{$moduleName} = 1;
                }
                else
                {
                    main::EMD_PERL_ERROR("Cannot parse line: '$line'");
                }
            }
            elsif ($directive eq 'AddModule')
            {
                $$modulesRef{$argument} = 1;
            }
            elsif (lc($directive) eq 'include')
            {
                my $fileName = $argument;
                if (-e $fileName)
                {
                    parseConfFileImpl($fileName,
                                      $filesRef,
                                      $modulesRef,
                                      $definesRef,
                                      $hashRef,
                                      $simpleHashRef,
                                      $contextRef,
                                      $simpleContextRef);
                }
                else
                {
                    main::EMD_PERL_ERROR("Include file does not exist: '$fileName'");
                }
            }
            else
            {
                my @list;
                my $oldListRef = $$hashRef{$directive};
                if ($oldListRef)
                {
                    push(@list, @{$oldListRef});
                }
                push(@list, $argument);
                $$hashRef{$directive} = \@list;
                $$simpleHashRef{$directive} = \@list;
            }
        }
        elsif ($line =~ /^(\S+)$/)
        {
            # directive is a simple directive with no arguments
            my $directive = $1;

            if ($directive eq 'ClearModuleList')
            {
                %{$modulesRef} = ();
            }
            else
            {
                my @list;
                my $oldListRef = $$hashRef{$directive};
                if ($oldListRef)
                {
                    push(@list, @{$oldListRef});
                }
                push(@list, '');
                $$hashRef{$directive} = \@list;
                $$simpleHashRef{$directive} = \@list;
            }
        }
        else
        {
            # unknown case
            main::EMD_PERL_ERROR("Cannot parse line: '$line'");
        }
    }
}

sub nextLine
{
    my $linesRef = $_[0];
    my $line = "";
    while (@{$linesRef} > 0)
    {
        $line = shift(@{$linesRef});

        # Trim comments
        if ($line =~ /^(.*?)\#.*$/)
        {
            $line = $1;
        }
        # Trim whitespace
        if ($line =~ /^\s*(.*?)\s*$/)
        {
            $line = $1;
        }
        # Skip blank lines
        if (length($line) == 0)
        {
            next;
        }
        last;
    }
    return $line;
}

sub skipToEndTag
{
    my $lines = $_[0];
    my $tagName = $_[1];
    my $depth = 1;
    my $re = qr/<$tagName\s.*>/;
    while ($depth > 0 && @{$lines} > 0)
    {
        my $line = nextLine($lines);
        if ($line =~ $re)
        {
            $depth = $depth + 1;
        }
        if ($line eq "</$tagName>")
        {
            $depth = $depth - 1;
        }
    }
    if ($depth > 0)
    {
        main::EMD_PERL_ERROR("End tag not found: '$tagName'");
    }
}

#
# Returns the last item in an array.
#
# Arguments
#   0: A reference to an array, or undef.
#
# Returns: The last item in the array, or the default value if the array reference is not defined.
#
sub peek
{
    my $ref = $_[0];
    if (!defined($ref))
    {
        return undef;
    }
    return ${@$ref}[-1];
}

return 1;
