#!/usr/local/bin/perl
# 
# $Header: getListenAddresses.pl 17-apr-2008.17:23:52 jsmoler Exp $
#
# getListenAddresses.pl
# 
# Copyright (c) 2004, 2008, Oracle. All rights reserved.  
#
#    NAME
#      getListenAddresses.pl - Apache ECM integration: get listen addresses
#
#    DESCRIPTION
#      Command-line utility called by the Agent to retrieve a list of listening
#      addresses for Apache.
#
#      Implements the ListenAddresses metric defined in oracle_apache.xml.
#
#      Command-line arguments: <oracle home>
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    jsmoler     04/17/08 - convert all directive names to lower case
#    jsmoler     06/26/06 - use new output functions 
#    jsmoler     07/14/06 - use parseApacheConf (bug 5224028)
#    jsmoler     11/02/05 - rearrange perl 
#    jsmoler     05/19/05 - dynamically determine root conf file 
#    jsmoler     03/03/05 - use vhost utility
#    jsmoler     02/17/05 - change matching to vhost
#    jsmoler     01/28/05 - apache -> Apache
#    jsmoler     12/16/04 - make protocol lowercase 
#    jsmoler     12/15/04 - add protocol 
#    jsmoler     10/05/04 - split port and ip
#    jsmoler     10/04/04 - jsmoler_ecm01
#    jsmoler     08/18/04 - Creation
# 

use strict;
use ias::ohs::confFileParser;
use ias::asecm;

my $oracleHome = $ARGV[0];
setOracleHome($oracleHome);

# Metric: ListenAddresses
setMetricFormat(
    { 'type' => 'string', 'key' => 1 }, # IP
    { 'type' => 'number', 'key' => 1 }, # port
    { 'type' => 'string' },             # protocol
);

my $hash = parseApacheConf();

my $listenRef = $$hash{'Listen'};
if (!$listenRef)
{
    exit;
}
my @listenAddrs = @{$listenRef};

my $topSSLRef = peek($$hash{'SSLEngine'});
my $topProtocol = 'http';
if ($topSSLRef && ($topSSLRef eq 'on'))
{
    $topProtocol = 'https';
}

my @vhosts = getApacheVHosts($hash);

for my $addr (@listenAddrs)
{
    my $ip;
    my $port = $addr;
    if ($addr =~ /^(.*):(.*)/)
    {
        $ip = $1;
        $port = $2;
    }
    my $protocol = $topProtocol;
    my $vhost = findMatchingVHost($port, \@vhosts);
    if (defined($vhost))
    {
        my $props = $$vhost{'properties'};
        my $sslRef = peek($$props{'SSLEngine'});
        if ($sslRef)
        {
            if ($sslRef eq 'on')
            {
                $protocol = 'https';
            }
            elsif ($sslRef eq 'off')
            {
                $protocol = 'http';
            }
        }
    }
    printMetricResult($ip, $port, $protocol);
}

sub findMatchingVHost
{
    my $port = $_[0];
    my $vhosts = $_[1];

    for my $vhost (@{$vhosts})
    {
        my $vhostPort = $$vhost{'port'};
        if (defined($vhostPort) && ($vhostPort eq $port))
        {
            return $vhost;
        }
    }
    for my $vhost (@vhosts)
    {
        my $vhostPort = $$vhost{'port'};
        if (!defined($vhostPort) || ($vhostPort eq '*'))
        {
            return $vhost;
        }
    }
    return undef;
}
