# 
# $Header: propertiesFileParser.pm 06-mar-2006.10:02:37 jsmoler Exp $
#
# propertiesFileParser.pm
# 
# Copyright (c) 2004, 2006, Oracle. All rights reserved.  
#
#    NAME
#      propertiesFileParser.pm - Properties file parser
#
#    DESCRIPTION
#       This module parses a Java properties file, as defined in the documentation
#       for java.util.Properties.
#
#       Entry points:
#           parseProperiesFile
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    jsmoler     03/06/06 - add package statement 
#    jsmoler     10/05/04 - 
#    jsmoler     10/04/04 - jsmoler_ecm01
#    jsmoler     08/18/04 - Creation
# 

package ias::propertiesFileParser;

use strict;
use Exporter 'import';

our @EXPORT = qw(
    parsePropertiesFile
);

#
# Parses a Java properties file. Returns a hash mapping property names to
# values.
#
# Arguments
#   0: Properties file name
#
# Returns: A hash mapping property names to property values.
#
sub parsePropertiesFile
{
    my $fileName = $_[0];
    
    if (! -e $fileName)
    {
        die "File does not exist: '$fileName'";
    }
    
    my %hash;
    
    my $file;
    open($file, $fileName);
    
    while (1)
    {
        my $line = readline($file);
        # Remove whitespace
        $line = trim($line);
        if (!defined($line))
        {
            last;
        }
        
        # Skip empty lines
        if (length($line) == 0)
        {
            next;
        }
        
        # Handle commented lines. Comments start with either # or !
        if ($line =~ /^[#!].*/)
        {
            next;
        }

        # Concatenate lines continued with backslashes
        while ($line =~ /^(.*)\\$/)
        {
            $line = $1;
            my $nextLine = readline($file);
            $nextLine = trim($nextLine);
            if (defined($nextLine))
            {
                $line = $line . $nextLine;
            }
        }

        my $i;
        my $key = '';
        for ($i = 0; $i < length($line); $i++)
        {
            my $char = substr($line, $i, 1);
            if ($char eq "\\")
            {
                $i = $i + 1;
                $char = substr($line, $i, 1);
            }
            elsif ($char =~ /[\s:=]/)
            {
                $i = $i + 1;
                last;
            }
            $key = $key . $char;
        }
        $key = trim($key);
        my $val = '';
        for ( ; $i < length($line); $i++)
        {
            my $char = substr($line, $i, 1);
            if ($char eq "\\")
            {
                $i = $i + 1;
                $char = substr($line, $i, 1);
            }
            $val = $val . $char;
        }
        $val = trim($val);
        
        $hash{$key} = $val;
    }
    
    close($file);
    
    return %hash;
}

#
# Removes trailing and leading whitespace from a string.
#
# Arguments
#   0: A string
#
# Returns: The string without trailing or leading whitespace.
#
sub trim
{
    my $str = $_[0];    
    if (!defined($str))
    {
        return undef;
    }    
    $str =~ /^\s*(.*?)\s*$/;
    return $1;
}

return 1;
