# $Header: iasotherresourceusage.pl 26-sep-2005.11:51:38 klmichae Exp $
#
# iasotherresourceusage.pl
# 
# Copyright (c) 2003, 2005, Oracle. All rights reserved.  
#
#    NAME
#      iasotherresourceusage.pl - gets status, start time, uptime, cpu, and
#       memory usage for non-opmn managed components
#
#    DESCRIPTION
#      iasotherresourceusage.pl <emd_root> <oracle_home> 
#
#      where:
#         <emd_root> is the root of the emd tree
#         <oracle_home> is the oracle home in which the iAS is installed
#
#    returns: 
#      em_result=<component-name>|<status>|<uptime>|<starttime>|<cpuUsage>|
#      <physicalMemoryUsage>
#
#      where:
#         <component-name> the display name of the component
#         <status> the up/down status of the component (1 or 0)
#         <uptime> the uptime of the component, in milliseconds
#         <starttime> the start time of the component in milliseconds since the
#          unix epoch
#         <cpuUsage> percentage of CPU time being used by the component
#         <physicalMemoryUsage> amount of physical memory (MB) being used by  
#          component
#
#    NOTES
#
#
#    MODIFIED   (MM/DD/YY)
#    klmichae   09/26/05 - pass emdroot to iasntresourceusage 
#    msasidha   06/28/05 - bug 3917246 changes 
#    klmichae   06/15/04 - add space at end of oracle_home 
#    klmichae   05/27/04 - handle management components with multiple 
#                          processes 
#    jsutton    04/28/04 - Merge 3451947 changes to MAIN 
#    jsutton    09/24/03 - Fixes for Win32 support 
#    klmichae   08/22/03 - handle not being able to get memory 
#    jsutton    07/02/03 - Use platform-specific separator
#    klmichae   06/16/03 - fix ifs support
#    klmichae   06/04/03 - add JServ Support
#    klmichae   05/29/03 - klmichae_bug-2888766
#    klmichae   05/22/03 - Initial revision

use strict;
use File::Basename;

require "iasresourceusage.pl";
require "iasntresourceusage.pl";

# prepare to invoke perl scripts
my $PERL5LIB = getPERL5LIB();
$ENV{'PERL5LIB'} = $PERL5LIB;

# Get process information
executePSCommand();
my $processors = getNumCpus( $ARGV[0] );

# Get information for forms
getComponentInfo( $ARGV[0], $ARGV[1], $processors, "Forms", "FormsProcessInfo.pl" );

# Get information for iFS
getComponentList( $ARGV[0], $ARGV[1], $processors, "IFSProcessInfo.pl" );

if (( $^O =~ "Windows" ) || ( $^O =~ "MSWin32" ))
{
   getNTResourceUsage( $ARGV[0], $ARGV[1] );
}
else
{
   # Get information for JServ
   getComponentInfo( $ARGV[0], $ARGV[1], $processors, "JServ", "JServProcessInfo.pl" );

   # Get information for EM processes
   my $ps_command = getPsCommand();
   getEMProcessInfo( $ARGV[0], $ARGV[1], $ps_command, $processors );
}

# Get information about a component.  If the perl script is found this function
# will output a line with the following information:
#      em_result=<component-name>|<status>|<uptime>|<starttime>|<cpuUsage>|
#      <physicalMemoryUsage>
# Parameters:
#  emd_root the root of the emd tree
#  oracle_home the oracle home that the component is running in
#  processors the number of processors on the system
#  name the display name of the component
#  script the name of the script to execute
#
sub getComponentInfo
{
  my $emd_root = shift(@_);
  my $oracle_home = shift(@_);
  my $processors = shift(@_);
  my $name = shift(@_);
  my $script = shift(@_);
  my $perl_exec = $^X;
  my $scriptsDir = dirname($0);

  # execute the perl script if it exists
  if( -e "$scriptsDir/$script" )
  {
    # Execute the script.  It will return:
    # [start-pid]|[comma-separated-pid-list]
    my $script_output = `$perl_exec -w $scriptsDir/$script $scriptsDir $oracle_home`;
    if( index( $script_output, '|' ) > 0 ) 
    {

      # Get the pids and get all the resource information from them
      my ($start_pid, $pid_list) = split( '\|', $script_output );
      getProcessInfoFromPids( getTimeInformation( $start_pid ),
        $pid_list, $name, $emd_root, $processors );
    } else {
        # No pids return; the component is down
        print "em_result=$name|0||||\n";
    }

  }
}

# Get information about a list of components.  If the perl script is found this 
# function will output one or more lines with the following information:
#      em_result=<component-name>|<status>|<uptime>|<starttime>|<cpuUsage>|
#      <physicalMemoryUsage>
# Parameters:
#  emd_root the root of the emd tree
#  oracle_home the oracle home that the component is running in
#  processors the number of processors on the system
#  script the name of the script to execute
#
sub getComponentList
{
  my $emd_root = shift(@_);
  my $oracle_home = shift(@_);
  my $processors = shift(@_);
  my $script = shift(@_);
  my $perl_exec = $^X;
  my $scriptsDir = dirname($0);

  # execute the perl script if it exists
  if( -e "$scriptsDir/$script" )
  {
    # Execute the script.  It will return lists like:
    # {name}|[start-pid]|[comma-separated-pid-list]
    my $script_output = `$perl_exec -w $scriptsDir/$script $scriptsDir $oracle_home`;

    # Split the lines and walk through them 
    my ($name, $start_pid, $pid_list);
    my @lines = split( "\n", $script_output );
    foreach my $line (@lines)
    {
      # Get the name and pids and get all the resource information from them
      my ($name,$start_pid, $pid_list) = split( '\|', $line );
      if( $start_pid ne "" ) 
      {
        getProcessInfoFromPids( getTimeInformation( $start_pid ),
          $pid_list, $name, $emd_root, $processors );
      } else {
        # No pids return; the component is down
        print "em_result=$name|0||||\n";
      }
    } # end foreach

  } # end if( -e "$scriptsDir/$script" )
}

# Get process information from pids.  This function will output process 
# information from a list of pids for a component. It will output the 
# following information:
#      em_result=<component-name>|<status>|<uptime>|<starttime>|<cpuUsage>|
#      <physicalMemoryUsage>
# Parameters:
#  $time_info start time information for the process.  It is in the form:
#   <uptime>|<starttime>
#  pid_list a command separated list of pids for the process
#  name the display name of the component
#  emd_root the root of the emd tree
#  processors the number of processors on the system
sub getProcessInfoFromPids
{
  my $time_info = shift(@_);
  my $pid_list = shift(@_);
  my $name = shift(@_);
  my $emd_root = shift(@_);
  my $processors = shift(@_);

  if( $time_info eq "\|" ) 
  {
    # No pids return; the component is down
    print "em_result=$name|0||||\n";
  } else {  
    # If we have some pids, get the cpu and memory
    my $cpu = 0; 
    my $memory = 0;
    if( $pid_list ne "" ) 
    {
      $cpu = getCPUUsageForPids( $pid_list, $processors ); 
      $memory = getMemoryUsageForPids( $pid_list, $emd_root );
      if( $memory == -1 ) 
      {
        $memory = "";
      }
    } 
    
    # the component is up, show its metrics
    print "em_result=$name|1|$time_info|$cpu|$memory\n";
  } 
}

# Get information about EM process.  This function will output a line for each 
# EM process with the following information:
#      em_result=<component-name>|<status>|<uptime>|<starttime>|<cpuUsage>|
#      <physicalMemoryUsage>
# The EM processes are: agent, java console, opmn and em watchdog
# Parameters:
#  emd_root the root of the emd tree
#  oracle_home the oracle home that the component is running in
#  ps_command ps command to get the full command line
#  processors the number of processors on the system
#
sub getEMProcessInfo
{
  my $emd_root = shift(@_);
  my $oracle_home = shift(@_);
  my $ps_command = shift(@_);
  my $processors = shift(@_);

  my $ps_output = `$ps_command`;
  my @lines = split( "\n", $ps_output );

  my (@opmn_lines, @agent_lines, @watchdog_lines, @console_lines);
  my ($cur_line, $agent_exe, $watchdog_exe, $i);

  # agent process 
  $agent_exe = "$emd_root/bin/emagent";
  if (-e "$oracle_home/bin/emagent")
  {
    $agent_exe = "$oracle_home/bin/emagent";
  }

  # watchdog process
  $watchdog_exe = "$emd_root/bin/emwd.pl";
  if (-e "$oracle_home/bin/emwd.pl")
  {
    $watchdog_exe = "$oracle_home/bin/emwd.pl";
  }

  for ($i = 0; $i < @lines; $i++)
  {
    $cur_line = $lines[$i];
    # opmn process
    # search for opmn -d pattern as opmn itself can occur in discoverer 
    # processes also and can lead to incorrect mgmt process size
    if (index ($cur_line, "$oracle_home/opmn/bin/opmn -d") >= 0)
    {
      push (@opmn_lines, $cur_line);
      next;
    }
    # agent process 
    if (index ($cur_line, $agent_exe) >= 0)
    {
      push (@agent_lines, $cur_line);
      next;
    }
    # watchdog process 
    if (index ($cur_line, $watchdog_exe) >= 0)
    {
      push (@watchdog_lines, $cur_line);
      next;
    }
    # em java process
    if (index ($cur_line, "EMDROOT") >= 0)
    {
      if (index ($cur_line, "ORACLE_HOME=$oracle_home ") >= 0)
      {
        push (@console_lines, $cur_line);
        next;
      }
    }
  }

  getProcessInfoNew( "opmn", $emd_root, $processors, \@opmn_lines );
  getProcessInfoNew( "agent", $emd_root, $processors, \@agent_lines );
  getProcessInfoNew( "watchdog", $emd_root, $processors, \@watchdog_lines );
  getProcessInfoNew( "console", $emd_root, $processors, \@console_lines );
}

# Get information about an EM process.  This function will output a line for  
# this EM process with the following information:
#      em_result=<name>|<status>|<uptime>|<starttime>|<cpuUsage>|
#      <physicalMemoryUsage>
# Parameters:
#  ps_command ps command to get the full command line
#  name the component name
#  emd_root the root of the emd tree
#  processors the number of processors on the system
#
sub getProcessInfo
{
  my $ps_command = shift(@_);
  my $name = shift(@_);
  my $emd_root = shift(@_);
  my $processors = shift(@_);

  # Execute the ps command to find the processes
  my $ps_output = `$ps_command`;
  my @lines = split( "\n", $ps_output );

  getProcessInfoNew($name, $emd_root, $processors, \@lines);
}

sub getProcessInfoNew
{
  my $name = shift(@_);
  my $emd_root = shift(@_);
  my $processors = shift(@_);
  my $lines = shift(@_);

  if( $#$lines > -1 ) 
  {
    # Get the first line
    my @tokens = split( " ", $lines->[0] );
    my $time_info = getTimeInformation( $tokens[1] );
    my $oldest_time_info = $time_info;
    my $uptime = 0;
    my $start_time;
    if( $time_info ne "/|" ) {
      ($uptime, $start_time) = split( '\|', $time_info );
    }
    my $longest_uptime = $uptime;
    my $pid_list = $tokens[1];
    
    # check for many pids
    if( $#$lines > 0 ) 
    {
      for( my $i = 1; $i <= $#$lines; $i ++ ) 
      {
        @tokens = split( " ", $lines->[$i] );
        $pid_list = "$pid_list,$tokens[1]";

        # see if this pid has an earlier start time
        $time_info = getTimeInformation( $tokens[1] );
        if( $time_info ne "/|" ) {
          ($uptime, $start_time) = split( '\|', $time_info );
          if( $uptime > $longest_uptime ) {
            $oldest_time_info = $time_info;
            $longest_uptime = $uptime;
          }
        }        
      }
    }

    # Get the resource information for the pids
    getProcessInfoFromPids( 
      $oldest_time_info, $pid_list, $name, $emd_root, $processors );

  } else {
    # no pids, process is down
    print "em_result=$name|0||||\n";
  }
}

# 
#
# getPERL5LIB()
#
# Return the equated value of $PERL5LIB
#
sub getPERL5LIB
{
    my $cpSep = ":";
    if (( $^O =~ "Windows" ) || ($^O =~ "MSWin32"))
    {
        $cpSep = ";";
    }
    my $PERL5LIB;
    for (my $i = 0; $i < @INC; $i++)
    {
        if ($i == 0)
        {
            $PERL5LIB = $INC[$i];
        }
        else
        {
            $PERL5LIB = $PERL5LIB . $cpSep . $INC[$i];
        }
    }

    return $PERL5LIB;
}

