# $Header: emas/sysman/admin/scripts/iasresourceusage.pl /main/29 2012/04/15 03:56:09 rahgupta Exp $
#
# iasresourceusage.pl
# 
# Copyright (c) 2002, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      iasresourceusage.pl - gets cpu and memory usage for a list of pids
#
#    DESCRIPTION
#      iasresourceusage.pl 
#
#    returns: 
#      em_result=<cpuUsage>|<physicalMemoryUsage>
#
#      where:
#         <cpuUsage> percentage of CPU time being used by the specified pid list
#         <physicalMemoryUsage> amount of physical memory (MB) being used by  
#          specified pid list
#
#    NOTES
#
#
#    MODIFIED   (MM/DD/YY)
#    cmrozien   04/02/12 - use EMSTATE for sysman/log writing
#    smariswa   04/05/11 - Remove usage of ResponseMetric class usage
#    msasidha   11/08/10 - more changes
#    msasidha   07/09/08 - bug 9907750 
#    msasidha   07/09/08 - bug 7142510
#    sradhakr   03/05/08 - review comments
#    sradhakr   02/28/08 - Bug#6840316 - RESPONSE MENTRIC OF STAND ALON
#                          WEBCACHE IS NOT WORKING.
#    klmichae   07/27/05 - handle multiple processor systems differently on 
#                          solaris 
#    klmichae   07/13/05 - change computation of cpus on Linux 
#    jsutton    01/24/05 - Make TMPFILE name unique per PID 
#    nsharma    09/03/04 - Changes for Mac
#    rrawat     05/31/04 - Bug-3647367
#    rlal       08/10/04 - Fix for bug 3677794 on aix 
#    nsharma    07/01/04 - Linux osLoad.pl output is shifted one extra place
#    klmichae   05/27/04 - handle low numbered pids 
#    klmichae   05/03/04 - correct expression to find 9.0.2/9.0.3 oc4j 
#    rlal       04/30/04 - Fix for performance issue on Tru64 
#    nsharma    04/27/04 - Bug 3597014 
#    nsharma    04/11/04 - Bug 3556116 
#    nsharma    04/02/04 - Added support for AIX and Tru64 
#    klmichae   03/10/04 - add EMDROOT and agent version parameters to webcache pid program
#    klmichae   03/02/04 - add support for monitoring ias 902/903 on NT 
#    jsutton    02/13/04 - Bring up to snuff 
#    klmichae   12/22/03 - add getTotalCpuTime
#    jsutton    12/10/03 - Fix key for pre-904 OC4J instances 
#    klmichae   12/03/03 - change nmupm arguments 
#    klmichae   12/03/03 - merge in HP changes 
#    klmichae   11/21/03 - only get cpu and memory usage
#    klmichae   09/22/03 - get ostype in functions 
#    ccleavel   08/29/03 - Add getWebcacheIsUp
#    klmichae   09/03/03 - fix uptime less than one day 
#    klmichae   08/22/03 - recover from nmb errors 
#    klmichae   07/31/03 - change resource computations
#    klmichae   05/22/03 - refactor code
#    klmichae   05/16/03 - change ps commands for other platforms
#    klmichae   05/15/03 - remove calls to opmnadmin
#    klmichae   03/31/03 - account for multiple cpus
#    klmichae   03/21/03 - handle non-existent processes
#    jmcclung   12/12/02 - rename nmp to nmb
#    klmichae   12/02/02 - call pmap program
#    klmichae   11/18/02 - report an error for ias installed as different user
#    klmichae   10/23/02 - set oracle_home before calling webcachectl
#    klmichae   10/15/02 - remove use of meminfo
#    klmichae   10/14/02 - set oracle home environment variable
#    klmichae   08/23/02 - klmichae_bug-2519611
#    klmichae   08/19/02 - Initial revision

use strict;
use Config;
use Fcntl ':mode';

require "iasntresourceusage.pl";
require "emd_common.pl";
require "semd_common.pl";

# This is the output from the ps command: ps -A -o "pid pcpu rss etime vsz cmd"
# This ps command is executed when calling executePSCommand()
my %process_hash = ();
my $PS_CMD_PID_COL = 0;
my $PS_CMD_CPU_COL = 1;
my $PS_CMD_MEM_COL = 2;  
my $PS_CMD_TIME_COL = 3;  
my $PS_CMD_VMEM_COL = 4;  
my $PS_CMD_CMD_COL = 5;  
my $now;

# Columns from long ps comannds that contains the complete command line that
# we will grep for the processes that we are interested in.  This is returned by
# getPsCommand.  It returns the following
# USER       PID %CPU %MEM   SZ  RSS TT       S    START  TIME COMMAND
#my $LONG_PS_CMD_USER_COL = 0;
my $LONG_PS_CMD_PID_COL = 1;
my $LONG_PS_CMD_PCPU_COL = 2;
#my $LONG_PS_CMD_PMEM_COL = 3;  
my $LONG_PS_CMD_VSZ_COL = 4;  
my $LONG_PS_CMD_RSS_COL = 5;  
#my $LONG_PS_CMD_TT_COL = 6;  
#my $LONG_PS_CMD_STATE_COL = 7;  
#my $LONG_PS_CMD_START_COL = 8;  
#my $LONG_PS_CMD_TIME_COL = 9;  
my $LONG_PS_CMD_CMD_COL = 10;  
    
# Determine the OSType
my $osType=" ";
sub computeOSType
{
  if( $osType ne " " ) 
  {
    return;
  }

  my $OSNAME = $Config{'osname'};
  if ($OSNAME eq 'MSWin32')
  {
    $osType = "MSWin32";
    # set up indexes into nmupm info for these columns
    $PS_CMD_CMD_COL = 1;
    $PS_CMD_CPU_COL = 4;
    $PS_CMD_MEM_COL = 5;
    $PS_CMD_VMEM_COL = 6;
  }
  else
  {
    # See what OS we have so that we can determine the algorithm to get info
    my $os;
    my $ver;
    $os = $^O;
    SWITCH: 
    {
      (($os eq "solaris") || ($os eq "sunos")) && do
      {
        chomp ($ver = `/usr/bin/uname -r`) or die "Failed to run the uname command";
        if ( $ver !~ /^4./ ) 
        {
          $osType = "Sol";
          last SWITCH;
        }
      };
      $os eq "hpux" && do
      {
        $osType = "HP";
        $ENV{UNIX95} = "XPG4";
        last SWITCH;
      };
      $os eq "linux" && do
      {
        $osType = "LNX";
        last SWITCH;
      };
      $os eq "dec_osf" && do
      {
        $osType = "OSF1";
        last SWITCH;
      };
      $os eq "aix" && do
      {
        $osType = "AIX";
        last SWITCH;
      };
      $os eq "darwin" && do
      {
        $osType = "Darwin";
        last SWITCH;
      };
      die "Unsupported Operating System\n";
    }
  }
}

#
# Execute the ps command that will be used to get process information
#
sub executePSCommand
{
  # execute the ps command
  computeOSType();
  my $oraHome = $ENV{ORACLE_HOME};
  my $agentStateDir = $ENV{"EMSTATE"};
  my ($ps_cmd, $process_output, $splitChar, @process_output_lines);
  
  $now = time();

  # handle Win32
  if ($osType eq "MSWin32")
  {
    $splitChar = '\|';
    my $TMPFILE = "$agentStateDir/sysman/log/topprocs.tmp".$$;
    system("$oraHome/bin/nmupm.exe topProcs > $TMPFILE");

    my ($proc);
    open (TOPPROCs, $TMPFILE) || die "Can not open file $TMPFILE: $!\n";
    while ($proc = <TOPPROCs>) 
    {
      # get output after "em_result="; columns are
      # <pid>|<imageName>|N/A|<pctMem>|<pctCpu>|<workingSet>|<vmSize>|0
      chomp $proc;
      push (@process_output_lines, substr($proc, index($proc, "=")+1));
    }
    close TOPPROCs;
    unlink $TMPFILE;
  }
  else
  {
    $splitChar = " ";
    if ($osType eq "LNX") 
    {
      $ps_cmd = "ps -A -o \"pid pcpu rss etime vsz cmd\"";
    } 
    elsif ($osType eq "HP") 
    {
      $ps_cmd = "ps -A -o \"pid,pcpu,sz,etime\"";
    } 
    elsif ($osType eq "OSF1") 
    {
      $ps_cmd = "ps -A -o \"pid,pcpu,sz,etime,vsz,cmd\"";
    } 
    elsif ($osType eq "AIX") 
    {
      $ps_cmd = "/usr/sysv/bin/ps -A -o \"pid,pcpu,rss,etime\"";
    } 
    elsif ($osType eq "Darwin") 
    {
      $ps_cmd = "ps  ax -o \"pid,pcpu,rsz,start,vsz,command\"";
    } 
    else
    {
      $ps_cmd = "ps -A -o \"pid pcpu rss etime\"";
    }
    $process_output = `$ps_cmd`;
    @process_output_lines = split("\n", $process_output);	 
  }

  # Create a hash table by pid
  my @process_info;
  for( my $i=1; $i <= $#process_output_lines; $i++ )
  {
    # pull off leading and embedded spaces and then break into tokens
    $process_output_lines[$i] =~ s/^\s+//;
    $process_output_lines[$i] =~ s/\s+/ /g;
    @process_info = split($splitChar, $process_output_lines[$i]);
    $process_hash{$process_info[$PS_CMD_PID_COL]} = 
      $process_output_lines[$i];
  }  
}

# Get physical memory metrics for the system.  Return them in the form: 
# {TotalMemoryInMB}|{FreeMemoryRaw}|{PageSize}
# Parameters:
# emdRoot the root directory for the emd
sub getPhysicalMemoryMetrics
{
  my $emdroot = shift(@_);

  # get the osLoad info from nmupm.  The result will be
  # em_result={cpuLoad_1min}|{cpuLoad_5min}|{cpuLoad_15min}|{PageScanRaw}|
  # {noOfProcs}|{noOfUsers}|{transfers}|{idleTimeRaw}|{userTimeRaw}|
  # {systemTimeRaw}|{waitTimeRaw}|{pageSize}|{realMem}|{freeMemRaw}|
  # {usedSwapRaw}|{freeSwapRaw}
  my $PAGE_SIZE_COL = 11;
  my $REAL_MEM_COL = 12;
  my $FREE_MEM_RAW_COL = 13;
  my $mem_output;
  computeOSType();

  if ( $osType eq "MSWin32" )
  {
    # nmupm output on Win32, as of 9.0.4:
    # em_result={cpuQLen}|{noOfProcs}|{noOfUsers}|{idleTimeRaw}|{userTimeRaw}|
    #  {systemTimeRaw}|{intrTimeRaw}|{pgTransferRaw}|{freeMemKb}|{realMemKb}|
    #  {freeSwapKb}|{totalSwapKb}|{usedSawpKb}
    $FREE_MEM_RAW_COL = 8;
    $REAL_MEM_COL = 9;
    $mem_output = `$emdroot/bin/nmupm Load`;
    my @win_mem_lines = split( '\|', $mem_output );

    # pull out realMem and freeMemRaw; pageSize = 512 [?]
    my $winTotalMem = $win_mem_lines[$REAL_MEM_COL]/1024; 
    return "$winTotalMem|$win_mem_lines[$FREE_MEM_RAW_COL]*1024|512";
  }
  if ( $osType eq "LNX" ) 
  {
    $PAGE_SIZE_COL = 12;
    $REAL_MEM_COL = 13;
    $FREE_MEM_RAW_COL = 14;
    $mem_output = `$emdroot/perl/bin/perl $emdroot/sysman/admin/scripts/osLoad.pl`;
  }
  else
  {
    $mem_output = `$emdroot/bin/nmupm osLoad`;
  }
  my @mem_lines = split( '\|', $mem_output );

  # pull out pageSize, realMem and freeMemRaw
  my $totalMem = $mem_lines[$REAL_MEM_COL]/1024; 
  return "$totalMem|$mem_lines[$FREE_MEM_RAW_COL]|$mem_lines[$PAGE_SIZE_COL]";
}

#
# Get the CPU usage for a list of pids.  When doing this computation take into
# account the number of processors on the system.
#
# Note: executePSCommand must be called before calling this function.
#
# Parameters:
#  pid_list a comma separated list of pids
#  processors the number of processors
sub getCPUUsageForPids
{
  my $pid_list = shift(@_);
  my $processors = shift(@_);
  my $total = 0;
  if( index( $pid_list, ',' ) == -1 ) 
  {
    # Get the CPU usage for the single pid
    $total = getCPUUtilization( $pid_list );
  }
  else
  {
    # Get the total CPU for all pids
    my @pid_array = split( ',', $pid_list );
    my $pid;
    foreach $pid (@pid_array)
    {
      $total += getCPUUtilization( $pid );
    }  
  }

  # The ps command on Solaris already accounts for multiple processors.
  # The ps command on Linux does not account for multiple processors, so
  # we need to divide the result by the number of processors
  if( $osType ne "Sol") 
  {
    $total = $total/$processors;
  }
  return $total;
}

#
# Get the memory usage in MB for a list of pids.  Return -1 if we 
# cannot compute the memory
#
# Note: executePSCommand must be called before calling this function.
#
# Parameters:
#  pid_list a comma separated list of pids
#  emdroot the location of the emd root directory
sub getMemoryUsageForPids
{
  my $pid_list = shift(@_);
  my $local_pid_list = $pid_list;
  my $emdroot = shift(@_);
  my @pid_array;
  my $memoryKB = 0;

  # On Linux the ps command returns process and threads.  But, we only want 
  # count processes, so assume that any process with the same vsz, rss and
  # command all represent the same process 
  if ($osType eq "LNX") 
  {
    my @process_info;
    my %linx_process_hash = ();
    my $key;

    # Go through all the pids
    $local_pid_list = "";
    @pid_array = split( ',', $pid_list );
    foreach my $pid (@pid_array)
    {
      # find the pid in the process hash
      if( defined($process_hash{$pid}) ) 
      {
        # Get the process info for the pid
        @process_info = split(" ", $process_hash{$pid});

        # See if this is a unique process; use vsz**rss**command as the key
        $key = "$process_info[$PS_CMD_VMEM_COL]**$process_info[$PS_CMD_MEM_COL]**$process_info[$PS_CMD_CMD_COL]";
        if (!defined $linx_process_hash{$key})
        {
          # This is a process that we should count
          if( $local_pid_list eq "" ) 
          { 
            $local_pid_list = "$pid";
           } 
           else
           {
             $local_pid_list = "$local_pid_list,$pid";
           }

           # add it to the hash table, so we don't found it again
           $linx_process_hash{$key} = '1';
           #print "DEBUG: pid=$pid, key = $key\n";
        }
      } # end if pid in hash
    } # end $i
  }

  # Create an array of pids and get the number usage
  @pid_array = split( ",",$local_pid_list );

  if ($osType eq "MSWin32")
  {
    # Get the total Memory for all pids
    my $pid;
    foreach $pid (@pid_array)
    {
      $memoryKB += getMemoryUtilization( $pid );
    }  
  }
  else
  {
    # compute the memory usage and convert it to MB
    $memoryKB = getPhysicalMemoryIncludingShared( \@pid_array, $emdroot );
  }
  if( $memoryKB == -1 ) 
  {
    return -1;
  }
  else
  {
    return $memoryKB/1024;
  }
}

#
# Get the CPU usage for the specified pid.  This gets the raw cpu percentage.
# It does not account for the number of processors on the machine.
#
# Note: executePSCommand must be called before calling this function.
#
# Parameters:
# pid the process to get cpu info for
sub getCPUUtilization
{
  my $pid = shift(@_);
  chomp ($pid);
  return getUtilization($pid, $PS_CMD_CPU_COL );
}

# Get the Memory usage for the specified pid
#
# Parameters:
# pid the process to get cpu info for
sub getMemoryUtilization
{
  my $pid = shift(@_);
  chomp ($pid);
  return getUtilization($pid, $PS_CMD_MEM_COL );
}

#
# Get the uptime and starttime for the pid
#
# Note: executePSCommand must be called before calling this function.
#
# Parameters:
#  pid the process to get uptime and starttime for
# returns <uptime>|<starttime>
sub getTimeInformation
{
  my $pid = shift(@_);
  my $oraHome = $ENV{ORACLE_HOME};
  my $nmupmLine;

  # Get the uptime.  It will be in the form [[ddd-]hh:]mm:ss
  # we must convert it to seconds
  my $uptime_string;
  if ($osType eq "MSWin32")
  {
    $nmupmLine = `$oraHome/bin/nmupm procinfo $pid`;
    if ((rindex($nmupmLine, "|")+1) > 0)
    {
      $uptime_string = substr($nmupmLine, rindex($nmupmLine, "|")+1);
    }
    else
    {
      $uptime_string = "0";
    }
  }
  else
  {
    $uptime_string = getUtilization($pid, $PS_CMD_TIME_COL );
  }
  if( $uptime_string eq "0" ) 
  {
    return "|";
  }
  my $uptime_ms = convertTimeToSecs( $uptime_string )*1000;

  # now get the start time
  my $starttime_ms = $now*1000 - $uptime_ms;

  return "$uptime_ms|$starttime_ms";  
}

sub convertTimeToSecs
{
  my $time = shift(@_);
  my $secs = 0;
  my ($d, $h, $m, $s);

  # See if there are days and account for them
  if( index( $time, '-' ) != -1 )
  {
    my $days;
    ($days, $time) = split( '-', $time );
    $secs = $days*86400;
  }

  # add in the hours, minutes and seconds
  my @t_array = split(":", $time);
  if(@t_array == 3) 
  {
    ($h,$m,$s) = @t_array;
    $secs += ($h*3600);
  }
  else
  {
    ($m, $s) = @t_array;
  }
  $secs += ($m*60)+$s;
  return $secs;
}

#
# Get the Resource usage for the specified pid
#
# Note: executePSCommand must be called before calling this function.
#
# Parameters:
# pid the process to get info for
# $process_column the column of process information to read
sub getUtilization
{
  my $pid = shift(@_);
  my $process_column = shift(@_);
  my $splitChar = ' ';

  # Get the result from the ps command
  my @process_info;
  if( defined($process_hash{$pid} ) ) 
  {
    if ($osType eq "MSWin32")
    {
      $splitChar = '\|';
    }
    @process_info = split($splitChar, $process_hash{$pid});
    return $process_info[$process_column];
  }
  return 0;
}

# Get the apache child pid from dms.  It is the pid.value metric column 
# from the dms_cProcessInfo metric
#
# Note
#  This function has only been tested on NT when monitoring a 902/903
#  iAS installation
#
# Parameter
#  oracleHome the ias oracle home
sub getApacheChildPidFromDMS
{
  my $oracleHome = shift(@_);

  # Run dmstool and get the dms_cProcessInfo table
  my $dms_output = `$oracleHome/bin/dmstool -table dms_cProcessInfo 2>&1`;

  # Get the output lines and walk through them looking for the pid.value column.
  # The following line is an example of what we are looking for:
  # /jsutton-pc/Apache:856:6003/ProcessInfo/pid.value        856
  my $line;
  my @lines = split( "\n", $dms_output );
  my @tokens;
  foreach $line (@lines)
  {
    if( index( $line, "pid.value" ) > 0 ) 
    { 
      my ($junk, $pid);
      ($junk, $pid) = split( " ", $line );
      #print "KLM: found apache pid $pid\n";
      return( $pid );
    }
  }
  
  # We didn't find a pid
  return( "" );
 
}

# Get the pid list for the specified process.  It is the pid.value metric 
# column from the opmn_process metric
#
# Note
#  This function has only been tested on NT when monitoring a 902/903
#  iAS installation
#
# Parameter
#  oracleHome the ias oracle home
#  name the name of the component whose pid we are looking for.  If this is 
#   empty, return all pids
sub getPidForComponentFromDMS
{
  my $oracleHome = shift(@_);
  my $name = shift(@_);
  my $pidList = "";

  # Run dmstool and get the opmn_process table
  my $dms_output = `$oracleHome/bin/dmstool -table opmn_process 2>&1`;

  # Get the output lines and walk through them looking for the pid.value column.
  # The following line is an example of what we are looking for:
  # /jsutton-pc/Opmn:271:6003/allProcesses/jsutton-pc:home.306787/pid.value  404
  my $line;
  my @lines = split( "\n", $dms_output );
  my @tokens;
  foreach $line (@lines)
  {
    # Is this a pid column?
    if( index( $line, "pid.value" ) > 0 ) 
    { 
      my ($key, $pid);
      ($key, $pid) = split( " ", $line );

      # Is this one of the components that we are interested in?
      if( $name eq "" || index( $key, ":$name." ) > 0 ) 
      {
        if( $pidList eq "" )
        {
          $pidList = $pid;
        } else {
          $pidList = "$pidList,$pid";
        }
      }
    }
  }
  
  #print "KLM: pid list for component '$name': $pidList\n";
  return( $pidList );

}
 
# Get a ps command that returns the the username followed by the pid
# plus the full command line.  Note that this command is plateform
# specific.
sub getPsCommand
{
  computeOSType();
  SWITCH:
  {
    $osType eq "Sol" && do
    {
      return "/usr/ucb/ps auxww";
    };
    $osType eq "HP" && do
    {
      $ENV{UNIX95} = "XPG4";

#    %MEM not available on HP-UX, printing pcpu instead as it is not used.  
      return "/bin/ps -efxo user,pid,pcpu,pcpu,vsz,sz,tty,state,stime,time,args";
    };
    $osType eq "LNX" && do
    {
      return "/bin/ps auxww";
    };
    $osType eq "OSF1" && do
    {
      return "/usr/bin/ps axww";
    };
    $osType eq "AIX" && do
    {
      return "/usr/bin/ps auxww";
    };
    $osType eq "Darwin" && do
    {
      return "/bin/ps auxww";
    };
    die "Unsupported Operating System\n";
  }
}

# Get the pid list for the Http Server 
# parameters:
#   oracle_home - the oracle home in which the iAS instance is running
sub getHttpPids 
{
  my $oracle_home = shift(@_);
  my $pidList = "";

  computeOSType();

  # on Win32, get the pids from DMS
  if ($osType eq "MSWin32")
  {
    # Get the apache child process pid and the parent pid
    $pidList = getApacheChildPidFromDMS( $oracle_home );
    my $parentPid =  getPidForComponentFromDMS( $oracle_home, "HTTP_Server" );
    if( $pidList eq "" ) 
    {
      $pidList = $parentPid;
    } else {
      $pidList = "$pidList,$parentPid";
    }
    
  } else {
    # The ps command is platform specific
    my $ps_command = getPsCommand();

    # Get the apache pids from ps
    # The output will be in the following form:
    # USER       PID %CPU %MEM   SZ  RSS TT       S    START  TIME COMMAND
    my $key = getHttpServerKey( $oracle_home );
    my $ps_output = `$ps_command | egrep "$key"`;

    # Get the ps output lines and walk through them Looking for the pids
    my $line;
    my @lines = split( "\n", $ps_output );
    my @tokens;
    foreach $line (@lines)
    {
      @tokens = split( " ", $line );
      # Get the pid; it is the second token
      if( $pidList eq "" )
      {
        $pidList = "$tokens[$LONG_PS_CMD_PID_COL]";
      } 
      else
      {
        $pidList = "$pidList,$tokens[$LONG_PS_CMD_PID_COL]";
      }
    } # end foreach line
  } # end if MSWin32

  return $pidList;

}

# Get properties from the emd.properties file.  This includes:
# 1. the URL for the EMD in this agent home (EMD_URL)
# 2. get the agent's classpath (CLASSPATH)
# 3. the agent version (agentVersion)
# 4. the location of the wallet (emdWalletDest)
# @param emdroot the agent's root directory
# Returns an array with all the property values 
sub getEmdProperties
{

  my $emdroot = shift(@_);

  # Get the EMD_URL from emd.properties
  my $propertiesFile = $emdroot . "/sysman/config/emd.properties";
  my @propNames = ( "EMD_URL", "CLASSPATH", "agentVersion", "emdWalletDest" );
  my @propValues = ( "", "", "", "" );
  my @propInfo;
  my $propertiesLine;
  my $foundAll;

  if( -e $propertiesFile )
  {
    open( PROPERTIES_FILE_READER, $propertiesFile );
    while( $propertiesLine = <PROPERTIES_FILE_READER> )
    {
      chomp( $propertiesLine );

      # Check for each property
      for my $i (0 .. $#propNames )
      {
  
        # Look for the property if we have not yet found it
        if( $propValues[$i] eq "" &&  
         index( $propertiesLine, $propNames[$i] ) eq 0 )
        {
 
          @propInfo = split( /=/ ,$propertiesLine );
          if( $propInfo[0] eq $propNames[$i] )
          {
            $propValues[$i] = $propInfo[1];
            #print "KLM: $propNames[$i] = $propValues[$i]\n";
            last; # no need to look at this line again
          }
        }
      }

      # Are we done?
      $foundAll = 1;
      for my $i (0 .. $#propValues )
      {
        if( $propValues[$i] eq "" ) 
        {
          $foundAll = 0;
          last;
        }
      }
      if( $foundAll ) 
      {
        last;
      }
    }

    close PROPERTIES_FILE_READER;

  }

  # Make sure that we were able to get all values.  Otherwise return an error
  for my $i (0 .. $#propValues )
  {
    if( $propValues[$i] eq "" ) 
    {
      print "em_error=Cannot get $propNames[$i]\n";
      exit 1;
    }
  }

  return( \@propValues );

}

 
# Get the pid for the Web Cache Server 
# parameters:
#   oracle_home - the oracle home in which the iAS instance is running
sub getWebcachePidWin32
{
  my $oracle_home = shift(@_);

  # Determine the location of emdroot
  my $emdroot = $ENV{EMDROOT};

  # Determine the location of windows for getting wmic binary
  my $windir = $ENV{WINDIR};

  # Get webcachectl command to get status
  my $status_program_command = "$oracle_home/bin/webcachectl status";
  EMD_PERL_DEBUG("webcachectl command = $status_program_command");

  my $output = `$status_program_command`;
  EMD_PERL_DEBUG("webcachectl output = $output");

  my $pid = "";
  # Walk through the output to determine status
  my @lines = split( "\n", $output );

  my $cmd_part = $oracle_home;
  EMD_PERL_DEBUG("webcache oraclehome = $oracle_home");
  #convert single slash to double for use with wmic syntax
  $cmd_part =~ s/\\/\\\\/g;
  EMD_PERL_DEBUG("webcache modified oraclehome = $cmd_part");

  foreach my $line( @lines )
  {
    EMD_PERL_DEBUG("line = $line");
    # see if status returned is UP
    if( $line =~m/WebCache is running/ )
    {
      EMD_PERL_DEBUG("match found - webcache is UP");
      my $pid_program_cmd = "$windir/system32/wbem/wmic.exe process where executablepath=\"$cmd_part\\\\bin\\\\webcached.exe\" get name,processid /FORMAT:csv";
      EMD_PERL_DEBUG("wmic command to get pid of webcached $pid_program_cmd");
      my $output2 = `$pid_program_cmd`;
      EMD_PERL_DEBUG("wmic results is $output2");

      my @firstlines = split( "\n", $output2 );
      foreach my $line2( @firstlines ) {
        EMD_PERL_DEBUG("line is $line2");
        # remove ctrl-M
        $line2 =~ s/\x0D//;
        my @fields=split(/,/,$line2);
        EMD_PERL_DEBUG("webcache pid $fields[2]");
        $pid = $fields[2];
      }
      EMD_PERL_DEBUG("pid = $pid");
      return( "$pid" );
    }
  }
  EMD_PERL_DEBUG("pid = $pid");
  return( "$pid" );

}


# See if webcache is up.  If we can find the process, it is up
# parameters:
#   oracle_home - the oracle home in which the iAS instance is running
sub getWebcacheIsUp
{
  my $oracle_home = shift(@_);
  computeOSType();

  # on Win32, get the pids from DMS
  if ($osType eq "MSWin32")
  {
    my $pid = getWebcachePidWin32( $oracle_home ); 
    return( $pid ne "" );
  } else {
    # The ps command is platform specific
    my $ps_command = getPsCommand();

    # Get the search key for web cache processes.  Exclude the monitor process.
    my $key = getWebcacheKeyNoMonitor( $oracle_home );

    # Get the Web Cache pids from ps
    # Exclude the admin process, which is marked with -A.
    # Note that the space character before the -A is important.  Otherwise
    # the -A is interpreted as a qualifier on egrep instead of as the parameter.
    # The output will be in the following form:
    # USER       PID %CPU %MEM   SZ  RSS TT       S    START  TIME COMMAND
    my $ps_output = `$ps_command | egrep "$key" | egrep -v ' \-A'`;

    # The cache is up if and only if there's something in the filtered ps output.
    # returns 1 or empty string
    return ($ps_output ne "");
  } # end if MSWin32
}

# Get the pid list for the specified OC4J Instance
# parameters
#   oracle_home - the oracle home in which the iAS instance is running
#   oc4j_name - the oc4j instance name (if empty, return all instances)
sub getOc4jPids 
{
  # Get the arguments
  my $oracle_home = shift(@_);
  my $oc4j_name = shift(@_);
  
  my $pidList = "";

  computeOSType();

  # on Win32, get the pids from DMS
  if ($osType eq "MSWin32")
  {
    $pidList = getPidForComponentFromDMS( $oracle_home, $oc4j_name );
  } else {
 
    # The ps command is platform specific
    my $ps_command = getPsCommand();

    # Get the oc4j pids from ps
    # The output will be in the following form:
    # USER       PID %CPU %MEM   SZ  RSS TT       S    START  TIME COMMAND
    my $key = getOc4jKey( $oracle_home, $oc4j_name );
    my $ps_output = `$ps_command | egrep "$key"`;

    # Get the ps output lines and walk through them Looking for the pids
    my $line;
    my @lines = split( "\n", $ps_output );
    my @tokens;
    foreach $line (@lines)
    {
      @tokens = split( " ", $line );
      # Get the pid; it is the second token
      if( $pidList eq "" )
      {
        $pidList = "$tokens[$LONG_PS_CMD_PID_COL]";
      } 
      else
      {
        $pidList = "$pidList,$tokens[$LONG_PS_CMD_PID_COL]";
      }
    } # end foreach line

  } # end if MSWin32

  return $pidList;
}

# Compute the resource usage of the OC4J instance. This information includes
# cpu and physical memory.
# parameters:
#   emdroot - the emd root directory
#   oracle_home - the oracle home in which the iAS instance is running
#   oc4j_name - the oc4j instance name (if empty, return all instances)
sub computeOc4jResourceUsage
{
  # Get the arguments
  my $emdroot = shift(@_);
  my $oracle_home = shift(@_);
  my $oc4j_name = shift(@_);

  computeOSType();
  
  # on Win32, get the pids from DMS
  if( $osType eq "MSWin32" )
  {
    # Get information about all processes
    executePSCommand();
    my $cpu = 0;
    my $memory = 0;

    my $pidList = getPidForComponentFromDMS( $oracle_home, $oc4j_name );
    my @pidArray = split( ",", $pidList );
    my $pid;
    foreach my $pid( @pidArray )
    {
      $cpu += getCPUUtilization( $pid )/1024;
      $memory += getMemoryUtilization( $pid )/1024;;
    }
    print "em_result=$cpu|$memory\n";

  } else { # not MSWin32 
    # Look for the specified oc4j Instance 
    my $key = getOc4jKey( $oracle_home, $oc4j_name );
    getProcessInfoFromPsOutput( $emdroot, "\"$key\"" );
  } # end if MSWin32
  
}

# Get the key to look for to identify the OC4J processes
# parameters:
#   oracle_home - the oracle home in which the iAS instance is running
#   oc4j_name - the oc4j instance name or empty string for all oc4j instances
sub getOc4jKey
{
  # Get the arguments
  my $oracle_home = shift(@_);
  my $oc4j_name = shift(@_);
  my $oc4j_key;

  if( $oc4j_name eq "" )
  {
    $oc4j_key = "oracle\.ons\.instanceid";
  } 
  else
  {
    $oc4j_key = "#$oc4j_name#";
  }

  return "[D]oracle\.ons\.oraclehome=$oracle_home.+$oc4j_key";
}

# Compute the resource usage of the HTTP Server. This information includes
# cpu and physical memory.
# parameters:
#   emdroot - the emd root directory
#   oracle_home - the oracle home in which the iAS instance is running
sub computeHttpResourceUsage
{
  # Get the arguments
  my $emdroot = shift(@_);
  my $oracle_home = shift(@_);

  computeOSType();
  # on Win32, get the pids from DMS
  if( $osType eq "MSWin32" )
  {
    # Get the apache child process pid and the parent pid
    my $childPid = getApacheChildPidFromDMS( $oracle_home );
    my $parentPid =  getPidForComponentFromDMS( $oracle_home, "HTTP_Server" );

    # Get information about all processes
    executePSCommand();

    # get Memory and CPU information for the pids and convert from KB to MB
    my $cpu = (getCPUUtilization( $childPid ) + getCPUUtilization( $parentPid ))/1024;
    my $memory = (getMemoryUtilization( $childPid ) + getMemoryUtilization( $parentPid ))/1024;
    print "em_result=$cpu|$memory\n";

  } else {
    # Look for all apache processes
    my $key = getHttpServerKey( $oracle_home );
    getProcessInfoFromPsOutput( $emdroot, "\"$key\"" );
  } # end if MSWin32
}

# Get the key to look for to identify the HTTP Server processes
# parameters:
#   emdroot - the emd root directory
#   oracle_home - the oracle home in which the iAS instance is running
sub getHttpServerKey
{
  # Get the arguments
  my $oracle_home = shift(@_);
  return "$oracle_home/[A]pache/Apache";
}

# Compute the resource usage of the Web Cache Server. This information includes
# cpu and physical memory.
# parameters:
#   emdroot - the emd root directory
#   oracle_home - the oracle home in which the iAS instance is running
sub computeWebcacheResourceUsage
{
  # Get the arguments
  my $emdroot = shift(@_);
  my $oracle_home = shift(@_);

  computeOSType();

  # on Win32, get the pids from a special program
  if( $osType eq "MSWin32" )
  {
    # Get the webcache pid and then get information about all the processes
    my $pid = getWebcachePidWin32( $oracle_home ); 
    executePSCommand();

    # get Memory and CPU information for the pids and convert from KB to MB
    my $cpu = getCPUUtilization( $pid )/1024;
    my $memory = getMemoryUtilization( $pid )/1024;
    print "em_result=$cpu|$memory\n";

  } else {
    # Look for all webcache processes
    my $key = getWebcacheKey( $oracle_home );
    getProcessInfoFromPsOutput( $emdroot, "\"$key\"" );
  } #end if MSWin32
}

# Get the key to look for to identify the Web Cache processes (cache, admin, monitor)
# parameters:
#   oracle_home - the oracle home in which the iAS instance is running
sub getWebcacheKey
{
  # Get the arguments
  my $oracle_home = shift(@_);
  return "$oracle_home/[w]ebcache/bin/webcache";
}

# Get the key to look for to identify the Web Cache webcached processes
# (includes cache, admin, excludes monitor)
# parameters:
#   oracle_home - the oracle home in which the iAS instance is running
sub getWebcacheKeyNoMonitor
{
  # Get the arguments
  my $oracle_home = shift(@_);

  return "$oracle_home/[w]ebcache/bin/webcached";
}

# Compute the resource usage of the Application Server. This information includes
# cpu and physical memory.
# parameters:
#   emdroot - the emd root directory
#   oracle_home - the oracle home in which the iAS instance is running
sub computeIasResourceUsage
{
  # Get the arguments
  my $emdroot = shift(@_);
  my $oracle_home = shift(@_);

  computeOSType();

  # on Win32, get the pids from DMS
  if( $osType eq "MSWin32" )
  {
    # Get the Http Server, webcache and oc4j pids
    my $httpPid = getApacheChildPidFromDMS( $oracle_home );
    my $webPid = getWebcachePidWin32( $oracle_home ); 
    my $pidList = getPidForComponentFromDMS( $oracle_home, "" );
    
    # Get information about all processes
    executePSCommand();
    my $cpu = 0;
    my $memory = 0;

    # put all the pids into an array
    my @pidArray = split( ",", $pidList );
    push( @pidArray, $httpPid ); 
    push( @pidArray, $webPid ); 
    foreach my $pid( @pidArray )
    {
      $cpu += getCPUUtilization( $pid )/1024;
      $memory += getMemoryUtilization( $pid )/1024;
    }
    print "em_result=$cpu|$memory\n";

  } else { # not MSWin32
    # Look for all oc4j, http server and webcache processes
    my $oc4j_key = getOc4jKey( $oracle_home, "" );
    my $http_key = getHttpServerKey( $oracle_home );
    my $web_key = getWebcacheKey( $oracle_home );

    # Look for the specified oc4j, http server and webcache processes
    getProcessInfoFromPsOutput( $emdroot, "\"$oc4j_key|$http_key|$web_key\"" );
  } # end if WSWin32
}

# Get resource usage information from process info
# parameters
#   emdroot - the emd root directory
#   grep_command - the command to identify the desired processes from ps 
sub getProcessInfoFromPsOutput
{
  my $emdroot = shift(@_);
  my $grep_command = shift(@_);
  #print "DEBUG: grep command = $grep_command\n";

  # The ps command is platform specific
  computeOSType();
  my $ps_command = getPsCommand();

  # Look for all the required processes
  # The output will be in the following form:
  # USER       PID %CPU %MEM   SZ  RSS TT       S    START  TIME COMMAND
  my $ps_output = `$ps_command | egrep $grep_command`; 
  #print "DEBUG: ps output = $ps_output\n";

  # Get the ps output lines 
  my $line;
  my @lines = split( "\n", $ps_output );
 
  # compute the cpu usage for the processes
  my $cpu_component = getCpuFromPsOutput( \@lines );

  # Get the number of the processors on the
  # machine so we can compute the true cpu time
  # For Linux, we need to divide by the number of processors on system
  # For Solaris, ps already takes the number of processors into account
  my $cpu_util = $cpu_component;
  if( $osType ne "Sol") 
  {
    my $num_cpus = getNumCpus( $emdroot );
    #print "DEBUG: number of cpus =  $num_cpus \n";
    $cpu_util = $cpu_util/$num_cpus;
  }

  # compute the memory usage for the processes and convert it from KB to MB
  my $physical_mem_kb = getMemoryFromPsOutput( $emdroot, \@lines );
  my $physical_mem;
  if( $physical_mem_kb == -1 ) {
    $physical_mem = "";
  } else {
    $physical_mem = $physical_mem_kb/1024;
  }

  # Create the results:
  print "em_result=$cpu_util|$physical_mem\n";

}

# Get the cpu usage for a list of processes
# parameters
#  ps_lines - an array of lines with ps output
#
sub getCpuFromPsOutput
{
  my $ps_lines = shift(@_);

  my $cpu = 0;

  my $i;
  my @tokens;
  for $i (0 .. $#$ps_lines)
  {
    @tokens = split( " ", $ps_lines->[$i] );
    $cpu = $cpu + $tokens[$LONG_PS_CMD_PCPU_COL];
  } # end $i

  return $cpu;
}

# Get the percentage of cpu time being used on the system.  This will return 
# the amount of cpu time used by all running processes on the system.  It does
# not take into account the number of processors on the system.  On a 
# multi-processor system, divide this value by the number of cpus.
#
# Note: executePSCommand must be called before calling this function.
sub getTotalCpuTime
{

  my $cpu = 0;

  # Walk through the information for all processes
  my $process_details;
  my @process_array;
  my $pid;
  while( ($pid, $process_details) = each %process_hash ) 
  {
     @process_array = split(" ", $process_details);
    $cpu = $cpu + $process_array[$PS_CMD_CPU_COL];
  }

  return $cpu;
}

# Get the memory usage for a list of processes.  Returns the memory in KB
# or -1 if it could not get the memory.
# parameters
#   emdroot - the emd root directory
#   ps_lines - an array of lines with ps output
#
sub getMemoryFromPsOutput
{
  my $emdroot = shift(@_);
  my $ps_lines = shift(@_);

  my $memory;
  my $i;
  my @tokens;
  my %processHash = ();
  my $pidList = "";
  my $key;

  # walk through all the pids and create a list
  for $i (0 .. $#$ps_lines)
  { 
    @tokens = split( " ", $ps_lines->[$i] );

    # On Linux the ps command returns process and threads.  But, we only want 
    # count processes, so assume that any process with the same vsz, rss and
    # command all represent the same process
    if ($osType eq "LNX") 
    {
      # use vsz**rss**command as the key
      $key = "$tokens[$LONG_PS_CMD_VSZ_COL]**$tokens[$LONG_PS_CMD_RSS_COL]**$tokens[$LONG_PS_CMD_CMD_COL]";
      if (!defined $processHash{$key})
      {
        # This is a process that we should count
        if( $pidList eq "" )
        { 
          $pidList = "$tokens[$LONG_PS_CMD_PID_COL]";
        }
        else
        {
          $pidList = "$pidList,$tokens[$LONG_PS_CMD_PID_COL]";
        }

        # add it to the hash table, so we don't found it again
        $processHash{$key} = '1';
        #print "DEBUG: pid=$tokens[$LONG_PS_CMD_PID_COL], key = $key\n";
      }
    }
    else
    { # not Linux
      # add the pid to the list
      if( $pidList eq "" )
      { 
        $pidList = "$tokens[$LONG_PS_CMD_PID_COL]";
      }
      else
      {
        $pidList = "$pidList,$tokens[$LONG_PS_CMD_PID_COL]";
      }
    } # end if Linux
  } # end $i

  # Create an array of pids and get the number usage
  my @pid_array = split( ",",$pidList );
  return getPhysicalMemoryIncludingShared( \@pid_array, $emdroot );
}

# Get the physical memory usage (KB) for a set of pids.  Make sure to account  
# for shared memory. Return -1 if we cannot compute the memory for the 
# component
#
# Parameters:
#  pid_list - an array of pids to account for
#  emdroot            - the root of the emd tree
# 
sub getPhysicalMemoryIncludingShared
{
  my $pid_list = shift(@_);
  my $emdroot = shift(@_);
  
  my $totalPrivate = 0;
  my $maxShared = 0;
  my $private;
  my $shared;
  my $checkedNmb = 0;

  computeOSType();
  if ($osType eq "LNX")
  {
     return getLinuxPhysicalMemoryIncludingShared( $pid_list );
  }

  for my $i (0 .. $#$pid_list)
  {
    # Call nmb and get the shared and private memory usage
    my $pmap_output = `$emdroot/bin/nmb $pid_list->[$i] 2>&1`;

    # Make sure we don't get an error
    if( index( $pmap_output, "Error" ) == -1 ) 
    {
      # the output will be: {private-memory-in-kb} {shared-memory-in-kb}
      ($private, $shared) = split(  " ",  $pmap_output );
    
      # Add this to our totals
      $totalPrivate += $private;
      if( $shared > $maxShared )
      {
        $maxShared = $shared;
      }      
    }
    else
    {
      # we could not parse the output, so make sure nmb is protected properly
      if( $checkedNmb == 0 ) 
      {
        if( !validateNmbProtection($emdroot) ) 
        {
          return -1;
        }
        $checkedNmb = 1;
      }
    } # end if( index( ...
  } # end foreach

  # The total memory usage is SUM(private) + MAX(shared)
  return $totalPrivate + $maxShared;

}

# Get the physical memory usage (KB) for a set of pids.  Make sure to account
# for shared memory.  This is for Linux only.
#
# Parameters:
#  pid_list - an array of pids to account for
#  emdroot            - the root of the emd tree
#
sub getLinuxPhysicalMemoryIncludingShared
{
  my $pid_list = shift(@_);

  my $totalPrivate = 0;
  my $maxShared = 0;
  my $private = 0;
  my $shared;

  for my $i (0 .. $#$pid_list)
  {
    $private = 0;
    $shared = 0;
    my $file = "/proc/$pid_list->[$i]/statm";
    if ( -e $file )
    {
      my $line = `cat $file`;
      my @fields=split(/\s+/,$line); #field separator is space
      # The output of this command is
      # {program-pages} {memory-pages} {shared-pages} {code-pages}
      # {data/stack-pages} {library-pages} {dirty-pages}
      my $STATM_MEMORY_COL = 1;
      my $STATM_SHARED_COL = 2;
      $shared = $fields[$STATM_SHARED_COL];
      $private = $fields[$STATM_MEMORY_COL] - $shared;
    }

    $totalPrivate += $private;
    if( $shared > $maxShared )
    {
      $maxShared = $shared;
    }
  } # end foreach

  # compute the page size
  my $pagesize = `getconf PAGE_SIZE` or die "Failed to run getconf PAGE_SIZE";
  chomp($pagesize);

  # The total memory usage is SUM(private) + MAX(shared)
  # since this is in pages, divide by page size and then 
  # divide by 1024 to change from bytes to KB
  my $totalMemory = ($totalPrivate + $maxShared)*$pagesize/1024;
  return $totalMemory;

}

#
# Check is the protection on nmb is correct.  It should be owned by root and 
# have the sticky bit set.  If the protect is not correct, print a warning and
# return that it is not protected correctly 
#
# Parameters:
#  emdroot            - the root of the emd tree
#
sub validateNmbProtection
{
  my $emdroot = shift(@_);

  # Get the stats for the file
  my @stat_array = stat("$emdroot/bin/nmb");

  # Make sure that it is owned by root (uid=0)
  if( $stat_array[4] == 0 ) 
  {
    # check that the sticky bit is set
    if( $stat_array[2] & S_ISUID )
    {
      # The protection is correct
      return 1;
    }
  }

  # The protection is incorrect
  print "em_warning=nmb does not have root access. Please see the release notes for information on setting the uid bit.\n";
  return 0;
}

# Get the number of CPUs for the system 
#
# Parameters:
#  emdroot            - the root of the emd tree
#
sub getNumCpus()
{
  # The linux computation is different
  computeOSType();
  if( $osType eq "LNX" )
  {
    my $cpu_stat = `grep cpu /proc/stat | tail -n +2 | wc -l`  or die "Failed to get processor information";
    return $cpu_stat;
  }

  # Execute nmupm asking for CPU information.  It will return one row per CPU
  my $emdroot = shift(@_);
  my $cpu_info = `$emdroot/bin/nmupm osCpuUsage`;
  my @lines = split("\n", $cpu_info);
  if( $#lines > 0 )
  {
    return ($#lines+1);
  }
  else
  {
    return 1;
  }
}

1;
