#!/usr/local/bin/perl
# 
# $Header: emfa/agent/scripts/EnabledOptions.pl /main/1 2011/04/29 17:45:34 vidykris Exp $
#
# EnabledOptions.pl
# 
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      EnabledOptions.pl - Enabled Options for Fusion Applications
#
#    DESCRIPTION
#      This script is used to collect the Enabled Options for FA Products and Product Families
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    vidykris    01/13/11 - Accomodate FS file location change
#    vidykris    11/19/10 - Accomodate Taxonomy file location change to ATGPF
#                           home
#    vidykris    07/21/10 - Creation
# 
use strict;
use File::Spec;
use ias::simpleXPath;

my ($appl_top,$fsFileHome,$hostName,$artifactName,$artifactType) = @ARGV;
my $uniqueIndex = 0;

my $fa_product_type = "fusion_apps_product";
my $fa_pf_type = "fusion_apps_pf";

# Get the FA Functional Setup file
my $appsFSFileName = File::Spec->catfile($fsFileHome, "fusionAppsFS.xml");

if ((-e $appsFSFileName) && (-r $appsFSFileName))
{  
  my $appsFSFile = ias::simpleXPath::parseFile($appsFSFileName);
  
  # Get a list of offerings
  my @offerings = ias::simpleXPath::queryForNodes($appsFSFile, 'ocm-collection-info/offering[@status="IMPLEMENTED"]');
  
  #Iterate through each offering and get the desired information
  foreach my $offering (@offerings)
  {
    # increase the index
    $uniqueIndex++;
    
    my $offeringName = ((ias::simpleXPath::queryForAttributes($offering, 'offering'))[0])->{'short-name'};  

    # Get a list of provisioned J2EE applications for this offering
    my @provisioned_j2ee_applications;
    my @provisioned_j2ee_app_nodes = ias::simpleXPath::queryForAttributes($offering, 'offering/application');
    foreach my $provisioned_j2ee_app_node  (@provisioned_j2ee_app_nodes) 
    {
      my $provisioned_j2ee_app_name = $provisioned_j2ee_app_node->{'short-name'};
      push (@provisioned_j2ee_applications, $provisioned_j2ee_app_name);  
    }
      
    #Get the j2ee apps for this artifact
    my @product_j2ee_apps = get_j2ee_apps($appl_top,$artifactName,$artifactType);
    
    # Now check to see if at least one element in the provisioned list is in the list of apps
    my $collect_offering = 'false';
    
    if(defined(@product_j2ee_apps))
    {
      my %product_j2ee_apps_hash = map { $_ => 1 } @product_j2ee_apps;          
      foreach my $provisioned_j2ee_application (@provisioned_j2ee_applications)
      {   
        if(exists($product_j2ee_apps_hash{$provisioned_j2ee_application}))
        {
          $collect_offering = 'true';
          last;
        }
      }
    }
    
    #This product has the j2ee app and thus should be collected
    if($collect_offering eq 'true')
    {
      # Write the offering as an em_result       
      my $id = generate_id($uniqueIndex);      
      print "em_result=$id|Offering|$offeringName|\n";
      
      # Print the options for this offering
      print_options($id, $offering);        
    } #if($collect_offering eq 'true')
  } #foreach my $offering (@offerings)
} #if ((-e $appsFSFileName) && (-r $appsFSFileName))

# subroutine print_options
# Print the Options Recursively
sub print_options
{  
  my $parent_id = $_[0];
  my $optionNode = $_[1];
  
  # Query the Options  
  my @options = ias::simpleXPath::queryForNodes($optionNode, 'offering/option');
  
  # Check to see if there's nothing returned. If yes, then we might be looking at the options node
  if(scalar(@options) == 0)
  {
    @options = ias::simpleXPath::queryForNodes($optionNode, 'option/option'); 
  }
  foreach my $option (@options)
  {
    # Options found. Generate an id and call recursively
    my $optionName = ((ias::simpleXPath::queryForAttributes($option, 'option'))[0])->{'short-name'};  
    $uniqueIndex++;
    my $optionId = generate_id($uniqueIndex);
    
    #Write the option as an em_result and recursively call print_options
    print "em_result=$optionId|Option|$optionName|$parent_id\n";    
    print_options($optionId, $option);
  }  
}

# subroutine generate_id
# Generate the id
# $_[0] - Unique ID
sub generate_id
{ 
  my ($index) = $_[0];
  my $id = $hostName . "@" . $appl_top . "@" . $index;
  return $id;
}

# subroutine get_j2ee_apps
# Get the j2ee apps for this artifact
sub get_j2ee_apps
{
  my $appl_top = $_[0];
  my $fusion_artifact_name = $_[1];
  my $fusion_artifact_type = $_[2];
  my @artifact_j2ee_apps;
  
  # Get the taxonomy file for this product
  my $appsTaxonomyFileName = get_taxonomy_file_name($appl_top, $fusion_artifact_name, $fusion_artifact_type);     
  if ((defined($appsTaxonomyFileName)) && (-e $appsTaxonomyFileName) && (-r $appsTaxonomyFileName))
  {
    my $appsTaxonomyFile = ias::simpleXPath::parseFile($appsTaxonomyFileName);
  
    # Get a list of j2ee applications for this product
    my @artifact_j2ee_app_nodes; 
    
    # Check to see if the type is a product or pf
    if($fusion_artifact_type eq $fa_pf_type)
    {      
      @artifact_j2ee_app_nodes = ias::simpleXPath::queryForAttributes($appsTaxonomyFile, 'ProductFamily/ListOfApplications/Application'); 
    }
    elsif($fusion_artifact_type eq $fa_product_type)
    {
      my $product = (ias::simpleXPath::queryForNodes($appsTaxonomyFile, 'ProductFamily/ListOfProducts/Product[@Name="' . $fusion_artifact_name . '"]'))[0];    
      @artifact_j2ee_app_nodes = ias::simpleXPath::queryForAttributes($product, 'Product/ProvidedByApplications/Application'); 
    }
    
    if(defined(@artifact_j2ee_app_nodes))
    {
      foreach my $artifact_j2ee_app_node  (@artifact_j2ee_app_nodes) 
      {
        my $artifact_j2ee_app_name = $artifact_j2ee_app_node->{'name'};
        push (@artifact_j2ee_apps, $artifact_j2ee_app_name);         
      }
    }
  }
  return @artifact_j2ee_apps;
}

# subroutine get_taxonomy_file_name
# Get the product family specific taxonomy file name
sub get_taxonomy_file_name
{  
  my $appl_top = $_[0];
  my $fusion_artifact_name = $_[1];
  my $fusion_artifact_type = $_[2];
  my $taxonomyFileName; 
  
  # Get the ATGPF home
  my ($volume, $directories, $file) = File::Spec->splitpath($appl_top);
  my @dirs = File::Spec->splitdir($directories);
  pop @dirs;
  my $fmw_home = File::Spec->catdir(@dirs);
  my $atgpf_home = File::Spec->catdir($fmw_home, 'atgpf', 'atgpf', 'applcore', 'db', 'data', 'fa_taxonomy');
  
  if(-e $atgpf_home)
  {
    # Get a list of all taxonomy files in ApplTop 
    my @appsTaxonomyFileNames = <$atgpf_home/*TaxonomyData.xml>;
    
    #Iteratively parse the files to see if there's a match
    foreach my $appsTaxonomyFileName (@appsTaxonomyFileNames)
    {
      if(-r $appsTaxonomyFileName)
      {
        my $appsTaxonomyFile = ias::simpleXPath::parseFile($appsTaxonomyFileName); 
        
        # Check to see if the type is a product or pf
        if($fusion_artifact_type eq $fa_pf_type)
        {
          # Parse the file to see if we have a matching name          
          my @pfs = ias::simpleXPath::queryForAttributes($appsTaxonomyFile, 'ProductFamily/[@Name="' . $fusion_artifact_name . '"]');
          if(scalar(@pfs) > 0)
          {
            # The right taxonomy file found. Return the file
            $taxonomyFileName = $appsTaxonomyFileName;
          }
        }
        elsif($fusion_artifact_type eq $fa_product_type)
        {
          # Parse the file to see if we have a matching name          
          my @products = ias::simpleXPath::queryForAttributes($appsTaxonomyFile, 'ProductFamily/ListOfProducts/Product[@Name="' . $fusion_artifact_name . '"]');
          if(scalar(@products) > 0)
          {
            # The right taxonomy file found. Return the file
            $taxonomyFileName = $appsTaxonomyFileName;
          } #if(scalar(@products) > 0)
        } #elsif($fusion_artifact_type eq $fa_product_type)              
      } #if(-r $appsTaxonomyFileName)
    } #foreach my $appsTaxonomyFileName (@appsTaxonomyFileNames)
  } #if(-e $atgpf_home)
  return $taxonomyFileName;
}
