#!/usr/local/bin/perl
# 
# $Header: emfa/agent/scripts/faTargetOracleHome.pl /main/1 2011/07/12 18:18:35 skasyano Exp $
#
# faTargetOracleHome.pl
# 
# Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      faTargetOracleHome.pl - This script returns oracle home of the Fusion target
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      Oracle home for PFs and products is determined from the mappings in the pfOUImapping.xml file.
#      For all other Fusion target types or in case the mapping is missing in the above file/file inaccessible, applTop
#      property value is returned.
#
#    MODIFIED   (MM/DD/YY)
#    skasyano    07/10/11 - Creation
# 

require "emd_common.pl";

use strict;
use File::Spec;
use ias::simpleXPath;

my $targetName = shift(@ARGV);
my $targetType = shift(@ARGV);
my $applTop = shift(@ARGV);

EMAGENT_PERL_DEBUG("\nStarted faTargetOracleHome script - faTargetOracleHome.pl");
EMAGENT_PERL_DEBUG("\ntargetName, targetType, applTop: $targetName, $targetType, $applTop"); 

my $oracle_home;
if (defined($targetName) && defined($targetType) && defined($applTop)) {
    if (($targetType =~ "fusion_apps_pf") || ($targetType =~ "fusion_apps_product"))
    {  
        $oracle_home = get_oracle_home($targetName, $targetType, $applTop);
        if (!defined($oracle_home))
        {
            EMAGENT_PERL_DEBUG("\nUnable to determine Oracle home using the mappings file. Check if $applTop/pfOUImapping.xml is accessible.");
            $oracle_home = $applTop;
        }
    }
    else
    {
        $oracle_home = $applTop;
    }
    
    EMAGENT_PERL_DEBUG("\nOracle home: $oracle_home");
    print "em_result=$oracle_home";
} else 
{
    EMAGENT_PERL_DEBUG("\nParameters are missing, check target properties.");
    print "em_warning=Mandatory parameters are missing: targetName=$targetName, targetType=$targetType, applTop=$applTop.";
}

# subroutine get_oracle_home
sub get_oracle_home
{
    my $oracle_home;
    my $targetName = @_[0];
    my $targetType = @_[1];
    my $applTop = @_[2];
    
    my $pfOUIMappingFilePath = "$applTop/pfOUImapping.xml";
    if (!(-e $pfOUIMappingFilePath) || !(-r $pfOUIMappingFilePath))
    {
        # Value remains undefined
        return $oracle_home;
    }
    
    # Assign default value
    $oracle_home = $applTop;
    
    my $pfOUIMappingFile = ias::simpleXPath::parseFile($pfOUIMappingFilePath);
    
    my @properties;
    if ($targetType =~ "fusion_apps_pf")
    {
        @properties = ias::simpleXPath::queryForNodes($pfOUIMappingFile, 'ProductFamilyOUIMappings/Mapping[@TYPE="product_family"][@NAME="' . $targetName . '"]/Property[@NAME="oracle_home"]');
    }
    
    if ($targetType =~ "fusion_apps_product")
    {
        @properties = ias::simpleXPath::queryForNodes($pfOUIMappingFile, 'ProductFamilyOUIMappings/Mapping/Mapping[@TYPE="product"][@NAME="' . $targetName . '"]/Property[@NAME="oracle_home"]');
    }
    
    if (defined(@properties)) {
        foreach my $property (@properties) {
            my $oracle_home = ((ias::simpleXPath::queryForAttributes($property, 'Property'))[0])->{'VALUE'};
            if (defined($oracle_home))
            {
                my @directories = File::Spec->splitdir($applTop);
                pop @directories;
                $oracle_home = File::Spec->catdir(File::Spec->catdir(@directories), $oracle_home);  
                return "$oracle_home";
            }
        }
    }
    
    return "$oracle_home";
}
