#!/usr/local/bin/perl
#
# $Header: emcore/builtin_target_types/oracle.sysman.oh/agent/scripts/OHConfigCollector.pl /main/13 2012/05/30 23:07:19 irraju Exp $
#
# OHConfigCollector.pl
#
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      OHConfigCollector.pl - Oracle Home Configuration Collector
#
#    DESCRIPTION
#     This is a connector script to the OUI and WLS APIs packages. Depending on the arguments 
#     provided and the options set, it parses relavent XML files and prints the metric results; 
#
#    NOTES
#     Collection logs will be written to the emd_perl.trc file
#     in the log directory. Users can specify their own log directory by either
#     1. setting ENV variable {EMAGENT_PERL_TRACE_DIR}
#     2. specifying it as an argument with option --logdir or as a fifth argument if not 
#        using options.
#     If {EMAGENT_PERL_TRACE_DIR} is not set and logDir is not specified in the arguments, 
#     then logs will
#     be written to the {EMSTATE} and if {EMSTATE} doesn't exist
#     then to {ORACLE_HOME}/syman/log
#     This program takes ORACLE HOME LOCATION, MIDDLEWARE/CENTRAL INVENTORY LOCATION and METRIC
#     NAME and HOMETYPE. If the HOMETYPE is not specified, it is considered as OUI.
#     If for the oui home type if the inventory location is not speficied, then it is assumed
#     to be default central inventory.
#
#    MODIFIED   (MM/DD/YY)
#    irraju      05/14/12 - primary columns for duplicate elimination
#    irraju      02/28/12 - enabling pluign collections even with duplicates
#    irraju      02/03/12 - collect configured plugins only
#    hmodawel    12/22/11 - 12c agent compatible
#    hmodawel    12/13/11 - getOptions handle optional params
#    hmodawel    12/11/11 - collect data from plugin homes
#    hmodawel    12/08/11 - add support for multiple jars
#    hmodawel    11/12/11 - handle nonoui
#    hmodawel    04/21/11 - fix return value
#    chkaushi    03/11/11 - Changing HOME_BASE to MW_HOME
#    ravpatel    01/21/11 - CCR Convergence
#    irraju      03/28/10 - Collection script that calls various metrics after
#                           validation
#    irraju      03/28/10 - Creation
#


## USE DIRECTIVES
 use strict;
 use Cwd 'abs_path';
 use File::Basename;
 use File::Spec();
 use File::Spec::Functions;
 use File::Path;
 use ValidArgs;
 use FileHandle;
 use OH_OUI;
 use OH_WLS;
 use Getopt::Long;
 use OH_Error;
 use OH_ConfigClasses;
 use OH_Utilities;

use emdcommon_ocm;

## VARIABLES
#create error, warnings and error objects for fetching metric info
    my $error      = new Error();#Error Object
    my $warnings   = new Warn(); # Warnings Object
    my $metricInfo = new OHConfigMetric();# metric data object
## VALIDATIONS
    # -- mandatory arguments
    my ($homeLoc, $metricName,$homeType,$inventory, $mwhome);
    # -- parse the options passed 
    my $resultOfCLArgs = GetOptions(
                     'help'        => \&helpMessage,
                     'home=s'      => \$homeLoc,
                     'metric=s'    => \$metricName,
                     'type=s'      => \$homeType,
                     'inventory:s' => \$inventory,
                     'mw_home:s' => \$mwhome);

    # -- if something is wrong with the options then print the help message
    if((!$resultOfCLArgs) )
    {
      # error parsing the arguments
      $error->setError(CODE  => OH_Error::INVALID_ARGUMENTS); 
      # print the usage and exit
      helpMessage();
    }

    #TODO: Do we need this ????
    # -- if options are not used but arguments are provided in the requested order then set the arguments
    $homeLoc    =  $ARGV[0]  unless (defined($homeLoc   ));
    $metricName = ($ARGV[1]) unless (defined($metricName));#case insensitive
    $homeType   = ($ARGV[2]) unless (defined($homeType  ));#case insensitive
    $inventory  =  $ARGV[3]  unless (defined($inventory ));
    $mwhome     = $ARGV[4] unless (defined($mwhome )) ;    
    # all arguments are passed?
    if(!defined($homeLoc)    or
       !defined($metricName) or
       !defined($homeType))  #others are optional, We'll flag appropriate error later if collection cant proceed 
    {
      # invalid option
      $error->setError(CODE  => OH_Error::MAND_ARG_MISSING);
      helpMessage();#print usage and exit
    }

    # -- convert the arguments to lower case to ease the comparison
    $metricName = lc($metricName);
    $homeType   = lc($homeType);

    # -- print the arguments
    emdcommon_ocm::EMD_PERL_DEBUG("Arguments passed HOME = $homeLoc METRIC = $metricName HOME TYPE = $homeType INVENTORY = $inventory MW HOME = $mwhome");

    # -- check if valid type
    # -- use a hash to validate 
    #TODO: Commenting out for now. Need to discuss if its required.
#    if(!$isThisHomeTypeValid{$homeType})
#     {
       # invalid home type
#       $error->setError( CODE  => $OH_Error::INVALID_HOME_TYPE,
#                         ARG   => $homeType);
#       helpMessage();
#     }
    # -- validate arguments provided
    if($homeType eq "o")
    {
      if(!$isThisOuiMetricValid{$metricName})
       {
         # invalid metric
         $error->setError( CODE  => OH_Error::INVALID_METRIC_NAME,
                           ARG   => $metricName);
         helpMessage();
       }
    }
#    else
#    {
#     if(!$isThisWlsMetricValid{$metricName})
#      {
#         # invalid metric
#         $error->setError( CODE  => OH_Error::INVALID_METRIC_NAME,
#                           ARG   => $metricName);
#         helpMessage();
#      }
#    }

    # -- file permissions and existence are checked by OUI module

## CALL THE OUI API and GET THE METRIC INFO

    # -- create an OUI/WLS object depending on the home type
      # -- if OUI create OUI object else WLS object
      my $collectionObj;
      my @refHomes;
      my %globalMetricData = undef;# Hash to store the metric data across plugins.
      if(!($homeType eq "o")) #non-OUI homes
      {
          # Launch class oracle.sysman.core.agent.ohinv.engine.OHConfigCollector
          # The collector passes JAVA_HOME as CCR_JAVA_HOME
          #my $JAVA_HOME = $ENV{'JAVA_HOME'};
          #if (!$JAVA_HOME) {
          #  $JAVA_HOME = $ENV{'JRE_HOME'};
          #}
          my $JAVA_HOME = $ENV{'CCR_JAVA_HOME'};
          if (!$JAVA_HOME) {
            $JAVA_HOME = $ENV{'JAVA_HOME'}; # in case we decide to run in EMAgent in future
          }
          # Lot of hardcoding of names/paths here :-(, ideally could have been defined as constants
          # Lack of time as always. Hopefully these will not change
          my @command = (File::Spec->catfile($JAVA_HOME, 'bin', 'java'));
          my ($script, $scriptDir) = fileparse($0);
          $scriptDir = OH_Utilities::removeTrailingSlash($scriptDir);
          my ($sdir, $pluginRoot) = fileparse($scriptDir);
          my $jlibDir = File::Spec->catfile($pluginRoot, 'intg', 'jlib');
          my $classPath = File::Spec->catfile($jlibDir, 'em-core-ohagent-pojo.jar');
          my $cpSep = (($^O eq "Windows_NT") || ($^O  eq "MSWin32")) ? ';' : ':'; 
          
          my $intgJarClass = getIntgJar($homeType, $pluginRoot);
          if ($intgJarClass eq '') {
            emdcommon_ocm::EMD_PERL_ERROR("No jar class found for home type ".uc($homeType));
            exit (OH_Error::SUCCESS);
          } 
          my ($intgJar, $className) = split('\|', $intgJarClass);
          emdcommon_ocm::EMD_PERL_DEBUG("Found jar ".$intgJar."class ".$className."for home type ".uc($homeType));
          # if there are multiple jars separated by ',', add all of them to class path
          my @jars = split(',', $intgJar);
          foreach my $jar (@jars) {
             my $jarPath = File::Spec->catfile($jlibDir, $jar);
             $classPath = $classPath.$cpSep.$jarPath;
          }
          #Some collections like WLS are dependent on MW_HOME/utils/bsu/patch-client.jar -- NOT REQUIRED an more
          #my $bsu_jar = File::Spec->catfile($mwhome, "utils", "bsu", "patch-client.jar") if defined($mwhome);
          #$classPath = $classPath.$cpSep.$bsu_jar if defined ($bsu_jar);

          @command = (@command, '-Xmx512M', '-cp', $classPath);
          @command = (@command, 'oracle.sysman.core.agent.ohinv.engine.OHConfigCollector');
          @command = (@command, "--home", $homeLoc, "--type", uc($homeType), "--metric", uc($metricName), "--class", $className);
          emdcommon_ocm::EMD_PERL_DEBUG("Executing for HomeType ".uc($homeType). "---> ".join(' ', @command));
          #print join(' ', @command)."\n";
          my $returnVal = system(join (' ', @command));
          #TODO : check return val and log message
          $returnVal = $returnVal<<8;
          emdcommon_ocm::EMD_PERL_DEBUG("Completed execution for HomeType ".uc($homeType). " Return Value ", $returnVal);
          exit(OH_Error::SUCCESS);
      }
      else 
      {
          emdcommon_ocm::EMD_PERL_DEBUG("This is an OUI home so creating an OUI Object");
          #validate and then create the OUI object 
          $collectionObj= OH_OUI->new(  home        => $homeLoc,
                                        inventory   => $inventory,
                                        error_obj   => $error );

          # -- Collection Object will have undef on failure and OH_OUI object 
          # on success.
          if (!defined($collectionObj)) # error while creating new obj
          {
             # validataion of the argumens has failed
             # Read the info about the error from error object
             &printError($error);
             exit(OH_Error::SUCCESS);  # exit now
          }

          # Collection object has been created - carry on with the collection
          # -- get the subroutine to be called for this metric
          my $subRoutine = $ValidArgs::subRoutineFor{$metricName};
          emdcommon_ocm::EMD_PERL_DEBUG("calling $subRoutine on Collection object");

          # -- get the metric
          my $retVal = $collectionObj->$subRoutine( metric_info => $metricInfo,
                                                    warnings    => $warnings );
          # if collection has errors throw 
          if($retVal) #error occured
          {
            &printMetricResult($warnings->getWarnings(),"em_warning=","|");
            &printError($error);
            exit(OH_Error::SUCCESS);
          }
          # -- print result if no errors are found
          emdcommon_ocm::EMD_PERL_DEBUG("Printing $metricName metric result for $homeLoc ");
          &printMetricResult($metricInfo->getRows(),"em_result=","|");

          # Now collect information from plugin Homes (for Comps, Patches, Patch_comps, patch_bugs)
          if ($ValidArgs::isPluginHomeMetricValid{$metricName})
          {
             @refHomes = &getRefHomeList($collectionObj);
             foreach my $refhome (@refHomes)
             { 
                 # If we are here, that means there are pluginhomes for this core Oracle Home.
                 # back up the previous metric info data to check for duplicates later 
                 
                 # At this point globalMetricData is superset of all the metricdata from all the plugins
                 # and metricInfo from the previous collection is already been cleaned of duplicates. 
                 &updateGlobalMetricData(globalMetricData => \%globalMetricData, previousMetricData => $metricInfo);
                 $collectionObj= OH_OUI->new(  home => $refhome,
                                        inventory   => $inventory,
                                        error_obj   => $error );
                
                 if (defined($collectionObj))
                 {
                     emdcommon_ocm::EMD_PERL_DEBUG("calling $subRoutine on Collection object for $refhome");
                     # -- get the metric
                     my $retVal = $collectionObj->$subRoutine( metric_info => $metricInfo,
                                                    warnings    => $warnings );
                     # if collection has errors throw 
                     if($retVal) #error occured
                     {
                       &printMetricResult($warnings->getWarnings(),"em_warning=","|");
                       &printError($error);
                     }
                     else 
                     {
                       emdcommon_ocm::EMD_PERL_DEBUG("Printing $metricName metric result for $refhome ");
                       #remove the duplicates if any considering the metric data from all the plugin homes
                       &removeDuplicatesAcrossPluginHomes(globalMetricData =>\%globalMetricData, currentMetricData => $metricInfo);
                       &printMetricResult($metricInfo->getRows(),"em_result=","|");
                     }
                 }
             }
         }
    }
    
    exit (OH_Error::SUCCESS);


#***** SUB ROUTINES************************************************************
#
#
# If there are errors on command line usage, print error first and then print the Usage
sub helpMessage
{
  &printError($error);
  print  "Usage: OHConfigCollector.pl";
  print  " --home <HOME LOCATION> --metric <METRIC NAME> --type <HOME TYPE> --inventory <INVENTORY LOC> --mw_home <MW HOME>\n";
  print  "\t\t HOME LOCATION is obsolute Oracle Home path \n";
  print  "\t\t METRIC NAME can be any of the following (case insensitive) \n";
  print  "\t\t\t HOME_INFO for Oracle Home information metric \n";
  print  "\t\t\t COMPS for component metric \n";
  print  "\t\t\t TL_COMPS for top level components \n";
  print  "\t\t\t COMP_DEPS for component dependency metric \n";
  print  "\t\t\t PATCHES for patches(oneoffs and PSUs) \n";
  print  "\t\t\t PATCH_BUGS for bugs fixed by patches(oneoffs and PSUs) \n";
  print  "\t\t\t PATCH_COMPS for comps affected by patches(oneoffs and PSUs) \n";
  print  "\t\t\t PATCH_FILES for files affected by patches \n";
  print  "\t\t\t PATCHSETS for patchsets applied \n";
  print  "\t\t\t VER_PATCHES for version patches \n";
  print  "\t\t\t INST_TYPES for install types of the components \n";
  print  "\t\t\t HOME_GUID for OUI Home Guid of Oracle Home \n";
  print  "\t\t\t ARU_ID for aru id of Oracle Home \n";
  print  "\t\t\t DEP_HOMES for dependent Oracle Home list \n";
  print  "\t\t\t CRS_NODES for list of CRS nodes \n";
  print  "\t\t\t CLONE_PROPS for list of clone properties \n";
  print  "\t\t HOME TYPE can be any of the following: case insensitive \n";
  print  "\t\t\t O for the OUI Installed homes \n";
  print  "\t\t\t W for the WLS homes \n";
  print  "\t\t INVENTORY LOC is the central inventory in case of OUI homes\n";
  print  "\t\t MW HOME is Middleware home in case of wls\n";
  print  "TIP: If not using the options, arguments should be provided in the following order\n";
  print  "Usage: OHConfigCollector.pl <HOME LOCATION> <METRIC NAME> <HOME TYPE> <INVENOTRY/MIDDLEWARE HOME> \n";
 
  exit ($error->{CODE}); # letting this be there as it will help catch metric executions in case of typos
}
#print "\t Directory to which logging should be done can be set by setting the ENV variable EMAGENT_PERL_TRACE_DIR. Maximum size of the log file can be set by setting EMAGENT_PERL_TRACE_FILESIZE\n" ;



sub getIntgJar
{
  my($homeType, $pluginRoot) = @_;
  # Need to put logic to fetch jar name based on home type
  #my $jarName = 'em-core-ohJrockit-pojo.jar';
  my $intgCollection = File::Spec->catfile("$pluginRoot", "intgcollection.lst");
  my $jarName;
  my $className;
  my $homeTypeInFile;

  if (open(INTG,  "<$intgCollection"))
  {
    while(<INTG>) {
            chomp;
            ($homeTypeInFile, $jarName, $className) = split('\|');
            if (lc($homeType) eq lc($homeTypeInFile))
            {
              close (INTG);
              return ($jarName && $className) ? join('|', $jarName, $className) : '';
            }
    }
    close (INTG);
    return (''); # no matching entry found for home type
  }
  return (''); # file not found
}


#Get an array of reference Homes for this home
sub getRefHomeList
{
  my $invObj = shift;
  my @refHomes;
  my $refHomeCount = 0; 

  if(!defined($invObj->{INVENTORY_DATA}))#if not collected already
  {
    my $failed = OH_OUI::collectInvXml($invObj);
    if($failed)
    {
      #collection of inventory.xml failed
      return $failed;
    }
  }
  my $refHomeList = $invObj->{INVENTORY_DATA}->{REF_HOMES};
  if (defined($refHomeList))
   {
     foreach my $index( sort keys %$refHomeList )
     {
        #Add the ref  home only if it is not  unconfigured.
        if(isRefHomeConfigured($refHomeList->{$index}))
        {
           $refHomes[$refHomeCount++] =  $refHomeList->{$index};
        }
     }
   }
   return @refHomes;
}



#***************SUB printMetricResult******************************************
# Takes the arguments
#        1) MetricResult object
#        2) Prefix for metric row result
#        3) Prefix for metric warning
#        4) Delimiter
#
#
sub printMetricResult
{
  my $metricData   = shift;
  my $prefix       = shift;
  my $delimiter    = shift;
  # print Result rows
  foreach my $index (sort keys %{$metricData})
   {
     my $rowRef = $metricData->{$index};
     my $class     = ref($rowRef);
     if($class)
     {
      no strict "refs";
      my @cols = $class->getColumns();
      my %row = %{$rowRef};
      $delimiter = "|" unless defined ($delimiter);
      print "$prefix".join($delimiter,@row{@cols})."\n";
     }
     else
     {
       #some scalar values which doesn't require a class 
       print("$prefix".join($delimiter,$rowRef)."\n");
     }
   }

}


#********printError()**********************************************************
sub printError
{
 my $errorObj = shift;
 if(defined ($errorObj->{CODE}))
 {
   my $str = $errorObj->getErrorString();
   emdcommon_ocm::EMD_PERL_ERROR("$str");
   # don't print as it goes to stdin of GC and OCMagent
   # print "$str \n";
 }
}

sub isRefHomeConfigured
{
  my $isPluginHomeConfigured = 1;
  my $pluginHomeLoc = shift;
  my $OHPXmlFile =  File::Spec->catfile($pluginHomeLoc,OH_OUI::OH_PROP_XML);
  my $OHPObject  = OH_ParseOHPropXml->new(XML_FILE => $OHPXmlFile);
  my $plugInError      = new Error();#Error Object
  my $plugInWarnings   = new Warn();
  if(defined($OHPObject))
  {
    my $retVal = $OHPObject->getOHProperties($plugInWarnings,$plugInError);
    if($retVal)#failed
     {
       #no point in collecting the plug-in home
       $isPluginHomeConfigured = 0;
     }
    else
    {
      my $contentType = OH_OUI::getOracleHomeProperty("ContentType", $OHPObject);
      # ref homes like JDK will not have content type at all
      if((!defined($contentType)) or ($contentType =~ /Unconfigured/i))
       {
         $isPluginHomeConfigured  = 0;
       }
    }
  }
  return $isPluginHomeConfigured;
} 

sub updateGlobalMetricData
{
  my %args = @_;
  my $globalMetricDataHash   = $args{globalMetricData};
  my $currentMetricDataHash  = $args{previousMetricData}->getRows();
  my $delimiter = '|';

  # Global Metric Data Hash will not have any duplicates (property of hash)
  # CurrentMetricData Hash will not contain duplicates as it is cleaned while printing
  foreach my $index (sort keys %{$currentMetricDataHash})
   {
     my $rowRef = $currentMetricDataHash->{$index};
     my $class     = ref($rowRef);
     if($class)
     {
      no strict "refs";
      my @cols = $class->getPrimaryColumns();
      my %row = %{$rowRef};
      $globalMetricDataHash->{join($delimiter,@row{@cols})} = 1;
     }
   }


}

sub removeDuplicatesAcrossPluginHomes
{
  my %args = @_;
  my $globalMetricDataHash  = $args{globalMetricData};
  my $currentMetricDataHash = $args{currentMetricData}->getRows();
  my $delimiter = '|';
  
  # form the key out of all the data in the row
  # if its a duplicate, just log it and remove from the metric data
 foreach my $index (sort keys %{$currentMetricDataHash})
   {
     my $rowRef = $currentMetricDataHash->{$index};
     my $class     = ref($rowRef);
     if($class)
     {
       no strict "refs";
       my @cols = $class->getPrimaryColumns();
       my %row = %{$rowRef};
       if($globalMetricDataHash->{join($delimiter,@row{@cols})})
       {
         #duplicate found.. log it and remove
         emdcommon_ocm::EMD_PERL_DEBUG("Duplicate entry found : join ($delimiter,@row{@cols})");
         delete($currentMetricDataHash->{$index});
       }
     }
   }

}  
