#!/usr/local/bin/perl
# 
# $Header: emcore/builtin_target_types/oracle.sysman.oh/discovery/OracleHomeDiscovery.pl /main/14 2011/12/27 00:12:29 hmodawel Exp $
#
# OracleHomeDiscovery.pl
# 
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      OracleHomeDiscovery.pl - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    hmodawel    12/23/11 - 12c agent compatible
#    hmodawel    11/19/11 - call plugin discovery scripts 
#    hmodawel    11/16/11 - use host from Agent
#    hmodawel    10/19/11 - re-arrange the code
#    ravpatel    06/22/11 - Disabling HOME_NAME property
#    ravpatel    06/06/11 - Fix for null home name issue
#    ravpatel    05/18/11 - Ignore case AGENT_MODE
#    hmodawel    03/29/11 - handle bea home variants
#    chkaushi    03/11/11 - Correcting Target Name
#    ravpatel    08/04/10 - Including Agent inventory discovery
#    ravpatel    06/24/10 - Removing Spaces from Target Names
#    ravpatel    04/09/10 - Oracle Home Discovery Script
#    ravpatel    04/09/10 - Creation
#

use strict;
use warnings;
use File::Basename;
use File::Spec();
use File::Spec::Functions;
use File::Path;
use Fcntl qw(:DEFAULT :flock);
use Net::Domain qw(hostname hostfqdn hostdomain);

use OH_Discovery;
use OH_InventoryOrCompositeHome;
use OH_Error;

main (@ARGV);

#--
sub main
{
    my $thisScript = $0;

    my @args = @_;
    #As per discovery team, the agent home and hostname are passed to discovery scripts in this order
    my $agentHome = $args[0] ? $args[0] : $ENV{'ORACLE_HOME'};
    my $hostName = $args[1];
    $hostName = OH_Discovery::HOSTNAME_LONG if (!defined($hostName));

    my $emState  = $ENV{'EMSTATE'};
    my $agentMode = lc($ENV{'AGENT_MODE'});

    # create a warnings object
    my $warnings = new Warn();

    my $ocm = (($agentMode) && ($agentMode eq 'ocm')) ? 1 : 0;
    # For Discovery in CCR
    my $me_loc   = OH_Utilities::removeTrailingSlash($ENV{'ME_LOC'});
    my $me_type  = $ENV{'ME_TYPE'};
    my $fixInv  = $ENV{'FIXINV'} ? $ENV{'FIXINV'} : "No";
    my $isBea = 0;

    if($ocm)
    {
      my $beahome = $ENV{'BEA_HOME'} ? $ENV{'BEA_HOME'} : $ENV{'BEAHOME'};
      if ($beahome)
      {
        $me_loc = $agentHome = $emState = OH_Utilities::removeTrailingSlash($beahome);
        $me_type = 'mwh';
        $isBea = 1;
        # trigger the integrator scripts
        my ($scriptsName, $scriptsDir) = fileparse($thisScript);
        $scriptsDir = "." if (!defined($scriptsDir));
        my $pluginDir =  File::Spec->catfile("$scriptsDir", "intg", "scripts");
        my $intgDiscover = File::Spec->catfile("$scriptsDir", "intgdiscover.lst");
        if (open(INTG,  "<$intgDiscover"))
        {
          while(<INTG>) {
            chomp;
            my ($homeType, $intgScript) = split('\|');
            my $path2PluginScript = File::Spec->catfile($pluginDir, $intgScript);
            require($path2PluginScript);
            printTarget($beahome, $hostName, $homeType);
          }
          close(INTG);
        } # else intgdiscover.lst not founf in OH discovery home
      }
      elsif ($ENV{'LL_ORACLE_HOME'})
      {
        $me_loc = $agentHome = $emState = OH_Utilities::removeTrailingSlash($ENV{'LL_ORACLE_HOME'});
        $me_type = 'oh';
      }
      else # do nothing
      {
        return;
      }
    }

    my $discovery = OH_Discovery->new(EMDROOT=>$agentHome,
                                     EMSTATE=>$emState,
                                     WARNINGS=>$warnings);


    fixInventory ($discovery, $me_loc, $me_type) if ($fixInv eq "Yes");

    $discovery->setME(ME_LOC => $me_loc, ME_TYPE => $me_type);

    my $OHTargets_ref = $discovery->getHomeTargets();
 
    
    if ($ocm)
    {
      foreach my $target_name (sort keys %$OHTargets_ref)
      {
        my $targetProp_ref = $OHTargets_ref->{$target_name};

#        $targetProp_ref->{OLD_NAME} = (($targetProp_ref->{HOME_NAME}) && !$isBea) ? $targetProp_ref->{HOME_NAME} : $targetProp_ref->{INSTALL_LOCATION};
        my $oldName = (($targetProp_ref->{HOME_NAME}) && !$isBea) ? $targetProp_ref->{HOME_NAME} : $targetProp_ref->{INSTALL_LOCATION};
        if($isBea)
        {
          $targetProp_ref->{OcmProxyOracleHome} = $targetProp_ref->{INSTALL_LOCATION};
        }
        
       # we are deliberately using old naming convention for OCM because of backward compatibility issues,
       # and this may be required to be changed later. 
       # At that point of time we'd need to un-comment the the code that adds OLD_NAME property to targetProp_ref.        
 
        printOHTarget($oldName, $targetProp_ref, $hostName, $isBea);
      }
    }
    else
    {
      print "<Targets>\n";

      foreach my $target_name (sort keys %$OHTargets_ref)
      {
        my $targetProp_ref = $OHTargets_ref->{$target_name};
        printOHTarget($target_name, $targetProp_ref, $hostName);
      }
      printDiscoveryWarnings($warnings);
      print "</Targets>\n";
    }
}

sub fixInventory
{
    my $discovery = shift;
    my $me_loc = shift;
    my $me_type = shift;

    my $emState = $discovery->{EMSTATE};
    my $warnings = $discovery->{WARNINGS};

    if($me_loc && $me_type)
    {
      $me_loc = OH_Utilities::removeTrailingSlash($me_loc);

      if($me_type eq "oh")
      {
        if(OH_Utilities::isValidOUIHome($me_loc))
        {
          my $homeListRef = $discovery->getHomes();
          if ($homeListRef && (!(exists $homeListRef->{$me_loc})))
          {
            attachHome($emState, $me_loc, $me_type, $warnings);
          }
        }
        else
        {
          my $compXml = File::Spec->catfile($me_loc, "inventory", "ContentsXML", "comps.xml");
          $warnings->addWarning("$me_loc doesn\'t seem to be a valid Oracle Home. Please verify that file $compXml exists.");
        }
      }
      elsif($me_type eq "ch")
      {
        my $CHListRef = $discovery->getCompositeHomes();

        if(OH_Utilities::verify_Entity_Not_in_List($me_loc,$CHListRef))
        {
          attachHome($emState, $me_loc, $me_type, $warnings);
        }
      }

      elsif ($me_type eq "inv")
      {
        my $invListRef = $discovery->getInventories();

        if(OH_Utilities::verify_Entity_Not_in_List($me_loc,$invListRef))
        {
          my $invObj = new OH_InventoryOrCompositeHome( LOC=>$me_loc, WARNINGS=>$warnings);

          if ($invObj->getHomes())
          {
            &attachInventory($emState, $me_loc, $warnings);
          }
        }
      }
      elsif($me_type eq "mwh")
      {
        my $mwhListRef = $discovery->getMWHomes();

        if(OH_Utilities::verify_Entity_Not_in_List($me_loc,$mwhListRef))
        {
          if (OH_Utilities::isValidMWHome($me_loc))
          {
            &attachMWHome($emState, $me_loc, $warnings);
          }
          else
          {
            my $registry_xml = File::Spec->catfile($me_loc, "registry.xml"); 
            my $registry_dat = File::Spec->catfile($me_loc, "registry.dat");  
            $warnings->addWarning("$me_loc is not a valid MWHome, Please verify that $registry_dat or $registry_xml exists and agent has read permission on it");
          }
        }
      }
    }

    $discovery->reset();
}

sub attachHome
{
  my $emState = shift;
  my $home_loc = shift;
  my $type = shift;
  my $warnings = shift;

  my $oraInstLoc = File::Spec->catfile($home_loc, "oraInst.loc");

  if (-e $oraInstLoc)
  {
    my $inv = OH_Utilities::read_oraInst_loc($oraInstLoc, $warnings);
    if ($inv)
    {
      my $invObj = new OH_InventoryOrCompositeHome(LOC => $inv, WARNINGS => $warnings);
      my $homesRef = ($type eq "ch") ? $invObj->getCompositeHomes() : $invObj->getHomes();

      if (exists $homesRef->{$home_loc})
      {
        # That means the home is correctly attached to the inventory but the inventory is not discoverable
        return &append_to_OUIInventories_add($emState, $warnings, "inventory: $oraInstLoc");
      }
      else
      {
        # The home is not attached to inventory, ask the user to attach it
        &print_oui_attachHome_command($home_loc, $type, $warnings, $oraInstLoc);
        &append_to_OUIInventories_add($emState, $warnings, "inventory: $oraInstLoc");
        return 0;
      }
    }
  }
  # attach home to central Inventory
  &print_oui_attachHome_command($home_loc, $type, $warnings);
  return 0;
}


#*********************************************************************
sub print_oui_attachHome_command
{
  my $home_loc = shift;
  my $type     = shift;
  my $warnings = shift;
  my $oraInstLoc = shift;

  my $agentHome = $ENV{'ORACLE_HOME'};

  my $oui = File::Spec->catfile($agentHome, "oui", "bin", "runInstaller" );
  my $command = "$oui -silent -attachHome ";
  if($type eq "oh")
  {
    $command = $command." ORACLE_HOME=\"$home_loc\" ORACLE_HOME_NAME=\"Home Name\" ";
  }
  else
  {
    $command = $command." COMPOSITE_HOME=\"$home_loc\" ";
  }
  $command = $command." -invPtrLoc $oraInstLoc" if $oraInstLoc;

  $warnings->addWarning("$home_loc is a stray home. Please run following command on the host to attach it with an inventory : \'$command\'");
}

#*********************************************************************
sub attachInventory
{
    my $emState = shift;
    my $invLoc = shift;
    my $warnings = shift;

    my $oraInstLoc = File::Spec->catfile($invLoc, "oraInst.loc");

    if((-e $oraInstLoc) && ($invLoc eq OH_Utilities::read_oraInst_loc($oraInstLoc, $warnings)))
    {
      return &append_to_OUIInventories_add($emState, $warnings, "inventory: $oraInstLoc");
    }

    my $emStage = File::Spec->catfile($emState, "EMStagedPatches");

    if(!(-e $emStage))
    {
      mkdir $emStage || die "Could not create dir $emStage. Reason: $!";
    }
    my ($sec, $min, $hr, $day, $month, $year) = (localtime) [0,1,2,3,4,5];
    $month += 1;
    $year += 1900;
    $oraInstLoc = File::Spec->catfile($emStage, "oraInst.loc.".$year.$month.$day."_".$hr.$min.$sec);

    if (open (ORAINST_LOC, ">$oraInstLoc"))
    {
      print  ORAINST_LOC "inventory_loc=$invLoc\n";
      close(ORAINST_LOC);

      return &append_to_OUIInventories_add($emState, $warnings, "inventory: $oraInstLoc");
    }
    else
    {
      $warnings->addWarning("Could not create file $oraInstLoc. Reason: $!");
      return 0;
    }
    return 0;
}

#*********************************************************************
sub attachMWHome
{
    my $emState = shift;
    my $MWHome  = shift;
    my $warnings = shift;

    my $beaHomesList;

    my $OS = $^O;

    if (($OS eq "Windows_NT") || ($OS  eq "MSWin32"))
    {
      my $beaHomeFile = File::Spec->catfile($ENV{'SYSTEMDRIVE'}, "bea", "beahomelist");

      if(open(BEAFILE,"<$beaHomeFile"))
      {
        #Try to acquire a shared read lock
        if(!(flock(BEAFILE,LOCK_SH)))
        {
          $warnings->addWarning("Couldn't acquire read-lock on $beaHomeFile, Please check if the file is in consistent state.");
        }

        my @beaHomesListing = <BEAFILE>;
        close(BEAFILE);
        $beaHomesList = join ('',@beaHomesListing);
        $beaHomesList =~ s/\n//gs;
        $beaHomesList =~ s/(\s)*\;(\s)*/\;/gs ;


        if(open(BEAFILE,">>$beaHomeFile"))
        {
          if(!(flock(BEAFILE,LOCK_EX)))
          {
            $warnings->addWarning("Couldn't acquire write-lock on $beaHomeFile, Please check if the file is in consistent state.");
          }
          if((($beaHomesList =~ m/;$/) or ((!$beaHomesList) or ($beaHomesList =~ m/^\s*$/))))
          {
            print BEAFILE "$MWHome" ;
          }
          else
          {
            print BEAFILE ";$MWHome" ;
          }
          close(BEAFILE);
          return 1;
        }
        $warnings->addWarning("Could not open $beaHomeFile for append. Reason: $!");
        return 0;
      }
      $warnings->addWarning("Could not open $beaHomeFile for reading. Reason: $!");
      return 0;
    }
    else
    {
      return &append_to_OUIInventories_add($emState, $warnings, "middleware home: $MWHome");
    }
    return 0;
}


#*********************************************************************
sub append_to_OUIInventories_add
{
    my $emState = shift;
    my $warnings = shift;
    my $line    = shift;

    my $invAddFile = File::Spec->catfile($emState,'sysman','config','OUIinventories.add');
    if(open ( INV_FILE , ">>$invAddFile" ))
    {
      if(!(flock(INV_FILE, LOCK_EX)))
      {
        $warnings->addWarning("Couldn't acquire write-lock on $invAddFile, Please check if the file is in consistent state.");
      }
      print INV_FILE "\n$line\n";
      # close the file, it will also release the lock
      close (INV_FILE);
      return 1;
    }
    $warnings->addWarning("Could not open $invAddFile for append. Reason: $!");
    return 0;
}



sub printOHTarget
{
  my $target_name = shift;
  my $ref = shift;
  my $hostName = shift;
  my $isbea = shift;

  my $home_loc = $ref->{INSTALL_LOCATION};
  my $hostnameshort = OH_Discovery::HOSTNAME_SHORT;
  my $displayName;

  if ($ref->{HOME_NAME})
  {
    $displayName = $ref->{HOME_NAME}."_$hostnameshort";
  }
  else
  {
    my ($fname, $path) = fileparse($home_loc);
    $displayName = $fname."_$hostnameshort";
  }

  print "\t<Target TYPE=\"oracle_home\" NAME=\"$target_name\" DISPLAY_NAME=\"$displayName\""
       .($isbea ? " ON_HOST=\"$hostName\"": "") . ">\n";

  foreach my $prop (sort keys %$ref)
  {
    # Please note that HOME_NAME is not a target property anymore. It has been passed till here
    # only to be used in target name and display name.
    print "\t\t<Property NAME=\"$prop\" VALUE=\"$ref->{$prop}\" />\n" if ($prop ne "HOME_NAME");
  }

  print "\t</Target>\n";
}

sub printDiscoveryWarnings
{
  my $warnings  = shift;
  my $warnsRef = $warnings->getWarnings();

  my $hostname = hostfqdn();
  foreach my $index(keys %$warnsRef)
  {
    print "\t<DiscoveryWarning DISCOVERY_SCRIPT=\"$hostname : OracleHomeDiscovery\">";
    print $warnsRef->{$index};
    print "</DiscoveryWarning>\n";
  }
}

