#!/bin/sh
#
# $Header: emll/install/core/config.sh /main/44 2010/02/15 09:18:07 asunar Exp $
#
# config.sh
#
# Copyright (c) 2005, 2010, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      config.sh - CCR Package config software script
#
#    DESCRIPTION
#      This script is invoked by the Oracle CCR package deployment
#      process and is used to configure the software package.
#
#    NOTES
#      This script is written with the assumption that the error of
#      any command will terminate unless the error is being explicitly
#      handled.
#
#    MODIFIED   (MM/DD/YY)
#    asunar      02/01/10 - Matching the crontab entries by making the
#                           JAVA_HOME value as optional in grep
#    aghanti     10/08/09 - Reset shell flags only if there were any initially
#    aghanti     10/08/09 - Removal of dependency on xpg4 utilities
#    ckalivar    08/13/09 - Fix Bug-8366683: disable traps before checking
#                           crontab access in warnIfCronDisallowed
#    ckalivar    08/13/09 - Fix bug-8781449: disableExitonError and 
#                           resetShellflags are made inline for all ports.
#    ckalivar    06/24/09 - Fix Bug:8366683- warnIfCronDisallowed is added to
#                           check crontab accesss before doing anything with
#                           crontab
#    jsutton     05/02/08 - fix grep expression for crontab entries
#    ndutko      04/23/08 - Set protection on sched.properties
#    aghanti     03/11/08 - Ensuring that the script runs with xpg4 shell on
#                           SunOS
#    ndutko      10/17/07 - XbranchMerge ndutko_bug6502842 from st_emll_10.2.7
#    ndutko      10/17/07 - Handle scenarion where the OCM is specified in a
#                           CRONENTRY for SharedOH configs
#    jsutton     10/12/07 - XbranchMerge jsutton_bug-6491511 from
#                           st_emll_10.2.7
#    ndutko      10/11/07 - XbranchMerge ndutko_bug-6495218 from st_emll_10.2.7
#    ndutko      10/11/07 - Explicitly compare the va sharedHomesSupported
#                           against a value
#    jsutton     10/10/07 - XbranchMerge jsutton_bug-6475754 from
#                           st_emll_10.2.7
#    jsutton     10/10/07 - Make sure we have port-specific SED definitions
#    ndutko      10/09/07 - XbranchMerge ndutko_bug-6488100 from st_emll_10.2.7
#    ndutko      10/09/07 - Determine whether this version of install supports
#                           shared installs and use constants vs checking for a
#                           directory.
#    jsutton     10/03/07 - Create crontab file in postinstall, remove it in
#                           config
#    jsutton     09/10/07 - Handle crontab in shared home env
#    jsutton     09/14/07 - Move sched properties file to ccr_config_home
#    aghanti     09/09/07 - Bug 6372785 - fixing PE in main
#    aghanti     08/30/07 - Bug 6372785 -z argument should be in quotes
#    jsutton     08/29/07 - Handle upgrades from pre-shared-home OCM
#    jsutton     07/26/07 - Pick up existing shared OH work
#    ndutko      09/27/06 - XbranchMerge ndutko_bug-5567836 from st_emll_10.2.4
#    ndutko      09/27/06 - Correct the pattern match looking for a signature
#                           assignement (Bug 5567836)
#    pparida     08/15/06 - Do not add crontab and do not register if OCM is in 
#                           disconnected mode 
#    ndutko      04/04/06 - Disable the error handling around grep - bug 
#                           5128644 
#    cvaishna    03/29/06 - Fix for differnt behaviour of TRAP 
#    cvaishna    03/22/06 - Remove MKTEMP for Porting platforms 
#    cvaishna    03/22/06 - Bug Fix : 4998382 
#    kgupta      03/08/06 - Setting scheduler properties in correct order
#    ndutko      03/07/06 - Conditional location of UNAME 
#    ndutko      12/08/05 - Movement of emCCRCollector to lib 
#    ndutko      12/06/05 - Add blind acceptance of the license agreement 
#                           using the metalink id 
#    ndutko      12/05/05 - Add the configuration setting for the Scheduler to 
#                           perform collections 
#    amitkum     12/01/05 - Internally use emCCRCollector for registration. 
#    ndutko      11/30/05 - Conditionalize location of GREP for Solaris 
#    ndutko      11/16/05 - Fix problem where the config is not re-registering 
#                           on deployment of core 
#    ndutko      10/12/05 - == is not portable 
#    ndutko      10/06/05 - CCR Cutover 
#    ndutko      10/03/05 - allow the ability to disable registration in the 
#                           install path 
#    ndutko      09/29/05 - Register a trap handler such that if an error
#                           occurs, it is returned when the procedure 
#                           terminates.
#    ndutko      04/05/05 - Created
#

# Define the paths for /usr/bin. These are used in subsequent path
# specifications for native commands.
_usrBinDir=/usr/bin
_binDir=/bin

# Set the variables that map to explicit paths to prevent from trojan
# horse type attacks and allow for more consistent installation experience
# by eliminating the use of aliases.
AWK=${_binDir}/awk
CHMOD=${_binDir}/chmod
CRONTAB=${_usrBinDir}/crontab
DATE=${_binDir}/date
ECHO=${_binDir}/echo
MKTEMP=${_binDir}/mktemp
MV=${_binDir}/mv
RM=${_binDir}/rm
RMF="$RM -f"
SED=${_binDir}/sed
TOUCH=${_binDir}/touch

if [ -f ${_binDir}/egrep ]
then
    EGREP=${_binDir}/egrep
elif [ -f ${_usrBinDir}/egrep ]
then
    EGREP=${_usrBinDir}/egrep
fi

if [ -f ${_binDir}/cut ]
then
    CUT=${_binDir}/cut
elif [ -f ${_usrBinDir}/cut ]
then
    CUT=${_usrBinDir}/cut
fi

if [ -f ${_binDir}/uname ]
then
     UNAME=${_binDir}/uname
elif [ -f ${_usrBinDir}/uname ]
then
     UNAME=${_usrBinDir}/uname
fi

PLATFORM=`$UNAME | $CUT -f1`

if [ $PLATFORM = "OSF1" ]
then
  if [ -z "${BIN_SH}" ]
  then
    BIN_SH=xpg4
    export BIN_SH
    $0 "$@"
    exit $?
  fi
fi


#
# returns the exit code for the last command executed - successful or 
# not.
returnExitCodeOnExit()
{
    exit $?
}

setHostName()
{
  CCR_HOST_NAME=`hostname`

  export CCR_HOST_NAME
}

setOCMDirPaths()
{
  # Check whether its a shared OCM install
  if [ ${_sharedHomesSupported} -eq 0 ]
  then
    if [ -z "${ORACLE_CONFIG_HOME}" ]
    then
      CCR_CONFIG_HOME="${CCR_HOME}/hosts/${CCR_HOST_NAME}"
    elif [ -d ${ORACLE_CONFIG_HOME} ]
    then
      # strip any trailing dirsep
      ORACLE_CONFIG_HOME=`$ECHO ${ORACLE_CONFIG_HOME} | $SED -e 's/\/*$//'`
      CCR_CONFIG_HOME="${ORACLE_CONFIG_HOME}/ccr"
    else
      $ECHO '$ORACLE_CONFIG_HOME should point to the directory containing OCM configurations for the host.'
      exit $ERR_PREREQ_FAILURE
    fi
  else
    CCR_CONFIG_HOME=${CCR_HOME}
  fi

  export CCR_CONFIG_HOME
}

_trapFunc=returnExitCodeOnExit

if [ "${PLATFORM}" = "SunOS" ]; then
    _trapSig=0
elif [ "${PLATFORM}" = "Linux" ]; then
    _trapSig=EXIT
else
    _trapSig=ERR
fi


# Save the original shell flags
_originalShellFlags=$-

# Register a function to return the command execution on exit.
set -e
trap ${_trapFunc} ${_trapSig}

addCrontabEntry()
{
    if [ -f ${_crontabEntry} ]
    then
        $ECHO Adding record to temporary crontab entry
        $EGREP -v "#" ${_crontabEntry} >> ${_tmpCrontab}
        $ECHO Updating crontab file
        $CRONTAB ${_tmpCrontab}
        $ECHO Crontab updated
        $RMF ${_crontabEntry}
    fi
}

scheduleCollections()
{
    if [ -z "${CCR_DISABLE_CRON_ENTRY}" ]
    then
        $ECHO Scheduling collections
        #
        # Turn off command exit's terminating the shell temporarily, as 
        # this is a handled condition
        # Disable the shell exit on error
        set +e
        if [ "${PLATFORM}" = "SunOS" -o "${PLATFORM}" = "Linux" ]
        then
            trap 0
        else
            trap '' ERR
        fi

        #
        # Check to see if a crontab file already exists.
        $CRONTAB -l > /dev/null 2>&1
        if [ $? = 0 ]
        then
            # Reset original shell flags only if there were any
            if [ ! -z "${_originalShellFlags}" ]
            then
                set ${_originalShellFlags}
            fi
            # Re-enable traps
            set -e
            trap ${_trapFunc} ${_trapSig}


            $ECHO Crontab exists
            # One exists, so create the temporary file 
            $CRONTAB -l > ${_tmpCrontab}
    
            # If there was no error writing the file out (out of quota, permissions,
            # disk full) check to see if there was an entry for this path already.
            $ECHO Able to create tmpCrontab ${_tmpCrontab}
            # The quoted string is needed to make certain that the path
            # that we match is absolute and not some substring.

            #disable Exit on Error
            set +e
            if [ "${PLATFORM}" = "SunOS" -o "${PLATFORM}" = "Linux" ]
            then
                trap 0
            else
                trap '' ERR
            fi
            

            _entryFound=`$EGREP -c "0,15,30,45 \* \* \* \* (${JAVA_CRONPATH})?${G_ORACLE_CONFIG_HOME_VAR}${CCR_HOME}/bin/emCCR " ${_tmpCrontab}`
            _betaEntryFound=`$EGREP -c "^[0-9]+ [0-9]+ \* \* \*.* ${CCR_HOME}/bin/emCCR " ${_tmpCrontab}`

            # Reset original shell flags only if there were any
            if [ ! -z "${_originalShellFlags}" ]
            then
                set ${_originalShellFlags}
            fi
            # Re-enable traps
            set -e
            trap ${_trapFunc} ${_trapSig}

            # if the beta entry was found or there was no crontab entry
            if [ ${_entryFound} -eq 0 -o \
                 ${_betaEntryFound} -ne 0 ]
            then
                # If the beta entry was found remove it from the list of 
                # scheduled crontab jobs.
                if [ ${_betaEntryFound} -ne 0 ]
                then
                    #disable Exit on Error
                    set +e
                    if [ "${PLATFORM}" = "SunOS" -o "${PLATFORM}" = "Linux" ]
                    then
                        trap 0
                    else
                        trap '' ERR
                    fi
                    $EGREP -v "^[0-9]+ [0-9]+ \* \* \*.* ${CCR_HOME}/bin/emCCR " \
                              ${_tmpCrontab} > ${_tmpCrontab}.2
                    # Reset original shell flags only if there were any
                    if [ ! -z "${_originalShellFlags}" ]
                    then
                        set ${_originalShellFlags}
                    fi
                    # Re-enable traps
                    set -e
                    trap ${_trapFunc} ${_trapSig}

                    $MV ${_tmpCrontab}.2 ${_tmpCrontab}
                fi

                # Add the entry
                addCrontabEntry
            fi
        else
            # Reset original shell flags only if there were any
            if [ ! -z "${_originalShellFlags}" ]
            then
                set ${_originalShellFlags}
            fi
            # Re-enable traps
            set -e
            trap ${_trapFunc} ${_trapSig}

            $ECHO Crontab does not exist
            addCrontabEntry
        fi
 
        #
        # Clean up after the addition of the crontab entry.
        if [ -f ${_tmpCrontab} ]
        then
            $RM ${_tmpCrontab}
        fi
    fi
}

# This function looks at the scheduler properties and sets the scheduler 
# collection entry if the entry is not present already.
configureScheduler()
{
    if [ ! -f ${CCR_CONFIG_HOME}/config/default/sched.properties ]
    then
        _frequency=DAILY
        _hour=`$DATE +%H`
        _minute=`$DATE +%M`
        _dow=`$DATE +%u`
        $SED -e "s?%FREQUENCY%?$_frequency?g" \
             -e "s?%HOUR%?$_hour?g" \
             -e "s?%MINUTE%?$_minute?g" \
             -e "s?%DAY_OF_WEEK%?$_dow?g" \
                    ${CCR_HOME}/config/default/sched.properties.template > \
                    ${CCR_CONFIG_HOME}/config/default/sched.properties
                                                                                                       
        $CHMOD 600 ${CCR_CONFIG_HOME}/config/default/sched.properties
    fi
}

init()
{
    #
    # Determine if Shared Oracle Home support is available. This is determined
    # thru the new directory 'hosts' under the CCR_HOME.
    #
    if [ -d "${CCR_HOME}/hosts" ]; then
        _sharedHomesSupported=0
    else
        _sharedHomesSupported=1
    fi

    setHostName
    setOCMDirPaths

    # Declare a variable used conditionally if the config home is set and shared
    # home configuration.
    if [ ${_sharedHomesSupported} -eq 0 -a \
         ! -z "${ORACLE_CONFIG_HOME}" ]; then
        G_ORACLE_CONFIG_HOME_VAR="ORACLE_CONFIG_HOME=${ORACLE_CONFIG_HOME} "
    else
        G_ORACLE_CONFIG_HOME_VAR=""
    fi

    if [ ! -z "${JAVA_HOME_CCR}" ]
    then
        JAVA_CRONPATH="JAVA_HOME=${JAVA_HOME_CCR} "
    elif [ ! -z "${JAVA_HOME}" ]
        then
            JAVA_CRONPATH="JAVA_HOME=${JAVA_HOME} "
        else
            JAVA_CRONPATH=""
    fi

    if [ -z "${CCR_TEMP}" ]
    then
        CCR_TEMP=/tmp
    fi
    if [ "${PLATFORM}" = "Linux" ]; then 
        _tmpCrontab=`$MKTEMP ${CCR_TEMP}/crontab.XXXXXX`
    else
        _tmpCrontab=${CCR_TEMP}/crontab.$$
        $TOUCH ${_tmpCrontab}; 
    fi
}    

#
# Procedure to morph the ccr.properties to use https.
#
upgradeProperties()
{
    $ECHO "Upgrading properties"
    _propertiesFile=${CCR_HOME}/config/default/ccr.properties
    _tmpProperties=${_propertiesFile}-$$
    
    # 
    # If the file exists...
    if [ -f ${_propertiesFile} ]
    then
        if [ -f ${_tmpProperties} ]
        then
            $RMF ${_tmpProperties}
        fi
        
        # Check to make certain there is something to change.
        # Disable the shell exit on error
        set +e
        if [ "${PLATFORM}" = "SunOS" -o "${PLATFORM}" = "Linux" ]
        then
            trap 0
        else
            trap '' ERR
        fi

        _match=`$EGREP -c -e "liveLink.endpoint=.*//livelinktst.us.oracle.com" ${_propertiesFile}`
        # Reset original shell flags only if there were any
        if [ ! -z "${_originalShellFlags}" ]
        then
            set ${_originalShellFlags}
        fi
        # Re-enable traps
        set -e
        trap ${_trapFunc} ${_trapSig}

        if [ ${_match} > 0 ]
        then
            $SED -e "s?^liveLink.endpoint=.*?liveLink.endpoint=https://livelinkbeta.oracle.com?" \
                 ${_propertiesFile} > ${_tmpProperties}
                 
            # If the replacement was successful, backup and replace 
            # the original.
            $ECHO "Properties upgraded"
            $MV ${_propertiesFile} ${_propertiesFile}.bak
            $MV ${_tmpProperties} ${_propertiesFile}
        fi
        
        if [ -f {_tmpProperties} ]
        then
            $RMF ${_tmpProperties}
        fi
    fi              
}

#
# Register the installation if there is no security file with a key
registerInstallation()
{
    if [ -z "${CCR_INSTALL_CODEPATH}" ]
    then
        $ECHO "Registering installation"
        if [ ! -f ${CCR_CONFIG_HOME}/config/default/uplinkreg.bin ]
        then
            CCR_NOUPDATE=1
            export CCR_NOUPDATE
            ${CCR_HOME}/lib/emCCRCollector -silent register
            CCR_NOUPDATE=
            export CCR_NOUPDATE
        fi
    fi
}

# Accept the license agreement in the case of an upgrade where the 
# installation of core is an upgrade.
acceptLicense()
{
   if [ -z "${CCR_INSTALL_CODEPATH}" ]
   then
       if [ -f ${CCR_CONFIG_HOME}/config/collector.properties ]
       then
           # Disable the shell exit on error
           set +e
           if [ "${PLATFORM}" = "SunOS" -o "${PLATFORM}" = "Linux" ]
           then
               trap 0
           else
               trap '' ERR
           fi
           _Accepted=`$EGREP '^ccr.agreement_signer=' ${CCR_CONFIG_HOME}/config/collector.properties | \
                      $SED -e 's/^ccr.agreement=\s*//' -e 's/\s//g'`
           _metalinkId=`$EGREP '^ccr.metalink_id=' $CCR_HOME/config/default/ccr.properties | $SED -e "s?^ccr.metalink_id=??"`

           # Reset original shell flags only if there were any
           if [ ! -z "${_originalShellFlags}" ]
           then
               set ${_originalShellFlags}
           fi
           # Re-enable traps
           set -e
           trap ${_trapFunc} ${_trapSig}

           if [ -z "${_Accepted}" ]
           then
               # Disable the shell exit on error
               set +e
               if [ "${PLATFORM}" = "SunOS" -o "${PLATFORM}" = "Linux" ]
               then
                   trap 0
               else
                   trap '' ERR
               fi
               $EGREP -v '^ccr.agreement_signer=' ${CCR_CONFIG_HOME}/config/collector.properties > \
                     ${CCR_TEMP}/collector.properties.$$

               # Reset original shell flags only if there were any
               if [ ! -z "${_originalShellFlags}" ]
               then
                   set ${_originalShellFlags}
               fi
               # Re-enable traps
               set -e
               trap ${_trapFunc} ${_trapSig}

               $ECHO "ccr.agreement_signer=${_metalinkId}" >> \
                     ${CCR_TEMP}/collector.properties.$$
               $MV -f ${CCR_TEMP}/collector.properties.$$ \
                      ${CCR_CONFIG_HOME}/config/collector.properties
           fi
       fi
    fi
}

#warnIfCronDisallowed
#checks if user have crontab access if not it will set
#CCR_DISABLE_CRON_ENTRY which will prevent tampering of
#crontab further.
#
warnIfCronDisallowed()
{
    if [ -z "$CCR_DISABLE_CRON_ENTRY" ]; then
        # Turn off command exit's terminating the shell temporarily, as
        # this is a handled condition
        set +e
        if [ "${PLATFORM}" = "SunOS" -o "${PLATFORM}" = "Linux" ]; then
            trap 0
        else
            trap '' ERR
        fi

        $RMF ${_tmpCrontab}
        $CRONTAB -l > ${_tmpCrontab} 2>${_tmpCrontab}.err
        if [ $? -ne 0 ]; then
            #
            # a crontab -l with no crontab entry returns a non-zero status.
            # In this case, now try to re-add an empty line and see if a
            # error still occurs.

            $ECHO "#OCMComment" > ${_tmpCrontab}_comment
            $CRONTAB ${_tmpCrontab}_comment > /dev/null 2>&1
            if [ $? -ne 0 ]; then
                $ECHO "This account does not have permission to use cron. The Oracle Configuration Manager"
                $ECHO "will not be automatically started on system reboot. Add the command:"
                $ECHO ""
                $ECHO "    ${ORACLE_HOME}/ccr/bin/emCCR start -silent"
                $ECHO ""
                $ECHO "to your system startup procedures."
                CCR_DISABLE_CRON_ENTRY=true
                export CCR_DISABLE_CRON_ENTRY
            else
                $CRONTAB ${_tmpCrontab}
            fi
            $RMF ${_tmpCrontab}_comment
        fi
        $RMF ${_tmpCrontab}
        $RMF ${_tmpCrontab}.err
        #
        # Reset original shell flags only if there were any
        if [ ! -z "${_originalShellFlags}" ]
        then
            set ${_originalShellFlags}
        fi
        # Re-enable traps
        set -e
        trap ${_trapFunc} ${_trapSig}
    fi
}

#
# Main entry for the config.
#

init
_crontabEntry=${CCR_CONFIG_HOME}/config/default/crontab.entry
#upgradeProperties
acceptLicense
configureScheduler
_disconn=`$EGREP '^ccr.disconnected=' $CCR_CONFIG_HOME/config/collector.properties | $SED -e "s?^ccr.disconnected=??"`
# Schedule collections (which also adds crontab) and register only in connected mode
if [ ! "$_disconn" = "true" ]; then
    #check crontab has access or not before calling scheduleCollections
    warnIfCronDisallowed
    scheduleCollections
    registerInstallation
fi
$RMF ${_crontabEntry}
exit 0
