// ident "@(#)ConsoleUtility.java	1.8 99/07/22 SMI"

package com.sun.smartcard.mgt.util;

import java.util.*;
import java.net.*;
import java.io.*;
import java.awt.*;
import java.awt.image.*;
import java.rmi.server.*;
import java.security.*;
import javax.swing.*;

/**
 * This class coupled with the ResourceManager class will provide a great deal
 * of convenience to Tools.
 *
 * This class contains functions for loading image icons and their corresponding
 * disabled versions.
 *
 * For a full set of convenience functions regarding resources such as resource bundles,
 * help sets, help files, fonts and colors, see the ResourceManager class.
 *
 * @version 1.19, 05/17/99
 */
public class ConsoleUtility {

	protected static ImageFilter 	grayFilter 		= new VGrayFilter();
	protected static ImageIcon 		warningIcon 	= null;
	protected static boolean 		checkedJVM		= false;
	protected static boolean 		java2			= false;

	/**
	 * A method to load an ImageIcon given a name of an image and
	 * a Class which is from the same codebase as the image. Basically,
	 * this is a utility method for dynamically loaded classes to load
	 * their images, without providing the common code themselves, and
	 * takes a step toward common code re-use. NOTE: Do not use this
	 * method inline in things like renderers and such for repeated
	 * loads of the same image. This utility method retrieves an image
	 * from the class's resource stream. If you require the same image
	 * multiple times, load it once with this method and keep the reference.
	 * Calling this method every time will repeatedly load the image from
	 * the resource stream, not from cache.
	 *
	 * Note: the imageFile name should be the path to the image in the
	 * jar which contains the image, such as a jar with the following
	 * contents listing:
	 * /UsrMgr/client/images/image.gif
	 * would send the parameters "/UsrMgr/client/images/image.gif"
	 * and this.getClass(). Also, do NOT place images at the root of
	 * the jar archive as naming conflicts may arise. You should follow
	 * the java package naming conventions for resources like images as
	 * well as java code. ie, place images for class com.sun.admin.usermgr
	 * in a directory called /com/sun/admin/usrmgr/images.
	 *
	 * @param	imageFile	The String name of the image file
	 * @param	toolClass	A Class object which has the same codebase as
	 *						the image to be loaded.
	 *
	 * @return	The loaded ImageIcon, null if the image could
	 * 			not be loaded.
	 */
	public static ImageIcon loadImageIcon(String imageFile, Class toolClass){
		if(!checkedJVM){
			try {
				if(System.getProperty("java.version").indexOf("1.2") != -1)
    				java2 = true;
               	else
                   	java2 = false;
         	} catch (Exception e){
       			e.printStackTrace();
         		java2 = false;
       		}
           	checkedJVM = true;
       	}
    	ImageIcon icon = null;
     	try{
        	if(java2)
         		icon = Java2ImageLoader.loadImageIcon(imageFile, toolClass);
      		else {
             	InputStream is = toolClass.getResourceAsStream(imageFile);
            	if(is != null){
                  	ByteArrayOutputStream baos=new ByteArrayOutputStream();
                  	int c;
         			while((c=is.read())>=0){
                  		baos.write(c);
                  	}
               		icon = new ImageIcon(Toolkit.getDefaultToolkit().createImage(baos.toByteArray()));
               	} else {
                 	icon = new ImageIcon(Toolkit.getDefaultToolkit().getImage(imageFile));
           		}
        	};
   		} catch (Exception e){
         	System.err.println(e);
		}
	 	return icon;
	}

	/**
	 * Convenience routine for Tools.
	 * Return the contents of a text file given the parameters.
	 * This method is for text file resources which are not necessarily localized. The ResourceManager class
	 * should be used for loading some localized text files such as HTML help files. See that class for
	 * more information. The contents of this file may be localized, and properly loaded using the encoding
	 * parameter, but this method does not automatically apply a search heuristic for a localized resource
	 * as the ResourceManager class does.
	 *
	 * Text file resources can be registered as add-on libraries or included in the
	 * Tool's original jar (along with the required default resource bundle).
	 * For example: if com.sun.UserTool registered a jar with text file resources,
	 * it should follow the namespace requirement in the jar directory structure
	 * such as com/sun/UserTool/<...>/file.html. The parameters to the method
	 * below would then be:
	 * loadTextFile("<...>/file.html", <tool class>, encoding);
	 *
	 * The method will form a locator for the file using the tool class package directory structure
	 * and append the relative path. The file contents will be decoded using the supplied file encoding
	 * or if null, using the local default file encoding.
	 *
	 * @param relativePath	This String should be the path to the file relative to the directory structure of
	 *						the Java class package passed as the Class parameter. It will be appended
	 *						to the Java package name directory.
	 *
	 * @param toolClass		The class to use to determine the apropriate ClassLoader to load the resource,
	 *						as well as the Java package directory structure to locate the file.
	 *
	 * @param encoding		The encoding value to use to recreate the text from its byte[] representation.
	 *						If null, the system's default character encoding will be used.
	 */
	public static String loadTextFile(String relativePath, Class toolClass, String encoding){
		if(!checkedJVM){
			try {
				if(System.getProperty("java.version").indexOf("1.2") != -1)
					java2 = true;
				else
					java2 = false;
			} catch (Exception e){
				e.printStackTrace();
				java2 = false;
			}
			checkedJVM = true;
		}
		String fileContents = null;
		try{
			byte[] rawData = loadRawFile(relativePath, toolClass);
			if(rawData != null){
				if(encoding != null)
					fileContents = new String(rawData, encoding);
				else
					fileContents = new String(rawData);
			}
		} catch (Exception e){
			System.err.println(e);
		}
		return fileContents;
	}

	/**
	 * Convenience routine for Tools.
	 * Return the contents of a binary file given the parameters.
	 * This method is for binary file resources which are not necessarily localized. The ResourceManager class
	 * should be used for loading some localized binary files. See that class for
	 * more information. This method does not automatically apply a search heuristic for a localized resource
	 * as the ResourceManager class does.
	 *
	 * Binary file resources can be registered as add-on libraries or included in the
	 * Tool's original jar (along with the required default resource bundle).
	 * For example: if com.sun.UserTool registered a jar with binary file resources,
	 * it should follow the namespace requirement in the jar directory structure
	 * such as com/sun/UserTool/<...>/file.au. The parameters to the method
	 * below would then be:
	 * loadRawFile("<...>/file.au", <tool class>);
	 *
	 * The method will form a locator for the file using the tool class package directory structure
	 * and append the relative path.
	 *
	 * @param relativePath	This String should be the path to the file relative to the directory structure of
	 *						the Java class package passed as the Class parameter. It will be appended
	 *						to the Java package name directory.
	 *
	 * @param toolClass		The class to use to determine the apropriate ClassLoader to load the resource,
	 *						as well as the Java package directory structure to locate the file.
	 */
	public static byte[] loadRawFile(String relativePath, Class toolClass){
		if(!checkedJVM){
			try {
				if(System.getProperty("java.version").indexOf("1.2") != -1)
					java2 = true;
				else
					java2 = false;
			} catch (Exception e){
				e.printStackTrace();
				java2 = false;
			}
			checkedJVM = true;
		}
		byte[] rawData = null;
		try{
			if(java2) {
				rawData = Java2FileLoader.loadFile(relativePath, toolClass);
			} else {
				InputStream is = toolClass.getResourceAsStream(relativePath);
				if(is != null){
					ByteArrayOutputStream baos=new ByteArrayOutputStream();
					int ct = 0;
					while((ct=is.read())>=0){
						baos.write(ct);
					}
            		rawData = baos.toByteArray();
				}
			}
		} catch (Exception e){
			System.err.println(e);
		}
		return rawData;
	}

	/**
	 * A method to return the disabled icon of an ImageIcon using a predefined
	 * image filter. The image filter is a gray filter which gives the image a
	 * gray-ed out effect. This provides for standard looking disabled icons
	 * rather than components providing their own disabled icons.
	 *
	 * @param	icon	The ImageIcon to filter
	 * @return	The disabled ImageIcon, null if the icon could
	 * 			not be created.
	 */
	public static ImageIcon getDisabledIcon(ImageIcon icon){
		ImageIcon dIcon = null;
		if(icon != null){
			try{
				dIcon = new ImageIcon(
					Toolkit.getDefaultToolkit().createImage(
						new FilteredImageSource(icon.getImage().getSource(), grayFilter)
					)
				);
			} catch(Exception e){;}
		}
		return dIcon;
	}

	/**
	 * A method to return the parent frame of any AWT component.
	 * Handy for creating modal dialogs which require the parent frame
	 * for creation.
	 *
	 * @param	c	The Component to find the parent frame for.
	 * @return	The Components parent Frame, null if an error occurs
	 */
	public static Frame getFrame(Component c){
		try{
			while(c != null && !(c instanceof Frame))
        		c = c.getParent();
     		return (Frame)c;
		} catch (Exception e){
			return null;
		}
	}

}

/**
 * A Utility ImageFilter class to provide a common method
 * of creating disabled icons given the orignal icons. Provides
 * for a common look for disabled icons across all components.
 *
 * @version 1.19, 05/17/99
 */
class VGrayFilter extends RGBImageFilter {
	public VGrayFilter() {
		canFilterIndexColorModel = true;
	}

	public int filterRGB(int x, int y, int rgb){
		int	a = rgb & 0xff000000;
		int r = (((rgb & 0xff0000) + 0x1800000)/3) & 0xff0000;
		int g = (((rgb & 0x00ff00) + 0x018000)/3) & 0x00ff00;
		int b = (((rgb & 0x0000ff) + 0x000180)/3) & 0x0000ff;
		return a | r | g | b;
	}
}

